#!/usr/bin/env python3
import configparser
import glob
import os
import os.path
import shlex
import shutil
import string
import subprocess
import textwrap

GIT_REPO_DIR = "/usr/local/src/"
GOPHER_DIR = "/var/gopher/software/"

def get_cmd_output(*args, **kwargs):
    """
    A simple wrapper around subprocess.checkoutput to handle UTF-8
    decoding and newline removal.
    """

    output = subprocess.check_output(*args, **kwargs)
    return output.decode("UTF-8").strip()

def main():
    """
    Visit each subdirectory of GIT_REPO_DIR which:
    i) contains a .git/git-daemon-export-ok file
    ii) does not contain a .git/gitpher-hide file
    and create a corresponding subdirectory in GOPHER_DIR, plus create
    a master gophemap in GOPHER_DIR.
    """

    # Process individual repositories
    handled_repos = []
    for repo in os.listdir(GIT_REPO_DIR):
        if (os.path.exists(os.path.join(GIT_REPO_DIR, repo, ".git", "git-daemon-export-ok")) and
            not os.path.exists(os.path.join(GIT_REPO_DIR, repo, ".git", "gitpher-hide"))):
            print("Handling %s..." % repo)
            handled_repos.append(handle_repo(repo))
        else:
            print("Skipping %s..." % repo)

    # Create master repository list
    handled_repos.sort(key=lambda x: x[1])
    generate_master_gophermap(handled_repos)

def handle_repo(repository):
    """
    Do everything for the single git repository located at
    `repository`.  Return a tuple containing the name of the
    repository's gopher directory, plus a human-readable label
    for use in the master gophermap.
    """

    # Create a gopher directory
    repo_dir = os.path.join(GIT_REPO_DIR, repository)
    gopher_dir = os.path.join(GOPHER_DIR, repository)
    if not os.path.exists(gopher_dir):
        os.makedirs(gopher_dir)

    # Create symbolic links for human-readable text files which are
    # likely to be of interest.
    textfiles = glob.glob(os.path.join(repo_dir, "README*"))
    textfiles.extend(glob.glob(os.path.join(repo_dir, "INSTALL*")))
    for textfile in textfiles:
        linkfile = os.path.join(gopher_dir, os.path.basename(textfile))
        if not os.path.exists(linkfile):
            os.symlink(textfile, linkfile)

    # Create recent commit file and zip archive
    subprocess.call("git -C %s log -5 > %s" % (repo_dir,
        os.path.join(gopher_dir, "recent-commits.txt")), shell=True)
    subprocess.call(shlex.split("git -C %s archive --prefix %s/ --output %s master" % (repo_dir,
        repository, os.path.join(gopher_dir, "%s-latest.zip" % repository))))

    # Attempt to read metadata
    meta = get_metadata(repo_dir)

    # Create the gophermap
    fp = open(os.path.join(gopher_dir, "gophermap"), "w")
    fp.write(generate_gophermap(repo_dir, meta, (os.path.basename(t) for t in textfiles)))
    fp.close()

    # Return main gophermap entry
    menu_line = meta["name"]
    if "short_descr" in meta:
        menu_line += " - " + meta["short_descr"]
    return (repository, menu_line)

def get_metadata(repository):
    """
    Read a INI-style .gitpher file from a git repository, and
    return a dictionary of its values.  If the file doesn't
    exist, insert some default values for the most essential
    variables.
    """

    meta = configparser.ConfigParser()
    meta.read(os.path.join(repository,".gitpher"))
    if "gitpher" in meta:
        meta = meta["gitpher"]
    else:
        meta = {}
    if "name" not in meta:
        meta["name"] = os.path.basename(repository)
    return meta

def generate_gophermap(repository, metadata, textfiles):
    """
    Returns a string containing the gophermap for the git
    repository at `repo`.
    """

    # Wrangle some strings
    repo_name = os.path.basename(repository)
    zipfilename = "%s-latest.zip" % repo_name

    # Get some git statistics
    commit_count = get_cmd_output(shlex.split("git -C %s rev-list --count master" % repository))
    first_commit = get_cmd_output(
        """git -C %s log --reverse --pretty=format:"%%ad" | head -n 1""" % repository,
        shell=True)
    last_commit = get_cmd_output(
        """git -C %s log --pretty=format:"%%ad" | head -n 1""" % repository,
        shell=True)

    # Gophermap template
    gophermap = string.Template(""" === $name ===

$description

 === Git repository details ===

Commits to master branch: $commit_count
First commit: $first_commit
Latest commit: $last_commit
0View recent commits\trecent-commits.txt

Clone the repo from: git://circumlunar.space/$repo_name
Send patches to: $patch_email
9${zipfilename}\t${zipfilename}
""")

    # Substitute variables from above into template
    variables = metadata
    for k,v in locals().items():
        variables[k] = str(v)
    gophermap = gophermap.safe_substitute(variables,
        description = "\n".join(textwrap.wrap(metadata.get("description",""))))

    # Add links to human readable files, if any
    if textfiles:
        gophermap += """
 === Selected files ===

The following human-readable files from the repository may be of
interest:

"""
        for tf in textfiles:
            gophermap += "0%s\t%s\n" % (tf,tf)

    # Done!
    return gophermap

def generate_master_gophermap(handled_repos):
    """
    Create the gophermap file in GOPHER_DIR, containing menu items
    for each of the repos for which gopher diretories were generated.
    """

    header_file = os.path.join(GOPHER_DIR, ".gitpher_header")
    gophermap_file = os.path.join(GOPHER_DIR, "gophermap")
    if os.path.exists(header_file):
        shutil.copy(header_file, gophermap_file)
        fp = open(gophermap_file, "a")
    else:
        fp = open(gophermap_file, "w")
    for selector, name in handled_repos:
        fp.write("1%s\t%s\n" % (name, selector))
    fp.close()

if __name__ == "__main__":
    main()
