/****************************************************************************

Copyright (c) 2014 Robert Cunnings, NW8L

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

****************************************************************************/

/*

server.js

Entry point for gopher service. Parses config file and launches the
service, opens and closes log if logging is enabled. Log file location
assumed to be in same directory, with name 'node-gopher.config'.

*/

var fs = require('fs');
var path = require('path');
var gopher = require('./gopher');
var logger = require('./logger');
var requestHandlers = require("./requestHandlers");

// request handlers
var handlers = {};
handlers["/"] = requestHandlers.file;  // ensure root path is mapped
handlers["/URL:"] = requestHandlers.url;  // ensure URL: selector is mapped

// cgi virtual path mappings (user configurable)
var basePaths = {};
basePaths["/"] = '/';
basePaths["/URL:"] = '/';  // dummy path for URL

// file extension/cgi script mappings (user configurable)
var scripts = {};
scripts[".py"] = 'python ';
scripts[".pl"] = 'perl ';

// basic settings (user configurable)
var settings = {};
settings["host"] = "localhost";
settings["port"] = "70";
settings["log-filename"] = "./node-gopher.log";
settings["log-enable"] = "1";
settings["log-rot-interval"] = parseInt(24);

// file extension/gopher-type mappings
var fileTypes = {};
fileTypes[".txt" ] = '0';
fileTypes[".log" ] = '0';
fileTypes[".gif" ] = 'g';
fileTypes[".jpg" ] = 'I';
fileTypes[".jpeg"] = 'I';
fileTypes[".bmp" ] = 'I';
fileTypes[".png" ] = 'I';
fileTypes[".ico" ] = 'I';
fileTypes[".xpm" ] = 'I';
fileTypes[".exe" ] = '9';
fileTypes[".dll" ] = '9';
fileTypes[".zip" ] = '9';
fileTypes[".gz"  ] = '9';
fileTypes[".tgz" ] = '9';
fileTypes[".jar" ] = '9';
fileTypes[".msi" ] = '9';
fileTypes[".dmg" ] = '9';
fileTypes[".iso" ] = '9';
fileTypes[".obj" ] = '9';
fileTypes[".o"   ] = '9';
fileTypes[".doc" ] = '9';
fileTypes[".ppt" ] = '9';
fileTypes[".xls" ] = '9';
fileTypes[".mov" ] = ';';
fileTypes[".mpg" ] = ';';
fileTypes[".mpe" ] = ';';
fileTypes[".mpeg"] = ';';
fileTypes[".pdf" ] = 'P';
fileTypes[".mp2" ] = 's';
fileTypes[".mp3" ] = 's';
fileTypes[".wav" ] = 's';
fileTypes[".aac" ] = 's';
fileTypes[".htm" ] = 'h';
fileTypes[".html"] = 'h';

function readSettings(filename, cb) {
    var section = -1;
    var settingsData = ""
    var settingsDataLines = [];
    function updateMaps(element, index, array) {
        var name = "";
        var value = "";
        var tokens = [];
        if (element.length == 0)
            return;
        if (element.search(/\[server\]/) != -1) {
            section = 0;
            return;
        }
        else if (element.search(/\[item-types\]/) != -1) {
            section = 1;
            return;
        }
        else if (element.search(/\[alias-map\]/) != -1) {
            section = 2;
            return;
        }
        else if (element.search(/\[cgi-alias-map\]/) != -1) {
            section = 3;
            return;
        }
        else if (element.search(/\[cgi-script-map\]/) != -1) {
            section = 4;
            return;
        }
        else if (element.search(/\[url-map\]/) != -1) {
            section = 5;
            return;
        }
        if (section == -1)
            return;
        tokens = element.split(/=/);
        if (tokens[0].charAt(0) == '#' || tokens[0].charAt(0) == ';') {
            return;
        }
        if (tokens.length < 2)
            return;
        // name/value pair
        name = tokens[0].trim();
        value = tokens[1].trim();
        switch (section) {
        case 0:
            if (name.length > 0 && value.length > 0) {
                settings[name] = value;
            }
            console.log("Server: [server] " + name + " = " + settings[name]);
            break;
        case 1:
            if (name.length > 0 && value.length > 0) {
                fileTypes[name] = value;
            }
            console.log("Server: [item-types] " + name + " = " + fileTypes[name]);
            break;
        case 2:
            if (name.length > 0 && value.length > 0) {
                basePaths[name] = value;
                handlers[name] = requestHandlers.file;
            }
            console.log("Server: [alias-map] " + name + " = " + basePaths[name]);
            break;
        case 3:
            if (name.length > 0 && value.length > 0) {
                basePaths[name] = value;
                handlers[name] = requestHandlers.cgi;
            }
            console.log("Server: [cgi-alias-map] " + name + " = " + basePaths[name]);
            break;
        case 4:
            if (name.length > 0) {
                scripts[name] = value;
            }
            console.log("Server: [cgi-script-map] " + name + " = " + scripts[name]);
            break;
        case 5:
            if (name.length > 0) {
                basePaths[name] = value;
                handlers[name] = requestHandlers.url;
            }
            console.log("Server: [url-map] " + name + " = " + scripts[name]);
            break;
        }
    }
    var rs = fs.createReadStream(path.normalize(filename));
    rs.on('data', function(data) {
            settingsData = settingsData + data.toString();
    });
    rs.on('end', function() {
        settingsDataLines = settingsData.split(/\n/);
        settingsDataLines.forEach(updateMaps);
        cb(null);
    });
    rs.on('error', function() {
        cb("Server: error reading configuration file.");
    });
}

var logStream = null;

function startUp(files) {
    readSettings(files, function (err) {
        if (err) {
            console.log(err);
        } else {
            // settings loaded, intialize and start gopher server
            var host = settings["host"];
            var port = parseInt(settings["port"]);
            var logFilename = settings["log-filename"];
            var logEnable = settings["log-enable"];
            var logRotationInterval = settings["log-rot-interval"];
            if (parseInt(logEnable) == 1)
                logger.initLog(logFilename, logRotationInterval, false);
            gopher.start(handlers, basePaths, fileTypes, scripts, host, port);
        }
    });
}

process.on('SIGINT', function() {
    console.log( "\nServer: shutting down.\n");
    logger.closeLog();
    process.exit();
});

startUp(path.resolve(__dirname, "./node-gopher.config"));

