/****************************************************************************

Copyright (c) 2014 Robert Cunnings, NW8L

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

****************************************************************************/

/*

requestHandlers.js

Contains handlers for various gopher request types as well as error handlers.
Includes support for binary file, text file and directory selectors as well
as selectors for CGI scripts and HTTP URL selectors.

*/

var fs = require('fs');
var path = require('path');
var util = require('util');
var exec = require("child_process").exec;

function handleFile(selector, filePath, socket, fileTypes) {
    var type = "0";
    var filename = ""
    var extension = ""
    var loc = selector.lastIndexOf('/');
    if (loc != -1)
        filename = selector.substring(loc);
    loc = filename.lastIndexOf('.');
    if (loc != -1)
        extension = filename.substring(loc).toLowerCase();
    //console.log("File handler: extension is [ " + extension + " ]");
    if (typeof fileTypes[extension] === 'string')
        type = fileTypes[extension];
    //console.log("File handler: type is [ " + type + " ]");
    var rs = fs.createReadStream(filePath);
    rs.on('data', function(data) {
        socket.write(data);
    });
    rs.on('end', function() {
        if (type == "0")
            socket.end('\r\n.\r\n');
        else
            socket.end();
    });
    rs.on('error', function() {
        handleInternalServerError(
            selector, socket, "File handler: error reading file.");
    });
}

function handleDirectory(selector, filePath, socket, fileTypes, host, port) {
    fs.readdir(filePath, function(err, files) {
        var gopherMapLinesIn = [];
        var gopherMapLinesOut = [];
        var excludedFiles = [];
        var state = 0;
        var stopGopherMapProc = 0;
        if(err) {
            handleDirNotAvailableError(selector, socket)
        }
        else {
            function processGopherLines(element, index, array) {
                if (stopGopherMapProc == 1)
                    return;
                var itemType = "i";
                var name = "";
                var fileSelector = "";
                var fileHost = host;
                var filePort = port;
                var tokens = [];
                var temp = "";
                tokens = element.split(/\t/);
                if (tokens.length == 0)
                    return;
                // get item type
                itemType = tokens[0].charAt(0);
                if (itemType == "#") { // line commented out
                    return;
                }
                else if (itemType == "*") { // stop and list directory
                    state = 1;
                    stopGopherMapProc = 1;
                    return;
                }
                else if (itemType == ".") { // stop and don't list directory
                    state = 2;
                    stopGopherMapProc = 1;
                    return;
                }
                else if (itemType == "-") { // exclude files from listing
                    temp = tokens[0].substr(1).trim();
                    if (temp.length > 0) {
                        // convert wildcard pattern to regex
                        temp = temp.replace(/\./g, "\\.");
                        temp = temp.replace(/\*/g, ".*");
                        temp = temp.replace(/\?/g, ".");
                        temp = temp.replace(/\(/g, "\\(");
                        temp = temp.replace(/\)/g, "\\)");
                        temp = temp.replace(/\{/g, "\\{");
                        temp = temp.replace(/\}/g, "\\}");
                        temp = temp.replace(/\[/g, "\\[");
                        temp = temp.replace(/\]/g, "\\]");
                        temp = temp.replace(/\+/g, "\\+");
                        temp = temp.replace(/\-/g, "\\-");
                        temp = temp.replace(/\$/g, "\\$");
                        temp = temp.replace(/\^/g, "\\^");
                        temp = "^" + temp + "$";
                        excludedFiles.push(new RegExp(temp));
                    }
                    //console.log("Gophermap: exclude [ " + temp + " ]");
                    return;
                }
                else if (itemType == "D") { // print directory path (selector)
                    temp = util.format("iDirectory: %s/\t\t\t\r\n", selector);
                    temp = temp.replace("//", "/");
                    gopherMapLinesOut.push(temp);
                    return;
                }
                else {
                    // get name
                    name = tokens[0].substr(1);
                }
                if (tokens.length > 1) {
                    // get selector
                    temp = tokens[1].trim();
                    if (temp.length > 0) {
                        // replace '\' with '/'
                        temp = temp.replace(/\\/, "/");
                    }
                    if (temp.charAt(0) == '/') {
                        // absolute path
                        fileSelector = temp;
                        if (fileSelector.search(/\.\.|%/) != -1) {
                            return;
                        }
                    } else {
                        // relative path
                        var pos = temp.search(/URL:/);
                        if (pos == 0) {
                            fileSelector = temp;
                        } else {
                            fileSelector = selector + "/" + temp;
                            fileSelector = fileSelector.replace("//", "/");
                            if (fileSelector.search(/\.\.|%/) != -1) {
                                return;
                            }
                        }
                    }
                    if (tokens.length > 2) {
                        // get host name
                        temp = tokens[2].trim();
                        if (temp.length > 0)
                            fileHost = temp;
                        if (tokens.length > 3) {
                            // get port number
                            temp = tokens[3].trim();
                            if (temp.length > 0)
                                filePort = temp;
                        }
                    }
                }
                temp = util.format("%s%s\t%s\t%s\t%s\r\n", itemType, name, fileSelector, fileHost, filePort);
                gopherMapLinesOut.push(temp);
                //console.log("Gophermap: gophermap line [ " + temp + " ]");
            }
            function printGopherMapLine(element, index, array) {
                    socket.write(element);
            }
            function checkIsExcluded(name) {
                for (var i = 0; i < excludedFiles.length; i++) {
                    if (excludedFiles[i].test(name)) {
                        //console.log("Directory handler: excluded [ " + name + " ]");
                        return true;
                    }
                }
                return false;
            }
            function printFileLine(element) {
                if (element != "gophermap") {
                    var type = "0";
                    var extension = "";
                    var loc = element.lastIndexOf('.');
                    if (loc != -1)
                        extension = element.substring(loc).toLowerCase();
                    //console.log("Directory handler: extension is [ " + extension + " ]");
                    if (typeof fileTypes[extension] === 'string')
                        type = fileTypes[extension];
                    //console.log("Directory handler: file type is [ " + type + " ]");
                    var tempSelector = selector + "/" + element;
                    tempSelector = tempSelector.replace("//", "/");
                    if (tempSelector.search(/\.\.|%/) != -1) {
                        return;
                    }
                    socket.write(util.format("%s%s\t%s\t%s\t%d\r\n", type, element, tempSelector, host, port));
                }
            }
            function printDirLine(element) {
                    var tempSelector = selector + "/" + element;
                    tempSelector = tempSelector.replace("//", "/");
                    if (tempSelector.search(/\.\.|%/) != -1) {
                        return;
                    }
                    socket.write(util.format("1%s\t%s\t%s\t%d\r\n", element, tempSelector, host, port));
            }
            function printDirTitle() {
                var temp = util.format("iDirectory: %s/\t\t\t\r\n\r\n", selector);
                temp = temp.replace("//", "/");
                socket.write(temp);
                if (selector != "/") {
                    temp = selector;
                    var loc = selector.lastIndexOf('/');
                    if (loc = selector.length - 1) {
                        temp = temp.substring(0, loc);
                        loc = temp.lastIndexOf('/');
                    }
                    temp = temp.substring(0, loc);
                    socket.write(util.format("1..\t%s\t%s\t%d\r\n", temp, host, port));
                }
            }
            function processListLines(element, index, array) {
                var tempPath = path.join(filePath + "/" + element)
                try {
                    var stats = fs.lstatSync(tempPath);
                    if (stats.isDirectory()) {
                        if (!checkIsExcluded(element))
                            printDirLine(element);
                    }
                    else if (stats.isFile()) {
                        if (!checkIsExcluded(element))
                            printFileLine(element);
                    }
                } catch (e) {
                    console.log("Directory handler: error checking [ " + tempPath + " ]");
                    return;
                }
            }
            // look for gophermap and process if found...
            var gopherMapPath = filePath;
            if (gopherMapPath.lastIndexOf() != '/')
                gopherMapPath = gopherMapPath + "/gophermap";
            else
                gopherMapPath = gopherMapPath + "gophermap";
            gopherMapPath = path.normalize(gopherMapPath);
            fs.stat(gopherMapPath, function (err, stats) {
                if (err) {
                    printDirTitle();
                    files.sort(function(a, b) {
                        var x = a.toLowerCase(), y = b.toLowerCase();
                        return x < y ? -1 : x > y ? 1 : 0;
                    }).forEach(processListLines);
                    socket.end('\r\n.\r\n');
                }
                else if (stats.isFile()) {
                    var gopherMapData = ""
                    var rs = fs.createReadStream(gopherMapPath);
                    rs.on('data', function(data) {
                        gopherMapData = gopherMapData + data.toString();
                    });
                    rs.on('end', function() {
                        //console.log("Directory handler: using Gophermap.");
                        gopherMapLinesIn = gopherMapData.split(/\n/);
                        gopherMapLinesIn.forEach(processGopherLines);
                        gopherMapLinesOut.forEach(printGopherMapLine);
                        if (state == 1) {
                            files.sort(function(a, b) {
                                var x = a.toLowerCase(), y = b.toLowerCase();
                                return x < y ? -1 : x > y ? 1 : 0;
                            }).forEach(processListLines);
                        }
                        socket.end('\r\n.\r\n');
                    });
                    rs.on('error', function() {
                        state = 0;
                        handleInternalServerError(
                            selector, socket, "Directory handler: Error reading Gophermap.");
                    });
                }
            });
        }
    });
}

function handleCgiFile(selector, searchString, filePath, socket, scripts, host, port) {
    var cmd = "";
    var filename = ""
    var extension = "";
    var loc = selector.lastIndexOf('/');
    if (loc != -1)
        filename = selector.substring(loc);
    loc = filename.lastIndexOf('.');
    if (loc != -1)
        extension = filename.substring(loc).toLowerCase();
    //console.log("CGI handler: extension is [ " + extension + " ]");
    if (typeof scripts[extension] === 'string')
        cmd = scripts[extension];
    cmd = cmd + " " + filePath;
    //console.log("CGI handler: cmd is [ " + cmd + " ]");

    var execEnv = process.env;
    execEnv.QUERY_STRING = searchString;
    execEnv.SCRIPT_NAME = selector;
    execEnv.SERVER_NAME = host;
    execEnv.SERVER_PORT = port;
    execEnv.REMOTE_ADDR = socket.remoteAddress;
    execEnv.SERVER_PROTOCOL = "Gopher";
    execEnv.SERVER_SOFTWARE = "White Mesa node-gopher/0.1.1";
    var config = {
        maxBuffer: 10000 * 1024,
        env: execEnv
    };

    exec(cmd, function (error, stdout, stderr) {
        if (error == null) {
            socket.write(stdout);
            socket.end();
            //console.log('CGI handler: CGI request fulfilled.');
        }
        else {
            handleInternalServerError(
                selector, socket, "CGI handler: execution error.");
        }
    });
}

function handleURL(selector, socket, url) {
    socket.write(util.format(
        "<HTML>\r\n<HEAD>\r\n"
      + "<META HTTP-EQUIV=\"refresh\" content=\"5;URL='%s'\">\r\n "
      + "<META name=\"generator\" content=\"White Mesa node-gopher v0.1\">\r\n"
      + "</HEAD>\r\n<BODY>\r\n" +
      + "You are following a link from gopher to a web site. You will be "
      + "automatically taken to the web site shortly. If you do not get sent "
      + "there, please click <A HREF=\"%s"
      + "\">here</A> to go to the web site.\r\n "
      + "<P>The URL linked is:<P><A HREF=\"%s\">%s</A>\r\n "
      + "<P>Thanks for using gopher!\r\n</BODY>\r\n</HTML>\r\n", url, url, url, url));
    socket.end();
    socket.logStats["status"] = 303;  // HTTP "see other"
    console.log("URL handler: selector [ " + selector + " ]");
}

function handleFileNotFound(selector, socket) {
    socket.end("File: '" + selector + "' not found.\r\n.\r\n");
    socket.logStats["status"] = 404; // HTTP "not found"
    console.log('Request handler: file not found.');
}

function handleDirNotAvailableError(selector, socket) {
    socket.end("Directory: '" + selector + "' not available.\r\n.\r\n");
    socket.logStats["status"] = 403; // HTTP "forbidden"
    console.log('Request handler: directory not available.');
}

function handleBadSelector(selector, socket) {
    socket.end("Bad selector: '" + selector + "'.\r\n.\r\n");
    socket.logStats["status"] = 400; // HTTP "bad request"
    console.log('Request handler: bad selector.');
}

function handleInternalServerError(selector, socket, consoleMsg) {
    socket.end("Error processing '" + selector + "'.\r\n.\r\n");
    socket.logStats["status"] = 500; // HTTP "internal server error"
    console.log(consoleMsg);
}

function file(selector, filePath, searchString, socket, fileTypes, scripts, host, port) {
    if (selector.search(/\.\.|%/) != -1) {
        handleBadSelector(selector, socket);
    }
    fs.lstat(filePath, function (err, stats) {
        if (err)
            handleFileNotFound(selector, socket);
        else if (stats.isFile())
            handleFile(selector, filePath, socket, fileTypes);
        else if (stats.isDirectory())
            handleDirectory(selector, filePath, socket, fileTypes, host, port);
        else
            handleFileNotFound(selector, socket);
    });
}

function cgi(selector, filePath, searchString, socket, fileTypes, scripts, host, port) {
    if (selector.search(/\.\.|%/) != -1) {
        handleBadSelector(selector, socket);
    }
    fs.lstat(filePath, function (err, stats) {
        if (err)
            handleFileNotFound(selector, socket);
        else if (stats.isFile())
            handleCgiFile(selector, searchString, filePath, socket, scripts, host, port);
        else
            handleFileNotFound(selector, socket);
    });
}

function url(selector, filePath, searchString, socket, fileTypes, scripts, host, port) {
    var pos = selector.search(/URL:/);
    if (pos != -1) {
        var url = selector.substr(pos + 4);
        handleURL(selector, socket, url);
    } else {
        handleInternalServerError(selector, socket, "URL handler: error parsing URL.");
    }
}

exports.file = file;
exports.cgi = cgi;
exports.url = url;

