/****************************************************************************

Copyright (c) 2014 Robert Cunnings, NW8L

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

****************************************************************************/

/*

logger.js

Helper functions for gopher service activity logging, including log file
rotation at specified interval.

*/

var fs = require('fs');

var initialized = false;
var writeToConsole = false;
var logFilename = ""
var logStream = null;
var timer = null;
var rotPeriod = 0;
var minutes = 0;

function onInterval() {
    if (rotPeriod > 0) {
        //console.log("Logger: timer fired, count is " + minutes + ".");
        ++minutes;
        if ((minutes / 60) >= rotPeriod) {
            minutes = 0;
            if (logStream != null) {
                logStream.end();
                logStream = null;
            }
            var dt = new Date();
            var month = ("0" + (dt.getUTCMonth() + 1)).slice(-2);
            var date = ("0" + dt.getUTCDate()).slice(-2);
            var hrs = ("0" + dt.getUTCHours()).slice(-2);
            var mins = ("0" + dt.getUTCMinutes()).slice(-2);
            var fn = logFilename;
            var ext = "";
            var loc = logFilename.lastIndexOf('.');
            if (loc != -1) {
                fn = logFilename.substring(0, loc);
                ext = logFilename.substr(loc);
            }
            var newLogFilename = fn + "_" + dt.getUTCFullYear() + month + date + hrs + mins + "UTC" + ext;
            fs.rename(logFilename, newLogFilename, function (err) {
                if (err) {
                    console.log("Logger: rotating log, error renaming " + logFilename + " to " + newLogFilename);
                } else {
                    console.log("Logger: rotating log, renamed " + logFilename + " to " + newLogFilename);
                }
                logStream = fs.createWriteStream(logFilename, options);
            });
        }
    }
}

function isInitialized() {
    return initialized;
}

function writeLog(msg) {
    if (initialized && logStream != null) {
        logStream.write(msg);
        if (writeToConsole)
            console.log(msg);
    }
}

function initLog(filename, rotationInterval, enableConsoleWrite) {
    options = { flags : 'a' };
    logStream = fs.createWriteStream(filename, options);
    writeToConsole = enableConsoleWrite;
    logFilename = filename;
    rotPeriod = rotationInterval;
    initialized = true;
    timer = setInterval(onInterval, 60000);
    console.log("Logger: opened log file " + logFilename + " with rotation every " + rotationInterval + " hours.");
}

function closeLog() {
    if (logStream != null) {
        logStream.end();
        //console.log("Logger: closed log file " + logFilename);
    }
    if (timer !== null) {
        clearInterval(timer);
        //console.log("Logger: closed timer.");
    }
}

exports.writeLog = writeLog;
exports.initLog = initLog;
exports.closeLog = closeLog;
exports.isInitialized = isInitialized;

