/****************************************************************************

Copyright (c) 2014 Robert Cunnings, NW8L

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

****************************************************************************/

/*

gopher.js

Holds core gopher service functionality. Incoming requests are parsed and
dispatched to handlers in requestHandlers.js. When done servicing request,
makes log entry in RFC 1413 common log format.

*/

var net = require('net');
var path = require('path');
var util = require('util');
var logger = require('./logger');

var months = {};
months[0]="Jan";
months[1]="Feb";
months[2]="Mar";
months[3]="Apr";
months[4]="May";
months[5]="Jun";
months[6]="Jul";
months[7]="Aug";
months[8]="Sep";
months[9]="Oct";
months[10]="Nov";
months[11]="Dec";

/*
common log format: client ip addr,
                   client RFC 1413 user identifier (not implemented, always '-'),
                   client userid  (not implemented, always '-'),
                   local date:time and timezone,
                   Gopher selector,
                   HTTP status code (same semantics for Gopher as for HTTP),
                   gopher search string.
Examples:

192.168.1.8 - - [26/Oct/2013:19:58:05 -0700] "/" "OK" 2301 "-"
192.168.1.8 - - [26/Oct/2013:23:23:02 -0700] "/" "OK" 2301 "javascript"
*/

/*
 * Method executed to write log entry
 */
function logTransaction(socket) {
    // create log record
    var dt = new Date();
    var offset = dt.getTimezoneOffset() / 60;
    offset = offset * 100;
    var neg = false;
    if (offset < 0)
        offset = offset * -1;
    var tzoffset = ("000" + offset).slice(-4);
    if (neg)
        tzoffset = "+" + tzoffset;
    else
        tzoffset = "-" + tzoffset;
    var line = util.format("%s - - [%d/%s/%d:%d:%d:%d %s] \"%s\" %d %d \"%s\"\n",
                           socket.logStats["addr"],
                           dt.getDate(),
                           months[dt.getMonth()],
                           dt.getFullYear(),
                           dt.getHours(),
                           dt.getMinutes(),
                           dt.getSeconds(),
                           tzoffset,
                           socket.logStats["selector"],
                           socket.logStats["status"],
                           socket.bytesWritten,
                           socket.logStats["search"]
                           );
    logger.writeLog(line);
}

/*
 * Method executed to dispatch requests to handlers
 */
function dispatch(selector, searchString, socket, handlers, basePaths, fileTypes, scripts, host, port) {
    var	handlerKey = "";
    var tempSelector = "";
    var filePath = "";
    var len = 0;
    for (key in handlers) {
        var pos = selector.search(key);
        //console.log("Gopher: searching for: [ " + key + " ] in: " + selector + " pos: " + pos.toString()); 
        if (pos == 0) {
            if (key.length > len) {
                len = key.length;
                handlerKey = key;
                // preserve root path '/'
                if (selector.length > 1)
                    tempSelector = selector.substr(len);
            }
        }
    }
    var basePath = basePaths[handlerKey];
    filePath = path.join(basePath, tempSelector);
    //console.log("Gopher: selector is [ " + selector + " ]");
    //console.log("Gopher: handlerKey is [ " + handlerKey + " ]");
    //console.log("Gopher: basePath is [ " + basePath + " ]");
    if (typeof handlers[handlerKey] === 'function') {
        handlers[handlerKey](selector, filePath, searchString, socket, fileTypes, scripts, host, port);
    } else {
        socket.write("File: '" + selector + "' not found.\r\n.\r\n");
        socket.logStats["status"] = 404; // HTTP "not found"
        console.log("Gopher: no handler found for [ " + selector + " ]");
    }
}

/*
 * Callback method executed when data is received from a socket
 */
function receiveData(socket, data, handlers, basePaths, fileTypes, scripts, host, port) {
    // extract selector and any search string
    var request = data.toString().trim();
    var reqTokens = request.split('\t');
    var selector = "";
    var searchString = "";
    if (reqTokens.length > 0)
        selector = reqTokens[0].trim();
    if (reqTokens.length > 1) {
        // RFC 1436 search string found
        searchString = reqTokens[1].trim();
    }
    else {
        // use RFC1738 style 'query string' appended to selector
        // as search string if present
        var queryTokens = selector.split('?');
        if (queryTokens.length > 1) {
            searchString = queryTokens[1].trim();
            selector = queryTokens[0].trim();
        }
    }
    if (selector.length == 0)
        selector = "/";
    else if (selector.charAt(0) != '/')
        selector = "/" + selector;  // force absolute path for logging, dir listings etc.
    // intialize log stats
    socket.logStats["selector"] = selector;
    if (searchString.length = 0)
        socket.logStats["search"] = "-";
    else
        socket.logStats["search"] = searchString;
    socket.logStats["status"] = 200; // HTTP "OK" optimistically assigned
    socket.logStats["addr"] = socket.remoteAddress;
    // now dispatch the request
    dispatch(selector, searchString, socket, handlers, basePaths, fileTypes, scripts, host, port);
    console.log("Gopher: file selector [ " + selector + " ]");
}

/*
 * Start the gopher server on specified port
 */
 function start(handlers, basePaths, fileTypes, scripts, host, port) {
    //Callback method executed when a new TCP socket is opened.
    function newSocket(socket) {
        console.log('\r\nGopher: new connection.');
        socket.logStats = {};
        socket.on('data', function(data) {
            receiveData(socket, data, handlers, basePaths, fileTypes, scripts, host, port);
        });
        socket.on('end', function() {
            if (logger.isInitialized()) {
                logTransaction(socket);
            }
            console.log("Gopher: connection closed.");
        });
        socket.on('error', function() {
            console.log("Gopher: socket error, connection closed.");
        });
    }
    var server = net.createServer(newSocket);
    server.listen(port, function (port) {
        var address = server.address();
        console.log("Gopher: listening on %j", address);
    });
 }

exports.start = start;

