/* This file defines a JavaScript class that can solve
 * Sudoku puzzles.  The algorithm that is used is a brute
 * force algorithm that came from the following Wikipedia
 * page:
 * http://en.wikipedia.org/wiki/Algorithmics_of_sudoku
 *
 * Author: Eric Oulashin (AKA Nightfox)
 * BBS: Digital Distortion
 * BBS address: digdist.bbsindex.com
 *
 * Date       User              Description
 * 2009-11-07 Eric Oulashin     Started
 * 2009-11-08 Eric Oulashin     Added a function to return whether
 *                              the Sudoku grid is empty (all zeroes).
 * 2009-11-09 Eric Oulashin     Added numSetNumbers().
 */

////////////////////////////////////////////////////////////////////////////////
// SudokuSolver class stuff

function SudokuSolver()
{
   this.solve = SudokuSolver_Solve;
   this.solveCell = SudokuSolver_SolveCell;
   this.valSafe = SudokuSolver_ValSafe;
   this.numSolutions = SudokuSolver_NumSolutions;
   this.getSolutionValue = SudokuSolver_GetSolutionValue;
   this.reset = SudokuSolver_Reset;
   this.isEmpty = SudokuSolver_IsEmpty;
   this.numSetNumbers = SudokuSolver_NumSetNumbers;

   // Create the data members by caling reset().
   this.reset();
}
// For the SudokuSolver class: Solves the Sudoku puzzle.
function SudokuSolver_Solve()
{
   this.solveCell(0, 0);
}
// For the SudokuSolver class: Solves a specific cell in the
// puzzle.
//
// Paramters:
//  pRow: The puzzle row (0-based)
//  pCol: The puzzle column (0-based)
function SudokuSolver_SolveCell(pRow, pCol)
{
   if (pRow == 9)
	{
      // A solution to the puzzle has been found.
      // Add the solution to the solutions array.

      // Copy the values from this.grid into the last
      // element of this.solutions.
      var solIndex = this.solutions.length; // Array index for the solution
      this.solutions[solIndex] = new SudokuSolution();
      for (var row = 0; row < 9; ++row)
      {
         for (var col = 0; col < 9; ++col)
            this.solutions[solIndex].grid[row][col] = this.grid[row][col];
      }
	}
	else
	{
      var t = 0;
      for (var value = 1; value <= 9; ++value)
		{
         if (this.valSafe(pRow, pCol, value))
         {
            t = this.grid[pRow][pCol];
            this.grid[pRow][pCol] = value;
            if (pCol == 8)
               this.solveCell(pRow + 1, 0);
            else
               this.solveCell(pRow, pCol + 1);

            this.grid[pRow][pCol] = t;
         }
      }
   }
}
// For the SudokuSolver class: Returns whether or not a
// value is valid for a given row & column in the puzzle.
function SudokuSolver_ValSafe(pRow, pCol, pVal)
{
   var row = 0, col = 0;

   if (this.grid[pRow][pCol] == pVal) return true;
   if (this.grid[pRow][pCol] != 0) return false;
   for (col = 0; col < 9; ++col)
      if (this.grid[pRow][col] == pVal) return false;
   for (row = 0; row < 9; ++row)
      if (this.grid[row][pCol] == pVal) return false;
   var br = Math.floor(pRow / 3);
   var bc = Math.floor(pCol / 3);
   for (row = br * 3; row < (br + 1) * 3; ++row)
   {
      for (col = bc * 3; col < (bc + 1) * 3; ++col)
         if (this.grid[row][col] == pVal) return false;
   }

   return true;
}
// For the SudokuSolver class: Returns the number of solutions.
function SudokuSolver_NumSolutions()
{
   return this.solutions.length;
}
// For the SudokuSolver class: This is a convenience method
// that returns a value from one of the solutions.
//
// Parameters:
//  pSolutionIndex: The index of the solution (0-based)
//  pRow: The row in the solution grid (0-based)
//  pCol: The column in the solution grid (0-based)
//
// Return value: The value from the solution grid at the given
//               index, row, & column.  If any of those values
//               are invalid, this function will return -1.
function SudokuSolver_GetSolutionValue(pSolutionIndex, pRow, pCol)
{
   var solutionValue = -1;

   if ((pSolutionIndex >= 0) && (pSolutionIndex < this.solutions.length))
   {
      if ((pRow >= 0) && (pRow < 9) && (pCol >= 0) && (pCol < 9))
         solutionValue = this.solutions[pSolutionIndex].grid[pRow][pCol];
   }

   return solutionValue;
}
// For the SudokuSolver class: Resets the data members.
function SudokuSolver_Reset()
{
   // Create the grid as a 9x9 array of integers
   this.grid = new Array(9);
   for (var row = 0; row < 9; ++row)
   {
      this.grid[row] = new Array(9);
      for (var col = 0; col < 9; ++col)
         this.grid[row][col] = 0;
   }

   // solutions will be an array containing SudokuSolution
   // objects that contain the solution(s) to the puzzle.
   // This is an array in case multiple solutions are found.
   this.solutions = new Array();
}
// For the SudokuSolver class: Returns whether or not it's
// empty (all zeroes).
function SudokuSolver_IsEmpty()
{
   var allZeroes = true;

   var continueOn = true;
   for (var row = 0; (row < 9) && continueOn; ++row)
   {
      for (var col = 0; (col < 9) && continueOn; ++col)
      {
         if (this.grid[row][col] != 0)
         {
            allZeroes = false;
            break;
         }
      }
   }

   return allZeroes;
}
// For the SudokuSolver class: Returns the number of non-zero numbers
// set in the grid.
function SudokuSolver_NumSetNumbers()
{
   var numNumbersSet = 0;

   for (var row = 0; row < 9; ++row)
   {
      for (var col = 0; col < 9; ++col)
      {
         if ((this.grid[row][col] > 0) && (this.grid[row][col] < 10))
            ++numNumbersSet;
      }
   }

   return numNumbersSet;
}

////////////////////////////////////////////////////////////////////////////////
// SudokuSolution class stuff

function SudokuSolution()
{
   // Create the grid as a 9x9 array of integers
   this.grid = new Array(9);
   for (var row = 0; row < 9; ++row)
   {
      this.grid[row] = new Array(9);
      for (var col = 0; col < 9; ++col)
         this.grid[row][col] = 0;
   }
}