/* This is the ANSI version of my door app for Synchronet
 * that can solve a Sudoku puzzle.
 *
 * Author: Eric Oulashin (AKA Nightfox)
 * BBS: Digital Distortion
 * BBS address: digdist.bbsindex.com
 *
 * Date       User              Description
 * 2009-11-08 Eric Oulashin     Worked on the ANSI display and user
 *                              input for the ANSI version..  Also,
 *                              added getANSIDigitInputLocations().
 * 2009-11-09 Eric Oulashin     Worked on the user interface some more.
 * 2009-11-11 Eric Oulashin     Made some minor updates.
 * 2009-11-14 Eric Oulashin     Updated to make use of the new
 *                              solveSudokuPuzzle() function (in
 *                              DDSudokuSolver_Misc.js) to solve
 *                              the puzzle.
 * 2009-11-15 Eric Oulashin     Updated so that when the user presses
 *                              Q to quit after the puzzle has been
 *                              solved, the correct value will be
 *                              written on the cell on the screen
 *                              before quitting.
 */

// If the user's terminal doesn't support ANSI, then show an
// error and exit.
if (!console.term_supports(USER_ANSI))
{
   console.print("Sorry, your terminal does not support ANSI.\r\n\1p");
   exit(0);
}


// Keyboard input characters
var CTRL_H = "\x08";
var BACKSPACE = CTRL_H;

// Box-drawing/border characters: Single-line
var UPPER_LEFT_SINGLE = "";
var HORIZONTAL_SINGLE = "";
var UPPER_RIGHT_SINGLE = "";
var VERTICAL_SINGLE = "";
var LOWER_LEFT_SINGLE = "";
var LOWER_RIGHT_SINGLE = "";
var T_SINGLE = "";
var LEFT_T_SINGLE = "";
var RIGHT_T_SINGLE = "";
var BOTTOM_T_SINGLE = "";
var CROSS_SINGLE = "";
// Box-drawing/border characters: Double-line
var UPPER_LEFT_DOUBLE = "";
var HORIZONTAL_DOUBLE = "";
var UPPER_RIGHT_DOUBLE = "";
var VERTICAL_DOUBLE = "";
var LOWER_LEFT_DOUBLE = "";
var LOWER_RIGHT_DOUBLE = "";
var T_DOUBLE = "";
var LEFT_T_DOUBLE = "";
var RIGHT_T_DOUBLE = "";
var BOTTOM_T_DOUBLE = "";
var CROSS_DOUBLE = "";
// Box-draing/border characters: Double horizontal, single vertical
var LEFT_T_HD_SV = "";
var RIGHT_T_HD_SV = "";
// Box-drawing/border characters: Double vertical, single horizontal
var T_DV_SH = "";
var BOTTOM_T_DV_SH = "";
var LEFT_T_DV_SH = "";
var RIGHT_T_DV_SH = "";
var CROSS_DV_SH = "";


// gStatusPos contains the screen coordinate for status text.
var gStatusPos = new Object();
gStatusPos.x = 23;
gStatusPos.y = 3;

// gKeyInfoPos contains the screen coordinate of where to start writing
// information about the valid keys.
var gKeyInfoPos = new Object();
gKeyInfoPos.x = 25;
gKeyInfoPos.y = 7;

// gEditColor stores the color to use for inputting numbers.
var gEditColor = "nw";

// Create a SudokuSolver object to use for solving the puzzle.
var gSolver = new SudokuSolver();

// Draw the Sudoku grid, status border, and help text on the screen.
var gSudokuGridPos = new Object;
gSudokuGridPos.x = 1;
gSudokuGridPos.y = 2;
console.clear();
console.print("ng     Sudoku grid");
drawSudokuGrid(gSudokuGridPos.x, gSudokuGridPos.y, "nc", false);
drawStatusBorder(22, 2);
printKeyInfo(gKeyInfoPos.x, gKeyInfoPos.y, true);

// Input the numbers from the user.
console.print(gEditColor);
var loopObj;
var continueOn = true;
while (continueOn)
{
   loopObj = doPuzzleInputLoop();
   if (loopObj.action == "SOLVE")
   {
      // If the puzzle is empty or doesn't have the minimum number
      // of set numbers, then refuse to solve it.  Otherwise, go
      // ahead and try to solve it.
      if (gSolver.isEmpty())
      {
         printxy(gStatusPos, "nyhCan't solve a blank puzzle.n.");
         printxy(gStatusPos, "n                           ");
      }
      else if (gSolver.numSetNumbers() < 17)
      {
         printxy(gStatusPos, "nyhThe puzzle must start with at least 17 numbers.n.");
         printxy(gStatusPos, "n                                               ");
      }
      else
      {
         // Try to solve the puzzle.
         printxy(gStatusPos, "nyhSolving...n");
         solveSudokuPuzzle();
         printxy(gStatusPos, "n                       ");
         // If a solution was found, then copy the solution
         // numbers into the puzzle.  Otherwise, tell the
         // user that no solution was found.
         if (gSolver.numSolutions() > 0)
         {
            for (var row = 0; row < 9; ++row)
            {
               for (var col = 0; col < 9; ++col)
               {
                  console.gotoxy(getScreenInputLocation(gSudokuGridPos.x, gSudokuGridPos.y, row, col));
                  console.print(gSolver.getSolutionValue(0, row, col));
               }
            }
         }
         else
         {
            printxy(gStatusPos, "nyhNo solutions found.n");
            console.gotoxy(gStatusPos.x, gStatusPos.y+2);
            console.pause();
            printxy(gStatusPos, "n                   ");
         }
      }
   }
   else if (loopObj.action == "QUIT")
   {
      continueOn = false;
      printxy(gStatusPos, "nyhHave a nice day!n.");
   }
}

exit(0);
// End of script execution.


////////////////////////////////////////////////////////////////////////////////
// ANSI-mode functions

// Performs user input for the puzzle.
//
// Return value: An object containing the following properties:
//               action: A string describing the action to take, such
//                       as "QUIT" or "SOLVE"
function doPuzzleInputLoop()
{
   // Create the return object
   var retObj = new Object();
   retObj.action = "QUIT";

   // Perform the user input loop
   var userInput = 0;
   var validKeys = " CQS?0" + KEY_UP + KEY_DOWN + KEY_LEFT + KEY_RIGHT + KEY_HOME
                 + KEY_END + BACKSPACE
   var row = 0;
   var col = 0;
   var screenPos;
   var solveIt = false;
   var continueOn = true;
   while (continueOn)
   {
      screenPos = getScreenInputLocation(gSudokuGridPos.x, gSudokuGridPos.y, row, col);
      console.gotoxy(screenPos);
      userInput = console.getkeys(validKeys, 9);
      if ((userInput == " ") || (userInput == "0"))
      {
         gSolver.grid[row][col] = 0;
         // Move to the cell to the right.
         ++col;
         if (col > 8)
         {
            // We're past the rightmost cell, so go to the first
            // cell on the next line.
            if (row < 8)
            {
               ++row;
               col = 0;
            }
            else
               --col;
         }

         if (userInput == "0")
            printxy(screenPos, " ");
      }
      // Q: Quit
      else if (userInput == "Q")
      {
         // Print the appropriate character in the current cell on the
         // screen, and then set the return action to "QUIT".
         if (gSolver.numSolutions() > 0)
            printxy(screenPos, gSolver.getSolutionValue(0, row, col));
         else
            printxy(screenPos, gSolver.grid[row][col] != 0 ? gSolver.grid[row][col] : " ");
         continueOn = false;
         retObj.action = "QUIT";
         break;
      }
      // S: Solve the puzzle
      else if (userInput == "S")
      {
         printxy(screenPos, gSolver.grid[row][col] != 0 ? gSolver.grid[row][col] : " ");
         continueOn = false;
         solveIt = true;
         retObj.action = "SOLVE";
         break;
      }
      // C: Clear the grid
      else if (userInput == "C")
      {
         gSolver.reset();
         row = 0;
         col = 0;
         var screenPos;
         for (var myRow = 0; myRow < 9; ++myRow)
         {
            for (var myCol = 0; myCol < 9; ++myCol)
            {
               screenPos = getScreenInputLocation(gSudokuGridPos.x, gSudokuGridPos.y, myRow, myCol);
               printxy(screenPos, " ");
            }
         }
      }
      // ?: Display help
      else if (userInput == "?")
      {
         // Display help, then re-draw the main screen
         displayGeneralHelp();
         printKeyInfo(1, 15, false);
         console.clear();
         console.print("ng     Sudoku grid");
         drawSudokuGrid(gSudokuGridPos.x, gSudokuGridPos.y, "nc", true);
         drawStatusBorder(22, 2);
         printKeyInfo(gKeyInfoPos.x, gKeyInfoPos.y, true);
         console.print(gEditColor);
      }
      else if (userInput == BACKSPACE)
      {
         printxy(screenPos, gSolver.grid[row][col] != 0 ? gSolver.grid[row][col] : " ");
         // Move to the cell to the left, and erase the number that is there.
         var movedBack = true;
         if (col > 0)
            --col;
         else
         {
            // We're at the left side, so go up to the last cell
            // on the previous row.
            if (row > 0)
            {
               --row;
               col = 8;
            }
            else
               movedBack = false;
         }

         // If the cursor moved back, then erase/remove the
         // number at the current position.
         if (movedBack)
         {
            gSolver.grid[row][col] = 0;
            var pos = getScreenInputLocation(gSudokuGridPos.x, gSudokuGridPos.y, row, col);
            printxy(pos, " ");
         }
      }
      else if (userInput == KEY_LEFT)
      {
         printxy(screenPos, gSolver.grid[row][col] != 0 ? gSolver.grid[row][col] : " ");
         // Move to the cell to the left.
         if (col > 0)
            --col;
         else
         {
            // We're at the left side, so go up to the last cell
            // on the previous row.
            if (row > 0)
            {
               --row;
               col = 8;
            }
         }
      }
      else if (userInput == KEY_RIGHT)
      {
         printxy(screenPos, gSolver.grid[row][col] != 0 ? gSolver.grid[row][col] : " ");
         // Move to the cell to the right.
         ++col;
         if (col > 8)
         {
            // We're past the rightmost cell, so go to the first
            // cell on the next line.
            if (row < 8)
            {
               ++row;
               col = 0;
            }
            else
               --col;
         }
      }
      else if (userInput == KEY_UP)
      {
         printxy(screenPos, gSolver.grid[row][col] != 0 ? gSolver.grid[row][col] : " ");
         // Move one row up.
         if (row > 0)
            --row;
      }
      else if (userInput == KEY_DOWN)
      {
         printxy(screenPos, gSolver.grid[row][col] != 0 ? gSolver.grid[row][col] : " ");
         // Move one row down.
         if (row < 8)
            ++row;
      }
      else if (userInput == KEY_HOME)
      {
         printxy(screenPos, gSolver.grid[row][col] != 0 ? gSolver.grid[row][col] : " ");
         // Move to the first cell in the current row.
         if (col > 0)
            col = 0;
      }
      else if (userInput == KEY_END)
      {
         printxy(screenPos, gSolver.grid[row][col] != 0 ? gSolver.grid[row][col] : " ");
         // Move to the last cell in the current row.
         if (col < 8)
            col = 8;
      }
      // Digit: Put it into the puzzle
      else if (/^[1-9]$/.test(userInput))
      {
         // The user entered a number, so set it in the gSolver
         // and advance to the next cell to the right.
         gSolver.grid[row][col] = userInput;
         ++col;
         if (col > 8)
         {
            // We're past the rightmost cell, so go to the first
            // cell on the next line.
            if (row < 8)
            {
               ++row;
               col = 0;
            }
            else
               --col;
         }
      }
   }

   return retObj;
}

// Draws the Sudoku grid on the screen.
//
// Parameters:
//  pX: The column of the upper-left coordinate of the grid
//  pY: The row of the upper-left coordinate of the grid
//  pColor: The color to use to draw the grid
//  pWritePuzzleVals: Boolean - Whether or not to also write the
//                    puzzle values on the screen.  This is optional;
//                    by default, the puzzle values are not written.
function drawSudokuGrid(pX, pY, pColor, pWritePuzzleVals)
{
   // This function draws a 3x3 grid of border characters.
   //
   // Parameters:
   //  pX: The column of the upper-left coordinate of the grid
   //  pY: The row of the upper-left coordinate of the grid
   function drawGrid3x3(pX, pY)
   {
      console.gotoxy(pX, pY);
      console.print(UPPER_LEFT_SINGLE);
      var numCharsDrawn = 1;
      var endX = pX + 5;
      // Top row
      var X = 0;
      for (X = pX; X < endX; ++X)
      {
         console.print(numCharsDrawn % 2 == 0 ? T_SINGLE : HORIZONTAL_SINGLE);
         ++numCharsDrawn;
      }
      console.print(UPPER_RIGHT_SINGLE);
      // Number rows
      var Y = pY + 1;
      ++endX;
      for (var numRow = 0; numRow < 3; ++numRow)
      {
         numCharsDrawn = 0;
         //console.gotoxy(pX, Y);
         for (var X = pX; X <= endX; X += 2)
         {
            console.gotoxy(X, Y);
            console.print(VERTICAL_SINGLE);
            ++numCharsDrawn;
         }
         // Print the row's bottom border
         ++Y;
         console.gotoxy(pX, Y);
         console.print(numRow == 2 ? LOWER_LEFT_SINGLE : LEFT_T_SINGLE);
         numCharsDrawn = 0;
         for (var X = pX+1; X < endX; ++X)
         {
            if (numCharsDrawn % 2 == 0)
               console.print(HORIZONTAL_SINGLE);
            else
               console.print(numRow == 2 ? BOTTOM_T_SINGLE : CROSS_SINGLE);
            ++numCharsDrawn;
         }
         console.print(numRow == 2 ? LOWER_RIGHT_SINGLE : RIGHT_T_SINGLE);
         ++Y;
      }
   }

   // Draw 9 3x3 grids
   console.print(pColor);
   var endX = pX + 19;
   var endY = pY + 19;
   for (var X = pX; X < endX; X += 7)
   {
      for (var Y = pY; Y < endY; Y += 7)
         drawGrid3x3(X, Y);
   }

   // If pWritePuzzleVals is true, then write the puzzle
   // values on the screen.
   if (pWritePuzzleVals)
   {
      console.print(gEditColor);
      var screenPos;
      // If the puzzle has already been solved, then write
      // the solved puzzle values; otherwise, write the
      // unsolved values.
      if (gSolver.numSolutions() > 0)
      {
         for (var row = 0; row < 9; ++row)
         {
            for (var col = 0; col < 9; ++col)
            {
               screenPos = getScreenInputLocation(pX, pY, row, col);
               printxy(screenPos, gSolver.getSolutionValue(0, row, col));
            }
         }
      }
      else
      {
         for (var row = 0; row < 9; ++row)
         {
            for (var col = 0; col < 9; ++col)
            {
               screenPos = getScreenInputLocation(pX, pY, row, col);
               if (gSolver.grid[row][col] > 0)
                  printxy(screenPos, gSolver.grid[row][col]);
            }
         }
      }
   }
}

// Draws the status area border.
//
// Parameters:
//  pX: The screen column of the upper-left corner of the border
//      grid (1-based).
//  pY: The screen row of the upper-left corner of the border
//      grid (1-based).
function drawStatusBorder(pX, pY)
{
   console.gotoxy(pX, pY);
   console.print("nb" + UPPER_LEFT_DOUBLE);
   var widthAvailable = console.screen_columns - pX - 1;
   var halfWidthAvailable = Math.floor(widthAvailable / 2);
   // Top border
   var endX = pX + halfWidthAvailable - 5;
   for (var X = pX+1; X < endX; ++X)
      console.print(HORIZONTAL_DOUBLE);
   console.print(RIGHT_T_HD_SV + "yh4Messagesnb" + LEFT_T_HD_SV);
   for (var X = pX + halfWidthAvailable + 5; X < pX + widthAvailable; ++X)
      console.print(HORIZONTAL_DOUBLE);
   console.print(UPPER_RIGHT_DOUBLE);
   // Vertical border chars
   console.gotoxy(pX, pY + 1);
   console.print(VERTICAL_DOUBLE);
   console.gotoxy(pX + widthAvailable, pY + 1);
   console.print(VERTICAL_DOUBLE);
   // Bottom border
   console.gotoxy(pX, pY + 2);
   console.print(LOWER_LEFT_DOUBLE);
   endX = pX + widthAvailable;
   for (var X = pX+1; X < endX; ++X)
      console.print(HORIZONTAL_DOUBLE);
   console.print(LOWER_RIGHT_DOUBLE);
}

// This function returns an object with x and y properties
// representing the screen location for a number input for
// the ANSI Sudoku grid.
//
// Parameters:
//  pX: The screen column of the upper-left corner of the Sudoku
//      grid (1-based).
//  pY: The screen row of the upper-left corner of the Sudoku
//      grid (1-based).
//  pRow: The Sudoku grid row index (0-based)
//  pCol: The Sudoku grid column index (0-based)
function getScreenInputLocation(pX, pY, pRow, pCol)
{
   var screenCoord = new Object();
   screenCoord.x = pX + 1 + (pCol*2);
   screenCoord.y = pY + 1 + (pRow*2);
   if (pCol > 5)
      screenCoord.x += 2;
   else if (pCol > 2)
      screenCoord.x += 1;
   if (pRow > 5)
      screenCoord.y += 2;
   else if (pRow > 2)
      screenCoord.y += 1;

   return screenCoord;
}

// This function prints a string at a specified location on
// the screen.
//
// Parameters:
//  pScreenpos: An object containing x and y properties representing a
//              location on the screen.
//  pStr: The string to print on the screen
function printxy(pScreenpos, pStr)
{
   console.gotoxy(pScreenpos);
   console.print(pStr);
}

// This function prints the key information on the screen.
//
// Parameters:
//  pX: The column on the screen at which to start writing the key
//      information
//  pY: The row on the screen at which to start writing the key
//      information
//  pMoveCursor: Boolean - Whether or not to move the cursor to
//               control text alignment.  If false, this function
//               will ignore pX and pY and simply output a line
//               feed between lines.
function printKeyInfo(pX, pY, pMoveCursor)
{
   if (pMoveCursor)
   {
      var pos = new Object();
      pos.x = pX;
      pos.y = pY;
      printxy(pos, "nwhKeys");
      ++pos.y;
      printxy(pos, "k");
      ++pos.y;
      printxy(pos, "c1-9y: bEnter a number into the puzzle");
      ++pos.y;
      printxy(pos, "cSPACE ncor h0y: bSkip the current cell");
      ++pos.y;
      printxy(pos, "cBACKSPACEy: bGo back & erase previous value");
      ++pos.y;
      printxy(pos, "c" + UP_ARROW_CHAR + "y: bMove up one row");
      ++pos.y;
      printxy(pos, "c" + DOWN_ARROW_CHAR + "y: bMove up one row");
      ++pos.y;
      printxy(pos, "c" + LEFT_ARROW_CHAR + "y: bMove left one column");
      ++pos.y;
      printxy(pos, "c" + RIGHT_ARROW_CHAR + "y: bMove right one column");
      ++pos.y;
      printxy(pos, "cHOMEy: bMove to the start of the row");
      ++pos.y;
      printxy(pos, "cENDy: bMove to the end of the row");
      ++pos.y;
      printxy(pos, "cSy: bSolve the puzzle");
      ++pos.y;
      printxy(pos, "cCy: bClear the grid");
      ++pos.y;
      printxy(pos, "cQy: bQuit");
      ++pos.y;
      printxy(pos, "c?y: bDisplay help");
   }
   else
   {
      console.print("nwhKeys\r\n");
      console.print("k\r\n");
      console.print("c1-9y: bEnter a number into the puzzle\r\n");
      console.print("cSPACEy: bSkip the current cell\r\n");
      console.print("cBACKSPACEy: bGo back & erase previous value\r\n");
      console.print("c" + UP_ARROW_CHAR + "y: bMove up one row\r\n");
      console.print("c" + DOWN_ARROW_CHAR + "y: bMove up one row\r\n");
      console.print("c" + LEFT_ARROW_CHAR + "y: bMove left one column\r\n");
      console.print("c" + RIGHT_ARROW_CHAR + "y: bMove right one column\r\n");
      console.print("cHOMEy: bMove to the start of the row\r\n");
      console.print("cENDy: bMove to the end of the row\r\n");
      console.print("cSy: bSolve the puzzle\r\n");
      console.print("cCy: bClear the grid\r\n");
      console.print("cQy: bQuit\r\n");
      console.print("c?y: bDisplay help\r\n");
   }
}