/* This is a script that lists the users on the BBS.
 *
 * Author: Eric Oulashin (AKA Nightfox)
 * BBS: Digital Distortion
 * BBS address: digdist.bbsindex.com
 *
 * Date       User              Description
 * 2009-05-02 Eric Oulashin     Created
 * 2009-05-03 Eric Oulashin     Added a Synchronet version check.
 * 2009-05-04 Eric Oulashin     Changed the User variable's name from user
 *                              to theUser to avoid stomping on the user
 *                              object declared in sbbsdefs.js.
 * 2009-05-07 Eric Oulashin     Updated to allow the user to sort the
 *                              list alphabetically by user alias.
 * 2009-05-10 Eric Oulashin     Removed the "donePrompt" color from the colors
 *                              array, because it wasn't being used.
 * 2009-05-13 Eric Oulashin     Version 1.01
 *                              Updated to be free of DigitalDistortionDefs.js.
 *                              Added VERSION and VER_DATE, although they aren't
 *                              displayed to the user.
 * 2009-07-23 Eric Oulashin     Version 1.02
 *                              Minor update: Commented out VERSION and VER_DATE
 *                              to avoid re-declaration problems.  These weren't
 *                              being displayed to the user anyway.  Also,
 *                              changed the special character variables to be
 *                              declared with "var" rather than "const" to avoid
 *                              re-declaration errors.
 * 2009-07-31 Eric Oulashin     Updated centerText() to get the width of the
 *                              text after using strip_ctrl() so that it
 *                              doesn't count control characters.
 * 2009-08-17 Eric Oulashin     Version 1.03
 *                              Updated to read configuration options from a
 *                              configuration file, DigitalDistortionListUsers.cfg.
 *                              Added options there for excluding the sysop,
 *                              sorting the list (true/false/ask), whether
 *                              or not to display QWK accounts, and colors.
 * 2009-08-19 Eric Oulashin     Version 1.04
 *                              Bug fix: Now skips invalid user numbers, so it
 *                              will display the complete list of users (rather
 *                              than looping through 1 to system.stats.total_users,
 *                              because user numbers aren't necessarily sequential).
 * 2010-02-17 Eric Oulashin     Version 1.05
 *                              Updated to skip inactive user slots.
 * 2011-02-13 Eric Oulashin     Version 1.06
 *                              Moved the user lister code into its own class.
 *                              Added the getHTML function for use in a web page.
 * 2011-11-23 Eric Oulashin     Version 1.07
 *                              Fixed a typo in a variable name.  Also fixed the
 *                              error text displayed when the Synchronet version
 *                              is not current enough.  Thanks to Slinky for
 *                              catching these.
 * 2011-11-25 Eric Oulashin     Version 1.08
 *                              Bug fix in DDUserLister_GetHTML(): For Synchronet 3.11+,
 *                              made lastuser equal to lastuser = system.lastuser + 1
 *                              instead of just system.lastuser.  Thanks to Slinky
 *                              for reporting this.
 * 2013-11-17 Eric Oulashin     Version 1.09
 *                              Bug fix in DDUserLister_GetHTML(): Now uses the correct
 *                              user numbers for the user information links when the list
 *                              is sorted.
 * 2018-08-03 Eric Oulashin     Version 1.10
 *                              Updated to delete instances of User objects that
 *                              are created, due to an optimization in Synchronet
 *                              3.17 that leaves user.dat open
 * 2018-09-22 Eric Oulashin     Version 1.11
 *                              Fixed an issue where a user object property
 *                              was being set after the object was set to
 *                              undefined, which was causing an error
 * 2023-02-21 Eric Oulashin     Version 1.12
 *                              Now makes use of command-line arguments as a loadable
 *                              module for UL_ALL, UL_SUB, or UL_DIR.
 * 2023-02-22 Eric Oulashin     Version 1.13
 *                              Refactored. Added the ability to sort by last login.
 */

"use strict";

// This script requires Synchronet version 3.11 or newer.
// Exit if the Synchronet version is below the minimum.
if (system.version_num < 31100)
{
	var message = "\x01n\x01h\x01y\x01i* Warning:\x01n\x01h\x01w Digital Distortion List Users "
	            + "requires version \x01g3.11\x01w or\r\n"
	            + "higher of Synchronet.  This BBS is using version \x01g" + system.version
	            + "\x01w.  Please notify the sysop.";
	console.crlf();
	console.print(message);
	console.crlf();
	console.pause();
	exit();
}

//var VERSION = "1.13";
//var VER_DATE = "2023-02-22";

if (typeof(require) === "function")
{
	require("sbbsdefs.js", "K_NOCRLF");
	require("text.js", 'UserListFmt');
}
else
{
	load("sbbsdefs.js");
	load("text.js");
}

// Column inner lengths
var USER_NUM_LEN = 5;
var USER_NAME_LEN = 18;
var LOCATION_LEN = 25;
var LAST_CALL_DATE_LEN = 8;
var NUM_CALLS_LEN = 7;
var CONNECTION_TYPE_LEN = 8;

// Single-line box drawing/border characters.
// Note: These are declared using "var" rather
// than "const" to avoid re-declaration problems.
var UPPER_LEFT_SINGLE = "";
var HORIZONTAL_SINGLE = "";
var UPPER_RIGHT_SINGLE = "";
var VERTICAL_SINGLE = "";
var LOWER_LEFT_SINGLE = "";
var LOWER_RIGHT_SINGLE = "";
var T_SINGLE = "";
var LEFT_T_SINGLE = "";
var RIGHT_T_SINGLE = "";
var BOTTOM_T_SINGLE = "";
var CROSS_SINGLE = "";
// Double-line box drawing/border characters.
var UPPER_LEFT_DOUBLE = "";
var HORIZONTAL_DOUBLE = "";
var UPPER_RIGHT_DOUBLE = "";
var VERTICAL_DOUBLE = "";
var LOWER_LEFT_DOUBLE = "";
var LOWER_RIGHT_DOUBLE = "";
var T_DOUBLE = "";
var LEFT_T_DOUBLE = "";
var RIGHT_T_DOUBLE = "";
var BOTTOM_T_DOUBLE = "";
var CROSS_DOUBLE = "";

///////////////////////////
// Script execution code //
///////////////////////////

var gConfig = ReadConfigFile();


// Default to listing all users.  If a command-line argument is given,
// determine what mode it is.
var LIST_MODE = UL_ALL;
var executeScriptCode = true;
if (argv.length > 0)
{
	var valNum = parseInt(argv[0], 10);
	if (!isNaN(valNum))
		LIST_MODE = valNum;
	else
	{
		if (typeof(argv[0]) === "string")
			executeScriptCode = (argv[0].toUpperCase() == "TRUE");
		else if (typeof(argv[0]) === "boolean")
			executeScriptCode = argv[0];
	}
}

if (executeScriptCode)
	ListAllUsers(LIST_MODE);

// End of script execution


///////////////////////////////////////////////////////////////////////////////////
// User list functions


// Reads the configuration file.  Returns an object with the configuration settings.
function ReadConfigFile()
{
	// Determine the script's startup directory.
	// This code is a trick that was created by Deuce, suggested by Rob Swindell
	// as a way to detect which directory the script was executed in.  I've
	// shortened the code a little.
	var startup_path = '.';
	try { throw dig.dist(dist); } catch(e) { startup_path = e.fileName; }
	startup_path = backslash(startup_path.replace(/[\/\\][^\/\\]*$/,''));

	var settings = {
		excludeSysop: false,    // Whether or not to exclude the sysop
		sortUserList: false,    // Whether or not to sort the list
		sortType: "alpha",      // Alphabetical sort
		displayQWKAccts: false, // Whether or not to display QWK accounts
		colors: {
			veryTopHeader: "\x01n\x01h\x01w",
			border: "\x01n\x01c",
			colLabelText: "\x01n\x01" + "4\x01h\x01w",
			userNum: "\x01h\x01c",
			userName: "\x01h\x01y",
			location: "\x01h\x01m",
			lastCallDate: "\x01h\x01r",
			connectionType: "\x01h\x01b",
			numCalls: "\x01n\x01h\x01g"
		}
	};

	// Open the configuration file
	var cfgFile = new File(startup_path + "DigitalDistortionListUsers.cfg");
	if (cfgFile.open("r"))
	{
		// Behavior settings
		var behaviorSettings = cfgFile.iniGetObject("BEHAVIOR");
		if (behaviorSettings.hasOwnProperty("ExcludeSysop") && typeof(behaviorSettings.ExcludeSysop) === "boolean")
			settings.excludeSysop = behaviorSettings.ExcludeSysop;
		if (behaviorSettings.hasOwnProperty("SortList"))
		{
			var optType = typeof(behaviorSettings.SortList);
			if (optType === "boolean")
				settings.sortUserList = behaviorSettings.SortList;
			else if (optType === "string")
			{
				var sortUpper = behaviorSettings.SortList.toUpperCase();
				if (sortUpper == "ASK")
					settings.sortUserList = "ask";
				else if (sortUpper == "ALPHA")
				{
					settings.sortUserList = true;
					settings.sortType = "alpha";
				}
				else if (sortUpper == "LASTON")
				{
					settings.sortUserList = true;
					settings.sortType = "lastOn";
				}
			}
		}
		if (behaviorSettings.hasOwnProperty("DisplayQWKAccts") && typeof(behaviorSettings.DisplayQWKAccts) === "boolean")
			settings.displayQWKAccts = behaviorSettings.DisplayQWKAccts;

		// Colors
		var colorSettings = cfgFile.iniGetObject("COLORS");
		for (var prop in colorSettings)
			settings.colors[prop] = colorSettings[prop];

		cfgFile.close();
	}

	return settings;
}

// Displays a User record in the list.
//
// Parameters:
//  pUser: A User object
function DisplayUserInfo(pUser)
{
	if (pUser != null)
	{
		var lineText = gConfig.colors.border + VERTICAL_SINGLE
		             + gConfig.colors.userNum + centerText(format("%" + USER_NUM_LEN + "d", pUser.number), USER_NUM_LEN)
		             //+ gConfig.colors.userNum + format("%-" + USER_NUM_LEN + "d", pUser.number)
		             + gConfig.colors.border + VERTICAL_SINGLE
		             + gConfig.colors.userName + centerText(pUser.alias.substring(0, USER_NAME_LEN), USER_NAME_LEN)
		             //+ gConfig.colors.userName + format("%-" + USER_NAME_LEN + "s", pUser.alias.substring(0, USER_NAME_LEN))
		             + gConfig.colors.border + VERTICAL_SINGLE
		             + gConfig.colors.location + centerText(pUser.location.substring(0, LOCATION_LEN), LOCATION_LEN)
		             //+ gConfig.colors.location + format("%-" + LOCATION_LEN + "s", pUser.location.substring(0, LOCATION_LEN))
		             + gConfig.colors.border + VERTICAL_SINGLE
		             + gConfig.colors.lastCallDate + centerText(strftime("%m/%d/%y", pUser.stats.laston_date), LAST_CALL_DATE_LEN)
		             + gConfig.colors.border + VERTICAL_SINGLE
		             + gConfig.colors.connectionType + centerText(pUser.connection.substring(0, CONNECTION_TYPE_LEN), CONNECTION_TYPE_LEN)
		             + gConfig.colors.border + VERTICAL_SINGLE
		             + gConfig.colors.numCalls + centerText(format("%" + NUM_CALLS_LEN + "d", pUser.stats.total_logons), NUM_CALLS_LEN)
		             + gConfig.colors.border + VERTICAL_SINGLE;
		console.center(lineText);
	}
}
// Lists all users.
//
// Parameters:
//  pListMode: The mode to list users with - From sbbsdefs:
//             UL_ALL: List all active users in user database
//             UL_SUB: List all users with access to cursub
//             UL_DIR: List all users with access to curdir
function ListAllUsers(pListMode)
{
	// If gSortUserList is "ask", then ask the user if they want the list
	// sorted alphabetically.
	if ((typeof(gConfig.sortUserList) == "string") && (gConfig.sortUserList == "ask"))
	{
		console.attributes = "N";
		console.crlf();
		console.mnemonics("Sort by User ~Name, ~Last-on Date or [#]: ");
		var sort = console.getkeys("NL#");
		if (sort != "N" && sort != "L")
			gConfig.sortUserList = false;
		else
		{
			gConfig.sortUserList = true;
			if (sort == "N")
				gConfig.sortType = "alpha";
			else if (sort == "L")
				gConfig.sortType = "lastOn";
			console.putmsg(bbs.text(CheckingSlots));
		}
		console.attributes = "N";
	}

	// Clear the screen
	console.clear();

	// The following line which sets console.line_counter to 0 is a
	// kludge to disable Synchronet's automatic pausing after a
	// screenful of text.
	console.line_counter = 0;


	// Write the header
	console.print(gConfig.colors.veryTopHeader);
	console.center("User List");
	// Upper border lines
	// User number section
	var lineText = gConfig.colors.border + UPPER_LEFT_SINGLE;
	for (var i = 0; i < USER_NUM_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// User name section
	for (var i = 0; i < USER_NAME_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// User location section
	for (var i = 0; i < LOCATION_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// Last call date section
	for (var i = 0; i < LAST_CALL_DATE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// Connection type section
	for (var i = 0; i < CONNECTION_TYPE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// # calls section
	for (var i = 0; i < NUM_CALLS_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += UPPER_RIGHT_SINGLE;
	console.center(lineText);
	// Column labels
	lineText = gConfig.colors.border + VERTICAL_SINGLE + gConfig.colors.colLabelText
	         + centerText("User#", USER_NUM_LEN) + gConfig.colors.border + VERTICAL_SINGLE
	         + gConfig.colors.colLabelText + centerText("User Name", USER_NAME_LEN) + gConfig.colors.border
	         + VERTICAL_SINGLE + gConfig.colors.colLabelText
	         + centerText("Location", LOCATION_LEN) + gConfig.colors.border
	         + VERTICAL_SINGLE + gConfig.colors.colLabelText + centerText("Last On", LAST_CALL_DATE_LEN)
	         + gConfig.colors.border + VERTICAL_SINGLE + gConfig.colors.colLabelText
	         + gConfig.colors.colLabelText + centerText("Via", CONNECTION_TYPE_LEN)
	         + gConfig.colors.border + VERTICAL_SINGLE + gConfig.colors.colLabelText
	         + centerText("# calls", NUM_CALLS_LEN) + gConfig.colors.border + VERTICAL_SINGLE;
	console.center(lineText);
	// Lower border lines for header
	// User number section
	var lineText = gConfig.colors.border + LEFT_T_SINGLE;
	for (var i = 0; i < USER_NUM_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// User name section
	for (var i = 0; i < USER_NAME_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// User location section
	for (var i = 0; i < LOCATION_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// Last call date section
	for (var i = 0; i < LAST_CALL_DATE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// Connection type section
	for (var i = 0; i < CONNECTION_TYPE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// # calls section
	for (var i = 0; i < NUM_CALLS_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += RIGHT_T_SINGLE;
	console.center(lineText);


	var listMode = UL_ALL; // List all users
	if (typeof(pListMode) === "number")
		listMode = pListMode;

	// If the user wants the user list sorted, then display a
	// sorted list; otherwise, just go through and display the
	// users.
	if (typeof(gConfig.sortUserList) === "boolean" && gConfig.sortUserList)
	{
		var userArray = getUserListArray(!(gConfig.excludeSysop), gConfig.displayQWKAccts, listMode);
		if (gConfig.sortType == "alpha")
			userArray.sort(userListSortByAlias);
		else if (gConfig.sortType == "lastOn")
			userArray.sort(userListSortByLastOnNewestFirst);
		for (var i = 0; i < userArray.length; ++i)
		{
			DisplayUserInfo(userArray[i]);
			userArray[i] = undefined; // Destructs the object now, rather than with 'delete'
		}
	}
	else
	{
		// Determine the last user number
		var lastuser = 1;
		if (system.lastuser == undefined)	// v3.10
			lastuser = system.stats.total_users;
		else							// v3.11
			lastuser = system.lastuser;
		// List the users
		var theUser = null;
		var userArray = null;
		for (var userNum = 1; userNum <= lastuser; ++userNum)
		{
			// If the user's name is blank, then skip it.
			if (system.username(userNum).length == 0)
				continue;

			// If the sysop is to be excluded, then skip user # 1 (user #1
			// should be the sysop).
			if (gConfig.excludeSysop && (userNum == 1))
				continue;

			// Load the user record
			theUser = new User(userNum);
			
			
			// If  this is a deleted or inactive user slot, then continue to
			// the next user.
			if (((theUser.settings & USER_DELETED) == USER_DELETED) || ((theUser.settings & USER_INACTIVE) == USER_INACTIVE))
			{
				theUser = undefined; // Destructs the object now, rather than with 'delete'
				continue;
			}

			// Skip this user if it's a guest account
			if (theUser.security.restrictions & UFLAG_G)
			{
				theUser = undefined; // Destructs the object now, rather than with 'delete'
				continue;
			}

			// Now, we should have a valid user account.

			// If gConfig.displayQWKAccts is false and this is a QWK account, then
			// skip it.
			if (!(gConfig.displayQWKAccts) && theUser.compare_ars("REST Q"))
			{
				theUser = undefined; // Destructs the object now, rather than with 'delete'
				continue;
			}

			// For list modes that are not UL_ALL
			// Users with access to cursub
			if (listMode == UL_SUB && !theUser.compare_ars(msg_area.sub[bbs.cursub_code].ars))
			{
				theUser = undefined;
				continue;
			}
			// Users with access to curdir
			else if (listMode == UL_DIR && !theUser.compare_ars(file_area.dir[bbs.curdir_code].ars))
			{
				theUser = undefined;
				continue;
			}

			// Display the user information.
			DisplayUserInfo(theUser);
			theUser = undefined; // Destructs the object now, rather than with 'delete'
		}
	}

	// Write the end border
	// User number section
	var lineText = gConfig.colors.border + LOWER_LEFT_SINGLE;
	for (var i = 0; i < USER_NUM_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// User name section
	for (var i = 0; i < USER_NAME_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// User location section
	for (var i = 0; i < LOCATION_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// Last call date section
	for (var i = 0; i < LAST_CALL_DATE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// Connection type section
	for (var i = 0; i < CONNECTION_TYPE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// # calls section
	for (var i = 0; i < NUM_CALLS_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += LOWER_RIGHT_SINGLE;
	console.center(lineText);
	console.print("\x01h\x01w" + system.stats.total_users + " \x01btotal users.\r\n");
	// Display the system's pause/"hit a key" prompt
	console.print("\x01p");
}
// Returns a string containing HTML to be displayed on the web.
//
// Parameters:
//  pTableClass: The class to use for the table.
function DDUserLister_GetHTML(pTableClass)
{
	var userListHTML = "";

	var userArray = getUserListArray(!(gConfig.excludeSysop), gConfig.displayQWKAccts);
	// If the user wants the user list sorted, then sort it
	if ((typeof(gConfig.sortUserList) == "boolean") && gConfig.sortUserList)
	{
		if (gConfig.sortType == "alpha")
			userArray.sort(userListSortByAlias);
		else if (gConfig.sortType == "lastOn")
			userArray.sort(userListSortByLastOnNewestFirst);
	}
	// Generate & return HTML
	if (userArray.length > 0)
	{
		userListHTML = "<table ";
		if ((pTableClass != null) && (typeof(pTableClass) != "undefined") && (pTableClass.length > 0))
			userListHTML += "class='" + pTableClass + "' ";
		// Columns: Alias, Location, Connection, Last On
		userListHTML += "cellspacing='1' cellpadding='2' summary='List of users for " + system.name + "'>\n"
							  + "<tr nowrap style='color: #ffffff; background-color: #0000bb; text-align: center;'>"
							  + "<th>Alias</th><th>Location</th><th>Connection</th><th>Last On</th></tr>\n";
		for (var i = 0; i < userArray.length; ++i)
		{
			// Append the row to the HTML.
			// Columns: Alias, Location, Connection, Last On
			userListHTML += "<tr nowrap style='background-color: #000000; text-align: center;'>"
							 // Show the user's handle with a link to their profile.
							 + "<td style='color: #00ffff;'><a href='/members/viewprofile.ssjs?showuser="
							 + userArray[i].number + "' style='color: #ffff00;'>" + userArray[i].alias + "</a></td>"
							 // Location
							 + "<td style='color: #ff00ff;'>" + userArray[i].location + "</td>"
							 // Connection
							 + "<td style='color: #0000ff;'>" + userArray[i].connection + "</td>"
							 // Last On
							 + "<td style='color: #ff0000; text-align: right;'>" + strftime("%Y-%m-%d", userArray[i].stats.laston_date) + "</td>"
							 + "</tr>\n";
		}
		userListHTML += "</table>\n";
	}
	else
		userListHTML = "<h3>There are currently no users on this system.</h3>";

	return userListHTML;
}



///////////////////////////////////////////////////////////////////////////////////
// General functions

// Returns an array of the users on the BBS
//
// Parameters:
//  pIncludeSysop: Whether or not to include the sysop
//  pIncludeQWKAccts: Whether or not to include QWK accounts
//  pListMode: The mode to list users with - From sbbsdefs:
//             UL_ALL: List all active users in user database
//             UL_SUB: List all users with access to cursub
//             UL_DIR: List all users with access to curdir
function getUserListArray(pIncludeSysop, pIncludeQWKAccts, pListMode)
{
	var listMode = UL_ALL; // List all users
	if (typeof(pListMode) === "number")
		listMode = pListMode;

	var userArray = [];

	// Determine the last user number
	var lastuser = 1;
	if (system.lastuser == undefined)	// v3.10
		lastuser = system.stats.total_users;
	else							// v3.11
		lastuser = system.lastuser;
	// Insert all the users into the array.
	var theUser = null;
	for (var userNum = 1; userNum <= lastuser; ++userNum)
	{
		// If the user's name is blank, then skip it.
		if (system.username(userNum).length == 0)
			continue;

		// If the sysop is to be excluded, then skip user # 1 (user #1
		// should be the sysop).
		if (!pIncludeSysop && (userNum == 1))
			continue;

		// Load the user record
		theUser = new User(userNum);

		// If  this is a deleted or inactive user slot, then don't
		// get this user.
		if (((theUser.settings & USER_DELETED) == USER_DELETED) ||
			 ((theUser.settings & USER_INACTIVE) == USER_INACTIVE))
		{
			theUser = undefined; // Destructs the object now, rather than with 'delete'
			continue;
		}

		// Skip this user if it's a guest account
		if (theUser.security.restrictions & UFLAG_G)
		{
			theUser = undefined; // Destructs the object now, rather than with 'delete'
			continue;
		}

		// Now, we should have a valid user account.

		// If this.displayQWKAccts is false and this is a QWK account, then
		// skip it.
		if (!pIncludeQWKAccts && theUser.compare_ars("REST Q"))
		{
			theUser = undefined; // Destructs the object now, rather than with 'delete'
			continue;
		}
		
		// For list modes that are not UL_ALL
		// Users with access to cursub
		if (listMode == UL_SUB && !theUser.compare_ars(msg_area.sub[bbs.cursub_code].ars))
		{
			theUser = undefined;
			continue;
		}
		// Users with access to curdir
		else if (listMode == UL_DIR && !theUser.compare_ars(file_area.dir[bbs.curdir_code].ars))
		{
			theUser = undefined;
			continue;
		}

		// Insert the user into the array
		userArray.push(theUser);

		// TODO: Should theUser be deleted here?
		theUser = undefined; // Destructs the object now, rather than with 'delete'
	}

	return userArray;
}

// Array sort function for sorting users alphabetically by alias
function userListSortByAlias(pA, pB)
{
	// Return -1, 0, or 1, depending on whether user A's alias
	// comes before, is equal to, or comes after user B's alias.
	var returnValue = 0;
	if (pA.alias < pB.alias)
		returnValue = -1;
	else if (pA.alias > pB.alias)
		returnValue = 1;
		
	return returnValue;
}

// Array sort function for sorting users by last time (oldest first)
function userListSortByLastOnOldestFirst(pA, pB)
{
	// Return -1, 0, or 1, depending on whether user A's alias
	// comes before, is equal to, or comes after user B's alias.
	var returnValue = 0;
	if (pA.stats.laston_date < pB.stats.laston_date)
		returnValue = -1;
	else if (pA.stats.laston_date > pB.stats.laston_date)
		returnValue = 1;
		
	return returnValue;
}

// Array sort function for sorting users by last time (newest first)
function userListSortByLastOnNewestFirst(pA, pB)
{
	// Return -1, 0, or 1, depending on whether user A's alias
	// comes before, is equal to, or comes after user B's alias.
	var returnValue = 0;
	if (pA.stats.laston_date < pB.stats.laston_date)
		returnValue = 1;
	else if (pA.stats.laston_date > pB.stats.laston_date)
		returnValue = -1;
		
	return returnValue;
}

// Centers some text within a specified field length.
//
// Parameters:
//  pText: The text to center
//  pFieldLen: The field length
function centerText(pText, pFieldLen)
{
	var centeredText = "";

	var textLen = strip_ctrl(pText).length;
	// If pFieldLen is less than the text length, then truncate the string.
	if (pFieldLen < textLen)
		centeredText = pText.substring(0, pFieldLen);
	else
	{
		// pFieldLen is at least equal to the text length, so we can
		// center the text.
		// Calculate the number of spaces needed to center the text.
		var numSpaces = pFieldLen - textLen;
		if (numSpaces > 0)
		{
			var rightSpaces = (numSpaces/2).toFixed(0);
			var leftSpaces = numSpaces - rightSpaces;
			// Build centeredText
			for (var i = 0; i < leftSpaces; ++i)
				centeredText += " ";
			centeredText += pText;
			for (var i = 0; i < rightSpaces; ++i)
				centeredText += " ";
		}
		else
		{
			// pFieldLength is the same length as the text.
			centeredText = pText;
		}
	}

	return centeredText;
}

// Removes multiple, leading, and/or trailing spaces
// The search & replace regular expressions used in this
// function came from the following URL:
//  http://qodo.co.uk/blog/javascript-trim-leading-and-trailing-spaces
//
// Parameters:
//  pString: The string to trim
//  pLeading: Whether or not to trim leading spaces (optional, defaults to true)
//  pMultiple: Whether or not to trim multiple spaces (optional, defaults to true)
//  pTrailing: Whether or not to trim trailing spaces (optional, defaults to true)
function trimSpaces(pString, pLeading, pMultiple, pTrailing)
{
	var leading = true;
	var multiple = true;
	var trailing = true;
	if(typeof(pLeading) != "undefined")
		leading = pLeading;
	if(typeof(pMultiple) != "undefined")
		multiple = pMultiple;
	if(typeof(pTrailing) != "undefined")
		trailing = pTrailing;

	// To remove both leading & trailing spaces:
	//pString = pString.replace(/(^\s*)|(\s*$)/gi,"");

	if (leading)
		pString = pString.replace(/(^\s*)/gi,"");
	if (multiple)
		pString = pString.replace(/[ ]{2,}/gi," ");
	if (trailing)
		pString = pString.replace(/(\s*$)/gi,"");

	return pString;
}
