{
                                  SBDSP
                          Version 1.03 (9/23/94)
                         Written by Ethan Brodsky
          Copyright 1994 by Ethan Brodsky.  All rights reserved.

This library is distributed AS IS.  The author specifically disclaims
any responsibility for any loss of profit or any consequential,
incidental, or other damages.  SBDSP is freeware and is distributed
with full Turbo Pascal source code.  You are free to incorporate parts
of the code into your own programs as long as you give credit to Ethan
Brodsky.  This source code may only be distributed in it's original
form, including this documentation file.

------------------------------------------------------------------------

    You may have used my SBVox and SBVoice units.  They played VOC files
on a Sound Blaster using Creative Labs' CT-VOICE driver.  Since they
used the CT-VOICE driver, they wouldn't work on other sound cards.  The
driver needed to be included with the program, either in a separate file
or linked into the executable.

    SBDSP performs the same functions as the SBVox unit without using
the CT-VOICE driver.  It has only been tested on a SB16 and PAS16, but
it should work on all Sound Blaster compatible sound cards.  By using
DMA transfers, it plays sound without using the CPU, saving processor
cycles for your program.

    I have many improvements planned, including 16-bit sound, stereo
effects, and mixing, along with new library for FM music.  But I NEED
FEEDBACK!  If you use my code, tell me about it!  If you make any
modifications, send them to me!  If you have any suggestions for
improvements, tell me about them!  If you want me to write a C version,
or a version to play WAV files, tell me!

   You don't need to pay me for using this unit.  All you have to do
is put my name in the credits for your product.  I'd appreciate it if
you'd send me a message telling me how you used SBDSP.  (If you used
it in a game, tell me where I can get it)  And finally, if you ever
have a sound programming job, think of me.

    You can find out most of the things you need to know in order to
use this library by looking at the PlayVOC example program, but I'll
go over it again.  The first thing you need to do is to reset the DSP,
initialize SBDSP's internal variables, and install the interrupt
handler.  In order to do this, you need to know the sound cards base
address, IRQ number, and 8-bit DMA channel.   If this is being used
on a Sound Blaster, this information can be obtained from the BLASTER
environment variable.  I don't know whether other cards use this.  You
can use the EnvironmentSet function to find out if the environment
variable is set.  If it is, you can call the function InitSBFromEnv.
Otherwise, you'll have to find out the settings some other way and pass
them to the InitSB function.

    Use the LoadVOCFile function to allocate a sound buffer.  Make sure
that you save the value returned from this function.  It is the size of
the allocated buffer.  It will be needed when you deallocate the buffer.
The memory needed for Sound will be allocated inside this function. You
do NOT need to allocate it beforehand.

    Before you can play any sounds, you have to turn on the speaker
output.  Do this by calling TurnSpeakerOn.  Make sure you turn it off
at the end of the program.  If you want to install a marker handler,
make sure you do it now by calling SetMarkerProc.  A marker handler
will be called each time a marker block is reached.  Before you install
your marker handler, save the old one using GetMarkerProc.  If the value
returned is not nil, then another marker procedure has been installed.
Call it each time your marker procedure is called.  This is a good
practice to get into when setting up a handler such as this.  It will
make it possible to install more than one marker procedure.

    To play a sound, pass a pointer to the sound buffer to PlaySound.
Any sound output in progress will be stopped.  To find out if the sound
is finished, check the SoundPlaying variable.  The VOC file format has
a provision for repeating sounds.  The sound can be set to repeat for
a number of times (Or forever)  You can break out of the loop by calling
BreakLoop.  The current iteration will finish and it will continue to
the next block.  When the program is completely finished playing sound,
call the ShutDownSB procedure.  This will stop any sound output in
progress and remove the interrupt handler.  You should deallocate all
sound buffers by using FreeBuffer.  The pointer to the buffer should be
typecasted as a pointer.  Make sure that you pass the buffer size that
was returned by LoadVOCFile so that the right amount of memory is
deallocated.

    This library will not allow you to play 16 bit or stereo VOC files.
It will not work in protected mode since it uses DMA transfers.  If you
have any other questions, feel free to ask.  If you would like me to
make any modifications or a customized version of this unit to use in
your program, contact me and we can work out some arrangements.

There are several ways to contact me:
    E-Mail:  ericbrodsky@psl.wisc.edu    (Preferred)
    Phone:   (608) 238-4830
    Mail:
        Ethan Brodsky
      4010 Cherokee Dr.
      Madison, WI 53711

Bug fixes and other announcements will be posted in:
    comp.lang.pascal
    comp.sys.ibm.pc.soundcard
    comp.sys.ibm.pc.soundcard.tech
    rec.games.programmer
}

{       SBDSP is Copyright 1994 by Ethan Brodsky.  All rights reserved.      }
{$I DEFINES.INC}
{.$A+,B-,D+,E+,F+,G+,I-,L+,N-,O+,P-,Q-,R-,S-,T-,V+,X+}
UNIT SBDSP;

INTERFACE

USES VOICE;

CONST
  On  = TRUE;
  Off = FALSE;

TYPE Proc = PROCEDURE;

FUNCTION InitSB(IRQ : BYTE; BaseIO : WORD; DMAChannel : BYTE) : BOOLEAN;
  {This function must be called before any sound is played.  It will }
  {initialize internal variables, reset the DSP chip, and install the}
  {interrupt handler.                                                }
  {IRQ:           The sound card's IRQ setting (Usually 5 or 7)      }
  {BaseIO:        The sound card's base IO address (Usually $220)    }
  {DMAChannel:    The sound card's 8-bit DMA channel (Usually 1)     }
  {Returns:                                                          }
  {    TRUE:      Sound card initialized correctly                   }
  {    FALSE:     Error initializing sound card                      }
FUNCTION EnvironmentSet : BOOLEAN;
  {Returns:                                                          }
  {    TRUE:  The BLASTER environment variable is set                }
  {    FALSE: The BLASTER environment variable isn't set             }
FUNCTION InitSBFromEnv : BOOLEAN;
  {This function initializes the sound card from the settings stored }
  {in the BLASTER environment variable.  I'm not sure if all sound   }
  {cards use the enviroment variable.                                }
  {Returns:                                                          }
  {    TRUE:  Environment variable found and sound card initialized  }
  {    FALSE: Environment variable not set or error initializing card}
PROCEDURE ShutDownSB;
  {This procedure must be called at the end of the program.  It stops}
  {sound output, removes the interrupt handler, and restores the old }
  {interrupt handler.                                                }
PROCEDURE InstallHandler;
  {This procedure will reinstall the interrupt handler.              }
PROCEDURE UninstallHandler;
  {This procedure will remove the interrupt handler.  You should not }
  {need to call this.  If you do, sound output won't work until the  }
  {handler is reinstalled.                                           }
FUNCTION ResetDSP : BOOLEAN;
  {This function resets the sound card's DSP chip.                   }
  {Returns:                                                          }
  {    TRUE:    The sound card's DSP chip was successfully reseted   }
  {    FALSE:   The chip couldn't be initialized (Don't use it)      }
FUNCTION GetDSPVersion : STRING;
  {This function returns a string containing the DSP chip version.   }
PROCEDURE TurnSpeakerOn;
  {This procedure turns on the speaker.  This should be called before}
  {a sound is played, but after the sound card is initialized.       }
PROCEDURE TurnSpeakerOff;
  {Turn off the speaker so that sound can't be heard.  You should do }
  {this when your program is finished playing sound.                 }
FUNCTION GetSpeakerState : BOOLEAN;
  {Returns the state of the speaker.  Only works on SBPro and higher.}
  {Returns:                                                          }
  {    TRUE:    Speaker is on                                        }
  {    FALSE:   Speaker is off                                       }
PROCEDURE PlaySound(Sound : PSound);
  {Stops any sound in progress and start playing the sound specified.}
  {Sound:       Pointer to buffer that the VOC file was loaded into  }
PROCEDURE PauseSound;
  {Pauses the sound output in progress.                              }
PROCEDURE ContinueSound;
  {Continues sound output stopped by Pause.                          }
PROCEDURE BreakLoop;
  {Stops the loop at the end of the current iteration and continues  }
  {with the next block.                                              }
PROCEDURE SetMarkerProc(MarkerProcedure : POINTER);
  {Installs a marker handler.  Each time a marker block is reached,  }
  {the procedure specified is called.  Before installing a handler,  }
  {you should store the old handler.  Your handler should also call  }
  {the old handler.  Look in the example program to see how this is  }
  {done.                                                             }
  {MarkerProcedure:  Pointer to the marker procedure                 }
PROCEDURE GetMarkerProc(VAR MarkerProcedure : POINTER);
  {Gets the current marker procedure.                                }
  {MarkerProcedure:  Current marker procedure (nil if none)          }

VAR
  SoundPlaying  : BOOLEAN;
  Looping       : BOOLEAN;
  UnknownBlock  : BOOLEAN;
  UnplayedBlock : BOOLEAN;
  LastMarker    : WORD;

IMPLEMENTATION

USES {CRT,} DOS, MEM, FGMISC;

CONST
   {DSP Commands}
    CmdDirectDAC       = $10;
    CmdNormalDMADAC    = $14;
    Cmd2BitDMADAC      = $16;
    Cmd2BitRefDMADAC   = $17;
    CmdDirectADC       = $20;
    CmdNormalDMAADC    = $24;
    CmdSetTimeConst    = $40;
    CmdSetBlockSize    = $48;
    Cmd4BitDMADAC      = $74;
    Cmd4BitRefDMADAC   = $75;
    Cmd26BitDMADAC     = $76;
    Cmd26BitRefDMADAC  = $77;
    CmdSilenceBlock    = $80;
    CmdHighSpeedDMADAC = $91;
    CmdHighSpeedDMAADC = $99;
    CmdHaltDMA         = $D0;
    CmdSpeakerOn       = $D1;
    CmdSpeakerOff      = $D3;
    CmdGetSpeakerState = $D8;
    CmdContinueDMA     = $D4;
    CmdGetVersion      = $E1;
    DACCommands : ARRAY[0..3] OF BYTE = (CmdNormalDMADAC, Cmd4BitDMADAC, Cmd26BitDMADAC, Cmd2BitDMADAC);

VAR
    ResetPort    : WORD;
    ReadPort     : WORD;
    WritePort    : WORD;
    PollPort     : WORD;

    PICPort      : BYTE;
    IRQStartMask : BYTE;
    IRQStopMask  : BYTE;
    IRQIntVector : BYTE;
    IRQHandlerInstalled : BOOLEAN;

    DMAStartMask : BYTE;
    DMAStopMask  : BYTE;
    DMAModeReg   : BYTE;

    OldIntVector : POINTER;
   {OldExitProc  : POINTER;}

    MarkerProc   : POINTER;

VAR
    VoiceStart     : LONGINT;
    CurPos         : LONGINT;
    CurPageEnd     : LONGINT;
    VoiceEnd       : LONGINT;
    LeftToPlay     : LONGINT;
    TimeConstant   : BYTE;
    SoundPacking   : BYTE;
    CurDACCommand  : BYTE;

    LoopStart      : PBlock;
    LoopsRemaining : WORD;
    EndlessLoop    : BOOLEAN;

    SilenceBlock   : BOOLEAN;

    CurBlock       : PBlock;
    NextBlock      : PBlock;

PROCEDURE EnableInterrupts;
INLINE($FB); {STI}

PROCEDURE DisableInterrupts;
INLINE($FA); {CLI}

PROCEDURE WriteDSP(Value : BYTE);
INLINE($8B / $16 / > WritePort / {MOV   DX, WritePort (Variable)  }
       $EC /                     {IN    AL, DX                    }
       $24 / $80 /               {AND   AL, 80h                   }
       $75 / $FB /               {JNZ   -05                       }
       $58 /                     {POP   AX                        }
       $8B / $16 / > WritePort / {MOV   DX, WritePort (Variable)  }
       $EE);                     {OUT   DX, AL                    }

FUNCTION ReadDSP : BYTE;
INLINE($8B / $16 / > PollPort / {MOV   AL, PollPort  (Variable)  }
       $EC /                    {IN    AL, DX                    }
       $24 / $80 /              {AND   AL, 80h                   }
       $74 / $FB /              {JZ    -05                       }
       $8B / $16 / > ReadPort / {MOV   DX, ReadPort  (Variable)  }
       $EC);                    {IN    AL,DX                     }

FUNCTION InitSB(IRQ : BYTE; BaseIO : WORD; DMAChannel : BYTE) : BOOLEAN;
CONST
  IRQIntNums : ARRAY[0..15] OF BYTE = ($08,$09,$0A,$0B,$0C,$0D,$0E,$0F,
                                       $70,$71,$72,$73,$74,$75,$76,$77);
VAR
  Success    : BOOLEAN;
BEGIN
  IF IRQ <= 7 THEN PICPort := $21   {INTC1}
              ELSE PICPort := $A1;  {INTC2}
  IRQIntVector := IRQIntNums[IRQ];
  IRQStopMask  := 1 SHL (IRQ MOD 8);
  IRQStartMask := NOT(IRQStopMask);

  ResetPort := BaseIO + $6;
  ReadPort  := BaseIO + $A;
  WritePort := BaseIO + $C;
  PollPort  := BaseIO + $E;

  DMAStartMask := DMAChannel + $00; {000000xx}
  DMAStopMask  := DMAChannel + $04; {000001xx}
  DMAModeReg   := DMAChannel + $48; {010010xx}

  Success := ResetDSP;
  IF Success THEN InstallHandler;
  InitSB := Success;
END;

FUNCTION EnvironmentSet : BOOLEAN;
BEGIN
  EnvironmentSet := GetEnv('BLASTER') <> '';
END;

FUNCTION GetSetting(BLASTER : STRING; Letter : CHAR; Hex : BOOLEAN; VAR Value : WORD) : BOOLEAN;
VAR
  EnvStr    : STRING;
  NumStr    : STRING;
  ErrorCode : INTEGER;
BEGIN
  EnvStr := BLASTER + ' ';
  DELETE(EnvStr,1,POS(Letter,EnvStr));
  NumStr := COPY(EnvStr,1,POS(' ',EnvStr) - 1);
  IF Hex THEN VAL('$' + NumStr,Value,ErrorCode)
         ELSE VAL(NumStr,Value,ErrorCode);
  IF ErrorCode <> 0 THEN GetSetting := FALSE
                    ELSE GetSetting := TRUE;
END;

FUNCTION GetSettings(VAR BaseIO, IRQ, DMAChannel : WORD) : BOOLEAN;
VAR
  EnvStr : STRING;
  I      : BYTE;
BEGIN
  EnvStr := GetEnv('BLASTER');
  FOR I := 1 TO LENGTH(EnvStr) DO EnvStr[I] := UPCASE(EnvStr[I]);
  GetSettings := TRUE;
  IF EnvStr = '' THEN GetSettings := FALSE ELSE BEGIN
    IF NOT(GetSetting(EnvStr,'A',TRUE,BaseIO)) THEN GetSettings := FALSE;
    IF NOT(GetSetting(EnvStr,'I',FALSE,IRQ)) THEN GetSettings := FALSE;
    IF NOT(GetSetting(EnvStr,'D',FALSE,DMAChannel)) THEN GetSettings := FALSE;
  END;
END;

FUNCTION InitSBFromEnv : BOOLEAN;
VAR
  IRQ,BaseIO,DMAChannel : WORD;
BEGIN
  IF GetSettings(BaseIO,IRQ,DMAChannel) THEN InitSBFromEnv := InitSB(IRQ,BaseIO,DMAChannel)
                                        ELSE InitSBFromEnv := FALSE;
END;

PROCEDURE ShutDownSB;
BEGIN
  ResetDSP;
  UninstallHandler;
END;

FUNCTION ResetDSP : BOOLEAN;
VAR
  I : BYTE;
BEGIN
  Port[ResetPort] := 1;
 {DELAY(4);}
  FG_WaitFor(1);
  Port[ResetPort] := 0;
  I := 1;
  WHILE (ReadDSP <> $AA) AND (I < 100) DO INC(I);
  IF I < 100 THEN ResetDSP := TRUE ELSE ResetDSP := FALSE;
END;

FUNCTION GetDSPVersion : STRING;
VAR
  MajorByte,MinorByte : BYTE;
  MajorStr,MinorStr : STRING;
BEGIN
  WriteDSP(CmdGetVersion);
  MajorByte := ReadDSP; STR(MajorByte, MajorStr);
  MinorByte := ReadDSP; STR(MinorByte, MinorStr);
  GetDSPVersion := MajorStr + '.'  + MinorStr;
END;

PROCEDURE TurnSpeakerOn;
BEGIN
  WriteDSP(CmdSpeakerOn);
END;

PROCEDURE TurnSpeakerOff;
BEGIN
  WriteDSP(CmdSpeakerOff);
END;

FUNCTION GetSpeakerState : BOOLEAN;
VAR
  SpeakerByte : BYTE;
BEGIN
  WriteDSP(CmdGetSpeakerState);
  SpeakerByte := ReadDSP;
  IF SpeakerByte = 0 THEN GetSpeakerState := Off ELSE GetSpeakerState := On;
END;

PROCEDURE StartDMADSP;
VAR
  Page          : BYTE;
  Offset        : WORD;
  Length        : WORD;
  NextPageStart : LONGINT;
BEGIN
  Page   := CurPos SHR 16;
  Offset := CurPos MOD 65536;
  IF VoiceEnd < CurPageEnd THEN LENGTH := LeftToPlay - 1
                           ELSE LENGTH := CurPageEnd - CurPos;

  INC(CurPos,LONGINT(LENGTH) + 1);
  DEC(LeftToPlay,LONGINT(LENGTH) + 1);
  INC(CurPageEnd,65536);

  WriteDSP(CmdSetTimeConst);
  WriteDSP(TimeConstant);
  Port[$0A] := DMAStopMask;
  Port[$0C] := $00;
  Port[$0B] := DMAModeReg;
  Port[$02] := LO(Offset);
  Port[$02] := HI(Offset);
  Port[$03] := LO(LENGTH);
  Port[$03] := HI(LENGTH);
  Port[$83] := Page;
  Port[$0A] := DMAStartMask;
  WriteDSP(CurDACCommand);
  WriteDSP(LO(LENGTH));
  WriteDSP(HI(LENGTH));
END;

PROCEDURE CallMarkerProc;
BEGIN
  IF MarkerProc <> NIL THEN Proc(MarkerProc);
END;

FUNCTION HandleBlock(Block : PBlock) : BOOLEAN;
BEGIN
  HandleBlock := FALSE;
  CASE Block^.BlockType OF
    EndBlockNum :
    BEGIN
      SoundPlaying := FALSE;
      HandleBlock  := TRUE;
    END;
    VoiceBlockNum :
    BEGIN
      VoiceStart    := GetAbsoluteAddress(Block) + 6;
      CurPageEnd    := ((VoiceStart SHR 16) SHL 16) + 65536 - 1;
      LeftToPlay    := BlockSize(Block) - 6;
      VoiceEnd      := VoiceStart + LeftToPlay;
      CurPos        := VoiceStart;
      TimeConstant  := PVoiceBlock(Block)^.SR;
      SoundPacking  := PVoiceBlock(Block)^.Packing;
      CurDACCommand := DACCommands[SoundPacking];
      StartDMADSP;
      HandleBlock   := TRUE;
    END;
    VoiceContinueBlockNum :
    BEGIN
      VoiceStart := GetAbsoluteAddress(Block) + 4;
      LeftToPlay := BlockSize(Block) - 4;
      VoiceEnd   := VoiceStart + LeftToPlay;
      CurPos     := VoiceStart;
      StartDMADSP;
      HandleBlock := TRUE;
    END;
    SilenceBlockNum :
    BEGIN
      SilenceBlock := TRUE;
      WriteDSP(CmdSetTimeConst);
      WriteDSP(PSilenceBlock(Block)^.SR);
      WriteDSP(CmdSilenceBlock);
      WriteDSP(LO(PSilenceBlock(Block)^.Duration + 1));
      WriteDSP(HI(PSilenceBlock(Block)^.Duration + 1));
      HandleBlock := TRUE;
    END;
    MarkerBlockNum :
    BEGIN
      LastMarker := PMarkerBlock(Block)^.Marker;
      CallMarkerProc;
    END;
    MessageBlockNum : BEGIN
                      END;
    RepeatBlockNum :
    BEGIN
      LoopStart := NextBlock;
      LoopsRemaining := PRepeatBlock(Block)^.Count + 1;
      IF LoopsRemaining = 0 {Wrapped around from $FFFF} THEN EndlessLoop := TRUE
                                                        ELSE EndlessLoop := FALSE;
      Looping := TRUE;
    END;
    RepeatEndBlockNum :
    BEGIN
      IF NOT (EndlessLoop) THEN BEGIN
        DEC(LoopsRemaining);
        IF LoopsRemaining = 0 THEN BEGIN
          Looping := FALSE;
          EXIT;
        END;
      END;
      NextBlock := LoopStart;
    END;
    NewVoiceBlockNum :
    BEGIN
      IF (PNewVoiceBlock(Block)^.Mode = NewStereo) OR (PNewVoiceBlock(Block)^.BitsPerSample = 16)
      THEN UnplayedBlock := TRUE ELSE BEGIN
        VoiceStart    := GetAbsoluteAddress(Block) + 16;
        CurPageEnd    := ((VoiceStart SHR 16) SHL 16) + 65536 - 1;
        LeftToPlay    := BlockSize(Block) - 16;
        VoiceEnd      := VoiceStart + LeftToPlay;
        CurPos        := VoiceStart;
        TimeConstant  := GetSRByte(PNewVoiceBlock(Block)^.SamplingRate);
        SoundPacking  := PNewVoiceBlock(Block)^.Compression;
        CurDACCommand := DACCommands[SoundPacking];
        StartDMADSP;
        HandleBlock := TRUE;
      END;
    END;
    ELSE UnknownBlock := TRUE;
  END;
END;

PROCEDURE ProcessBlocks;
BEGIN
  REPEAT
    CurBlock  := NextBlock;
    NextBlock := FindNextBlock(POINTER(CurBlock));
  UNTIL HandleBlock(CurBlock);
END;

PROCEDURE ClearInterrupt;
VAR
  Temp : BYTE;
BEGIN
  Temp      := Port[PollPort];
  Port[$20] := $20;
END;

PROCEDURE IntHandler; INTERRUPT;
BEGIN
  IF SilenceBlock {Interrupted because a silence block ended} THEN BEGIN
    SilenceBlock := FALSE;
    ProcessBlocks;
  END ELSE {Interrupted because a DMA transfer was completed}
  IF LeftToPlay <> 0 THEN StartDMADSP ELSE ProcessBlocks;
  ClearInterrupt;
END;

PROCEDURE PlaySound(Sound : PSound);
BEGIN
  PauseSound;
  NextBlock      := PBlock(Sound);
  SoundPlaying   := TRUE;
  Looping        := FALSE;
  LastMarker     := 0;
  UnknownBlock   := FALSE;
  UnplayedBlock  := FALSE;

  LoopStart      := NIL;
  LoopsRemaining := 0;
  EndlessLoop    := FALSE;

  ProcessBlocks;
END;

PROCEDURE PauseSound;
BEGIN
  WriteDSP(CmdHaltDMA);
END;

PROCEDURE ContinueSound;
BEGIN
  WriteDSP(CmdContinueDMA);
END;

PROCEDURE BreakLoop;
BEGIN
  LoopsRemaining := 1;
  EndlessLoop    := FALSE;
END;

PROCEDURE StopSBIRQ;
BEGIN
  Port[PICPort] := Port[PICPort] OR IRQStopMask;
END;

PROCEDURE StartSBIRQ;
BEGIN
  Port[PICPort] := Port[PICPort] AND IRQStartMask;
END;

PROCEDURE InstallHandler;
BEGIN
  DisableInterrupts;
  StopSBIRQ;
  GETINTVEC(IRQIntVector,OldIntVector);
  SETINTVEC(IRQIntVector,@IntHandler);
  StartSBIRQ;
  EnableInterrupts;
  IRQHandlerInstalled := TRUE;
END;

PROCEDURE UninstallHandler;
BEGIN
  DisableInterrupts;
  StopSBIRQ;
  SETINTVEC(IRQIntVector,OldIntVector);
  EnableInterrupts;
  IRQHandlerInstalled := FALSE;
END;

PROCEDURE SetMarkerProc(MarkerProcedure : POINTER);
BEGIN
  MarkerProc := MarkerProcedure;
END;

PROCEDURE GetMarkerProc(VAR MarkerProcedure : POINTER);
BEGIN
  MarkerProcedure := MarkerProc;
END;

{PROCEDURE SBDSPExitProc; Far;
BEGIN
  ExitProc := OldExitProc;
  ResetDSP;
  IF (IRQHandlerInstalled = TRUE) THEN UninstallHandler;
END;}

BEGIN
  MarkerProc   := NIL;
 {OldExitProc  := ExitProc;
  ExitProc     := @SBDSPExitProc;}
  SoundPlaying := FALSE;
END. 
