/*********************************************************************
** FILENAME: tg2ra.c                        VERSION: 1.00
**
** DESCRIPTION: Convert a Telegard 2.7 USER.LST file to a
**              RemoteAccess v2.0x USERS.BBS file
**
** NOTES: N/A
**
** AUTHOR: John Kristoff                START DATE: 04/21/94
**         Internet: jkristof@xroads.chigate.com
**                   jkristof@mica.meddean.luc.edu
**         FidoNet:  1:115/743
**
** VERSION  DATE     WHO  DETAIL
** 1.00     24Apr94  JK   Initial design and coding
**
**      Copyright John Kristoff, 1994.  All rights reserved.
**      You may use this program or any part there-of as desired
**      without restriction.
*/

#include <stdio.h>                      /* Standard i/o */
#include <stdlib.h>                     /* Standard library */
#include "telegard.h"                   /* Telegard structures */
#include "ra.h"                         /* RemoteAccess structures */

#define VERS       "1.00"               /* Program version */
#define FILEPATH   81                   /* Length of path + file + '\0' */

int main( int argc, char * argv[] );
void FixPath( char * OldPath, char * filename );
char * Tgstr2c( char * Tgstr );

int
main( int argc, char * argv[] )
{
    char TgUserlist[FILEPATH] = "USER.LST";
    char RaUserlist[FILEPATH] = "USERS.BBS";

    userrec urec;                       /* User record in USER.LST */
    USERS * users;                      /* User record in USERS.BBS */
    FILE * tgfp;                        /* Telegard USER.LST */
    FILE * rafp;                        /* RemoteAccess USERS.BBS */

    fprintf( stdout, "\n"
             "BYE-BYE Telegard v" VERS ", " __DATE__ ".\n"
              "Telegard USER.LST to RemoteAccess USERS.BBS convertor.\n"
              "Copyright John Kristoff, 1994.  All rights reserved.\n"
              "\n" );

    if( argc > 1 )
    {
        strncpy( TgUserlist, argv[1], FILEPATH - 1 );
        FixPath( TgUserlist, "USER.LST" );
    }

    if( (tgfp = fopen(TgUserlist, "rb")) == NULL )
    {
        fprintf( stderr, "ERROR (%d): Cannot open %s\n",
                 __LINE__ , TgUserlist );
        exit( EXIT_FAILURE );
    }

    if( argc > 2 )
    {
        strncpy( RaUserlist, argv[2], FILEPATH - 1 );
        FixPath( RaUserlist, "USERS.BBS" );
    }

    if( (rafp = fopen(RaUserlist, "wb")) == NULL )
    {
        fprintf( stderr, "ERROR (%d): Cannot open %s\n",
                 __LINE__, RaUserlist );
        exit( EXIT_FAILURE );
    }

    fprintf( stdout, "Processing..." );

    if( fread(&urec, sizeof(urec), 1, tgfp) == NULL )
    {
       fprintf( stderr, "ERROR (%d): Cannot read %s\n",
               __LINE__, TgUserlist );
       exit( EXIT_FAILURE );
    }

    while( !feof(tgfp) )
    {
        if( urec.flags.udeleted != 1 )
        {
            char Date[9] = "";          /* TG to RA date string conversion */

            users = (USERS *) calloc( 1, sizeof(USERS) );

            /* --- Let's get to it! --- */

            strncpy( users->name.string, Tgstr2c(urec.realname), 35 );
            users->name.l1 = strlen( users->name.string );

            strncpy( users->location, Tgstr2c(urec.citystate), 25 );
            users->l1 = strlen( users->location );

            strncpy( users->address1, Tgstr2c(urec.street), 50 );
            users->l3 = strlen( users->address1 );

            strncpy( users->address2, Tgstr2c(urec.citystate), 50 );
            users->l4 = strlen( users->address2 );

            strncpy( users->address3, Tgstr2c(urec.zipcode), 50 );
            users->l5 = strlen( users->address3 );

            strncpy( users->handle, Tgstr2c(urec.name), 35 );
            users->l6 = strlen( users->handle );

            strncpy( users->comment, Tgstr2c(urec.pw), 80 );
            users->l7 = strlen( users->comment );

            strncpy( users->voicephone, Tgstr2c(urec.ph), 15 );
            users->l9 = strlen( users->voicephone );

            strncpy( Date, Tgstr2c(urec.laston), 9 );
            Date[2] = '-';
            Date[5] = '-';
            users->lastdate.l1 = strlen( Date );
            strncpy( users->lastdate.Date, Date, 8 );

            users->attribute.ClearScreen = urec.flags.clsmsg;
            users->attribute.MorePrompt = urec.ac.pause;
            users->attribute.Ansi = urec.ac.ansi;
            users->attribute.NoKill = urec.ac.fnodeletion;
            users->Attribute2.HotKeys = urec.ac.onekey;
            users->Attribute2.Avatar = urec.ac.avatar;

            users->credit = urec.credit;

            users->security = urec.sl;
            users->msgsposted = urec.pubpost + urec.privpost + urec.netpost;
            users->nocalls = urec.loggedon;

            users->uploads = urec.uploads;
            users->downloads = urec.downloads;
            users->uploadsk = urec.uk;
            users->downloadsk = urec.dk;

            users->screenlength = urec.pagelen;

            strncpy( Date, Tgstr2c(urec.firston), 9 );
            Date[2] = '-';
            Date[5] = '-';
            users->firstdate.l1 = strlen( Date );
            strncpy( users->firstdate.Date, Date, 8 );

            strncpy( Date, Tgstr2c(urec.bday), 9 );
            Date[2] = '-';
            Date[5] = '-';
            users->birthdate.l1 = strlen( Date );
            strncpy( users->birthdate.Date, Date, 8 );

            users->screenwidth = urec.linelen;

            if( urec.sex == 'M' )
            {
                users->sex = 1;
            }
            else
            {
                users->sex = 2;
            }

            if( fwrite( users, sizeof(USERS), 1, rafp) == NULL )
            {
                fprintf( stderr, "ERROR (%d): Cannot write %s\n",
                        __LINE__, RaUserlist );
                exit( EXIT_FAILURE );
            }

            free(users);
        }

        fread(&urec, sizeof(userrec), 1, tgfp);
    }

    fclose( rafp );
    fclose( tgfp );

    fprintf( stdout, "Done!\n" );
    return( EXIT_SUCCESS);
}


/*********************************************************************
** Function: FixPath
**
**   Parameters: A DOS path
**               Filename to concatenate to path
**
**   Returns: N/A
**
** Description: Tack on a filename to a path but first check for a
**              trailing '\\' in the path specified and add one if
**              necessary.
*/

void
FixPath( char * Path, char * Filename )
{
    if( Path[strlen(Path) - 1] != '\\' )
    {
          strcat( Path, "\\" );
          strcat( Path, Filename );
    }
    else
    {
          strcat( Path, Filename );
    }
}



/*********************************************************************
** Function: Tgstr2c
**
**   Parameters:  Telegard formatted string.
**
**      Returns:  C formatted string.
**
**  Description:  Telegard data files contain "heart codes" in it's
**                character strings that Telegard uses to determine
**                the color of the text to follow.  These codes are
**                useless outside of Telegard (unless we want to
**                interpret them), so here we strip those codes out.
*/

char *
Tgstr2c( char * Tgstr )
{
    int register Length = Tgstr[0];    /* Pascal string length is here */
    static char Cstring[256];          /* New C string w/o heart codes */
    register int t = 1;                /* Character in Tgstr */
    register int c = 0;                /* Character in Cstring */

    while( t <= Length )
    {
        if( Tgstr[t] == 0x03 || Tgstr[t] == '^')
        {
            ++t;                       /* Skip over heart code crap */
        }
  
        else
        {
            Cstring[c++] = Tgstr[t];
        }

        ++t;                           /* Goto next char in string */
    }

    Cstring[c] = '\0';                 /* NULL terminate string */

    return( Cstring );
}
