// General, miscellaneous functions, Y2K
// Created 09 September 1995
// Revised 19 June 1998

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <share.h>
#include <dos.h>
#include <ctype.h>
#include <direct.h>
#include <io.h>
#include <fcntl.h>
#include <sys\stat.h>
#include "utility.h"
#include "config.h"
#include "msgs.h"
#include "users.h"
#include "fileopts.h"
#include "extern.h"

extern class Config Config_obj;

/************************************************************************/
/* Toggle the specified bit on/off */
void Utility::togglebit(unsigned *i, unsigned bit)
{
    *i = *i ^ bit;
}

/************************************************************************/
/* Determine if the specified bit is on or off */
int Utility::bitset(unsigned i, unsigned bit)
{
    unsigned tmp;

    tmp = i;
    togglebit(&tmp, bit);
    if(tmp < i)
    {
        return(TRUE);
    }
    return(FALSE);
}

/************************************************************************/
/* Turn the specified bit ON */
void Utility::setbit(unsigned *i, unsigned bit)
{
    *i = *i | bit;
}

/************************************************************************/
/* Turn the specified bit OFF */
void Utility::clearbit(unsigned *i, unsigned bit)
{
    unsigned tmp;

    tmp = *i;
    togglebit(&tmp, bit);
    if(tmp < *i)
    {
        *i = tmp;
    }
}

/***********************************************************************/
/* Return the archive format of the input file */
int Utility::get_arctype(char *inname)
{
    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
    short arctype;
/*
    char header[129], fn[129];
    short x;
    FILE *arcfile;
*/
    _splitpath(inname, drive, dir, fname, ext);

    /* If it is not an archive, UNKNOWN will be returned */
    arctype = UNKNOWN;

   /* If this is an icon file, get out of here now or it will be flagged as
      an ARC file instead
    if(stricmp(ext, ".ICN") == 0)
    {
        return(arctype);
    }
    if(stricmp(ext, ".DLL") == 0)
    {
        return(arctype);
    }
    if(stricmp(ext, ".OBJ") == 0)
    {
        return(arctype);
    }
*/

    if(stricmp(ext, ".ARC") == 0)
    {
        arctype = ARC;
    }
    if(stricmp(ext, ".ARJ") == 0)
    {
        arctype = ARJ;
    }
    if(stricmp(ext, ".HAP") == 0)
    {
        arctype = HAP;
    }
    if(stricmp(ext, ".LZH") == 0)
    {
        arctype = LZH;
    }
    if(stricmp(ext, ".PAK") == 0)
    {
        arctype = PAK;
    }
    if(stricmp(ext, ".SQZ") == 0)
    {
        arctype = SQZ;
    }
    if(stricmp(ext, ".ZIP") == 0)
    {
        arctype = ZIP;
    }
    if(stricmp(ext, ".ZOO") == 0)
    {
        arctype = ZOO;
    }
    if(stricmp(ext, ".UC2") == 0)
    {
        arctype = UC2;
    }

    if(stricmp(ext, ".RAR") == 0)
    {
        arctype = RAR;
    }

/*
    if(stricmp(ext, ".EXE") == 0)
    {
        if(is_sfx(inname) == SFX)
        {
            arctype = SFX;
        }
    }
*/
    /* If it doesn't recognize the extension, check the archive header
    if(arctype == -1)
    {
        sprintf(fn, "%s", inname);
        arcfile = _fsopen(fn, "rb", SH_DENYWR);
        if(arcfile)
        {
            fread(header, sizeof(header) - 1, 1, arcfile);
            header[sizeof(header)-1] = NULL;
            fclose(arcfile);

            /* Check for LZH
            if (header[2] == '-' && header[3] == 'l' && header[4] == 'h'
                && header[6] == '-')
            {
                return(LZH);
            }

            /* Check for ZOO
            if(header[0] == 'Z' && header[1] == 'O' && header[2] == 'O')
            {
                return(ZOO);
            }

            /* Check for ARJ
            if(header[0] == 0x60  &&  header[1] == 0xEA)
            {
                return(ARJ);
            }

            /* Check for ARC and PAK
            if (x >= 25 && header[0] == 0x1A)
            {
                if(header[1] > 0x14)
                {
                    return(ARC);
                }
                if(header[1]==0x0a || header[1]==0x0b)
                {
                    return(PAK);
                }
                return(ARC);
            }

            /* Check for ZIP
            if(header[0] == 'P' &&  header[1] == 'K')
            {
                return(ZIP);
            }

            /* Check for UC2
            if(header[0] == 'U' &&  header[1] == 'C'  &&  header[2] == '2')
            {
                return(UC2);
            }

            /* Check for RAR
            if(header[0] == 'R' &&  header[1] == 'a'  && header[2] == 'r')
            {
                return(RAR);
            }
        }
    }
*/
    return(arctype);
}

/************************************************************************/
/* Return whether or not this archive has authenticity validation */
int Utility::is_secure(char *inname)
{
    char header[129];
    FILE *arcfile;
    short rar_byte;

    arcfile = _fsopen(inname, "rb", SH_DENYWR);
    if(arcfile)
    {
	fread(header, sizeof(header) - 1, 1, arcfile);
        header[128] = NULL;
        fseek(arcfile, 10, SEEK_SET);
        fread(&rar_byte, sizeof(rar_byte), 1, arcfile);
        header[sizeof(header)-1] = NULL;

        /* Check for ARJ */
        if(header[0] == 0x60  &&  header[1] == 0xEA)
        {
            /* Check to see if this is a security envelope */
            if(((header[8] & 0x02) || (header[8] & 0x40)) ||
              (header[8]&0x40 || header[8]&0x02))
            {
                fclose(arcfile);
                return(TRUE);
            }
        }

        /* Check for ZIP */
	if(header[0] == 'P' &&  header[1] == 'K')
	{
	    fclose(arcfile);
	    return(check_zipav(inname));
        }

        /* Check for RAR */
        if(header[0] == 'R' &&  header[1] == 'a'  && header[2] == 'r')
        {
            if(bitset(rar_byte, BIT5))
            {
                fclose(arcfile);
                return(TRUE);
            }
        }
        fclose(arcfile);
    }
    return(FALSE);
}

/* Check for the AV code in a PKZIP file */
int Utility::check_zipav(char *fn)
{
    FILE *arcfile;
    short fnsize, fieldsize, done, AV;
    long compsize, uncompsize, offset;
    unsigned long id;

    AV = FALSE;
    arcfile = fopen(fn, "rb");
    if(arcfile)
    {

	done = FALSE;
	fseek(arcfile, 0, SEEK_SET);
	while(!done && !feof(arcfile) && !ferror(arcfile))
	{
	    fseek(arcfile, 18, SEEK_CUR);
	    if(feof(arcfile) || ferror(arcfile))
	    {
		break;
	    }

	    fread(&compsize, 4, 1, arcfile);
	    if(feof(arcfile) || ferror(arcfile))
	    {
		break;
	    }

	    fread(&uncompsize, 4, 1, arcfile);
	    if(feof(arcfile) || ferror(arcfile))
	    {
		break;
	    }

	    fread(&fnsize, 2, 1, arcfile);
	    if(feof(arcfile) || ferror(arcfile))
	    {
		break;
	    }

	    fread(&fieldsize, 2, 1, arcfile);
	    if(feof(arcfile) || ferror(arcfile))
	    {
		break;
	    }

	    fseek(arcfile, fnsize + fieldsize + compsize, SEEK_CUR);
	    if(feof(arcfile) || ferror(arcfile))
	    {
		break;
	    }

	    offset = ftell(arcfile);
	    fread(&id, 4, 1, arcfile);
	    if(feof(arcfile) || ferror(arcfile))
	    {
		break;
	    }

	    if(id == 0x02014b50)
	    {
		done = TRUE;
	    }
	    else
	    {
		fseek(arcfile, offset, SEEK_SET);
	    }
	}
	if(done)
	{
	    fseek(arcfile, 24, SEEK_CUR);
	    fread(&fnsize, 2, 1, arcfile);
	    fread(&fieldsize, 2, 1, arcfile);
	    fseek(arcfile, fnsize + 14, SEEK_CUR);
	    id = 0;
	    fread(&id, 2, 1, arcfile);
	    if(id == 7)
	    {
		AV = TRUE;
	    }
	}
	fclose(arcfile);
    }
    return(AV);
}

/**********************************************************************/
/* Return whether or not the archive is self-extracting */
int Utility::is_sfx(char *inname)
{
    char header[129];
    FILE *arcfile;
    short c, i, y;
    long x[4];

    arcfile = _fsopen(inname, "rb", SH_DENYWR);
    if(arcfile)
    {
         /* Test for SFX ARJ file, there are 4 different places this
            information may be stored depending on the ARJ version */
         x[0] = 14778;
         x[1] = 5779;
         x[2] = 16959;
         x[3] = 14858;
         for(i = 0; i < 4; i++)
         {
             fseek(arcfile, x[i], SEEK_SET);
             fread(header, sizeof(header) - 1, 1, arcfile);
             if(header[0] == 0x60 && header[1] == 0xea &&
                (header[8]&0x02 || header[8]&0x40))
             {
                 fclose(arcfile);
                 return(TRUE);
             }
             if(header[0] == 0x60 && header[1] == 0xea)
             {
                 fclose(arcfile);
                 return(TRUE);
             }
         }



         /* test for SFX LHARC file, two different offsets */
         x[0] = 0x653;
         x[1] = 1870;
         for(y = 0; y < 2; y++)
         {
             fseek(arcfile, x[y], SEEK_SET);
             fread(header, sizeof(header) - 1, 1, arcfile);
             for (c = 0, i = header[0]; i--; c += (header+2)[i]);
             if(((unsigned char)(c & 0x00FF)) == header[1]  && header[2] == '-'
                && header[3] == 'l' && header[4] == 'h' && header[6] == '-')
             {
                 fclose(arcfile);
                 return(TRUE);
             }
         }

         /* test for SFX LHA file */
         fseek(arcfile, 0x666, SEEK_SET);
         fread(header, sizeof(header) - 1, 1, arcfile);
         for(c = 0, i = header[0]; i--; c += (header+2)[i]);
         if(header[0] == '-'  && header[1] == 'l' && header[2] == 'h' && header[4] == '-')
         {
             fclose(arcfile);
             return(TRUE);
         }


           /* Test for SFX ZIP file, 2 offsets */
         x[0] = 15770;
         x[1] = 0;
         for(y = 0; y < 2; y++)
         {
             fseek(arcfile, x[y], SEEK_SET);
             fread(header, sizeof(header) - 1, 1, arcfile);
             if(header[0] == 'P'  && header[1] == 'K' && !(header[6] & 0x00))
             {
                 fclose(arcfile);
                 return(TRUE);
             }
             if(header[0] == 'P'  && header[1] == 'K' && (header[6] & 0x00))
             {
                 fclose(arcfile);
                 return(TRUE);
             }
         }


         /* test for SFX PKARC file */
         fseek(arcfile, 0x261e, SEEK_SET);
         fread(header, sizeof(header) - 1, 1, arcfile);
         if(header[0] == 0x1a)
         {
              if(header[1]>0x14  ||  header[1] == 0x0A  ||  header[1] == 0x0B)
              {
                  fclose(arcfile);
                  return(TRUE);
              }
         }
         fclose(arcfile);
    }
    return(FALSE);
}

/************************************************************************/
/* See if there is at least 'amount' bytes free */
int Utility::disk_ok(long amount, char *directory)
{
    long avail;
    int disk;
    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT], fn[129];
    struct diskfree_t free;

    disk = 0;
    if(*directory != NULL)
    {
        strcpy(fn, directory);
        _splitpath(fn, drive, dir, fname, ext);
        if(drive[0])
        {
            disk = (int) drive[0] - 'A' + 1;
        }
    }
    _dos_getdiskfree(disk, &free);
    avail =  (long) free.avail_clusters
             * (long) free.bytes_per_sector
             * (long) free.sectors_per_cluster;
    if(avail > amount)
    {
        return(TRUE);
    }
    return(FALSE);
}

/************************************************************************/
/* Make an entry to the logfile */
#pragma argsused
void Utility::logentry(char *str, int level)
{
#if defined(__AMUONLY__)
    char fn[129], tmplog[201], *ptr, *ptr2;
    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
    FILE *logfile;
    class Dates Date_Now;
    int errors_found = 0;

    if(!*str || (level > Config_obj.amu_cfgvar.logging_level))
    {
        return;
    }

    /* Create a L$G filename from the actual logfilename */
    _splitpath(Config_obj.amu_cfgvar.AMUlogfile, drive, dir, fname, ext);
    sprintf(fn, "%s%s.L$G", Config_obj.amu_cfgvar.AMUpath, fname);
    logfile = fopen(fn, "a+t");
    if(logfile)
    {
        if(*str == '?'  ||  *str == '!')
        {
           errors_found = TRUE;
        }

        /* Filter out any newline characters */
        ptr2 = tmplog;
        ptr = str;
        while(*ptr)
        {
            if(*ptr != '\n'  &&  *ptr != '\r')
            {
                *ptr2 = *ptr;
                ptr2++;
            }
            ptr++;
        }
	*ptr2 = NULL;
	fprintf(logfile, "%s %s  %s\n", Date_Now.getDateStr(Config_obj.amu_cfgvar.date_format),
					Date_Now.getTimeStr(DATES_HHMMSS24), tmplog);
	fclose(logfile);
    }
    if(errors_found)
    {
	AMU_ERRORS++;
	sprintf(fn, "%s%s", Config_obj.amu_cfgvar.AMUpath, ERRORS_AMU);
	logfile = fopen(fn, "a+t");
	if(logfile)
	{
	    fprintf(logfile, "%s %s  %s\n", Date_Now.getDateStr(Config_obj.amu_cfgvar.date_format),
					    Date_Now.getTimeStr(DATES_HHMMSS24), tmplog);
	    fclose(logfile);
	}
    }
#endif
}

/************************************************************************/
/* Check the specified file for a CRC-32 of the input string            */
/************************************************************************/
struct tracking_record *Utility::read_tracking(char *filename)
{
    FILE *infile;
    static struct tracking_record trackingvar;
    unsigned long crc;
    char mode[4], tracking_name[MAX_FILESPEC];
    int found;

    memset(&trackingvar, NULL, sizeof(trackingvar));
    sprintf(tracking_name, "%s%s", Config_obj.amu_cfgvar.AMUpath, TRACKING_AMU);
    found = FALSE;
    crc = Crc_obj.crc32(filename);
    strcpy(mode, "w+b");
    if(access(tracking_name, 0) == 0)
    {
        strcpy(mode, "r+b");
    }

    infile = _fsopen(tracking_name, mode, SH_DENYRW);
    if(infile)
    {
        fread(&trackingvar, sizeof(trackingvar), 1, infile);
        while(!found && !feof(infile) && !ferror(infile))
        {
            if(crc == trackingvar.filecrc)
            {
                found = TRUE;
            }
            else
            {
                fread(&trackingvar, sizeof(trackingvar), 1, infile);
            }
        }
        if(!found)
        {
            fseek(infile, 0, SEEK_END);
            memset(&trackingvar, NULL, sizeof(trackingvar));
            trackingvar.filecrc = crc;
            trackingvar.recnum = (filelength(fileno(infile)) / sizeof(trackingvar)) + 1;
            fwrite(&trackingvar, sizeof(trackingvar), 1, infile);
        }
        fclose(infile);
    }
    return(&trackingvar);
}

/************************************************************************/
/* Update the tracking file                                             */
/************************************************************************/
void Utility::update_tracking(struct tracking_record *trackvar)
{
    FILE *infile;
    char tracking_name[MAX_FILESPEC];

    sprintf(tracking_name, "%s%s", Config_obj.amu_cfgvar.AMUpath, TRACKING_AMU);
    infile = _fsopen(tracking_name, "r+b", SH_DENYWR);
    if(infile)
    {
        fseek(infile, (trackvar -> recnum - 1) * sizeof(struct tracking_record), SEEK_SET);
        fwrite(trackvar, sizeof(struct tracking_record), 1, infile);
        fclose(infile);
    }
}

/***********************************************************************/
/* Change drive/directory */
void Utility::change_dir(char *path)
{
    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
    char tmppath[MAX_FILESPEC];
    unsigned maxdrives;

    strcpy(tmppath, path);
    _splitpath(tmppath, drive, dir, fname, ext);
    strupr(drive);
    if(!tmppath[0] || !drive[0])
    {
        return;
    }
    _dos_setdrive(drive[0] - 'A' + 1, &maxdrives);
    Amustr_obj.noslash(tmppath);
    chdir(tmppath);
}

/***********************************************************************/
/* Clear the current directory */
void Utility::cleardir(void)
{
    int done;
    struct find_t fileinfo;

    done = _dos_findfirst("*.*", _A_ARCH, &fileinfo);
    while(!done)
    {
	chmod(fileinfo.name, S_IREAD | S_IWRITE);
	remove(fileinfo.name);
	done = _dos_findnext(&fileinfo);
    }

    done = _dos_findfirst("*.*", _A_SUBDIR, &fileinfo);
    while(!done)
    {
	if(strcmp(fileinfo.name, ".") != 0 && strcmp(fileinfo.name, "..") != 0)
	{
	    if(chdir(fileinfo.name) == 0)
	    {
		cleardir();
		chdir("..");
		rmdir(fileinfo.name);
	    }
	}
	done = _dos_findnext(&fileinfo);
    }
}

/***********************************************************************/
/* Return a filename with a new archive extension */
void Utility::get_newarc_ext(char *fn, int arctype)
{
    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];

    _splitpath(fn, drive, dir, fname, ext);

    switch(arctype)
    {
        case ARC: strcpy(ext, ".ARC");
                  break;
        case ARJ: strcpy(ext, ".ARJ");
                  break;
        case HAP: strcpy(ext, ".HAP");
                  break;
        case LZH: strcpy(ext, ".LZH");
                  break;
        case PAK: strcpy(ext, ".PAK");
                  break;
        case RAR: strcpy(ext, ".RAR");
                  break;
        case SQZ: strcpy(ext, ".SQZ");
                  break;
        case UC2: strcpy(ext, ".UC2");
                  break;
        case ZIP: strcpy(ext, ".ZIP");
                  break;
        case ZOO: strcpy(ext, ".ZOO");
                  break;
    }
    sprintf(fn, "%s%s%s%s", drive, dir, fname, ext);
}

/***********************************************************************/
/* Copy a file */
int Utility::copyfile(char *source, char *target)
{
    int sourcefile, targetfile, bufsize, status;
    unsigned int bytes_read, bytes_written;
    char *buf;
#ifdef __BORLANDC__
    unsigned int date, time;
#else
    unsigned short date, time;
#endif

    status = TRUE;
    bufsize = 4096;
    sourcefile = targetfile = -1;
    buf = (char *) malloc(bufsize);
    sourcefile = sopen(source, O_RDONLY | O_BINARY, SH_COMPAT | SH_DENYNO);
    targetfile = sopen(target, O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, SH_COMPAT | SH_DENYNO, S_IWRITE);

    if(sourcefile > -1  &&  targetfile > -1 && status)
    {
        _dos_getftime(sourcefile, &date, &time);
        _dos_read(sourcefile, buf, bufsize, &bytes_read);
        while(bytes_read > 0)
        {
            _dos_write(targetfile, buf, bytes_read, &bytes_written);
            _dos_read(sourcefile, buf, bufsize, &bytes_read);
        }
        _dos_setftime(targetfile, date, time);
        close(sourcefile);
        close(targetfile);
    }
    else
    {
        if(sourcefile == -1)
        {
            status = FALSE;
            sprintf(LOGSTR, "! [COPYFILE] Could not open source file '%s' - (%s)", source, strerror(errno));
            logentry(LOGSTR, LOG_MINIMAL);
        }
        if(targetfile == -1)
        {
            status = FALSE;
            sprintf(LOGSTR, "! [COPYFILE] Could not open target file '%s' - (%s)", target, strerror(errno));
            logentry(LOGSTR, LOG_MINIMAL);
        }

        if(sourcefile != -1)
        {
            close(sourcefile);
        }
        if(targetfile != -1)
        {
            close(targetfile);
        }
    }
    free(buf);
    return(status);
}

/*****************************************************************/
/* Toggle a value from Yes to No or vice versa */
void Utility::toggle(int *i)
{
    if(*i)
    {
        *i = FALSE;
    }
    else
    {
        *i = TRUE;
    }
}

/***********************************************************************/
/* Copy the temp logfile to the real one, clean up */
void Utility::cleanup(void)
{
#if defined(__AMUONLY__)
    char sourcefn[129], str[201];
    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
    FILE *sourcelog, *targetlog;
    struct msgpostinfo msgpostvar;
    class Msgs *Msgs_ptr;

    /* Create a new Msgs instance */
    Msgs_ptr = new Msgs();

    Config_obj.open_config(MSGAREA_RECORD);
    Config_obj.read_record(MSGAREA_RECORD, Config_obj.amu_cfgvar.amu_area);
    Config_obj.close_config(MSGAREA_RECORD);
    msgpostvar.area = &Config_obj.msgareavar;
    strcpy(msgpostvar.from, "AMU");
    strcpy(msgpostvar.to, Config_obj.amu_cfgvar.sysop);

    /* Create a L$G filename from the actual logfilename */
    _splitpath(Config_obj.amu_cfgvar.AMUlogfile, drive, dir, fname, ext);
    sprintf(sourcefn, "%s%s.L$G", Config_obj.amu_cfgvar.AMUpath, fname);
    if(Config_obj.amu_cfgvar.postlog)
    {
        strcpy(msgpostvar.subject, "Run completed");
        strcpy(msgpostvar.fn, sourcefn);
	Msgs_ptr->post_msg(&msgpostvar, NULL);
    }
    sourcelog = fopen(sourcefn, "rt");
    targetlog = fopen(Config_obj.amu_cfgvar.AMUlogfile, "a+t");
    if(sourcelog && targetlog)
    {
        fgets(str, sizeof(str)-2, sourcelog);
        while(!feof(sourcelog))
        {
            fprintf(targetlog, "%s", str);
	    fgets(str, sizeof(str)-2, sourcelog);
        }
        fprintf(targetlog, "\n");
    }
    if(sourcelog)
    {
        fclose(sourcelog);
        remove(sourcefn);
    }
    if(targetlog)
    {
        fclose(targetlog);
    }

    sprintf(sourcefn, "%s%s", Config_obj.amu_cfgvar.AMUpath, ERRORS_AMU);
    if(access(sourcefn, 0) == 0)
    {
        if(Config_obj.amu_cfgvar.posterrors)
        {
            strcpy(msgpostvar.subject, "Errors from last run");
            strcpy(msgpostvar.fn, sourcefn);
	    Msgs_ptr->post_msg(&msgpostvar, NULL);
        }
        remove(sourcefn);
    }
    sprintf(sourcefn, "%s%s", Config_obj.amu_cfgvar.AMUpath, TMPUSERS_AMU);
    remove(sourcefn);
    delete(Msgs_ptr);
#endif
}

/***********************************************************************/
/* Strip the existing extension from a filename and append the arc type
   extension.  I.g., ALLFILES.LST --> ALLFILES.ZIP
*/
char *Utility::arcext(char *instring, int arctype)
{
    static char tmp[81], *ptr;

    strncpy(tmp, instring, sizeof(tmp) - 1);

    /* Strip the current extension, if one exists */
    ptr = strchr(tmp, '.');
    if(ptr)
    {
        *ptr = NULL;
    }

    switch(arctype)
    {
        case ARC: strncat(tmp, ".ARC", sizeof(tmp) - strlen(tmp) - 1);
                  break;
        case ARJ: strncat(tmp, ".ARJ", sizeof(tmp) - strlen(tmp) - 1);
                  break;
        case HAP: strncat(tmp, ".HAP", sizeof(tmp) - strlen(tmp) - 1);
                  break;
        case LZH: strncat(tmp, ".LZH", sizeof(tmp) - strlen(tmp) - 1);
                  break;
        case PAK: strncat(tmp, ".PAK", sizeof(tmp) - strlen(tmp) - 1);
                  break;
        case SQZ: strncat(tmp, ".SQZ", sizeof(tmp) - strlen(tmp) - 1);
                  break;
        case ZIP: strncat(tmp, ".ZIP", sizeof(tmp) - strlen(tmp) - 1);
                  break;
        case ZOO: strncat(tmp, ".ZOO", sizeof(tmp) - strlen(tmp) - 1);
                  break;
        case UC2: strncat(tmp, ".UC2", sizeof(tmp) - strlen(tmp) - 1);
                  break;
        case RAR: strncat(tmp, ".RAR", sizeof(tmp) - strlen(tmp) - 1);
                  break;
    }
    return(tmp);
}

/***********************************************************************/
/* Remove or add all files in 'fn' from current directory */
void Utility::check_files(char *fn, int mode)
{
    FILE *infile;
    char str[MAX_FILESPEC], source[MAX_FILESPEC];
    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
    int done;
    struct find_t fileinfo;

    infile = _fsopen(fn, "rt", SH_DENYWR);
    if(infile)
    {
        fgets(str, sizeof(str)-2, infile);
        while(!feof(infile))
        {
            Amustr_obj.strip_n(str);
	    Amustr_obj.trimlead(str);
            Amustr_obj.trimend(str);
            if(str[0] && str[0] != ';')
            {
                _splitpath(str, drive, dir, fname, ext);
                done = _dos_findfirst(str, _A_ARCH, &fileinfo);
                while(!done)
                {
                    sprintf(source, "%s%s%s", drive, dir, fileinfo.name);
                    if(mode == ADD_FILES)
                    {
                        copyfile(source, fileinfo.name);
                    }
                    if(mode == REMOVE_FILES)
                    {
                        remove(fileinfo.name);
                    }
                    done = _dos_findnext(&fileinfo);
                }
            }
            fgets(str, sizeof(str)-2, infile);
	}
        fclose(infile);
    }
}

/***********************************************************************/
/* Function to remove all files pointed to by 'source' - wildcards ok */
void Utility::remove_all(char *source)
{
    struct find_t fileinfo;
    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
    char fn[MAX_FILESPEC];
    int done;

    _splitpath(source, drive, dir, fname, ext);
    done = _dos_findfirst(source, _A_ARCH, &fileinfo);
    while(!done)
    {
        sprintf(fn, "%s%s%s", drive, dir, fileinfo.name);
	chmod(fn, S_IREAD | S_IWRITE);
        remove(fn);
	done = _dos_findnext(&fileinfo);
    }
}

/***********************************************************************/
/* Check to see if 'param' is valid */
int Utility::check_param(int argc, char *argv[], char *param, int num)
{
    int x, begin, end;
    char *ptr, *ptr2, tmpstr[11];

    if(argc == 1)
    {
        return(TRUE);
    }
    for(x = 1; x < argc; x++)
    {
        strupr(argv[x]);
        if(strstr(argv[x], param))
        {
            if(num <= 0)
	    {
                return(TRUE);
            }
            if(!strchr(argv[x], '[') && !strchr(argv[x], ']'))
            {
                return(DEFAULT_PARAM);
            }
            ptr = strchr(argv[x], '[');
            ptr++;
            while(*ptr && *ptr != ']')
            {
                ptr2 = tmpstr;
                while(isdigit(*ptr)  && *ptr)
                {
                    *ptr2 = *ptr;
                    ptr++;
                    ptr2++;
		}
                *ptr2 = NULL;

                /* If this is a range of areas */
		if(*ptr == '-')
                {
                    begin = atoi(tmpstr);
                    ptr++;
                    ptr2 = tmpstr;
                    while(isdigit(*ptr)  && *ptr)
                    {
                        *ptr2 = *ptr;
                        ptr++;
                        ptr2++;
                    }
                    *ptr2 = NULL;
                     end = atoi(tmpstr);
                     if(num >= begin && num <= end)
                     {
                         return(TRUE);
                     }
                }
                else
                {
                    if(*ptr == ','  ||  *ptr == ']')
		    {
                        ptr++;
                        if(num == atoi(tmpstr))
                        {
                            return(TRUE);
                        }
                    }
                }
            }
            return(FALSE);
        }
    }
    return(FALSE);
}


/* See if the given filename is an AMU system filename */
int Utility::system_filename(char *inname)
{
    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
    char str[MAX_FILENAME];

    _splitpath(inname, drive, dir, fname, ext);
    if(ext[0] == NULL)
    {
        strcpy(ext, ".AMU");
    }
    sprintf(str, "%s%s", fname, ext);

    if(strstr(str, BULLETIN_AMU))
    {
        return(TRUE);
    }
    if(strstr(str, CONFIG_AMU))
    {
        return(TRUE);
    }
    if(strstr(str, ARCHIVES_AMU))
    {
        return(TRUE);
    }
    if(strstr(str, MAINT_AMU))
    {
        return(TRUE);
    }
    if(strstr(str, LOGS_AMU))
    {
        return(TRUE);
    }
    if(strstr(str, FILEAREA_AMU))
    {
        return(TRUE);
    }
    if(strstr(str, MSGAREA_AMU))
    {
        return(TRUE);
    }
    if(strstr(str, ENFORCE_AMU))
    {
        return(TRUE);
    }
    if(strstr(str, LISTS_AMU))
    {
        return(TRUE);
    }
    if(strstr(str, USERMSGS_AMU))
    {
        return(TRUE);
    }

    if(strstr(str, NOADOPT_CTL))
    {
        return(TRUE);
    }
    if(strstr(str, TRASH_CTL))
    {
        return(TRUE);
    }
    if(strstr(str, ADD_CTL))
    {
        return(TRUE);
    }
    if(strstr(str, LOCKED_CTL))
    {
        return(TRUE);
    }

    return(FALSE);
}

/* Return TRUE if 'fn' is a multivolume archive, false if not */
int Utility::is_multivolume(char *fn)
{
    FILE *fp;
    unsigned x;

    x = 0;

    /* Check for an ARJ archive */
    if(strstr(fn, ".ARJ"))
    {
	fp = _fsopen(fn, "rb", SH_DENYWR);
	if(fp != NULL)
	{
	    fseek(fp, 8, SEEK_SET);
	    fread(&x, 1, 1, fp);
	    fclose(fp);
	    if(bitset(x, BIT2) == TRUE)
	    {
		return(TRUE);
	    }
	}
	return(FALSE);
    }

    /* Check for a RAR archive */
    if(strstr(fn, ".RAR"))
    {
	fp = _fsopen(fn, "rb", SH_DENYWR);
	if(fp != NULL)
	{
	    fseek(fp, 10, SEEK_SET);
	    fread(&x, 1, 1, fp);
	    fclose(fp);
	    if(bitset(x, BIT0) == TRUE)
	    {
		return(TRUE);
	    }
	}
	return(FALSE);
    }
    return(FALSE);
}

/***********************************************************************/
// EOF - UTILITY.CPP
