// Functions to manipulate the user files, Y2K
// Created 21 March 1996
// Revised 21 May 1998

#include <stdio.h>
#include <share.h>
#include <string.h>
#include <io.h>
#include <dos.h>
#include <ctype.h>
#include <malloc.h>
#include <stdlib.h>
#include "extern.h"
#include "logs.h"
#include "users.h"
#include "screen.h"

/* External objects */
extern class Screen Screen_obj;
extern class Config Config_obj;

/************************************************************************/

/* Users constructor */
Users::Users(void)
{
    currec = toprec = tmprec = NULL;
}

/* Process user options */
void Users::process_users(int argc, char *argv[])
{
    int update;
    struct screeninfo s;

    if(!Utility_obj.check_param(argc, argv, "_MSG", 0) &&
       !Utility_obj.check_param(argc, argv, "ENFORCE", 0) &&
       !Utility_obj.check_param(argc, argv, "CLEAN", 0) &&
       !Utility_obj.check_param(argc, argv, "DAILYDL", 0) &&
       !Utility_obj.check_param(argc, argv, "DAILYUL", 0) &&
       !Utility_obj.check_param(argc, argv, "USERLIST", 0) &&
       !Utility_obj.check_param(argc, argv, "BDAYLIST", 0) &&
       !Utility_obj.check_param(argc, argv, "SUBLIST", 0) &&
       !Utility_obj.check_param(argc, argv, "TODAYCALL", 0) &&
       !Utility_obj.check_param(argc, argv, "TOPUSERS", 0))
    {
        return;
    }

    update = FALSE;
    strcpy(s.curoption, "Processing user options");
    if(open_users(READ_MODE))
    {
        strcpy(s.curarea, "Reading user file");
        Screen_obj.display(&s);
        if(read_users())
        {

	    /* Close the user file */
	    close_users();

	   /* All user message parameters end in _MSG */
	    if(Utility_obj.check_param(argc, argv, "_MSG", 0))
            {
                sprintf(LOGSTR, "Processing user messages");
                Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
                strcpy(s.curarea, LOGSTR);
                Screen_obj.display(&s);
                user_msgs(argc, argv);
            }

            if(Utility_obj.check_param(argc, argv, "ENFORCE", 0))
	    {
                sprintf(LOGSTR, "Processing enforcement records");
                Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
                strcpy(s.curarea, LOGSTR);
                Screen_obj.display(&s);
                enforce();
		update = TRUE;
            }

	    if(argc > 1 && Utility_obj.check_param(argc, argv, "CLEAN", 0))
            {
		Config_obj.amu_cfgvar.clean = TRUE;
            }
            if(Config_obj.amu_cfgvar.clean)
            {
                sprintf(LOGSTR, "Cleaning user records");
                Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
                strcpy(s.curarea, LOGSTR);
                Screen_obj.display(&s);
                clean_users();
                update = TRUE;
	    }

            if(update)
            {
                strcpy(s.curarea, "Updating user file");
		Screen_obj.display(&s);
		if(open_users(WRITE_MODE))
		{
		    write_users();
		    close_users();
		}
		else
		{
		    sprintf(LOGSTR, "! [PROCESS_USERS] Unable to open user file for update operations");
		    Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
		}
	    }
	    deallocate_users();
	}
        else
        {
            close_users();
            sprintf(LOGSTR, "! [PROCESS_USERS] Unable to read user file");
            Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        }
    }
}

/************************************************************************/
/* Open the user file; Return FALSE if unsuccessful */
int Users::open_users(int mode)
{
    char userfn[MAX_FILESPEC], xifn[MAX_FILESPEC], open_mode[5];
    int share_mode;

    switch(mode)
    {
	case READ_MODE  : strcpy(open_mode, "rb");
			  share_mode = SH_DENYNO;
			  break;
	case WRITE_MODE : strcpy(open_mode, "r+b");
			  share_mode = SH_DENYRW;
			  break;
    }

    userfile = xifile = NULL;
    switch(Config_obj.amu_cfgvar.BBStype)
    {
        case PB2  :
        case RA250:
        case RA2  : sprintf(userfn, "%sUSERS.BBS", Config_obj.amu_cfgvar.userpath);
                    user_recsize = sizeof(ra20uservar);
		    xi_recsize = 0;
                    break;

        case RA1: sprintf(userfn, "%sUSERS.BBS", Config_obj.amu_cfgvar.userpath);
                  sprintf(xifn, "%sUSERSXI.BBS", Config_obj.amu_cfgvar.userpath);
		  user_recsize = sizeof(ra10uservar);
		  xi_recsize = sizeof(ra10userxivar);
		  break;

	case QBBS: sprintf(userfn, "%sUSERS.BBS", Config_obj.amu_cfgvar.userpath);
		   sprintf(xifn, "%sALIAS.BBS", Config_obj.amu_cfgvar.userpath);
		   user_recsize = sizeof(q276uservar);
		   xi_recsize = sizeof(aliasvar);
		   break;

	case QBBSC: sprintf(userfn, "%sUSERS.BBS", Config_obj.amu_cfgvar.userpath);
		    sprintf(xifn, "%sALIAS.BBS", Config_obj.amu_cfgvar.userpath);
		    user_recsize = sizeof(q280uservar);
		    xi_recsize = sizeof(aliasvar);
		    break;

	case QBBSG: sprintf(userfn, "%sUSERS.DAT", Config_obj.amu_cfgvar.userpath);
		    sprintf(xifn, "%sALIAS.BBS", Config_obj.amu_cfgvar.userpath);
		    user_recsize = sizeof(q280uservar);
		    xi_recsize = sizeof(aliasvar);
		    break;

	case SBBS: sprintf(userfn, "%sUSERS.BBS", Config_obj.amu_cfgvar.userpath);
		   sprintf(xifn, "%sSUSERS.BBS", Config_obj.amu_cfgvar.userpath);
		   user_recsize = sizeof(s117uservar);
		   xi_recsize = sizeof(s117userxivar);
		   break;

	case LORA: sprintf(userfn, "%sUSERS.BBS", Config_obj.amu_cfgvar.userpath);
		   user_recsize = sizeof(lorauservar);
		   xi_recsize = 0;
		   break;

	case LORA3: sprintf(userfn, "%sUSERS.DAT", Config_obj.amu_cfgvar.userpath);
		    user_recsize = sizeof(l300uservar);
		    xi_recsize = 0;
		    break;

        case TG310:
	case TG3: sprintf(userfn, "%sUSERS.DAT", Config_obj.amu_cfgvar.userpath);
		  user_recsize = sizeof(t302uservar);
		  xi_recsize = 0;
		  break;

	case EZ: sprintf(userfn, "%sUSERS.BBS", Config_obj.amu_cfgvar.userpath);
		 sprintf(xifn, "%sUSERSEXT.BBS", Config_obj.amu_cfgvar.userpath);
		 user_recsize = sizeof(e120uservar);
		 xi_recsize = sizeof(e120userxivar);
		 break;

	case MAXIMUS2: sprintf(userfn, "%sUSER.BBS", Config_obj.amu_cfgvar.userpath);
		       user_recsize = sizeof(max2uservar);
		       xi_recsize = 0;
                       break;

        case MAXIMUS3: sprintf(userfn, "%sUSER.BBS", Config_obj.amu_cfgvar.userpath);
                       user_recsize = sizeof(max3uservar);
                       xi_recsize = 0;
                       break;

	case CONCORD: sprintf(userfn, "%sUSERINFO.DAT", Config_obj.amu_cfgvar.userpath);
                      user_recsize = sizeof(concuservar);
                      xi_recsize = 0;
                      break;
    }

    userfile = _fsopen(userfn, open_mode, share_mode);
    if(!userfile)
    {
	sprintf(LOGSTR, "? [USERS] Could not open '%s' - (%s)", userfn, strerror(errno));
	Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
	return(FALSE);
    }

    /* If defined, try to open the index file */
    if(xi_recsize)
    {
	xifile = _fsopen(xifn, open_mode, share_mode);
	if(!xifile)
	{
	    sprintf(LOGSTR, "? [USERS] Could not open '%s' - (%s)", userfn, strerror(errno));
	    Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
	    fclose(userfile);
	    return(FALSE);
	}
    }

    /* If this is Concord, seek past the header record */
    if(Config_obj.amu_cfgvar.BBStype == CONCORD)
    {
	fseek(userfile, sizeof(concuserhdrvar) + sizeof(concuservar), SEEK_SET);
    }
    return(TRUE);
}

/***********************************************************************/
/* Read the pertinant user file data into a linked list */
int Users::read_users(void)
{
    FILE *tmpusers;
    struct amu_users amu_uservar;
    int _eof, status;
    char *fn, tmpdate[25];
    struct user_array uservar;

    fn = (char *) malloc(MAX_FILESPEC);
    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.AMUpath, TMPUSERS_AMU);
    tmpusers = _fsopen(fn, "w+b", SH_DENYWR);
    if(!tmpusers)
    {
	sprintf(LOGSTR, "! [USERS] Unable to create user index file '%s' - (%s)", fn, strerror(errno));
	Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
	free(fn);
	return(FALSE);
    }
    free(fn);

    if(Config_obj.amu_cfgvar.BBStype == CONCORD)
    {
	fseek(userfile, sizeof(concuserhdrvar) + sizeof(concuservar), SEEK_SET);
    }
    else
    {
	fseek(userfile, 0, SEEK_SET);
    }
    if(xifile)
    {
	fseek(xifile, 0, SEEK_SET);
    }

    _eof = FALSE;
    status = TRUE;
    toprec = currec = tmprec = NULL;
    while(!_eof)
    {
        memset(&uservar, NULL, sizeof(uservar));
	switch(Config_obj.amu_cfgvar.BBStype)
        {
            case PB2  :
	    case RA250:
            case RA2  : fread(&ra20uservar, sizeof(ra20uservar), 1, userfile);
			if(feof(userfile))
			{
			    _eof = TRUE;
			    break;
			}
			Amustr_obj.pascaltoc(uservar.name, ra20uservar.name, (int) ra20uservar.l1);
			if(ra20uservar.l7 > 0)
			{
			    Amustr_obj.pascaltoc(uservar.alias, ra20uservar.handle, (int) ra20uservar.l7);
			}
			Amustr_obj.pascaltoc(uservar.city, ra20uservar.location, (int) ra20uservar.l2);
			Amustr_obj.pascaltoc(tmpdate, ra20uservar.birthdate, (int) ra20uservar.l14);
			uservar.BirthDate.createDate(tmpdate, NULL);
			Amustr_obj.pascaltoc(tmpdate, ra20uservar.subdate, (int) ra20uservar.l15);
			uservar.SubDate.createDate(tmpdate, NULL);
                        uservar.dls = ra20uservar.downloads;
			uservar.uls = ra20uservar.uploads;
			uservar.calls = ra20uservar.nocalls;
			uservar.posts = ra20uservar.msgsposted;
			uservar.dlkb = ra20uservar.downloadsk;
			uservar.ulkb = ra20uservar.uploadsk;
			uservar.security = ra20uservar.security;
                        uservar.credit = ra20uservar.credit;
                        if(Utility_obj.bitset(ra20uservar.attribute2, BIT3))
                        {
			    uservar.hidden = TRUE;
			}
                        break;

        case RA1: fread(&ra10uservar, sizeof(ra10uservar),1 , userfile);
                  fread(&ra10userxivar, sizeof(ra10userxivar),1 , xifile);
                  if(feof(userfile) || feof(xifile))
                  {
                      _eof = TRUE;
                      break;
                  }
                  Amustr_obj.pascaltoc(uservar.name, ra10uservar.name, ra10uservar.l1);
		  if(ra10userxivar.l1)
                  {
                      Amustr_obj.pascaltoc(uservar.alias, ra10userxivar.handle, ra10userxivar.l1);
                  }
                  Amustr_obj.pascaltoc(uservar.city, ra10uservar.location, ra10uservar.l2);
		  Amustr_obj.pascaltoc(tmpdate, ra10userxivar.subdate, ra10userxivar.l5);
		  uservar.SubDate.createDate(tmpdate, NULL);
		  Amustr_obj.pascaltoc(tmpdate, ra10userxivar.birthdate, ra10userxivar.l4);
		  uservar.BirthDate.createDate(tmpdate, NULL);
                  uservar.dls = ra10uservar.downs;
                  uservar.uls = ra10uservar.ups;
		  uservar.calls = ra10uservar.times;
                  uservar.posts = ra10uservar.timesposted;
                  uservar.dlkb = ra10uservar.downk;
		  uservar.ulkb = ra10uservar.upk;
                  uservar.security = ra10uservar.seclvl;
                  uservar.credit = ra10uservar.credit;
                  if(Utility_obj.bitset(ra10uservar.attribute2, BIT3))
                  {
		      uservar.hidden = TRUE;
		  }
		  break;

            case TG310:
	    case TG3  : fread(&t302uservar, sizeof(t302uservar), 1, userfile);
			if(feof(userfile))
			{
			    _eof = TRUE;
			    break;
			}
			Amustr_obj.pascaltoc(uservar.name, t302uservar.name, t302uservar.l2);
			if(t302uservar.l1 > 0)
			{
			    Amustr_obj.pascaltoc(uservar.alias, t302uservar.alias, t302uservar.l1);
			}
			Amustr_obj.pascaltoc(uservar.city, t302uservar.location, (int) t302uservar.l3);
			Amustr_obj.pascaltoc(tmpdate, t302uservar.birthdate, (int) t302uservar.l4);
			uservar.BirthDate.createDate(tmpdate, NULL);
			Amustr_obj.pascaltoc(tmpdate, t302uservar.expiredate, (int) t302uservar.l6);
			uservar.SubDate.createDate(tmpdate, NULL);
			uservar.dls = t302uservar.downloads;
			uservar.uls = t302uservar.uploads;
			uservar.calls = t302uservar.totalcalls;
			uservar.posts = t302uservar.pubpost + t302uservar.privpost + t302uservar.netpost;
			uservar.dlkb = t302uservar.dk;
			uservar.ulkb = t302uservar.uk;
			uservar.security = t302uservar.level;
			uservar.credit = t302uservar.credit;
			if(t302uservar.flags.hiddenlist)
			{
			    uservar.hidden = TRUE;
			}
			break;


	case QBBS: fread(&q276uservar, sizeof(q276uservar),1 , userfile);
                   if(feof(userfile))
                   {
                       _eof = TRUE;
                       break;
                   }
                   fseek(xifile, sizeof(aliasvar) * (q276uservar.aliasptr), SEEK_SET);
		   if(fread(&aliasvar, sizeof(aliasvar), 1, xifile))
		   {
		       Amustr_obj.pascaltoc(uservar.alias, aliasvar.alias, aliasvar.l1);
		   }
		   uservar.BirthDate.createDate(1600, (long) q276uservar.bdate);
		   uservar.SubDate.createDate(0, 0, 0, 0, 0, 0);
		   Amustr_obj.pascaltoc(uservar.name, q276uservar.name, q276uservar.ln1);
		   Amustr_obj.pascaltoc(uservar.city, q276uservar.location, q276uservar.ln2);
		   uservar.dls = q276uservar.downs;
		   uservar.uls = q276uservar.ups;
		   uservar.calls = q276uservar.times;
		   uservar.posts = q276uservar.timesposted;
		   uservar.dlkb = q276uservar.downk;
		   uservar.ulkb = q276uservar.upk;
		   uservar.security = q276uservar.seclvl;
		   uservar.credit = q276uservar.credit;
		   break;

	case QBBSG:
	case QBBSC: fread(&q280uservar, sizeof(q280uservar),1 , userfile);
		    if(feof(userfile))
		    {
			_eof = TRUE;
			break;
		    }
		    fseek(xifile, sizeof(aliasvar) * (q280uservar.aliasptr), SEEK_SET);
		    if(fread(&aliasvar, sizeof(aliasvar),1, xifile))
		    {
			Amustr_obj.pascaltoc(uservar.alias, aliasvar.alias, aliasvar.l1);
		    }
		    uservar.BirthDate.createDate(1600, (long) q280uservar.birthdate);
		    uservar.SubDate.createDate(1900, (long) q280uservar.expiredate);
		    Amustr_obj.pascaltoc(uservar.name, q280uservar.name, q280uservar.ln1);
		    Amustr_obj.pascaltoc(uservar.city, q280uservar.location, q280uservar.ln2);
		    uservar.dls = q280uservar.downs;
		    uservar.uls = q280uservar.ups;
		    uservar.calls = q280uservar.times;
		    uservar.posts = q280uservar.timesposted;
		    uservar.dlkb = q280uservar.downk;
		    uservar.ulkb = q280uservar.upk;
		    uservar.security = q280uservar.security;
		    uservar.credit = q280uservar.Credit;
		    break;

        case SBBS: fread(&s117uservar, sizeof(s117uservar),1 , userfile);
		   if(feof(userfile))
                   {
		       _eof = TRUE;
                       break;
                   }
                   fread(&s117userxivar, sizeof(s117userxivar), 1, xifile);
                   Amustr_obj.pascaltoc(uservar.name, s117uservar.name, s117uservar.ln1);
                   Amustr_obj.pascaltoc(uservar.city, s117uservar.location, s117uservar.ln2);
                   if(s117userxivar.l6)
                   {
                       Amustr_obj.pascaltoc(uservar.alias, s117userxivar.alias, s117userxivar.l6);
                   }
                   uservar.dls = s117uservar.downs;
                   uservar.uls = s117uservar.ups;
                   uservar.calls = s117uservar.times;
                   uservar.posts = s117uservar.msgsposted;
                   uservar.ulkb = s117uservar.upk;
                   uservar.dlkb = s117uservar.downk;
                   uservar.credit = s117uservar.credit;
		   Amustr_obj.pascaltoc(tmpdate, s117userxivar.birthday, s117userxivar.l2);
		   uservar.BirthDate.createDate(tmpdate, NULL);
		   Amustr_obj.pascaltoc(tmpdate, s117userxivar.expirationdate, s117userxivar.l10);
		   uservar.SubDate.createDate(tmpdate, NULL);
		   uservar.security = s117uservar.seclvl;
                   break;


	case LORA: fread(&lorauservar, sizeof(lorauservar),1 , userfile);
                   if(feof(userfile))
                   {
                       _eof = TRUE;
                       break;
                   }
                   strcpy(uservar.name, lorauservar.name);
                   strcpy(uservar.city, lorauservar.location);
		   strcpy(uservar.alias, lorauservar.handle);
		   uservar.BirthDate.createDate(lorauservar.birthdate, NULL);
		   uservar.SubDate.createDate(lorauservar.expiredate, NULL);
		   uservar.credit = lorauservar.credit;
		   uservar.dls = lorauservar.downloads;
		   uservar.uls = lorauservar.uploads;
		   uservar.calls = lorauservar.calls;
		   uservar.posts = lorauservar.posts;
		   uservar.dlkb = lorauservar.downloadk;
		   uservar.ulkb = lorauservar.uploadk;
		   uservar.security = lorauservar.security;
		   if(lorauservar.usrhidden)
		   {
		       uservar.hidden = TRUE;
		   }
		   break;

	case LORA3:fread(&l300uservar, sizeof(l300uservar),1 , userfile);
		   if(feof(userfile))
		   {
		       _eof = TRUE;
		       break;
		   }
		   strcpy(uservar.name, l300uservar.RealName);
		   strcpy(uservar.city, l300uservar.City);
		   strcpy(uservar.alias, l300uservar.Name);
		   uservar.BirthDate.createDate(l300uservar.BirthYear,
						 l300uservar.BirthMonth,
						 l300uservar.BirthDay,
						 0, 0, 0);
		   uservar.SubDate.createDate(0, 0, 0, 0, 0, 0);
		   uservar.credit = 0;
		   uservar.dls = l300uservar.DownloadFiles;
		   uservar.uls = l300uservar.UploadFiles;
		   uservar.calls = l300uservar.TotalCalls;
		   uservar.posts = 0;
		   uservar.dlkb = l300uservar.DownloadBytes / 1024;
		   uservar.ulkb = l300uservar.UploadBytes / 1024;
		   uservar.security = l300uservar.Level;
		   if(!l300uservar.InUserList)
		   {
		       uservar.hidden = TRUE;
		   }
		   break;

	case EZ  : fread(&e120uservar, sizeof(e120uservar),1 , userfile);
		   fread(&e120userxivar, sizeof(e120userxivar), 1, xifile);
		   if(feof(userfile))
		   {
		       _eof = TRUE;
		       break;
		   }
		   Amustr_obj.pascaltoc(uservar.name, e120uservar.Name, e120uservar.l1);
		   Amustr_obj.pascaltoc(uservar.alias, e120uservar.Alias, e120uservar.l2);
		   Amustr_obj.pascaltoc(uservar.city, e120userxivar.Location, e120userxivar.l1);
		   uservar.BirthDate.createDate(e120userxivar.DateOfBirth.Year,
						 e120userxivar.DateOfBirth.Month,
						 e120userxivar.DateOfBirth.Day,
						 0, 0, 0);
		   uservar.SubDate.createDate(0, 0, 0, 0, 0, 0);
		   uservar.credit = e120userxivar.Credit;
		   uservar.dls = e120userxivar.Downloads;
		   uservar.uls = e120userxivar.Uploads;
		   uservar.calls = e120userxivar.NoCalls;
		   uservar.posts = e120userxivar.MsgsPosted;
		   uservar.dlkb = e120userxivar.DownloadsK;
		   uservar.ulkb = e120userxivar.UploadsK;
		   uservar.security = e120uservar.Security;
		   break;


	case MAXIMUS2: fread(&max2uservar, sizeof(max2uservar),1 , userfile);
		       if(feof(userfile))
		       {
			   _eof = TRUE;
			   break;
		       }
                       strcpy(uservar.name, max2uservar.name);
                       strcpy(uservar.alias, max2uservar.alias);
		       strcpy(uservar.city, max2uservar.location);
		       uservar.BirthDate.createDate(0, 0, 0, 0, 0, 0);
		       uservar.SubDate.createDate(max2uservar.expiredate.datevar.year + 1980,
						   max2uservar.expiredate.datevar.month,
						   max2uservar.expiredate.datevar.day,
						   0, 0, 0);
		       uservar.credit = max2uservar.credit;
		       uservar.dls = uservar.dlkb = max2uservar.down;
		       uservar.uls = uservar.ulkb = max2uservar.up;
		       uservar.calls = max2uservar.times;
		       uservar.posts = 0;
		       uservar.security = max2uservar.security;
		       uservar.hidden = FALSE;
		       break;

	case MAXIMUS3: fread(&max3uservar, sizeof(max3uservar),1 , userfile);
		       if(feof(userfile))
		       {
			   _eof = TRUE;
			   break;
		       }
		       strcpy(uservar.name, max3uservar.name);
		       strcpy(uservar.alias, max3uservar.alias);
		       strcpy(uservar.city, max3uservar.city);
		       uservar.SubDate.createDate(max3uservar.expiredate.datevar.year + 1980,
						   max3uservar.expiredate.datevar.month,
						   max3uservar.expiredate.datevar.day,
						   0, 0, 0);
		       uservar.BirthDate.createDate(max3uservar.dob_year,
						     max3uservar.dob_month,
						     max3uservar.dob_day,
						     0, 0, 0);
		       uservar.credit = max3uservar.credit;
		       uservar.dls = max3uservar.ndown;
		       uservar.uls = max3uservar.nup;
		       uservar.dlkb = max3uservar.down;
		       uservar.ulkb = max3uservar.up;
		       uservar.calls = max3uservar.calls;
		       uservar.posts = max3uservar.msgs_posted;
		       uservar.security = max3uservar.security;
		       if(Utility_obj.bitset((unsigned) max3uservar.bits, BITS_NOULIST))
		       {
			    uservar.hidden = TRUE;
		       }
		       else
		       {
			   uservar.hidden = FALSE;
		       }
		       break;

	case CONCORD: fread(&concuservar, sizeof(concuservar),1 , userfile);
		      if(feof(userfile))
		      {
			  _eof = TRUE;
			  break;
		      }
		      Amustr_obj.pascaltoc(uservar.city, concuservar.city, concuservar.l3);
		      Amustr_obj.pascaltoc(uservar.name, concuservar.name, concuservar.l1);
		      Amustr_obj.pascaltoc(uservar.alias, concuservar.alias, concuservar.l2);
		      sprintf(tmpdate, "%02d-%02d-%02d", concuservar.birthday.month,
							 concuservar.birthday.day,
							 concuservar.birthday.year);
		      uservar.BirthDate.createDate(tmpdate, NULL);
		      sprintf(tmpdate, "%02d-%02d-%02d", concuservar.expiration.month,
							 concuservar.expiration.day,
							 concuservar.expiration.year);
		      uservar.SubDate.createDate(tmpdate, NULL);
		      uservar.dls = concuservar.downtimes;
		      uservar.uls = concuservar.uptimes;
		      uservar.calls = concuservar.timescalled;
		      uservar.posts = concuservar.publicmsgs + concuservar.privatemsgs;
		      uservar.dlkb = concuservar.downk;
		      uservar.ulkb = concuservar.upk;
		      uservar.security = concuservar.sec.seclvl;
		      uservar.credit = 0;
		      uservar.hidden = FALSE;
                      break;
        }

    /* If no alias, copy the real name to that field */
        if(!uservar.alias[0])
	{
            strcpy(uservar.alias, uservar.name);
        }

        if(!_eof)
        {
            tmprec = currec;
            currec = (struct user_array *)calloc(1, sizeof(struct user_array));
	    if(!currec)
            {
		sprintf(LOGSTR, "! [READ_USER] Not enough memory to read user file");
                Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
                deallocate_users();
                return(FALSE);
            }
            else
            {
                if(!toprec)
		{
                    toprec = currec;
                }
                if(tmprec)
                {
                    tmprec -> nextrec = currec;
                }
                memcpy(currec, &uservar, sizeof(uservar));
            }

            /* Write an entry to the AMU user file */
            strcpy(amu_uservar.realname, uservar.name);
	    strcpy(amu_uservar.handle, uservar.alias);
	    strcpy(amu_uservar.location, uservar.city);
	    amu_uservar.BirthDate = uservar.BirthDate;
	    amu_uservar.SubDate = uservar.SubDate;
	    amu_uservar.calls = uservar.calls;
	    amu_uservar.posts = uservar.posts;
	    amu_uservar.dls = uservar.dls;
	    amu_uservar.uls = uservar.uls;
            amu_uservar.ulkb = uservar.ulkb;
            amu_uservar.dlkb = uservar.dlkb;
            if(!uservar.hidden)
            {
                fwrite(&amu_uservar, sizeof(amu_uservar), 1, tmpusers);
            }
        }
    }
    fclose(tmpusers);
    return(status);
}

/***********************************************************************/
/* Update the user file */
void Users::write_users(void)
{
    long user_offset, xi_offset;
    char tmpstr[15];

    if(Config_obj.amu_cfgvar.BBStype == CONCORD)
    {
        fseek(userfile, sizeof(concuserhdrvar) + sizeof(concuservar), SEEK_SET);
    }
    else
    {
        fseek(userfile, 0, SEEK_SET);
    }
    if(xifile)
    {
        fseek(xifile, 0, SEEK_SET);
    }

    user_offset = ftell(userfile);
    xi_offset = 0;
    currec = toprec;
    while(currec)
    {
	strcpy(tmpstr, currec -> SubDate.getDateStr(DATES_MMDDYY));
	switch(Config_obj.amu_cfgvar.BBStype)
	{
	    case RA250:
	    case RA2  :
	    case PB2  : fread(&ra20uservar, sizeof(ra20uservar), 1, userfile);
			ra20uservar.security = currec -> security;
			if(currec -> daysforward)
			{
			    adjust_subdate(currec);
			    Amustr_obj.ctopascal(tmpstr, ra20uservar.subdate);
			}
			Amustr_obj.ctopascal(currec -> city, ra20uservar.location);
			fseek(userfile, user_offset, SEEK_SET);
			write(fileno(userfile), &ra20uservar, sizeof(ra20uservar));
			user_offset += sizeof(ra20uservar);
			break;

	    case TG310:
	    case TG3  : fread(&t302uservar, sizeof(t302uservar), 1, userfile);
			t302uservar.level = currec -> security;
			if(currec -> daysforward)
			{
			    adjust_subdate(currec);
			    Amustr_obj.ctopascal(tmpstr, t302uservar.expiredate);
			}
			Amustr_obj.ctopascal(currec -> city, t302uservar.location);
			fseek(userfile, user_offset, SEEK_SET);
			write(fileno(userfile), &t302uservar, sizeof(t302uservar));
			user_offset += sizeof(t302uservar);
			break;


	    case RA1: fread(&ra10uservar, sizeof(ra10uservar), 1, userfile);
		      fread(&ra10userxivar, sizeof(ra10userxivar),1, xifile);
		      ra10uservar.seclvl = currec -> security;
		      if(currec -> daysforward)
		      {
			  adjust_subdate(currec);
			  Amustr_obj.ctopascal(tmpstr, ra10userxivar.subdate);
		      }
		      Amustr_obj.ctopascal(currec -> city, ra10uservar.location);
		      fseek(userfile, user_offset, SEEK_SET);
		      fseek(xifile, xi_offset, SEEK_SET);
		      write(fileno(userfile), &ra10uservar, sizeof(ra10uservar));
		      write(fileno(xifile), &ra10userxivar, sizeof(ra10userxivar));
		      xi_offset += sizeof(ra10userxivar);
		      user_offset += sizeof(ra10uservar);
		      break;

	    case QBBS: fread(&q276uservar, sizeof(q276uservar), 1, userfile);
		       q276uservar.seclvl = currec -> security;
		       Amustr_obj.ctopascal(currec -> city, q276uservar.location);
		       fseek(userfile, user_offset, SEEK_SET);
		       write(fileno(userfile), &q276uservar, sizeof(q276uservar));
		       user_offset += sizeof(q276uservar);
		       break;

	    case QBBSC:
	    case QBBSG: fread(&q280uservar, sizeof(q280uservar), 1, userfile);
			q280uservar.security = currec -> security;
			Amustr_obj.ctopascal(currec -> city, q280uservar.location);
			if(currec -> daysforward)
			{
			    q280uservar.expiredate += currec -> daysforward;
			}
			fseek(userfile, user_offset, SEEK_SET);
                        write(fileno(userfile), &q280uservar, sizeof(q280uservar));
			user_offset += sizeof(q280uservar);
			break;

	    case SBBS: fread(&s117uservar, sizeof(s117uservar), 1, userfile);
		       fread(&s117userxivar, sizeof(s117userxivar), 1, xifile);
		       Amustr_obj.ctopascal(currec -> city, s117uservar.location);
		       s117uservar.seclvl = currec -> security;
		       if(currec -> daysforward)
		       {
			   adjust_subdate(currec);
			   Amustr_obj.ctopascal(tmpstr, s117userxivar.expirationdate);
		       }
		       fseek(userfile, user_offset, SEEK_SET);
		       fseek(xifile, xi_offset, SEEK_SET);
		       write(fileno(userfile), &s117uservar, sizeof(s117uservar));
		       write(fileno(xifile), &s117userxivar, sizeof(s117userxivar));
		       user_offset += sizeof(s117uservar);
		       xi_offset += sizeof(s117userxivar);
		       break;

	    case LORA: fread(&lorauservar, sizeof(lorauservar), 1, userfile);
		       lorauservar.security = currec -> security;
		       strcpy(lorauservar.location, currec -> city);
		       if(currec -> daysforward)
		       {
			   adjust_subdate(currec);
			   strcpy(lorauservar.expiredate, currec -> SubDate.getDateStr(DATES_DDMMMYY));
		       }
		       fseek(userfile, user_offset, SEEK_SET);
		       write(fileno(userfile), &lorauservar, sizeof(lorauservar));
		       user_offset += sizeof(lorauservar);
		       break;

	    case LORA3:fread(&l300uservar, sizeof(l300uservar), 1, userfile);
		       l300uservar.Level = currec -> security;
		       strcpy(l300uservar.City, currec -> city);
		       fseek(userfile, user_offset, SEEK_SET);
		       write(fileno(userfile), &l300uservar, sizeof(l300uservar));
		       user_offset += sizeof(l300uservar);
		       break;

	    case EZ  : fread(&e120uservar, sizeof(e120uservar), 1, userfile);
		       fread(&e120userxivar, sizeof(e120userxivar), 1, xifile);
		       e120uservar.Security = currec -> security;
		       Amustr_obj.ctopascal(currec -> city, e120userxivar.Location);
		       fseek(userfile, user_offset, SEEK_SET);
		       fseek(xifile, xi_offset, SEEK_SET);
		       write(fileno(userfile), &e120uservar, sizeof(e120uservar));
		       write(fileno(xifile), &e120userxivar, sizeof(e120userxivar));
		       user_offset += sizeof(e120uservar);
		       xi_offset += sizeof(e120userxivar);
		       break;

	    case MAXIMUS2: fread(&max2uservar, sizeof(max2uservar), 1, userfile);
			   max2uservar.security = currec -> security;
			   strcpy(max2uservar.location, currec -> city);
			   if(currec -> daysforward)
			   {
			       adjust_subdate(currec);
			       max2uservar.expiredate.datevar.month = currec -> SubDate.getMonth();
			       max2uservar.expiredate.datevar.day = currec -> SubDate.getDay();
			       max2uservar.expiredate.datevar.year = currec -> SubDate.getYear() - 1980;
			   }
			   fseek(userfile, user_offset, SEEK_SET);
			   write(fileno(userfile), &max2uservar, sizeof(max2uservar));
			   user_offset += sizeof(max2uservar);
			   break;

	    case MAXIMUS3: fread(&max3uservar, sizeof(max3uservar), 1, userfile);
			   max3uservar.security = currec -> security;
			   strcpy(max3uservar.city, currec -> city);
			   if(currec -> daysforward)
			   {
			       adjust_subdate(currec);
			       max3uservar.expiredate.datevar.month = currec -> SubDate.getMonth();
			       max3uservar.expiredate.datevar.day = currec -> SubDate.getDay();
			       max3uservar.expiredate.datevar.year = currec -> SubDate.getYear() - 1980;
			   }
			   fseek(userfile, user_offset, SEEK_SET);
			   write(fileno(userfile), &max3uservar, sizeof(max3uservar));
			   user_offset += sizeof(max3uservar);
                           break;

            case CONCORD: fread(&concuservar, sizeof(concuservar), 1, userfile);
                          concuservar.sec.seclvl = currec -> security;
                          Amustr_obj.ctopascal(currec -> city, concuservar.city);
                          if(currec -> daysforward)
                          {
                              adjust_subdate(currec);
                          }
			  concuservar.expiration.day = currec -> SubDate.getDay();
			  concuservar.expiration.month = currec -> SubDate.getMonth();
			  concuservar.expiration.year = currec -> SubDate.getYear() - 1900;
                          fseek(userfile, user_offset, SEEK_SET);
                          write(fileno(userfile), &concuservar, sizeof(concuservar));
                          user_offset += sizeof(concuservar);
                          break;

        }
        currec = currec -> nextrec;
    }
}

/*********************************************************************/
/* Close the user files */
void Users::close_users(void)
{
    if(userfile)
    {
	fclose(userfile);
    }
    if(xifile)
    {
	fclose(xifile);
    }
    xifile = userfile = NULL;
}

/***********************************************************************/
/* De-allocate the user file linked list */
void Users::deallocate_users(void)
{
    currec = toprec;
    while(currec)
    {
	tmprec = currec;
	currec = currec -> nextrec;
        free(tmprec);
    }
}

/************************************************************************/
/* Clean up the location field of the user file */
void Users::clean_users(void)
{
    int i;

    currec = toprec;
    while(currec != NULL)
    {
        currec -> city[0] = toupper(currec -> city[0]);
        for(i = 1; i < strlen(currec -> city); i++)
        {
            if(isspace(currec -> city[i-1])  ||
               ispunct(currec -> city[i-1]))
            {
                currec -> city[i] = toupper(currec -> city[i]);
            }
            else
            {
                currec -> city[i] = tolower(currec -> city[i]);
            }
        }
        currec = currec -> nextrec;
    }
}

// Adjust the expiration date
void Users::adjust_subdate(struct user_array *userptr)
{
    long seconds;

    seconds = userptr -> SubDate.getSecondsSince1970();
    seconds += userptr -> daysforward * DATES_SECONDS_PER_DAY;
    userptr -> SubDate.createDate(seconds);
}

/***********************************************************************/
/* Options to post messages to the users */
void Users::user_msgs(int argc, char *argv[])
{
    int x, post_msg, found, killdata;
    FILE *datafile;
    struct screeninfo s;
    struct userdata usersdatavar;
    struct msgpostinfo msgpostvar;
    struct track trackvar;
    class Dates Today;
    char fn[MAX_FILESPEC], sysop[41], tmpparam[41], *useruls;
    unsigned long num_days;

    useruls = (char *)malloc(1);
    *useruls = NULL;

    strcpy(s.curoption, "Posting messages to users");
    memset(&msgpostvar, NULL, sizeof(msgpostvar));
    s.curarea[0] = NULL;
    Screen_obj.display(&s);

    killdata = FALSE;
    if(Config_obj.amu_cfgvar.alias_system)
    {
        strcpy(sysop, Config_obj.amu_cfgvar.sysopalias);
    }
    else
    {
        strcpy(sysop, Config_obj.amu_cfgvar.sysop);
    }

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.AMUpath, USERINFO);
    datafile = _fsopen(fn, "rb", SH_DENYWR);
    if(!datafile)
    {
        sprintf(LOGSTR, "! [USER_MSGS] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return;
    }

    Config_obj.open_config(USERMSGS_RECORD);
    Config_obj.open_config(MSGAREA_RECORD);
    for(x = 1; x <= Config_obj.num_usermsgs; x++)
    {
        if(!Config_obj.read_record(USERMSGS_RECORD, x))
	{
	    sprintf(LOGSTR, "! [USER_MSGS] Unable to read record #%d", x);
	    Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
            continue;
        }
        if(!Config_obj.usermsgsvar.enabled)
        {
            continue;
        }

	switch(Config_obj.usermsgsvar.type)
        {
            case MSG_NEWUSER:  strcpy(tmpparam, "NEWUSER_MSG");
			       break;
            case MSG_CARRIER:  strcpy(tmpparam, "CARRIER_MSG");
                               break;
            case MSG_UPLOADS:  strcpy(tmpparam, "UPLOADS_MSG");
                               break;
            case MSG_BIRTHDAY: strcpy(tmpparam, "BDAY_MSG");
                               break;
            case MSG_CREDIT  : strcpy(tmpparam, "CREDIT_MSG");
                               break;
            case MSG_SUBEXP  : strcpy(tmpparam, "SUBEXP_MSG");
                               break;
            default: tmpparam[0] = NULL;
                     break;
        }
        if(!Utility_obj.check_param(argc, argv, tmpparam, 0))
        {
            continue;
        }

        sprintf(s.curarea, "Record #%d", x);
        Screen_obj.display(&s);

        if(Config_obj.usermsgsvar.type == MSG_NEWUSER ||
           Config_obj.usermsgsvar.type == MSG_UPLOADS ||
           Config_obj.usermsgsvar.type == MSG_CARRIER)
        {
            fseek(datafile, 0, SEEK_SET);
	    fread(&usersdatavar, sizeof(usersdatavar), 1, datafile);
	    killdata = TRUE;
	    while(!feof(datafile))
            {
                post_msg = FALSE;
                found = FALSE;

              /* Loop through and try to find the user's record */
                currec = toprec;
                while(currec && !found)
		{
                    if(strcmp(currec -> name, usersdatavar.username) == 0 ||
                       strcmp(currec -> alias, usersdatavar.username) == 0)
		    {
                        found = TRUE;
                    }
                    else
                    {
                        currec = currec -> nextrec;
                    }
                }
                if(!found)
                {
                    currec = NULL;
                }
                switch(Config_obj.usermsgsvar.type)
                {
                    case MSG_NEWUSER: if(usersdatavar.newuser)
                                      {
                                          if(!currec)
                                          {
                                              sprintf(LOGSTR, "[USER_MSGS] '%s' is a newuser but not in user file, skipping",
                                                          usersdatavar.username);
                                              Utility_obj.logentry(LOGSTR, LOG_EXTENSIVE);
                                          }
                                          else
                                          {
                                              sprintf(LOGSTR, "[USER_MSGS] Posting newuser message to '%s'",
                                                          usersdatavar.username);
                                              Utility_obj.logentry(LOGSTR, LOG_NORMAL);
					      Config_obj.read_record(MSGAREA_RECORD, Config_obj.usermsgsvar.user_area);
					      msgpostvar.area = &Config_obj.msgareavar;
					      strcpy(msgpostvar.subject, "Newuser");
                                              strcpy(msgpostvar.from, sysop);
                                              strcpy(msgpostvar.to, usersdatavar.username);
                                              strcpy(msgpostvar.fn, Config_obj.usermsgsvar.user_fn);
                                              Msgs_obj.post_msg(&msgpostvar, currec);
                                              post_msg = TRUE;
                                          }
                                      }
				      break;
                    case MSG_UPLOADS: if(usersdatavar.uploads && !strstr(useruls, usersdatavar.username))
                                      {
					  if(!currec)
                                          {
                                              sprintf(LOGSTR, "[USER_MSGS] '%s' is an uploader but not in user file, skipping",
                                                          usersdatavar.username);
                                              Utility_obj.logentry(LOGSTR, LOG_EXTENSIVE);
                                          }
                                          else
                                          {
                                              useruls = (char *) realloc(useruls, strlen(useruls) + strlen(usersdatavar.username) + 2);
                                              strcat(useruls, usersdatavar.username);

                                              sprintf(LOGSTR, "[USER_MSGS] Posting upload 'thank you' message to '%s'",
                                                          usersdatavar.username);
                                              Utility_obj.logentry(LOGSTR, LOG_NORMAL);
                                              Config_obj.read_record(MSGAREA_RECORD, Config_obj.usermsgsvar.user_area);
                                              msgpostvar.area = &Config_obj.msgareavar;
                                              strcpy(msgpostvar.subject, "Files uploaded");
                                              strcpy(msgpostvar.from, sysop);
                                              strcpy(msgpostvar.to, usersdatavar.username);
                                              strcpy(msgpostvar.fn, Config_obj.usermsgsvar.user_fn);
                                              Msgs_obj.post_msg(&msgpostvar, currec);
                                              post_msg = TRUE;
                                          }
                                      }
                                      break;
                    case MSG_CARRIER: if(usersdatavar.carrier)
                                      {
					  if(!currec)
					  {
					      sprintf(LOGSTR, "[USER_MSGS] '%s' dropped carrier but not in user file, skipping",
                                                          usersdatavar.username);
                                              Utility_obj.logentry(LOGSTR, LOG_EXTENSIVE);
                                          }
                                          else
                                          {
                                              sprintf(LOGSTR, "[USER_MSGS] Posting lost carrier message to '%s'",
                                                          usersdatavar.username);
					      Utility_obj.logentry(LOGSTR, LOG_NORMAL);
                                              Config_obj.read_record(MSGAREA_RECORD, Config_obj.usermsgsvar.user_area);
                                              msgpostvar.area = &Config_obj.msgareavar;
					      strcpy(msgpostvar.subject, "Improper logoff");
                                              strcpy(msgpostvar.from, sysop);
                                              strcpy(msgpostvar.to, usersdatavar.username);
                                              strcpy(msgpostvar.fn, Config_obj.usermsgsvar.user_fn);
                                              Msgs_obj.post_msg(&msgpostvar, currec);
                                              post_msg = TRUE;
                                          }
                                      }
                                      break;
                }

              /* If a message was posted, see if we need to post one to
                 the sysop as well */
                if(post_msg && Config_obj.usermsgsvar.sysop_area > 0 &&
                   Config_obj.usermsgsvar.sysop_fn[0])
                {
                    strcpy(msgpostvar.from, "AMU");
                    strcpy(msgpostvar.to, sysop);
                    strcpy(msgpostvar.fn, Config_obj.usermsgsvar.sysop_fn);
                    Config_obj.read_record(MSGAREA_RECORD, Config_obj.usermsgsvar.sysop_area);
                    msgpostvar.area = &Config_obj.msgareavar;
                    Msgs_obj.post_msg(&msgpostvar, currec);
                }
                fread(&usersdatavar, sizeof(usersdatavar), 1, datafile);
            }
	}
	else
	{
	    killdata = TRUE;
	    open_track(Config_obj.usermsgsvar.type);
	    currec = toprec;
	    while(currec)
	    {
		post_msg = FALSE;
		trackvar.crc = Crc_obj.crc32(currec -> name);
		switch(Config_obj.usermsgsvar.type)
		{
		    case MSG_BIRTHDAY: if(Today.getMonth() == currec -> BirthDate.getMonth()
					  && Today.getDay() == currec -> BirthDate.getDay())
                                       {
					   trackvar.year = currec -> BirthDate.getYear();
					   trackvar.month = currec -> BirthDate.getMonth();
					   trackvar.day = currec -> BirthDate.getDay();
                                           if(!entry_exists(&trackvar))
                                           {
                                               sprintf(LOGSTR, "[USER_MSGS] Posting birthday message to '%s'",
                                                       currec -> name);
                                               Utility_obj.logentry(LOGSTR, LOG_NORMAL);
                                               Config_obj.read_record(MSGAREA_RECORD, Config_obj.usermsgsvar.user_area);
                                               msgpostvar.area = &Config_obj.msgareavar;
                                               strcpy(msgpostvar.subject, "Happy birthday!");
                                               strcpy(msgpostvar.from, sysop);
                                               if(Config_obj.amu_cfgvar.alias_system)
                                               {
                                                   strcpy(msgpostvar.to, currec -> alias);
                                               }
                                               else
                                               {
                                                   strcpy(msgpostvar.to, currec -> name);
                                               }
                                               strcpy(msgpostvar.fn, Config_obj.usermsgsvar.user_fn);
                                               Msgs_obj.post_msg(&msgpostvar, currec);
                                               post_msg = TRUE;
                                           }
				       }
				       break;

		    case MSG_SUBEXP: trackvar.year = currec -> SubDate.getYear();
				     trackvar.day = currec -> SubDate.getDay();
				     trackvar.month = currec -> SubDate.getMonth();
				     num_days = Today.getDateDiff(&currec -> SubDate);
                                     if(num_days > 0 && num_days <= Config_obj.usermsgsvar.num_days)
                                     {
                                           if(!entry_exists(&trackvar))
                                           {
                                               sprintf(LOGSTR, "[USER_MSGS] Posting subscription message to '%s'",
                                                       currec -> name);
                                               Utility_obj.logentry(LOGSTR, LOG_NORMAL);
                                               Config_obj.read_record(MSGAREA_RECORD, Config_obj.usermsgsvar.user_area);
                                               msgpostvar.area = &Config_obj.msgareavar;
                                               strcpy(msgpostvar.subject, "Subscription expiration");
                                               strcpy(msgpostvar.from, sysop);
                                               if(Config_obj.amu_cfgvar.alias_system)
                                               {
                                                   strcpy(msgpostvar.to, currec -> alias);
                                               }
                                               else
                                               {
                                                   strcpy(msgpostvar.to, currec -> name);
                                               }
                                               strcpy(msgpostvar.fn, Config_obj.usermsgsvar.user_fn);
                                               Msgs_obj.post_msg(&msgpostvar, currec);
                                               post_msg = TRUE;
                                           }
                                     }
                                     break;
                    case MSG_CREDIT: trackvar.value = currec -> credit;
                                     if(currec -> credit < Config_obj.usermsgsvar.num_credit
					&& !(!Config_obj.usermsgsvar.post_zero && currec -> credit == 0))

				     {
                                           if(!entry_exists(&trackvar))
					   {
                                               sprintf(LOGSTR, "[USER_MSGS] Posting low credit message to '%s'",
                                                       currec -> name);
                                               Utility_obj.logentry(LOGSTR, LOG_NORMAL);
                                               Config_obj.read_record(MSGAREA_RECORD, Config_obj.usermsgsvar.user_area);
                                               msgpostvar.area = &Config_obj.msgareavar;
                                               strcpy(msgpostvar.subject, "Low credit");
                                               strcpy(msgpostvar.from, sysop);
                                               if(Config_obj.amu_cfgvar.alias_system)
                                               {
                                                   strcpy(msgpostvar.to, currec -> alias);
                                               }
                                               else
                                               {
                                                   strcpy(msgpostvar.to, currec -> name);
                                               }
                                               strcpy(msgpostvar.fn, Config_obj.usermsgsvar.user_fn);
                                               Msgs_obj.post_msg(&msgpostvar, currec);
                                               post_msg = TRUE;
                                           }
                                     }
                                     break;             }
              /* If a message was posted, see if we need to post one to
		 the sysop as well */
                if(post_msg && Config_obj.usermsgsvar.sysop_area > 0 &&
                   Config_obj.usermsgsvar.sysop_fn[0])
                {
                    strcpy(msgpostvar.from, "AMU");
                    strcpy(msgpostvar.to, sysop);
                    strcpy(msgpostvar.fn, Config_obj.usermsgsvar.sysop_fn);
                    Config_obj.read_record(MSGAREA_RECORD, Config_obj.usermsgsvar.sysop_area);
                    msgpostvar.area = &Config_obj.msgareavar;
                    Msgs_obj.post_msg(&msgpostvar, currec);
                }
                currec = currec -> nextrec;
            }
	    close_track(Config_obj.usermsgsvar.type);
	}
    }
    Config_obj.close_config(USERMSGS_RECORD);
    Config_obj.close_config(MSGAREA_RECORD);
    fclose(datafile);
    free(useruls);
    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.AMUpath, USERINFO);
    if(killdata)
    {
        remove(fn);
    }
}

/*************************************************************************/
/* Process the Enforcement options */
void Users::enforce(void)
{
    char msgtype[21];
    int index, post_msg;
    struct screeninfo s;
    struct msgpostinfo msgpostvar;
    class Dates Today;
    float x, ratio;

    strcpy(s.curoption, "Processing enforcement options");

    Config_obj.open_config(ENFORCE_RECORD);
    Config_obj.open_config(MSGAREA_RECORD);
    for(index = 1; index <= Config_obj.num_enforce; index++)
    {
        if(!Config_obj.read_record(ENFORCE_RECORD, index))
	{
	    sprintf(LOGSTR, "! [ENFORCE] Unable to read record #%d", index);
	    Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
	    continue;
	}
	if(!Config_obj.enforcevar.enabled)
	{
	    continue;
	}

	if(Config_obj.amu_cfgvar.BBStype == LORA3 && (Config_obj.enforcevar.type == POSTS || Config_obj.enforcevar.type == POSTCALL))
	{
	    sprintf(LOGSTR, "! [ENFORCE] Note - 'Posts' and 'Post:Call' options are incompatible with LoraBBS 3.00, skipping");
	    Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
	    continue;
	}

	sprintf(s.curarea, "Record #%d", index);
        Screen_obj.display(&s);

	currec = toprec;
	while(currec)
	{
	    post_msg = FALSE;
	    if(currec -> security != Config_obj.enforcevar.cur_level)
	    {
		currec = currec -> nextrec;
		continue;
	    }

	    switch(Config_obj.enforcevar.type)
            {
                case ULS: switch(Config_obj.enforcevar.comparison)
                          {
                              case GREATER: if(currec -> uls > Config_obj.enforcevar.num)
                                            {
						post_msg = TRUE;
                                            }
                                            break;
                              case LESS: if(currec -> uls < Config_obj.enforcevar.num)
                                         {
                                             post_msg = TRUE;
                                         }
                                         break;
                          }
                          if(post_msg)
                          {
                              strcpy(msgtype, "[ULS]");
                          }
                          break;

                case DLS: switch(Config_obj.enforcevar.comparison)
                          {
			      case GREATER: if(currec -> dls > Config_obj.enforcevar.num)
                                            {
                                                post_msg = TRUE;
                                            }
                                            break;
                              case LESS: if(currec -> dls < Config_obj.enforcevar.num)
                                         {
					     post_msg = TRUE;
                                         }
                                         break;
                          }
                          if(post_msg)
                          {
                              strcpy(msgtype, "[DLS]");
                          }
                          break;

		case CALLS: switch(Config_obj.enforcevar.comparison)
			    {
                                case GREATER: if(currec -> calls > Config_obj.enforcevar.num)
                                              {
                                                  post_msg = TRUE;
                                              }
                                              break;
                                case LESS: if(currec -> calls < Config_obj.enforcevar.num)
                                           {
                                               post_msg = TRUE;
                                           }
                                           break;
                            }
                            if(post_msg)
                            {
                                strcpy(msgtype, "[CALLS]");
                            }
                            break;

                case POSTS: switch(Config_obj.enforcevar.comparison)
                            {
                                case GREATER: if(currec -> posts > Config_obj.enforcevar.num)
					      {
						  post_msg = TRUE;
                                              }
                                              break;
                                case LESS: if(currec -> posts < Config_obj.enforcevar.num)
                                           {
                                               post_msg = TRUE;
                                           }
					   break;
                            }
                            if(post_msg)
                            {
                                strcpy(msgtype, "[POSTS]");
                            }
                            break;

            case ULDL:   if(!Config_obj.enforcevar.num2  ||  !currec -> dls)
                         {
			     break;
                         }
                         x = (float) currec -> uls / (float) currec -> dls;
			 ratio = (float) Config_obj.enforcevar.num / (float) Config_obj.enforcevar.num2;
                         switch(Config_obj.enforcevar.comparison)
                         {
                             case GREATER: if(x > ratio  &&  x > 0.0)
                                           {
                                               post_msg = TRUE;
                                           }
                                           break;
                             case LESS: if(x < ratio  &&  x > 0.0)
                                        {
                                            post_msg = TRUE;
                                        }
                                        break;
                         }
                         if(post_msg)
                         {
                             strcpy(msgtype, "[UL:DL]");
                         }
                         break;

	    case ULDLKB: if(!Config_obj.enforcevar.num2  ||  !currec -> dlkb)
                         {
                             break;
                         }
                         x = (float) currec -> ulkb / (float) currec -> dlkb;
                         ratio = (float) Config_obj.enforcevar.num / (float) Config_obj.enforcevar.num2;
                         switch(Config_obj.enforcevar.comparison)
			 {
                             case GREATER: if(x > ratio  &&  x > 0.0)
                                           {
                                               post_msg = TRUE;
                                           }
                                           break;
                             case LESS: if(x < ratio &&  x > 0.0)
                                        {
                                            post_msg = TRUE;
                                        }
					break;
                         }
                         if(post_msg)
                         {
                             strcpy(msgtype, "[ULKB:DLKB]");
                         }
                         break;

            case POSTCALL: if(!Config_obj.enforcevar.num2  ||  !currec -> calls)
                           {
                               break;
                           }
                           x = (float) currec -> posts / (float) currec -> calls;
                           ratio = (float) Config_obj.enforcevar.num / (float) Config_obj.enforcevar.num2;
                           switch(Config_obj.enforcevar.comparison)
                           {
                               case GREATER: if(x > ratio  &&  x > 0.0)
                                             {
                                                 post_msg = TRUE;
                                             }
                                             break;
                               case LESS: if(x < ratio  &&  x > 0.0)
                                          {
					      post_msg = TRUE;
                                          }
                                          break;
                           }
                           if(post_msg)
                           {
                               strcpy(msgtype, "[POST:CALL]");
			   }
                           break;

	    case SUBSCRIPTION: if(currec -> SubDate.getYear() <= 1900)
			       {
				   break;
			       }
			       if(currec -> SubDate.compareDates(&Today) > 0)
			       {
				   Utility_obj.logentry(LOGSTR, LOG_MINIMAL);

				   // See if we need to adjust the subscription date
                                   currec -> daysforward = Config_obj.enforcevar.days_forward;
                                   post_msg = TRUE;
                                   strcpy(msgtype, "[SUBSCRIPTION]");
                               }
                               break;
            }
            if(post_msg)
            {
                currec -> security = Config_obj.enforcevar.next_level;
                sprintf(LOGSTR, "%s Level changed from %d to %d for '%s'", msgtype,
                        Config_obj.enforcevar.cur_level, currec -> security,
                        currec -> name);
		Utility_obj.logentry(LOGSTR, LOG_NORMAL);

                if(Config_obj.enforcevar.user_fn[0])
                {
                    if(Config_obj.amu_cfgvar.alias_system)
                    {
                        strcpy(msgpostvar.to, currec -> alias);
                        strcpy(msgpostvar.from, Config_obj.amu_cfgvar.sysopalias);
                    }
                    else
                    {
                        strcpy(msgpostvar.to, currec -> name);
                        strcpy(msgpostvar.from, Config_obj.amu_cfgvar.sysop);
                    }
                    strcpy(msgpostvar.subject, "Security level change");
                    strcpy(msgpostvar.fn, Config_obj.enforcevar.user_fn);
		    Config_obj.read_record(MSGAREA_RECORD, Config_obj.enforcevar.user_area);
		    msgpostvar.area = &Config_obj.msgareavar;
		    Msgs_obj.post_msg(&msgpostvar, currec);
		}
		if(Config_obj.enforcevar.sysop_fn[0])
		{
		    strcpy(msgpostvar.to, Config_obj.amu_cfgvar.sysop);
		    strcpy(msgpostvar.from, "AMU");
		    strcpy(msgpostvar.fn, Config_obj.enforcevar.sysop_fn);
		    Config_obj.read_record(MSGAREA_RECORD, Config_obj.enforcevar.sysop_area);
		    msgpostvar.area = &Config_obj.msgareavar;
		    Msgs_obj.post_msg(&msgpostvar, currec);
		}
	    }
	    currec = currec -> nextrec;
	}
    }
    Config_obj.close_config(ENFORCE_RECORD);
    Config_obj.close_config(MSGAREA_RECORD);
}

/***********************************************************************/
/* Open the tracking file */
void Users::open_track(int type)
{
    strcpy(trackfn, Config_obj.amu_cfgvar.AMUpath);
    switch(type)
    {
        case MSG_BIRTHDAY: strcat(trackfn, BDAY_TRACK);
                           break;
        case MSG_SUBEXP  : strcat(trackfn, SUBEXP_TRACK);
                           break;
        case MSG_CREDIT  : strcat(trackfn, CREDIT_TRACK);
                           break;
    }
    trackfile = _fsopen(trackfn, "r+b", SH_DENYNO);
    if(!trackfile)
    {
        trackfile = _fsopen(trackfn, "w+b", SH_DENYNO);
    }
}

/***********************************************************************/
/* Return if the entry exists or not.  If not, make a new entry */
int Users::entry_exists(struct track *x)
{
    int status;

    status = FALSE;
    fseek(trackfile, 0, SEEK_SET);
    fread(&trackvar, sizeof(trackvar), 1, trackfile);
    while(!status && !feof(trackfile))
    {
        if(trackvar.crc == x -> crc)
        {
            status = TRUE;
        }
        else
        {
            fread(&trackvar, sizeof(trackvar), 1, trackfile);
        }
    }
    if(!status)
    {
	fseek(trackfile, 0, SEEK_END);
        x -> deleted = FALSE;
        fwrite(x, sizeof(trackvar), 1, trackfile);
    }
    return(status);
}

/***********************************************************************/
/* Close and pack the tracking file */
void Users::close_track(int type)
{
    char tmpfn[MAX_FILESPEC], tmp[11];
    FILE *tmpfile;
    struct dosdate_t d;
    unsigned long sub1, sub2;

    _dos_getdate(&d);
    d.year -= 1900;
    sprintf(tmp, "%d%d%d", d.year, d.month, d.day);
    sub1 = atol(tmp);
    sprintf(tmpfn, "%sTMPTRACK.AMU", Config_obj.amu_cfgvar.AMUpath);
    tmpfile = _fsopen(tmpfn, "w+b", SH_DENYNO);
    fseek(trackfile, 0, SEEK_SET);
    fread(&trackvar, sizeof(trackvar), 1, trackfile);
    while(!feof(trackfile))
    {
        switch(type)
        {
            case MSG_BIRTHDAY: if(trackvar.day != d.day || trackvar.month != d.month)
                               {
                                   trackvar.deleted = TRUE;
                               }
                               break;
            case MSG_SUBEXP: if(trackvar.year <= 80)
                             {
                                 trackvar.year += 100;
                             }
                             sprintf(tmp, "%d%d%d", trackvar.year,
                                     trackvar.month, trackvar.day);
                             sub2 = atol(tmp);
			  /* If today is later than the sub date, delete it */
                             if(sub1 > sub2)
                             {
                                 trackvar.deleted = TRUE;
                             }
			     break;
            case MSG_CREDIT: currec = toprec;
                             while(currec)
                             {
                                 if(Crc_obj.crc32(currec -> name) == trackvar.crc
                                    && currec -> credit >= trackvar.value)
                                 {
                                     trackvar.deleted = TRUE;
                                     currec = NULL;
                                 }
                                 else
                                 {
                                     currec = currec -> nextrec;
                                 }
                             }
                             break;
        }
        if(!trackvar.deleted)
        {
            fwrite(&trackvar, sizeof(trackvar), 1, tmpfile);
        }
        fread(&trackvar, sizeof(trackvar), 1, trackfile);
    }
    fclose(trackfile);
    fclose(tmpfile);
    Utility_obj.copyfile(tmpfn, trackfn);
    remove(tmpfn);
}

/************************************************************************/
/* EOF - USERS.CPP */
