// Functions to write messages to the message base, Y2K
// Created 23 February 1996
// Revised 07 August 1998

#include <stdio.h>
#include <string.h>
#include <io.h>
#include <dos.h>
#include <share.h>
#include <stdlib.h>
#include <fcntl.h>
#include <mem.h>
#include <ctype.h>
#include <time.h>
#include "extern.h"
#include "msgs.h"
#include "config.h"
#include "bbsinfo.h"
#include "users.h"
#if defined(__AMU32__)
  #include <i86.h>
#endif

// External objects
extern class Config Config_obj;

/***********************************************************************/
// Constructor for this class
Msgs::Msgs(void)
{
#if defined(__AMU16__) || defined(__AMU32__)
    union REGS regs;

    /* See if SHARE.EXE is installed */
    regs.h.ah = 0x0010;
    regs.h.al = 0x0000;
#if defined(__AMU16__)
    int86(0x002F, &regs,&regs);
#endif
#if defined(__AMU32__)
    int386(0x002F, &regs, &regs);
#endif
    if(regs.h.al == 0x00FF)
    {
        share_installed = TRUE;
    }
    else
    {
        share_installed = FALSE;
    }
#else
    share_installed = TRUE;
#endif
}


/*************************************************************************/
/* Get information about the current EZYCom area */
long Msgs::EZY_info(struct msgarea *area)
{
    unsigned short AreaCount;
    FILE *msghdr;
    char fn[81];

    sprintf(fn, "%sMSGCOUNT.BBS", Config_obj.amu_cfgvar.hmbpath);
    msghdr = _fsopen(fn, "rb", SH_DENYNO);
    if(msghdr)
    {
        fseek(msghdr, (area -> areanum - 1) * sizeof(short), SEEK_SET);
        fread(&AreaCount, sizeof(AreaCount), 1, msghdr);
        fclose(msghdr);
        return(AreaCount);
    }
    else
    {
        sprintf(LOGSTR, "? [EZY_INFO] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(-1);
    }
}


/*************************************************************************/
/* Get information about the current JAM area */
long Msgs::JAM_info(struct msgarea *area)
{
    JAMHDRINFO jamhdrvar;
    FILE *msghdr;
    char fn[81];

    sprintf(fn, "%s.JHR", area -> basepath);
    msghdr = _fsopen(fn, "rb", SH_DENYNO);
    if(msghdr)
    {
        fread(&jamhdrvar, sizeof(jamhdrvar), 1, msghdr);
        fclose(msghdr);
        return(jamhdrvar.ActiveMsgs);
    }
    else
    {
        sprintf(LOGSTR, "? [JAM_INFO] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(-1);
    }
}

/*************************************************************************/
/* Get information about the current Hudson area */
long Msgs::Hudson_info(struct msgarea *area)
{
    struct hmb_msginfo msginfovar;
    FILE *msginfo;
    char fn[81];

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGINFO.BBS");
    msginfo = _fsopen(fn, "rb", SH_DENYNO);
    if(msginfo)
    {
        fread(&msginfovar, sizeof(msginfovar), 1, msginfo);
        fclose(msginfo);
        return(msginfovar.totalonboard[area -> areanum - 1]);
    }
    else
    {
        sprintf(LOGSTR, "? [HMB_INFO] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(-1);
    }
}

/************************************************************************/
/* Get information about the current GoldBase area */
long Msgs::Gold_info(struct msgarea *area)
{
    struct gold_msginfo msginfovar;
    FILE *msginfo;
    char fn[81];

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGINFO.DAT");
    msginfo = _fsopen(fn, "rb", SH_DENYNO);
    if(msginfo)
    {
        fread(&msginfovar, sizeof(msginfovar), 1, msginfo);
        fclose(msginfo);
        return(msginfovar.totalonboard[area -> areanum - 1]);
    }
    else
    {
        sprintf(LOGSTR, "? [GOLD_INFO] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(-1);
    }
}

/*************************************************************************/
/* Get information about the current Squish area */
long Msgs::Squish_info(struct msgarea *area)
{
    FILE *infile;
    char fn[81];
    long num_msgs;

    sprintf(fn, "%s.SQD", area -> basepath);
    infile = _fsopen(fn, "rb", SH_DENYWR);
    if(infile)
    {
        fseek(infile, 4, SEEK_SET);
        fread(&num_msgs, sizeof(num_msgs), 1, infile);
        fclose(infile);
    }
    else
    {
        sprintf(LOGSTR, "? [SQ_INFO] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        num_msgs = -1;
    }
    return(num_msgs);
}

/*************************************************************************/
/* Get information on the *.MSG area */
long Msgs::MSG_info(struct msgarea *area)
{
    char filespec[MAX_FILESPEC];
    struct find_t fileinfo;
    int done;
    long num_msgs;

    sprintf(filespec, "%s*.MSG", area -> basepath);
    num_msgs = 0;
    done = _dos_findfirst(filespec, _A_ARCH, &fileinfo);
    while(!done)
    {
        num_msgs++;
        done = _dos_findnext(&fileinfo);
    }
    return(num_msgs);
}

/***********************************************************************/
/* Generic function to get information about the message area */
long Msgs::area_info(struct msgarea *area)
{
    long i;

    i = -1;
    switch(area -> areatype)
    {
        case HUDSON_TYPE: i = Hudson_info(area);
                          break;
        case JAM_TYPE   : i = JAM_info(area);
                          break;
        case EZY_TYPE   : i = EZY_info(area);
                          break;
        case GOLD_TYPE  : i = Gold_info(area);
                          break;
        case SQUISH_TYPE: i = Squish_info(area);
                          break;
        case MSG_TYPE   : i = MSG_info(area);
                          break;
        default: sprintf(LOGSTR, "? [MSG_INFO] Unsupported message base type - Area #%d",
                         area -> areanum);
                 Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
                 break;
    }
    return(i);
}


/*************************************************************************/
/* Post a message to an EZYCom area */
int Msgs::write_to_EZY(struct msgpostinfo *msgpostvar)
{
    unsigned short areacount;
    struct EZ_MsgFastRecord msgfastvar;
    struct EZ_MsgHdrRecord msghdrvar;
    FILE *msgcount, *msgtxt, *msgfast, *msghdr, *txtfile;
    char fn[81], msgtxtvar[256];
    unsigned i;
    unsigned short areabase;
    union bbstimedate packedtime;

    packedtime.t1.time.timevar.seconds = Today.getSeconds();
    packedtime.t1.time.timevar.minutes = Today.getMinutes();
    packedtime.t1.time.timevar.hours = Today.getHour24();
    packedtime.t1.date.datevar.day = Today.getDay();
    packedtime.t1.date.datevar.month = Today.getMonth();
    packedtime.t1.date.datevar.year = Today.getYear() - 1980;

    /* Calculate the base path for this area */
    /*   FileName:     <msgpath>\AREA<(<area>-1)/100+1>\MH<area>.BBS */
    areabase = (msgpostvar -> area -> areanum - 1) / 100 + 1;


    /* Attempt to open the file to be posted */
    sprintf(fn, "%s", msgpostvar->fn);
    txtfile = fopen(fn, "rt");
    if(!txtfile)
    {
        sprintf(LOGSTR, "? [POST_EZY] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }


    /* Attempt to lock the EZY base
       Locking specification:
       Lock the first byte of the MH*.BBS file
    */
    sprintf(fn, "%sAREA%u\\MH%05d.BBS", Config_obj.amu_cfgvar.hmbpath, areabase,
            msgpostvar -> area -> areanum);
    msghdr = _fsopen(fn, "r+b", SH_DENYNO);
    if(msghdr)
    {
        if(share_installed  &&  lock(fileno(msghdr), 0, 1))
        {
            sprintf(LOGSTR, "? [POST_EZY] Unable to lock '%s' - (%s)", fn, strerror(errno));
            Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
            fclose(txtfile);
            return(FALSE);
        }
    }
    else
    {
        sprintf(LOGSTR, "? [POST_EZY] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }

    /* Now open the rest of the EZY files */

    /* Open the MT*.BBS file */
    sprintf(fn, "%sAREA%u\\MT%05d.BBS", Config_obj.amu_cfgvar.hmbpath, areabase,
            msgpostvar -> area -> areanum);
    msgtxt = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgtxt)
    {
        sprintf(LOGSTR, "? [POST_EZY] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msghdr), 0, 1);
        fclose(msghdr);
        fclose(txtfile);
        return(FALSE);
    }

    /* Open the MSGCOUNT.BBS file */
    sprintf(fn, "%sMSGCOUNT.BBS", Config_obj.amu_cfgvar.hmbpath);
    msgcount = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgcount)
    {
        sprintf(LOGSTR, "? [POST_EZY] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msghdr), 0, 1);
        fclose(msghdr);
        fclose(txtfile);
        fclose(msgtxt);
        return(FALSE);
    }

    /* Open the MSGFAST.BBS file */
    sprintf(fn, "%sMSGFAST.BBS", Config_obj.amu_cfgvar.hmbpath);
    msgfast = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgfast)
    {
        sprintf(LOGSTR, "? [POST_EZY] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msghdr), 0, 1);
        fclose(msghdr);
        fclose(txtfile);
        fclose(msgtxt);
        fclose(msgcount);
        return(FALSE);
    }


    /* If we're at this point, the EZY base is open.  Start writing! */

    /* Update the MH*.BBS file */
    memset(&msghdrvar, NULL, sizeof(msghdrvar));

    msghdrvar.StartPosition = filelength(fileno(msgtxt));
    msghdrvar.MessageLength = filelength(fileno(txtfile)) + 1;
    msghdrvar.MsgAttr = BIT3 | BIT6;
    msghdrvar.PostTimeDate = packedtime.t2;
    msghdrvar.l1 = strlen(msgpostvar -> to);
    Amustr_obj.ctopascal(msgpostvar -> to, msghdrvar.WhoTo);
    msghdrvar.l2 = strlen(msgpostvar -> from);
    Amustr_obj.ctopascal(msgpostvar -> from, msghdrvar.WhoFrom);
    msghdrvar.l3 = strlen(msgpostvar -> subject);
    Amustr_obj.ctopascal(msgpostvar -> subject, msghdrvar.Subject);
    fseek(msghdr, 0, SEEK_END);
    fwrite(&msghdrvar, sizeof(msghdrvar), 1, msghdr);


    /* Update the  MT*.BBS file */
    fseek(msgtxt, 0, SEEK_END);
    memset(msgtxtvar, NULL, sizeof(msgtxtvar));
    _dos_read(fileno(txtfile), msgtxtvar, sizeof(msgtxtvar), &i);
    while(i > 0)
    {
        _dos_write(fileno(msgtxt), msgtxtvar, i, &i);
        memset(msgtxtvar, NULL, sizeof(msgtxtvar));
        _dos_read(fileno(txtfile), msgtxtvar, sizeof(msgtxtvar), &i);
    }
    msgtxtvar[0] = NULL;
    _dos_write(fileno(msgtxt), msgtxtvar, 1, &i);


    /* Update the MSGFAST.BBS file */
    msgfastvar.WhoTo = Crc_obj.zmodemcrc32(strupr(msgpostvar -> to));
    msgfastvar.MsgBoard = msgpostvar -> area -> areanum;
    msgfastvar.MsgNumber = filelength(fileno(msghdr)) / sizeof(msghdrvar);
    fseek(msgfast, 0, SEEK_END);
    fwrite(&msgfastvar, sizeof(msgfastvar), 1, msgfast);


    /* Update the MSGCOUNT.BBS file */
    fseek(msgcount, (msgpostvar -> area -> areanum - 1) * sizeof(short), SEEK_SET);
    fread(&areacount, sizeof(areacount), 1, msgcount);
    areacount++;
    fseek(msgcount, (msgpostvar -> area -> areanum - 1) * sizeof(short), SEEK_SET);
    fwrite(&areacount, sizeof(areacount), 1, msgcount);


    /* Unlock and close the EZY base */
    unlock(fileno(msghdr), 0, 1);
    fclose(msghdr);
    fclose(msgtxt);
    fclose(msgcount);
    fclose(msgfast);
    fclose(txtfile);
    return(TRUE);
}


/*************************************************************************/
/* Post a message to a JAM area */
int Msgs::write_to_JAM(struct msgpostinfo *msgpostvar)
{
    JAMHDRINFO jamhdrinfovar;
    JAMHDR jamhdrvar;
    JAMBINSUBFIELD jambinsubfieldvar;
    JAMIDXREC jamidxvar;
    FILE *msghdr, *msgtxt, *msgidx, *txtfile;
    char fn[81], msgtxtvar[256];
    unsigned i;
    long nextmsg;

    /* Attempt to open the file to be posted */
    sprintf(fn, "%s", msgpostvar->fn);
    txtfile = fopen(fn, "rt");
    if(!txtfile)
    {
        sprintf(LOGSTR, "? [POST_JAM] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }

    /* Attempt to lock the JAM base
       Locking specification:
       Lock the first byte of the .JHR file
    */

    sprintf(fn, "%s.JHR", msgpostvar -> area -> basepath);
    msghdr = _fsopen(fn, "r+b", SH_DENYNO);
    if(msghdr)
    {
        if(share_installed  &&  lock(fileno(msghdr), 0, 1))
        {
            sprintf(LOGSTR, "? [POST_JAM] Unable to lock '%s' - (%s)", fn, strerror(errno));
            Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
            fclose(txtfile);
            return(FALSE);
        }
    }
    else
    {
        sprintf(LOGSTR, "? [POST_JAM] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }

    /* Now open the rest of the JAM files */

    /* Open the .JDT file */
    sprintf(fn, "%s.JDT", msgpostvar -> area -> basepath);
    msgtxt = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgtxt)
    {
        sprintf(LOGSTR, "? [POST_JAM] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msghdr), 0, 1);
        fclose(msghdr);
        fclose(txtfile);
        return(FALSE);
    }

    /* Open the .JDX file */
    sprintf(fn, "%s.JDX", msgpostvar -> area -> basepath);
    msgidx = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgidx)
    {
        sprintf(LOGSTR, "? [POST_JAM] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msghdr), 0, 1);
        fclose(msghdr);
        fclose(msgtxt);
        fclose(txtfile);
        return(FALSE);
    }


    /* If we're at this point, the JAM base is open.  Start writing! */

    /* Update the Header Information Block in the .JHR file */
    fseek(msghdr, 0, SEEK_SET);
    fread(&jamhdrinfovar, sizeof(jamhdrinfovar), 1, msghdr);

    /* The JAM spec says this gets wrapped to zero when it reaches ffffffffH */
    if(jamhdrinfovar.ModCounter == 0xFFFFFFFF)
    {
        jamhdrinfovar.ModCounter = 0;
    }
    jamhdrinfovar.ModCounter++;
    jamhdrinfovar.ActiveMsgs++;
    fseek(msghdr, 0, SEEK_SET);
    fwrite(&jamhdrinfovar, sizeof(jamhdrinfovar), 1, msghdr);


    /* Update the .JDT file */
    jamhdrvar.TxtOffset = filelength(fileno(msgtxt));
    jamhdrvar.TxtLen = i = 0;
    fseek(msgtxt, 0, SEEK_END);
    memset(msgtxtvar, NULL, sizeof(msgtxtvar));
    _dos_read(fileno(txtfile), msgtxtvar, sizeof(msgtxtvar), &i);
    while(i > 0)
    {
        _dos_write(fileno(msgtxt), msgtxtvar, i, &i);
        jamhdrvar.TxtLen += i;
        memset(msgtxtvar, NULL, sizeof(msgtxtvar));
        _dos_read(fileno(txtfile), msgtxtvar, sizeof(msgtxtvar), &i);
    }

    /* Add a header record to the end of the .JHR file */
    jamidxvar.HdrOffset = filelength(fileno(msghdr));
    fseek(msghdr, 0, SEEK_END);
    /* Calculate the next message number
       - Total number of records in .JDX file + BaseMsgNum + 1
         The extra '1' is there since the .JDX file has not yet
         been updated for this message.
    */
    nextmsg = jamhdrinfovar.BaseMsgNum + (filelength(fileno(msgidx)) / sizeof(jamidxvar));
    strcpy(jamhdrvar.Signature, HEADERSIGNATURE);
    jamhdrvar.Revision = CURRENTREVLEV;
    jamhdrvar.ReservedWord = 0;
    jamhdrvar.TimesRead = 0;
    jamhdrvar.SubfieldLen = (3 * sizeof(jambinsubfieldvar)) +
                            strlen(msgpostvar -> to) + strlen(msgpostvar -> from)
                            + strlen(msgpostvar -> subject);
    jamhdrvar.MsgIdCRC = 0;
    jamhdrvar.ReplyCRC = 0;
    jamhdrvar.ReplyTo = 0;
    jamhdrvar.Reply1st = 0;
    jamhdrvar.ReplyNext = 0;
    jamhdrvar.DateWritten = Today.getSecondsSince1970();
    jamhdrvar.DateReceived = 0;
    jamhdrvar.DateProcessed = 0;
    jamhdrvar.MsgNum = nextmsg;
    jamhdrvar.Attribute = MSG_LOCAL + MSG_PRIVATE;
    jamhdrvar.Attribute2 = 0;
    jamhdrvar.PasswordCRC = 0xFFFFFFFF;
    jamhdrvar.Cost = 0;
    fwrite(&jamhdrvar, sizeof(jamhdrvar), 1, msghdr);

    /* Create the subfields for this message header */
    jambinsubfieldvar.LoID = JAMSFLD_RECVRNAME;
    jambinsubfieldvar.HiID = 0;
    jambinsubfieldvar.DatLen = strlen(msgpostvar -> to);
    fwrite(&jambinsubfieldvar, sizeof(jambinsubfieldvar), 1, msghdr);
    fwrite(msgpostvar -> to, jambinsubfieldvar.DatLen, 1, msghdr);

    jambinsubfieldvar.LoID = JAMSFLD_SENDERNAME;
    jambinsubfieldvar.HiID = 0;
    jambinsubfieldvar.DatLen = strlen(msgpostvar -> from);
    fwrite(&jambinsubfieldvar, sizeof(jambinsubfieldvar), 1, msghdr);
    fwrite(msgpostvar -> from, jambinsubfieldvar.DatLen, 1, msghdr);

    jambinsubfieldvar.LoID = JAMSFLD_SUBJECT;
    jambinsubfieldvar.HiID = 0;
    jambinsubfieldvar.DatLen = strlen(msgpostvar -> subject);
    fwrite(&jambinsubfieldvar, sizeof(jambinsubfieldvar), 1, msghdr);
    fwrite(msgpostvar -> subject, jambinsubfieldvar.DatLen, 1, msghdr);

    /* Update the .JDX file */
    jamidxvar.UserCRC = Crc_obj.JAMcrc32(msgpostvar -> to);
    fseek(msgidx, 0, SEEK_END);
    fwrite(&jamidxvar, sizeof(jamidxvar), 1, msgidx);

    /* Unlock and close the JAM base */
    unlock(fileno(msghdr), 0, 1);
    fclose(msghdr);
    fclose(msgtxt);
    fclose(msgidx);
    fclose(txtfile);
    return(TRUE);
}


/************************************************************************/
/* Post a message to a Hudson area */
int Msgs::write_to_Hudson(struct msgpostinfo *msgpostvar)
{
    struct hmb_msginfo msginfovar;
    struct hmb_msgidx msgidxvar;
    struct hmb_msghdr msghdrvar;
    struct hmb_msgtoidx msgtoidxvar;
    FILE *msginfo, *msgidx, *msghdr, *msgtxt, *msgtoidx, *txtfile, *semaphore;
    char fn[81], fn2[81], msgtxtvar[256], time[6], date[9];
    unsigned startblock, numblocks, k, x;


    /* The date MUST be in mm-dd-yy format to follow the Hudson specs */
    strcpy(date, Today.getDateStr(DATES_MMDDYY));
    strcpy(time, Today.getTimeStr(DATES_HHMM24));

    sprintf(fn, "%s", msgpostvar -> fn);
    txtfile = fopen(fn, "rt");
    if(!txtfile)
    {
        sprintf("? [HMB_POST] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }

    /* Attempt to lock the message base

       Locking specification:

       Use the INT 21 5Ch function to lock a region of the MSGINFO.BBS file.
       This region starts at byte 407, and is 1 byte long. Locking a region

    */
    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGINFO.BBS");
    msginfo = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msginfo)
    {
        sprintf(fn2, "%sMBUNLOCK.NOW", Config_obj.amu_cfgvar.hmbpath);
        semaphore = fopen(fn2, "w+t");
        fclose(semaphore);
        k = 0;
        while(share_installed  &&  k < 10  &&  !msginfo)
        {
            sleep(2);
            k++;
            msginfo = _fsopen(fn, "r+b", SH_DENYNO);
        }
        remove(fn2);

        if(!msginfo)
        {
            sprintf(LOGSTR, "? [HMB_POST] Unable to open '%s' - (%s)", fn, strerror(errno));
            Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
            fclose(txtfile);
            return(FALSE);
        }
        else
        {
            lock(fileno(msginfo), 406, 1);
        }
    }

    /* Now open the rest of the Hudson files */

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGIDX.BBS");
    msgidx = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgidx)
    {
        sprintf(LOGSTR, "? [HMB_POST] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msginfo), 406, 1);
        fclose(msginfo);
        fclose(txtfile);
        return(FALSE);
    }

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGTOIDX.BBS");
    msgtoidx = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgtoidx)
    {
        sprintf(LOGSTR, "? [HMB_POST] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msginfo), 406, 1);
        fclose(msginfo);
        fclose(msgidx);
        fclose(txtfile);
        return(FALSE);
    }

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGHDR.BBS");
    msghdr = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msghdr)
    {
        sprintf(LOGSTR, "? [HMB_POST] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msginfo), 406, 1);
        fclose(msgtoidx);
        fclose(msgidx);
        fclose(msginfo);
        fclose(txtfile);
        return(FALSE);
    }

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGTXT.BBS");
    msgtxt = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgtxt)
    {
        sprintf(LOGSTR, "? [HMB_POST] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msginfo), 406, 1);
        fclose(msgtoidx);
        fclose(msgidx);
        fclose(msginfo);
        fclose(msghdr);
        fclose(txtfile);
        return(FALSE);
    }


    /* Check to make sure the base isn't full */
    if((filelength(fileno(msghdr)) / sizeof(msghdrvar)) > 32765)
    {
        sprintf(LOGSTR, "! [HMB_POST] Hudson message base is full, aborting.");
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);

        /* Unlock and close the message base */
        unlock(fileno(msginfo), 406, 1);
        fclose(msginfo);
        fclose(msgtxt);
        fclose(msgidx);
        fclose(msghdr);
        fclose(msgtoidx);
        fclose(txtfile);
        return(FALSE);
    }


    /* If we've gotton this far the message base is open, time to write! */

    /* Update MSGINFO.BBS */
    fread(&msginfovar, sizeof(msginfovar), 1, msginfo);
    msginfovar.highmsg++;
    msginfovar.totalmsgs++;
    msginfovar.totalonboard[msgpostvar -> area -> areanum-1]++;
    rewind(msginfo);
    fwrite(&msginfovar, sizeof(msginfovar), 1, msginfo);

    /* Update MSGIDX.BBS */
    msgidxvar.msgnum = msginfovar.highmsg;
    msgidxvar.board = msgpostvar -> area -> areanum;
    fseek(msgidx, 0, SEEK_END);
    fwrite(&msgidxvar, sizeof(msgidxvar), 1, msgidx);

    /* Update MSGTOIDX.BBS */
    msgtoidxvar.l1 = strlen(msgpostvar -> to);
    Amustr_obj.ctopascal(msgpostvar -> to, msgtoidxvar.name);
    fseek(msgtoidx, 0, SEEK_END);
    fwrite(&msgtoidxvar, sizeof(msgtoidxvar), 1, msgtoidx);


    /* Update MSGTXT.BBS */
    numblocks = 0;
    startblock = filelength(fileno(msgtxt)) / 256;
    fseek(msgtxt, 0, SEEK_END);
    memset(msgtxtvar, NULL, sizeof(msgtxtvar));
    _dos_read(fileno(txtfile), msgtxtvar, 255, &x);
    while(x > 0)
    {
        fwrite(&x, 1, 1, msgtxt);
        fwrite(msgtxtvar, 255, 1, msgtxt);
        numblocks++;
        memset(msgtxtvar, NULL, sizeof(msgtxtvar));
        _dos_read(fileno(txtfile), msgtxtvar, 255, &x);
    }


    /* Update MSGHDR.BBS */
    msghdrvar.msgnum = msginfovar.highmsg;
    msghdrvar.prevreply = 0;
    msghdrvar.nextreply = 0;
    msghdrvar.timesread = 0;
    msghdrvar.startblock = startblock;
    msghdrvar.numblocks = numblocks;
    msghdrvar.destnet = 0;
    msghdrvar.destnode = 0;
    msghdrvar.orignet = 0;
    msghdrvar.orignode = 0;
    msghdrvar.destzone = 0;
    msghdrvar.origzone = 0;
    msghdrvar.cost = 0;
    msghdrvar.msgattr = 72; // Private Message
    msghdrvar.netattr = 0;
    msghdrvar.board = msgpostvar -> area -> areanum;
    msghdrvar.l1 = 5;
    Amustr_obj.ctopascal(time, msghdrvar.posttime);
    msghdrvar.l2 = 8;
    Amustr_obj.ctopascal(date, msghdrvar.postdate);
    msghdrvar.l3 = strlen(msgpostvar -> to);
    Amustr_obj.ctopascal(msgpostvar -> to, msghdrvar.whoto);
    msghdrvar.l4 = strlen(msgpostvar -> from);
    Amustr_obj.ctopascal(msgpostvar -> from, msghdrvar.whofrom);
    msghdrvar.l5 = strlen(msgpostvar -> subject);
    Amustr_obj.ctopascal(msgpostvar -> subject, msghdrvar.subject);
    fseek(msghdr, 0, SEEK_END);
    fwrite(&msghdrvar, sizeof(msghdrvar), 1, msghdr);

    /* Unlock and close the message base */
    unlock(fileno(msginfo), 406, 1);
    fclose(msginfo);
    fclose(msgtxt);
    fclose(msgidx);
    fclose(msghdr);
    fclose(msgtoidx);
    fclose(txtfile);

    return(TRUE);
}

/************************************************************************/
/* Post a message to a Goldbase area */
int Msgs::write_to_Gold(struct msgpostinfo *msgpostvar)
{
    struct gold_msginfo msginfovar;
    struct gold_msgidx msgidxvar;
    struct gold_msghdr msghdrvar;
    struct gold_msgtoidx msgtoidxvar;
    FILE *msginfo, *msgidx, *msghdr, *msgtxt, *msgtoidx, *txtfile, *semaphore;
    char fn[81], fn2[81], msgtxtvar[256], time[6], date[9];
    unsigned x;
    long startblock, numblocks, k;


    /* The date MUST be in mm-dd-yy format to follow the Hudson specs */
    strcpy(date, Today.getDateStr(DATES_MMDDYY));
    strcpy(time, Today.getTimeStr(DATES_HHMM24));

    sprintf(fn, "%s", msgpostvar -> fn);
    txtfile = fopen(fn, "rt");
    if(!txtfile)
    {
        sprintf(LOGSTR, "? [GOLD_POST] Unable to open '%s' - (%s)", fn, strerror(errno));
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }

    /* Attempt to lock the message base

       Locking specification:

       Use the INT 21 5Ch function to lock a region of the MSGINFO.BBS file.
       This region starts at byte 407, and is 1 byte long. Locking a region

    */
    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGINFO.DAT");
    msginfo = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msginfo)
    {
        sprintf(fn2, "%sMBUNLOCK.NOW", Config_obj.amu_cfgvar.hmbpath);
        semaphore = fopen(fn2, "w+t");
        fclose(semaphore);
        k = 0;
        while(share_installed  &&  k < 10  && !msginfo)
        {
            sleep(2);
            k++;
            msginfo = _fsopen(fn, "r+b", SH_DENYNO);
        }
        remove(fn2);

        /* If 'k' is up to 10, the lock failed */
        if(!msginfo)
        {
            sprintf(LOGSTR, "? [GOLD_POST] Unable to open '%s'", fn);
            Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
            fclose(txtfile);
            return(FALSE);
        }
        else
        {
            lock(fileno(msginfo), 407, 1);
        }
    }

    /* Now open the rest of the Hudson files */

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGIDX.DAT");
    msgidx = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgidx)
    {
        sprintf(LOGSTR, "? [GOLD_POST] Unable to open '%s'", fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msginfo), 407, 1);
        fclose(msginfo);
        fclose(txtfile);
        return(FALSE);
    }

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGTOIDX.DAT");
    msgtoidx = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgtoidx)
    {
        sprintf(LOGSTR, "? [GOLD_POST] Unable to open '%s'", fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msginfo), 407, 1);
        fclose(msginfo);
        fclose(msgidx);
        fclose(txtfile);
        return(FALSE);
    }

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGHDR.DAT");
    msghdr = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msghdr)
    {
        sprintf(LOGSTR, "? [GOLD_POST] Unable to open '%s'", fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msginfo), 407, 1);
        fclose(msgtoidx);
        fclose(msgidx);
        fclose(msginfo);
        fclose(txtfile);
        return(FALSE);
    }

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.hmbpath, "MSGTXT.DAT");
    msgtxt = _fsopen(fn, "r+b", SH_DENYNO);
    if(!msgtxt)
    {
        sprintf(LOGSTR, "? [GOLD_POST] Unable to open '%s'", fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        unlock(fileno(msginfo), 407, 1);
        fclose(msgtoidx);
        fclose(msgidx);
        fclose(msginfo);
        fclose(msghdr);
        fclose(txtfile);
        return(FALSE);
    }


    /* If we've gotton this far the message base is open, time to write! */

    /* Update MSGINFO.DAT */
    fread(&msginfovar, sizeof(msginfovar), 1, msginfo);
    msginfovar.highmsg++;
    msginfovar.totalmsgs++;
    msginfovar.totalonboard[msgpostvar -> area -> areanum -1]++;
    rewind(msginfo);
    fwrite(&msginfovar, sizeof(msginfovar), 1, msginfo);

    /* Update MSGIDX.DAT */
    msgidxvar.msgnum = msginfovar.highmsg;
    msgidxvar.board = msgpostvar -> area -> areanum;
    fseek(msgidx, 0, SEEK_END);
    fwrite(&msgidxvar, sizeof(msgidxvar), 1, msgidx);

    /* Update MSGTOIDX.DAT */
    msgtoidxvar.l1 = strlen(msgpostvar -> to);
    Amustr_obj.ctopascal(msgpostvar -> to, msgtoidxvar.name);
    fseek(msgtoidx, 0, SEEK_END);
    fwrite(&msgtoidxvar, sizeof(msgtoidxvar), 1, msgtoidx);


    /* Update MSGTXT.DAT */
    numblocks = 0;
    startblock = filelength(fileno(msgtxt)) / 256;
    fseek(msgtxt, 0, SEEK_END);
    memset(msgtxtvar, NULL, sizeof(msgtxtvar));
    _dos_read(fileno(txtfile), msgtxtvar, 255, &x);
    while(x > 0)
    {
        fwrite(&x, 1, 1, msgtxt);
        fwrite(msgtxtvar, 255, 1, msgtxt);
        numblocks++;
        memset(msgtxtvar, NULL, sizeof(msgtxtvar));
        _dos_read(fileno(txtfile), msgtxtvar, 255, &x);
    }


    /* Update MSGHDR.DAT */
    msghdrvar.msgnum = msginfovar.highmsg;
    msghdrvar.prevreply = 0;
    msghdrvar.nextreply = 0;
    msghdrvar.timesread = 0;
    msghdrvar.startblock = startblock;
    msghdrvar.numblocks = numblocks;
    msghdrvar.destnet = 0;
    msghdrvar.destnode = 0;
    msghdrvar.orignet = 0;
    msghdrvar.orignode = 0;
    msghdrvar.destzone = 0;
    msghdrvar.origzone = 0;
    msghdrvar.cost = 0;
    msghdrvar.msgattr = 72; // Private Message
    msghdrvar.netattr = 0;
    msghdrvar.board = msgpostvar -> area -> areanum;
    msghdrvar.l1 = 5;
    Amustr_obj.ctopascal(time, msghdrvar.posttime);
    msghdrvar.l2 = 8;
    Amustr_obj.ctopascal(date, msghdrvar.postdate);
    msghdrvar.l3 = strlen(msgpostvar -> to);
    Amustr_obj.ctopascal(msgpostvar -> to, msghdrvar.whoto);
    msghdrvar.l4 = strlen(msgpostvar -> from);
    Amustr_obj.ctopascal(msgpostvar -> from, msghdrvar.whofrom);
    msghdrvar.l5 = strlen(msgpostvar -> subject);
    Amustr_obj.ctopascal(msgpostvar -> subject, msghdrvar.subj);
    fseek(msghdr, 0, SEEK_END);
    fwrite(&msghdrvar, sizeof(msghdrvar), 1, msghdr);

    /* Unlock and close the message base */
    unlock(fileno(msginfo), 407, 1);
    fclose(msginfo);
    fclose(msgtxt);
    fclose(msgidx);
    fclose(msghdr);
    fclose(msgtoidx);
    fclose(txtfile);

    return(TRUE);
}

/************************************************************************/
/* Post a message to a Squish message base */
int Msgs::write_to_Squish(struct msgpostinfo *msgpostvar)
{
    time_t lt;
    unsigned status, i;
    long length;
    char txtvar[256], fn[81];
    XMSG xmsgvar;
    struct _sqmsghdr squishmsghdr;
    struct _squishhdr squishhdr;
    struct _sqidx squishidx;
    FILE *textfile, *datafile, *idxfile;

    status = TRUE;
    textfile = fopen(msgpostvar -> fn, "rt");
    if(!textfile)
    {
        sprintf(LOGSTR, "? [SQ_POST] Unable to open '%s'", msgpostvar -> fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }
    length = filelength(fileno(textfile));

    /* Attempt to open and lock the data file */
    sprintf(fn, "%s.SQD", msgpostvar -> area -> basepath);
    datafile = _fsopen(fn, "r+b", SH_DENYNO);
    if(!datafile)
    {
        fclose(textfile);
        sprintf(LOGSTR, "? [SQ_POST] Unable to open '%s'", fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }

    /* Try to lock the first byte of the .SQD file */
    if(share_installed && lock(fileno(datafile), 0, 1) != 0)
    {
        fclose(datafile);
        fclose(textfile);
        sprintf(LOGSTR, "? [SQ_POST] Unable to lock '%s'", fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }

    sprintf(fn, "%s.SQI", msgpostvar -> area -> basepath);
    idxfile = _fsopen(fn, "r+b", SH_DENYNO);
    if(!idxfile)
    {
        fclose(textfile);
        if(share_installed)
        { 
            unlock(fileno(datafile), 0, 1);
        }
        fclose(datafile);
        sprintf(LOGSTR, "? [SQ_POST] Unable to open '%s'", fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }

    /* If we made it this far, the Squish area has been opened */

    /* Read the static header record */
    fread(&squishhdr, sizeof(squishhdr), 1, datafile);

    /* Read the last frame, then update it to point to the one we're
       about to add.
    */
    if(squishhdr.begin_frame != 0)
    {
        fseek(datafile, squishhdr.last_frame, SEEK_SET);
        fread(&squishmsghdr, sizeof(squishmsghdr), 1, datafile);
        squishmsghdr.next_frame = filelength(fileno(datafile));
        fseek(datafile, squishhdr.last_frame, SEEK_SET);
        fwrite(&squishmsghdr, sizeof(squishmsghdr), 1, datafile);
    }
    else
    {
        squishhdr.begin_frame = sizeof(squishhdr);
    }
    fseek(datafile, 0, SEEK_END);

    /* Add a SQHDR for the new frame & update the .last_frame field */
    squishmsghdr.id = SQHDRID;
    squishmsghdr.next_frame = 0;
    squishmsghdr.prev_frame = squishhdr.last_frame;
    squishmsghdr.frame_length = squishmsghdr.msg_length = sizeof(xmsgvar) + length;
    squishmsghdr.clen = 0;
    squishmsghdr.frame_type = 0;
    squishhdr.last_frame = filelength(fileno(datafile));
    fwrite(&squishmsghdr, sizeof(squishmsghdr), 1, datafile);
    fseek(datafile, 0, SEEK_END);

    /* Add the XMSG structure */
    /* Fill up the XMSG structure */
    xmsgvar.attr = MSGPRIVATE | MSGLOCAL | MSGUID;
    strcpy(xmsgvar.from, msgpostvar -> from);
    strcpy(xmsgvar.to, msgpostvar -> to);
    strcpy(xmsgvar.subj, msgpostvar -> subject);
    xmsgvar.orig.zone = 0;
    xmsgvar.orig.net = 0;
    xmsgvar.orig.node = 0;
    xmsgvar.orig.point = 0;
    xmsgvar.dest.zone = 0;
    xmsgvar.dest.net = 0;
    xmsgvar.dest.node = 0;
    xmsgvar.dest.point = 0;

    xmsgvar.date_written.date.da = xmsgvar.date_arrived.date.da = Today.getDay();
    xmsgvar.date_written.date.mo = xmsgvar.date_arrived.date.mo = Today.getMonth();
    xmsgvar.date_written.date.yr = xmsgvar.date_arrived.date.yr = Today.getYear() - 1980;
    xmsgvar.date_written.time.ss = xmsgvar.date_arrived.time.ss = Today.getSeconds();
    xmsgvar.date_written.time.mm = xmsgvar.date_arrived.time.mm = Today.getMinutes();
    xmsgvar.date_written.time.hh = xmsgvar.date_arrived.time.hh = Today.getHour24();

    xmsgvar.utc_ofs = 0;
    xmsgvar.replyto = 0;
    for(i = 0; i < MAX_REPLY; i++)
    {
        xmsgvar.replies[i] = 0;
    }

    xmsgvar.umsgid = squishhdr.uid;
    lt = time(NULL);
    strcpy(xmsgvar.ftsc_date, ctime(&lt));

    fwrite(&xmsgvar, sizeof(xmsgvar), 1, datafile);

    /* Add the actual message text */
    fseek(datafile, 0, SEEK_END);
    _dos_read(fileno(textfile), txtvar, sizeof(txtvar), &i);
    while(i > 0)
    {
        _dos_write(fileno(datafile), txtvar, i, &i);
        _dos_read(fileno(textfile), txtvar, sizeof(txtvar), &i);
    }

    /* Now update the rest of the information in the static header and
       write it back to position 0 in the .SQD file.  The 'last_frame' field
       has already been updated.
    */
    squishhdr.num_msg++;
    squishhdr.high_msg++;
    squishhdr.uid++;
    squishhdr.end_frame = filelength(fileno(datafile));
    fseek(datafile, 0, SEEK_SET);
    fwrite(&squishhdr, sizeof(squishhdr), 1, datafile);

    /* Now update the index file */
    fseek(idxfile, sizeof(squishidx) * (squishhdr.num_msg - 1), SEEK_SET);
    squishidx.ofs = squishhdr.last_frame;
    squishidx.umsgid = xmsgvar.umsgid;
    squishidx.hash = get_squish_hash(xmsgvar.to);
    fwrite(&squishidx, sizeof(squishidx), 1, idxfile);


    /* Close the files & release the locks */
    fclose(textfile);
    if(share_installed)
    {
        unlock(fileno(datafile), 0, 1);
    }
    fclose(datafile);
    fclose(idxfile);

/*
    printf("Length         : %d\n", squishhdr.len);
    printf("Reserved       : %d\n", squishhdr.rsvd1);
    printf("Num_Msg        : %ld\n", squishhdr.num_msg);
    printf("High_Msg       : %ld\n", squishhdr.high_msg);
    printf("Skip_Msg       : %ld\n", squishhdr.skip_msg);
    printf("High_water     : %ld\n", squishhdr.high_water);
    printf("UID            : %ld\n", squishhdr.uid);
    printf("Base           : %s\n", squishhdr.base);
    printf("Begin_frame    : %ld\n", squishhdr.begin_frame);
    printf("Last_frame     : %ld\n", squishhdr.last_frame);
    printf("Free_frame     : %ld\n", squishhdr.free_frame);
    printf("Last_free_frame: %ld\n", squishhdr.last_free_frame);
    printf("End_frame      : %ld\n", squishhdr.end_frame);
    printf("Max_msg        : %ld\n", squishhdr.max_msg);
    printf("Keep_days      : %d\n", squishhdr.keep_days);
    printf("Hdrsize        : %d\n", squishhdr.sz_sqhdr);
    exit(1);
*/

    return(status);
}

/***********************************************************************/
/* Get the 'hash' value for a Squish area */
long Msgs::get_squish_hash(char *f)
{
    unsigned long hash = 0;
    unsigned long g;
    char *p;

    for(p = f; *p; p++)
    {
        hash = (hash << 4) + (unsigned long)tolower(*p);
        if((g=(hash & 0xf0000000L)) != 0L)
        {
            hash |= g >> 24;
            hash |= g;
        }
    }
    return(hash & 0x7fffffffLu);
}

/************************************************************************/
/* Post a message to a *.MSG message base */
int Msgs::write_to_MSG(struct msgpostinfo *msgpostvar)
{
    struct find_t fileinfo;
    struct netmsg msgvar;
    unsigned high_msg, done, tmp, i;
    char fn[MAX_FILESPEC], txtvar[256];
    char drive[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], ext[_MAX_EXT];
    FILE *msgfile, *textfile;

    textfile = fopen(msgpostvar -> fn, "rt");
    if(!textfile)
    {
        sprintf(LOGSTR, "? [MSG_POST] Unable to open '%s'", msgpostvar -> fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }

    memset(&msgvar, NULL, sizeof(msgvar));
    sprintf(fn, "%s*.MSG", msgpostvar -> area -> basepath);
    high_msg = 0;
    done = _dos_findfirst(fn, _A_ARCH, &fileinfo);
    while(!done)
    {
        _splitpath(fileinfo.name, drive, dir, fname, ext);
        tmp = atoi(fname);
        if(tmp > high_msg)
        {
            high_msg = tmp;
        }
        done = _dos_findnext(&fileinfo);
    }
    high_msg++;
    sprintf(fn, "%s%d.MSG", msgpostvar -> area -> basepath, high_msg);
    msgfile = _fsopen(fn, "w+b", SH_DENYNO);
    if(!msgfile)
    {
        sprintf(LOGSTR, "! [MSG_POST] Unable to create '%s'", fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }
    strcpy(msgvar.from, msgpostvar -> from);
    strcpy(msgvar.too, msgpostvar -> to);
    strcpy(msgvar.subj, msgpostvar -> subject);
    msgvar.attr = _PRIVATE | _LOCAL;
    sprintf(msgvar.date, Today.getDateStr(DATES_FIDODATE));

    write(fileno(msgfile), &msgvar, sizeof(msgvar));
    _dos_read(fileno(textfile), txtvar, sizeof(txtvar), &i);
    while(i > 0)
    {
        _dos_write(fileno(msgfile), txtvar, i, &i);
        _dos_read(fileno(textfile), txtvar, sizeof(txtvar), &i);
    }
    txtvar[0] = NULL;
    _dos_write(fileno(msgfile), txtvar, 1, &i);

    fclose(textfile);
    fclose(msgfile);
    return(TRUE);
}

/*************************************************************************/
/* Decide which style message base gets posted to */
int Msgs::post_msg(struct msgpostinfo *msgpostvar, struct user_array *userptr)
{
    int i;
    char newfn[MAX_FILESPEC], str[151], *buffer;
    FILE *infile, *outfile;

    /* Strip any spaces which may have gotton in there */
    Amustr_obj.trimlead(msgpostvar -> to);
    Amustr_obj.trimend(msgpostvar -> to);
    Amustr_obj.trimlead(msgpostvar -> from);
    Amustr_obj.trimend(msgpostvar -> from);
    Amustr_obj.trimlead(msgpostvar -> subject);
    Amustr_obj.trimend(msgpostvar -> subject);

    sprintf(newfn, "%s%s", Config_obj.amu_cfgvar.AMUpath, NO_POST);
    infile = fopen(newfn, "rt");
    if(infile)
    {
        i = FALSE;
        buffer = (char *)calloc(filelength(fileno(infile)) + 1, 1);
        fread(buffer, filelength(fileno(infile)), 1, infile);
        fclose(infile);
        strcpy(str, msgpostvar -> to);
        strupr(buffer);
        strupr(str);
        if(strstr(buffer, str))
        {
            sprintf(LOGSTR, "[POST_MSG] '%s' found in %s, message not posted",
                             msgpostvar -> to, NO_POST);
            Utility_obj.logentry(LOGSTR, LOG_EXTENSIVE);
            i = TRUE;
        }
        free(buffer);
        if(i)
        {
            return(TRUE);
        }
    }

    /* Check to make sure the message area is > 0 */
    if(msgpostvar -> area -> areanum <= 0)
    {
        sprintf(LOGSTR, "! [POST_MSG] Unable to post message - specified message area number must be greater than zero");
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }

    sprintf(newfn, "%s%s", Config_obj.amu_cfgvar.workpath, "TMPPOST.TXT");
    outfile = fopen(newfn, "w+t");
    if(!outfile)
    {
        sprintf(LOGSTR, "! [POST_MSG] Unable to create temp msg file '%s'", newfn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        return(FALSE);
    }
    infile = fopen(msgpostvar -> fn, "rt");
    if(!infile)
    {
        fclose(outfile);
        sprintf(LOGSTR, "! [POST_MSG] Unable to open msg file '%s'", msgpostvar -> fn);
        Utility_obj.logentry(LOGSTR, LOG_MINIMAL);
        remove(newfn);
        return(FALSE);
    }

    while(fgets(str, sizeof(str)-2, infile))
    {
        Amustr_obj.strip_n(str);
        MsgMacro_obj.set_userrecord(userptr);
        MsgMacro_obj.get_expansion(str, Header);
        fprintf(outfile, "%s\n", Header);
    }
    fclose(infile);
    fclose(outfile);
    strcpy(msgpostvar -> fn, newfn);

    switch(msgpostvar -> area -> areatype)
    {
        case HUDSON_TYPE: i = write_to_Hudson(msgpostvar);
                          break;
        case JAM_TYPE   : i = write_to_JAM(msgpostvar);
                          break;
        case EZY_TYPE   : i = write_to_EZY(msgpostvar);
                          break;
        case GOLD_TYPE  : i = write_to_Gold(msgpostvar);
                          break;
        case SQUISH_TYPE: i = write_to_Squish(msgpostvar);
                          break;
        case MSG_TYPE   : i = write_to_MSG(msgpostvar);
                          break;
        default         : sprintf(LOGSTR, "? [POST_MSG] Unrecognized message base type - Area #%d",
                                  msgpostvar -> area -> areanum);
                          i = FALSE;
                          break;
    }
    remove(newfn);
    return(i);
}

//************************************************************************
// EOF MSGS.CPP
