// Object to handle various types of dates.  Year 2000 compliant
// Created 29 October 1997
// Revised 21 May 1998

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <dos.h>
#include "dates.h"

// Default Constructor - creates object for today's date
Dates::Dates(void)
{
    struct dosdate_t d;
    struct dostime_t t;

    initializeDateTime();

    // Get the system date and time
    _dos_gettime(&t);
    _dos_getdate(&d);

    _seconds = t.second;
    _minutes = t.minute;
    _hour = t.hour;
    _day = d.day;
    _month = d.month;
    _year = d.year;

    convertToInternal(getSecondsSince1970());
    setDateSep('-');
    setTimeSep(':');
}

// Set the object up with today's date
void Dates::createDate(void)
{
    struct dosdate_t d;
    struct dostime_t t;

    initializeDateTime();

    // Get the system date and time
    _dos_gettime(&t);
    _dos_getdate(&d);

    _seconds = t.second;
    _minutes = t.minute;
    _hour = t.hour;
    _day = d.day;
    _month = d.month;
    _year = d.year;

    convertToInternal(getSecondsSince1970());
    setDateSep('-');
    setTimeSep(':');
}

// Creates object for explicit date and time information
void Dates::createDate(int year, int month, int day, int hour, int minutes, int seconds)
{
    initializeDateTime();

    _seconds = seconds;
    _minutes = minutes;
    _hour = hour;
    _day = day;
    _month = month;
    _year = year;

    // See if this is a 2 digit year being passed in
    if(_year < 100)
    {
	if(_year < 20)
	{
	    _year += 2000;
	}
	else
	{
	    _year += 1900;
	}
    }

    convertToInternal(getSecondsSince1970());
    setDateSep('-');
    setTimeSep(':');
}


// Constructor - accepts date in seconds since January 1, 1970
void Dates::createDate(long date)
{
    initializeDateTime();

    convertToInternal(date);
    setDateSep('-');
    setTimeSep(':');
}

// Constructor - accepts a date in MM-DD-YY and HH:MM format
void Dates::createDate(char *d, char *t)
{
    char j1, j2, mstr[4];

    initializeDateTime();
    _seconds = _minutes = _hour = 0;
    _month = _day = _year = 0;

    // Convert a date in MM-DD-YY format
    if(d != NULL && strlen(d) == 8)
    {
	sscanf(d, "%02d%c%02d%c%02d", &_month, &j1, &_day, &j2, &_year);

	// If the year is 0 - 19, assume this is in the 21st century, otherwise
	// assume this is the 20th century
	if(_year < 20)
	{
	    _year += 2000;
	}
	else
	{
	    _year += 1900;
	}
    }

    // Convert a date in MM-DD-YYYY format
    if(d != NULL && strlen(d) == 10)
    {
	sscanf(d, "%02d%c%02d%c%04d", &_month, &j1, &_day, &j2, &_year);
    }


    // Convert a date in DD MMM YY format
    if(d != NULL && strlen(d) == 9)
    {
	sscanf(d, "%02d%c%3s%c%02d", &_day, &j1, &mstr, &j2, &_year);

	// If the year is 0 - 19, assume this is in the 21st century, otherwise
	// assume this is the 20th century
	if(_year < 20)
	{
	    _year += 2000;
	}
	else
	{
	    _year += 1900;
	}

	for(_month = 1; _month <= 12; _month++)
	{
	    if(stricmp(mstr, getMonthStrAbbrv()) == 0)
	    {
		break;
	    }
	}
    }

    // Convert a date in DD MMM YYYY format
    if(d != NULL && strlen(d) == 11)
    {
	sscanf(d, "%02d%c%3s%c%04d", &_day, &j1, &mstr, &j2, &_year);

	for(_month = 1; _month <= 12; _month++)
	{
	    if(stricmp(mstr, getMonthStrAbbrv()) == 0)
	    {
		break;
	    }
	}
    }


    // Convert the time if it's in HH:MM format
    if(t != NULL && strlen(t) == 5)
    {
	sscanf(t, "%02d%c%02d", &_hour, &j1, &_minutes);
    }

    // Convert the time if it's in HH:MM:SS format
    if(t != NULL && strlen(t) == 8)
    {
	sscanf(t, "%02d%c%02d%c%02d", &_hour, &j1, &_minutes, &j2, &_seconds);
    }

    convertToInternal(getSecondsSince1970());
    setDateSep('-');
    setTimeSep(':');
}

// Constructor - creates object for the DOS packed date/time format
void Dates::createDate(unsigned short date, unsigned short time)
{
    struct d
    {
	unsigned short day  : 5,
		 month: 4,
		 year : 7;
    };
    union
    {
	struct d packed_date;
	unsigned short date;
    }dvar;

    struct t
    {
	unsigned short seconds: 5,
		 minutes: 6,
		 hours  : 5;
    };
    union
    {
	struct t packed_time;
	unsigned short time;
    }tvar;


    initializeDateTime();

    tvar.time = time;
    dvar.date = date;
    _seconds = tvar.packed_time.seconds * 2;
    _minutes = tvar.packed_time.minutes;
    _hour = tvar.packed_time.hours;
    _day = dvar.packed_date.day;
    _month = dvar.packed_date.month;
    _year = dvar.packed_date.year + 1980;

    convertToInternal(getSecondsSince1970());
    setDateSep('-');
    setTimeSep(':');
}


// Create an object from the number of days elapsed since a certain year
void Dates::createDate(int start_year, long num_days)
{
    int x, months[12];

    initializeDateTime();

    months[0] = 31;
    months[1] = 28;
    months[2] = 31;
    months[3] = 30;
    months[4] = 31;
    months[5] = 30;
    months[6] = 31;
    months[7] = 31;
    months[8] = 30;
    months[9] = 31;
    months[10] = 30;
    months[11] = 31;

    // Get to the appropriate year
    for(_year = start_year; ; _year++)
    {
	x = 365;
	months[1] = 28;

	// See if this is a leap year or a leap century
	if(_year % 100 == 0)
	{
	    if(_year % 400 == 0)
	    {
		x = 366;
		months[1] = 29;
	    }
	}
	else
	{
	    if(_year % 4 == 0)
	    {
		x = 366;
		months[1] = 29;
	    }
	}

	if(x > num_days)
	{
	    break;
	}
	else
	{
	    num_days -= x;
	}

	// See if we landed right on the last day of the year
	if(num_days == 0)
	{
	    _month = 12;
	    _day = 31;
	    _seconds = _minutes = _hour = 0;
	    convertToInternal(getSecondsSince1970());
	    setDateSep('-');
	    setTimeSep(':');
	    return;
	}
    }

    // Now figure out the month
    for(_month = 1; _month <= 12; _month++)
    {
	if(months[_month-1] > num_days)
	{
	    break;
	}
	else
	{
	    num_days -= months[_month-1];
	}
    }
    _day = num_days;



    _seconds = _minutes = _hour = 0;
    convertToInternal(getSecondsSince1970());
    setDateSep('-');
    setTimeSep(':');
}


// Convert this date into the number of seconds since 1970
long Dates::getSecondsSince1970(void)
{
    return(toSecondsSince1970(getYear(), getMonth(), getDay(),
			      getHour24(), getMinutes(), getSeconds()));
}

// Accessor for the day of the month
int Dates::getDay(void)
{
    return(_day);
}

// Accessor for the month
int Dates::getMonth(void)
{
    return(_month);
}

// Returns a string representation of the month
char *Dates::getMonthStr(void)
{
    static char s[11];

    switch(_month)
    {
	case 1 : strcpy(s, "January");
		 break;
	case 2 : strcpy(s, "February");
		 break;
	case 3 : strcpy(s, "March");
		 break;
	case 4 : strcpy(s, "April");
		 break;
	case 5 : strcpy(s, "May");
		 break;
	case 6 : strcpy(s, "June");
		 break;
	case 7 : strcpy(s, "July");
		 break;
	case 8 : strcpy(s, "August");
		 break;
	case 9 : strcpy(s, "September");
		 break;
	case 10: strcpy(s, "October");
		 break;
	case 11: strcpy(s, "November");
		 break;
	case 12: strcpy(s, "December");
		 break;
	default: strcpy(s, "Invalid");
    }
    return(s);
}

// Returns a three letter representation of the month
char *Dates::getMonthStrAbbrv(void)
{
    char str[10];
    static char s[4];

    strcpy(str, getMonthStr());
    str[3] = NULL;
    strcpy(s, str);
    return(s);
}

// Return the year
int Dates::getYear(void)
{
    return(_year);
}

// Return the day of the week (numeric)
int Dates::getDayOfWeek(void)
{
    return(_dayofweek);
}

// Return the day of the week (string)
char *Dates::getDayOfWeekStr(void)
{
    static char s[11];

    switch(_dayofweek)
    {
	case 1: strcpy(s, "Sunday");
		break;
	case 2: strcpy(s, "Monday");
		break;
	case 3: strcpy(s, "Tuesday");
		break;
	case 4: strcpy(s, "Wednesday");
		break;
	case 5: strcpy(s, "Thursday");
		break;
	case 6: strcpy(s, "Friday");
		break;
	case 7: strcpy(s, "Saturday");
		break;
	default: strcpy(s, "Invalid");
    }
    return(s);
}

// Return a three letter representation of the day of the week
char *Dates::getDayOfWeekStrAbbrv(void)
{
    char str[10];
    static char s[4];

    strcpy(str, getDayOfWeekStr());
    str[3] = NULL;
    strcpy(s, str);
    return(s);
}

// Return the day of the year
int Dates::getDayOfYear(void)
{
    return(_dayofyear);
}

// Return the seconds
int Dates::getSeconds(void)
{
    return(_seconds);
}

// Return the minutes
int Dates::getMinutes(void)
{
    return(_minutes);
}

// Return the hour (0-23)
int Dates::getHour24(void)
{
    return(_hour);
}

// Return the hour (1-12)
int Dates::getHour12(void)
{
    int x;

    x = getHour24();
    x = x % 12;
    if(x == 0)
    {
	x = 12;
    }
    return(x);
}


// Compare this date object with another date object
// Return < 1 if dateObj is less than this, 0 if equal, > 1 if greater than this
int Dates::compareDates(class Dates *dateObj)
{
    return(-1 * getDateDiff(dateObj));
}

// Figure out how many days old this date is
int Dates::getAge(void)
{
    class Dates todayDate;
    int days = -1;

    days = todayDate.getDaysSince(DATES_BASECOMPARISON) - getDaysSince(DATES_BASECOMPARISON);
    return(days);
}

// Figure out the difference in days between this date and another
int Dates::getDateDiff(class Dates *D)
{
    int days;

    days = getDaysSince(DATES_BASECOMPARISON) - D -> getDaysSince(DATES_BASECOMPARISON);
    return(days);
}


// Return a date in the format: Wed Jan 03 19:00:00 1998
char *Dates::getExtendedFormatStr(void)
{
    static char str[26];

    sprintf(str, "%s %s %02d %02d:%02d:%02d %04d", getDayOfWeekStrAbbrv(),
	    getMonthStrAbbrv(), getDay(), getHour24(), getMinutes(),
	    getSeconds(), getYear());
    return(str);
}


// Return the time in HH:MM format if mode == DATES_HHMMxx
// or in HH:MM:SS if mode == DATES_HHMMSSxx
char *Dates::getTimeStr(int format)
{
    static char str[10];
    char t;

    t = getTimeSep();
    switch(format)
    {
	case DATES_HHMMSS12: sprintf(str, "%02d%c%02d%c%02d", getHour12(), t,
				     getMinutes(), t, getSeconds());
			     break;
	case DATES_HHMMSS24: sprintf(str, "%02d%c%02d%c%02d", getHour24(), t,
				     getMinutes(), t, getSeconds());
			     break;
	case DATES_HHMM12  : sprintf(str, "%02d%c%02d", getHour12(), t,
				     getMinutes());
			     break;
	case DATES_HHMM24  : sprintf(str, "%02d%c%02d", getHour24(), t,
				     getMinutes());
			     break;
	default            : strcpy(str, "INVALID");
			     break;
    }
    return(str);
}


// Return the date in the specified format
char *Dates::getDateStr(int format)
{
    static char str[30];
    char d;

    d = getDateSep();
    switch(format)
    {
	case DATES_MMDDYY  : sprintf(str, "%02d%c%02d%c%02d", getMonth(), d,
				     getDay(), d, getYear() % 100);
			     break;
	case DATES_MMYYDD  : sprintf(str, "%02d%c%02d%c%02d", getMonth(), d,
				     getYear() % 100, d, getDay());
			     break;
	case DATES_DDMMYY  : sprintf(str, "%02d%c%02d%c%02d", getDay(), d,
				     getMonth(), d, getYear() % 100);
			     break;
	case DATES_DDYYMM  : sprintf(str, "%02d%c%02d%c%02d", getDay(), d,
				     getYear() % 100, d, getMonth());
			     break;
	case DATES_YYMMDD  : sprintf(str, "%02d%c%02d%c%02d", getYear() % 100, d,
				     getMonth(), d, getDay());
			     break;
	case DATES_YYDDMM  : sprintf(str, "%02d%c%02d%c%02d", getYear() % 100, d,
				     getDay(), d, getMonth());
			     break;

	case DATES_MMDDYYYY: sprintf(str, "%02d%c%02d%c%04d", getMonth(), d,
				     getDay(), d, getYear());
			     break;
	case DATES_MMYYYYDD: sprintf(str, "%02d%c%04d%c%02d", getMonth(), d,
				     getYear(), d, getDay());
			     break;
	case DATES_DDMMYYYY: sprintf(str, "%02d%c%02d%c%04d", getDay(), d,
				     getMonth(), d, getYear());
			     break;
	case DATES_DDYYYYMM: sprintf(str, "%02d%c%04d%c%02d", getDay(), d,
				     getYear(), d, getMonth());
			     break;
	case DATES_YYYYMMDD: sprintf(str, "%04d%c%02d%c%02d", getYear(), d,
				     getMonth(), d, getDay());
			     break;
	case DATES_YYYYDDMM: sprintf(str, "%04d%c%02d%c%02d", getYear(), d,
				     getDay(), d, getMonth());
			     break;

	case DATES_DDMMMYY : sprintf(str, "%02d %s %02d", getDay(),
				     getMonthStrAbbrv(), getYear() % 100);
			     break;
	case DATES_DDMMMYYYY: sprintf(str, "%02d %s %04d", getDay(),
				      getMonthStrAbbrv(), getYear());
			      break;

	//  Creates FidoNet date - 01 Jan 89 21:05:18
	case DATES_FIDODATE: sprintf(str, "%02d %s %02d %s", getDay(),
				     getMonthStrAbbrv(), getYear() % 100,
				     getTimeStr(DATES_HHMMSS24));
			     break;



	default            : strcpy(str, "INVALID");
			     break;
    }
    return(str);
}


// Get the date separation character
char Dates::getDateSep(void)
{
    return(_datesep);
}

// Get the time separation character
char Dates::getTimeSep(void)
{
    return(_timesep);
}

// Set the date separation character
void Dates::setDateSep(char s)
{
    _datesep = s;
}

// Set the time separation character
void Dates::setTimeSep(char s)
{
    _timesep = s;
}

// Return the date and time in the DOS packed format
void Dates::getPackedDateTime(unsigned short *date, unsigned short *time)
{
    struct d
    {
	unsigned short day  : 5,
		       month: 4,
		       year : 7;
    };
    union
    {
	struct d packed_date;
	unsigned short date;
    }dvar;

    struct t
    {
	unsigned short seconds: 5,
		       minutes: 6,
		       hours  : 5;
    };
    union
    {
	struct t packed_time;
	unsigned short time;
    }tvar;


    tvar.time = 0;
    dvar.date = 0;
    tvar.packed_time.hours = _hour;
    tvar.packed_time.minutes = _minutes;
    tvar.packed_time.seconds = _seconds / 2;
    dvar.packed_date.day = _day;
    dvar.packed_date.month = _month;
    dvar.packed_date.year  = _year - 1980;

    *time = tvar.time;
    *date = dvar.date;
}


// Get the number of days elapsed between start_year and the day of this object

long Dates::getDaysSince(int start_year)
{
    int x, months[12], year;
    long num_days;

    months[0] = 31;
    months[1] = 28;
    months[2] = 31;
    months[3] = 30;
    months[4] = 31;
    months[5] = 30;
    months[6] = 31;
    months[7] = 31;
    months[8] = 30;
    months[9] = 31;
    months[10] = 30;
    months[11] = 31;

    // Calculate the years
    num_days = 0;
    for(year = start_year; year <= getYear(); year++)
    {
	// See if this is a leap year or a leap century
	if(year % 100 == 0)
	{
	    if(year % 400 == 0)
	    {
		x = 366;
		months[1] = 29;
	    }
	}
	else
	{
	    if(year % 4 == 0)
	    {
		x = 366;
		months[1] = 29;
	    }
	}

	if(year == getYear())
	{
	    break;
	}
	num_days += x;
    }

    // Now figure out the month
    for(x = 1; x < getMonth(); x++)
    {
	num_days += months[x-1];
    }
    num_days += getDay();
    return(num_days);
}



//******************************************************************
//*                      Private Methods                           *
//******************************************************************

// Convert a date to SecondsSince1970
long Dates::toSecondsSince1970(int year, int month, int day, int hours,
			       int minutes, int seconds)
{
    long time;
    int x, months[12], begin_year, end_year;

    // Set up the months
    months[0] = 31;
    months[1] = 28;
    months[2] = 31;
    months[3] = 30;
    months[4] = 31;
    months[5] = 30;
    months[6] = 31;
    months[7] = 31;
    months[8] = 30;
    months[9] = 31;
    months[10] = 30;
    months[11] = 31;

    // Account for a leap year
    if(year % 4 == 0)
    {
	months[1] = 29;
    }


    if(year < 1970)
    {
	begin_year = year;
	end_year = 1970;
    }
    else
    {
	begin_year = 1970;
	end_year = year;
    }

    // Get the time component
    time = seconds;
    time += minutes * 60;
    time += hours * 3600;

    // Figure out how many seconds for each year so far since 1970
    for(x = begin_year; x < end_year; x++)
    {
	// 86,400 seconds per day, 365 days in a year
	time += 86400 * 365;

	// Take into account if it was a leap year
	if(x % 4 == 0)
	{
	    time += 86400;
	}
    }

    // Figure out the time elapsed so far this year
    for(x = 1; x < month; x++)
    {
	time += 86400 * months[x-1];
    }
    time += 86400 * (day - 1);

    if(end_year == 1970)
    {
	time *= -1;
    }

    return(time);
}

// Convert a date in seconds since 1970 to the internal format
void Dates::convertToInternal(long date)
{
    struct tm *d;
#if defined(__BORLANDC__)
    long newdate;
#else
    unsigned long newdate;
#endif


    newdate = date;
    d = gmtime(&newdate);
    if(_seconds == -1) _seconds = d -> tm_sec;
    if(_minutes == -1) _minutes = d -> tm_min;
    if(_hour == -1) _hour = d -> tm_hour;
    if(_day == -1) _day = d -> tm_mday;
    if(_month == -1) _month = d -> tm_mon + 1;
    if(_year == -1) _year = d -> tm_year + 1900;
    if(_dayofweek == -1) _dayofweek = d -> tm_wday + 1;
    if(_dayofyear == -1) _dayofyear = d -> tm_yday + 1;
}

// Initialize the variables
void Dates::initializeDateTime(void)
{
    _seconds = -1;
    _minutes = -1;
    _hour = -1;
    _day = -1;
    _month = -1;
    _year = -1;
    _dayofweek = -1;
    _dayofyear = -1;
}

// EOF - DATES.CPP
