// Member functions of the CONFIG class to access CONFIG.AMU, Y2K
// Created 29 May 1996
// Revised 21 May 1998

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <share.h>
#include <sys\stat.h>
#include <io.h>
#include <conio.h>
#include <dos.h>
#include "bbsinfo.h"
#include "extern.h"
#include "dates.h"
#include "screen.h"
#include "config.h"


#if defined(__AMU32__) || defined(__AMU95__)

#if defined(__AMU32__)
  #include <graph.h>
#endif

#define BLACK        0
#define BLUE         1
#define GREEN        2
#define CYAN         3
#define RED          4
#define MAGENTA      5
#define BROWN        6
#define LIGHTGRAY    7
#define DARKGRAY     8
#define LIGHTBLUE    9
#define LIGHTGREEN   10
#define LIGHTCYAN    11
#define LIGHTRED     12
#define LIGHTMAGENTA 13
#define YELLOW       14
#define WHITE        15
#define BLINK        128

#endif

extern class Screen Screen_obj;

/************************************************************************/
/* Constructor for this class                                           */
/************************************************************************/
Config::Config(void)
{
    char *ptr;

    memset(&areasvar, 0, AREAS_SIZE);
    BIT_ARRAY[0] = BIT0;
    BIT_ARRAY[1] = BIT1;
    BIT_ARRAY[2] = BIT2;
    BIT_ARRAY[3] = BIT3;
    BIT_ARRAY[4] = BIT4;
    BIT_ARRAY[5] = BIT5;
    BIT_ARRAY[6] = BIT6;
    BIT_ARRAY[7] = BIT7;
    BIT_ARRAY[8] = BIT8;
    BIT_ARRAY[9] = BIT9;
    BIT_ARRAY[10] = BIT10;
    BIT_ARRAY[11] = BIT11;
    BIT_ARRAY[12] = BIT12;
    BIT_ARRAY[13] = BIT13;
    BIT_ARRAY[14] = BIT14;
    BIT_ARRAY[15] = BIT15;

    ptr = getenv("COMSPEC");
    if(!ptr)
    {
#if defined(__DOS__)
        strcpy(comspec, "COMMAND.COM");
#endif
#if defined(__OS2__)
        strcpy(comspec, "CMD.EXE");
#endif
    }
    else
    {
        strcpy(comspec, ptr);
    }
    strupr(comspec);
    AMUcfgfile = LISTScfg = ARCHIVEScfg = ENFORCEcfg = NULL;
    LOGScfg = FILEAREAcfg = BULLETINcfg = USERMSGScfg = NULL;
    MSGAREAcfg = MAINTcfg = NULL;
}


/************************************************************************/
/* Create a new AMU configuration file.                                 */
/************************************************************************/
int Config::create_config(void)
{
    char *ptr, AMUenv[MAX_FILESPEC], tmpstr[MAX_FILESPEC];
    class Dates Today_Date;

    tmpstr[0] = NULL;
    ptr = getenv("AMU");
    if(ptr)
    {
      strcpy(cfg_filename, ptr);
//strcpy(cfg_filename, "D:\\AMU\\");

        strupr(cfg_filename);
        Amustr_obj.noslash(cfg_filename);
        Utility_obj.change_dir(cfg_filename);
        Amustr_obj.slash(cfg_filename);
        strcpy(tmpstr, cfg_filename);
        strcat(cfg_filename, CONFIG_AMU);

        strcpy(AMUenv, ptr);
        strupr(AMUenv);
        Amustr_obj.slash(AMUenv);
    }
    else
    {
        AMUenv[0] = NULL;
        strcpy(cfg_filename, CONFIG_AMU);
    }

    /* Remove all .AMU files so we get a fresh start and to make it
       harder for folks whose version has expired so they have to
       reconfigure the entire thing. */
    strcat(tmpstr, "*.AMU");
    Utility_obj.remove_all(tmpstr);

    AMUcfgfile = _fsopen(cfg_filename, "w+b", SH_DENYRW);
    if(!AMUcfgfile)
    {
        return(FALSE);
    }

   /* Set up the header record */
    memset(&amu_hdrvar, NULL, HDR_SIZE);
    amu_hdrvar.Revision = REVISION;
    amu_hdrvar.Begin_Date = Today_Date.getSecondsSince1970() / 86400;
    fwrite(&amu_hdrvar, HDR_SIZE, 1, AMUcfgfile);

    /* Set up the configuration record */
    memset(&amu_cfgvar, NULL, CFG_SIZE);
#if defined (__AMU16__)
    amu_cfgvar.swaptype = MODE_1;
#endif
    amu_cfgvar.left_bracket = '[';
    amu_cfgvar.right_bracket = ']';
    strcpy(amu_cfgvar.ext_desc_char, "+-!|");
    amu_cfgvar.ext_offset = 2;
    amu_cfgvar.counter_width = 2;
    amu_cfgvar.use_multiple_lines = TRUE;
    amu_cfgvar.max_single_length = 1024;
    strcpy(amu_cfgvar.missing_desc, "No description available");
    if(AMUenv[0] != NULL)
    {
        sprintf(amu_cfgvar.AMUpath, "%s", AMUenv);
        sprintf(amu_cfgvar.AMUlogfile, "%sAMU.LOG", AMUenv);
    }
    else
    {
        strcpy(amu_cfgvar.AMUlogfile, "AMU.LOG");
    }
    init_external();
    fwrite(&amu_cfgvar, CFG_SIZE, 1, AMUcfgfile);
    fseek(AMUcfgfile, 0, SEEK_SET);
    return(TRUE);
}


/************************************************************************/
/* Open the configuration file. Return TRUE if successful, FALSE if not */
/************************************************************************/
int Config::open_config(int type)
{
    long rec_size, length;
    char *AMUenv;
    FILE **cfgfile;

    close_config(type);
    AMUenv = getenv("AMU");
    cfg_filename[0] = NULL;
    if(AMUenv)
    {
       strcpy(cfg_filename, AMUenv);
//strcpy(cfg_filename, "D:\\AMU\\");

        Amustr_obj.noslash(cfg_filename);
        Utility_obj.change_dir(cfg_filename);
        Amustr_obj.slash(cfg_filename);
    }

    cfgfile = NULL;
    rec_size = length = 0;
    switch(type)
    {
        case CFG_RECORD     : strcat(cfg_filename, CONFIG_AMU);
                              rec_size = 1;
                              cfgfile = &AMUcfgfile;
                              break;
        case ALLFILE_RECORD : strcat(cfg_filename, LISTS_AMU);
                              rec_size = ALLNEW_SIZE + AREAS_SIZE;
                              cfgfile = &LISTScfg;
                              break;
        case ARCHIVE_RECORD : strcat(cfg_filename, ARCHIVES_AMU);
                              rec_size = ARC_SIZE;
                              cfgfile = &ARCHIVEScfg;
                              break;
        case ENFORCE_RECORD : strcat(cfg_filename, ENFORCE_AMU);
                              rec_size = ENFORCE_SIZE;
                              cfgfile = &ENFORCEcfg;
                              break;
        case LOGFILE_RECORD : strcat(cfg_filename, LOGS_AMU);
                              rec_size = LOG_SIZE;
                              cfgfile = &LOGScfg;
                              break;
        case FILEAREA_RECORD: strcat(cfg_filename, FILEAREA_AMU);
                              rec_size = FILEAREA_SIZE;
                              cfgfile = &FILEAREAcfg;
                              break;
        case BULLETIN_RECORD: strcat(cfg_filename, BULLETIN_AMU);
                              rec_size = BULLETIN_SIZE + AREAS_SIZE;
                              cfgfile = &BULLETINcfg;
                              break;
        case USERMSGS_RECORD: strcat(cfg_filename, USERMSGS_AMU);
                              rec_size = USERMSGS_SIZE + AREAS_SIZE;
                              cfgfile = &USERMSGScfg;
                              break;
        case MSGAREA_RECORD : strcat(cfg_filename, MSGAREA_AMU);
                              rec_size = MSGAREA_SIZE;
                              cfgfile = &MSGAREAcfg;
                              break;
        case MAINT_RECORD   : strcat(cfg_filename, MAINT_AMU);
                              rec_size = MAINT_SIZE;
                              cfgfile = &MAINTcfg;
                              break;
        default: return(FALSE);
    }

    *cfgfile = _fsopen(cfg_filename, "r+b", SH_DENYWR);
    if(!*cfgfile && type != CFG_RECORD)
    {
        *cfgfile = _fsopen(cfg_filename, "w+b", SH_DENYWR);
    }

    if(!*cfgfile)
    {
        return(FALSE);
    }

    length = filelength(fileno(*cfgfile)) / rec_size;
    if(length == 0)
    {
        add_record(type, ADD_NEW_RECORD);
        length = filelength(fileno(*cfgfile)) / rec_size;
    }
    switch(type)
    {
        case ALLFILE_RECORD : num_lists = length;
                              break;
        case ARCHIVE_RECORD : num_archives = length;
                              break;
        case ENFORCE_RECORD : num_enforce = length;
                              break;
        case LOGFILE_RECORD : num_logs = length;
                              break;
        case FILEAREA_RECORD: num_filearea = length - 1; // Adjust for the default record
                              break;
        case BULLETIN_RECORD: num_bulletin = length;
                              break;
        case USERMSGS_RECORD: num_usermsgs = length;
                              break;
        case MSGAREA_RECORD : num_msgarea = length;
                              break;
        case MAINT_RECORD   : num_maint = length;
                              break;
    }
    return(TRUE);
}

/************************************************************************/
/* Read the header and main configuration records.  Returns nothing.    */
/************************************************************************/
void Config::read_main_config(void)
{
    char *ptr;

    /* Read the header */
    fseek(AMUcfgfile, 0, SEEK_SET);
    fread(&amu_hdrvar, sizeof(amu_hdrvar), 1, AMUcfgfile);

    /* Check the revision level */
    if(amu_hdrvar.Revision != REVISION)
    {
        Screen_obj.clearscreen();
        printf("\n\n! '%s' is for an older version.  Aborting.\n",
               cfg_filename);
#if defined(__AMU2__)
        printf("Run 'AMUCFG2 -UPGRADE' to upgrade the configuration.\n");
#else
        printf("Run 'AMUCFG -UPGRADE' to upgrade the configuration.\n");
#endif
        close_config(CFG_RECORD);
#if defined(__AMU16__) || defined(__AMU2__)
        _setcursortype(_NORMALCURSOR);
#endif
        exit(NO_CONFIG);
    }

    /* Read the main config structure */
    fread(&amu_cfgvar, sizeof(amu_cfgvar), 1, AMUcfgfile);
    crc = Crc_obj.struct_crc((char *)&amu_hdrvar, HDR_SIZE)
          + Crc_obj.struct_crc((char *)&amu_cfgvar, CFG_SIZE);
    
    ptr = strchr(amu_cfgvar.ext_desc_char, ';');
    if(ptr)
    {
        *ptr = NULL;
    }
    if(amu_cfgvar.date_separator == NULL)
    {
        amu_cfgvar.date_separator = '-';
    }

    switch(amu_cfgvar.BBStype)
    {
        case RA2     : strcpy(bbs_software, "RemoteAccess 2.02");
                       break;
        case RA250   : strcpy(bbs_software, "RemoteAccess 2.52");
                       break;
        case SBBS    : strcpy(bbs_software, "SuperBBS 1.17");
                       break;
        case QBBS    : strcpy(bbs_software, "QuickBBS 2.76");
                       break;
        case RA1     : strcpy(bbs_software, "RemoteAccess 1.11");
                       break;
        case QBBSC   : strcpy(bbs_software, "QuickBBS 2.85/Hudson");
                       break;
        case QBBSG   : strcpy(bbs_software, "QuickBBS 2.85/GoldBase");
                       break;
        case PB2     : strcpy(bbs_software, "ProBoard 2.16");
                       break;
        case EZ      : strcpy(bbs_software, "EzyCom 1.20/1.48g");
                       break;
        case TG3     : strcpy(bbs_software, "Telegard 3.02");
                       break;
        case TG310   : strcpy(bbs_software, "Telegard 3.09g");
                       break;
        case CONCORD : strcpy(bbs_software, "Concord 0.01");
                       break;
        case LORA    : strcpy(bbs_software, "LoraBBS 2.40");
                       break;
        case LORA3   : strcpy(bbs_software, "LoraBBS 2.99");
                       break;
        case MAXIMUS2: strcpy(bbs_software, "Maximus 2.02");
                       break;
        case MAXIMUS3: strcpy(bbs_software, "Maximus 3.01");
                       break;
    }
}


/************************************************************************/
/* Close the configuration file.  Returns nothing.                      */
/************************************************************************/
void Config::close_config(int type)
{
    FILE **cfgfile;

    cfgfile = NULL;
    switch(type)
    {
        case CFG_RECORD     : cfgfile = &AMUcfgfile;
                              break;
        case ALLFILE_RECORD : cfgfile = &LISTScfg;
                              break;
        case ARCHIVE_RECORD : cfgfile = &ARCHIVEScfg;
                              break;
        case ENFORCE_RECORD : cfgfile = &ENFORCEcfg;
                              break;
        case LOGFILE_RECORD : cfgfile = &LOGScfg;
                              break;
        case FILEAREA_RECORD: cfgfile = &FILEAREAcfg;
                              break;
        case BULLETIN_RECORD: cfgfile = &BULLETINcfg;
                              break;
        case USERMSGS_RECORD: cfgfile = &USERMSGScfg;
                              break;
        case MSGAREA_RECORD : cfgfile = &MSGAREAcfg;
                              break;
        case MAINT_RECORD   : cfgfile = &MAINTcfg;
                              break;
    }
    if(cfgfile && *cfgfile)
    {
        fclose(*cfgfile);
        *cfgfile = NULL;
    }
}

/************************************************************************/
/* Read the specified record type & number.  Returns TRUE or FALSE.     */
/************************************************************************/
int Config::read_record(int type, int num)
{
    long offset, rec_size;
    FILE **cfgfile;

    if(num < 1 && num != DEFAULT_RECORD)
    {
        return(FALSE);
    }

    cfgfile = NULL;
    switch(type)
    {
        case ALLFILE_RECORD : cfgfile = &LISTScfg;
                              rec_size = ALLNEW_SIZE + AREAS_SIZE;
                              break;
        case ARCHIVE_RECORD : cfgfile = &ARCHIVEScfg;
                              rec_size = ARC_SIZE;
                              break;
        case ENFORCE_RECORD : cfgfile = &ENFORCEcfg;
                              rec_size = ENFORCE_SIZE;
                              break;
        case LOGFILE_RECORD : cfgfile = &LOGScfg;
                              rec_size = LOG_SIZE;
                              break;
        case FILEAREA_RECORD: cfgfile = &FILEAREAcfg;
                              rec_size = FILEAREA_SIZE;
                              break;
        case BULLETIN_RECORD: cfgfile = &BULLETINcfg;
                              rec_size = BULLETIN_SIZE + AREAS_SIZE;
                              break;
        case USERMSGS_RECORD: cfgfile = &USERMSGScfg;
                              rec_size = USERMSGS_SIZE + AREAS_SIZE;
                              break;
        case MSGAREA_RECORD : cfgfile = &MSGAREAcfg;
                              rec_size = MSGAREA_SIZE;
                              break;
        case MAINT_RECORD   : cfgfile = &MAINTcfg;
                              rec_size = MAINT_SIZE;
                              break;
    }

    // See if we should read the default (0th) record
    if(type == FILEAREA_RECORD && num == DEFAULT_RECORD)
    {
        offset = 0;
    }
    else
    {
        if(type != FILEAREA_RECORD)
        {
            num--;
        }
        offset = num * rec_size;
    }

    fseek(*cfgfile, offset, SEEK_SET);
    switch(type)
    {
        case ALLFILE_RECORD : fread(&allnewvar, ALLNEW_SIZE, 1, *cfgfile);
                              fread(&areasvar, AREAS_SIZE, 1, *cfgfile);
                              break;
        case ARCHIVE_RECORD : fread(&archivesvar, ARC_SIZE, 1, *cfgfile);
                              break;
        case ENFORCE_RECORD : fread(&enforcevar, ENFORCE_SIZE, 1, *cfgfile);
                              break;
        case LOGFILE_RECORD : fread(&logsvar, LOG_SIZE, 1, *cfgfile);
                              break;
        case FILEAREA_RECORD: fread(&fileareavar, FILEAREA_SIZE, 1, *cfgfile);
                              break;
        case BULLETIN_RECORD: fread(&bulletinvar, BULLETIN_SIZE, 1, *cfgfile);
                              fread(&areasvar, AREAS_SIZE, 1, *cfgfile);
                              break;
        case USERMSGS_RECORD: fread(&usermsgsvar, USERMSGS_SIZE, 1, *cfgfile);
                              fread(&areasvar, AREAS_SIZE, 1, *cfgfile);
                              break;
        case MSGAREA_RECORD : fread(&msgareavar, MSGAREA_SIZE, 1, *cfgfile);
                              break;
        case MAINT_RECORD   : fread(&maintvar, MAINT_SIZE, 1, *cfgfile);
                              break;
    }

    if(ferror(*cfgfile))
    {
        return(FALSE);
    }
    return(TRUE);
}

/************************************************************************/
/* Update the specified record type & number.  Returns TRUE or FALSE.   */
/************************************************************************/
int Config::update_record(int type, int num)
{
    long offset, rec_size;
    FILE **cfgfile;

    if(num < 1 && num != DEFAULT_RECORD)
    {
        return(FALSE);
    }

    cfgfile = NULL;
    switch(type)
    {
        case ALLFILE_RECORD : cfgfile = &LISTScfg;
                              rec_size = ALLNEW_SIZE + AREAS_SIZE;
                              break;
        case ARCHIVE_RECORD : cfgfile = &ARCHIVEScfg;
                              rec_size = ARC_SIZE;
                              break;
        case ENFORCE_RECORD : cfgfile = &ENFORCEcfg;
                              rec_size = ENFORCE_SIZE;
                              break;
        case LOGFILE_RECORD : cfgfile = &LOGScfg;
                              rec_size = LOG_SIZE;
                              break;
        case FILEAREA_RECORD: cfgfile = &FILEAREAcfg;
                              rec_size = FILEAREA_SIZE;
                              break;
        case BULLETIN_RECORD: cfgfile = &BULLETINcfg;
                              rec_size = BULLETIN_SIZE + AREAS_SIZE;
                              break;
        case USERMSGS_RECORD: cfgfile = &USERMSGScfg;
                              rec_size = USERMSGS_SIZE + AREAS_SIZE;
                              break;
        case MSGAREA_RECORD : cfgfile = &MSGAREAcfg;
                              rec_size = MSGAREA_SIZE;
                              break;
        case MAINT_RECORD   : cfgfile = &MAINTcfg;
                              rec_size = MAINT_SIZE;
                              break;
    }

    // See if we should update the default record
    if(type == FILEAREA_RECORD && num == DEFAULT_RECORD)
    {
        offset = 0;
    }
    else
    {
        if(type != FILEAREA_RECORD)
        {
            num--;
        }
        offset = num * rec_size;
    }

    fseek(*cfgfile, offset, SEEK_SET);
    switch(type)
    {
        case ALLFILE_RECORD : fwrite(&allnewvar, ALLNEW_SIZE, 1, *cfgfile);
                              fwrite(&areasvar, AREAS_SIZE, 1, *cfgfile);
                              break;
        case ARCHIVE_RECORD : fwrite(&archivesvar, ARC_SIZE, 1, *cfgfile);
                              break;
        case ENFORCE_RECORD : fwrite(&enforcevar, ENFORCE_SIZE, 1, *cfgfile);
                              break;
        case LOGFILE_RECORD : fwrite(&logsvar, LOG_SIZE, 1, *cfgfile);
                              break;
        case FILEAREA_RECORD: fwrite(&fileareavar, FILEAREA_SIZE, 1, *cfgfile);
                              break;
        case BULLETIN_RECORD: fwrite(&bulletinvar, BULLETIN_SIZE, 1, *cfgfile);
                              fwrite(&areasvar, AREAS_SIZE, 1, *cfgfile);
                              break;
        case USERMSGS_RECORD: fwrite(&usermsgsvar, USERMSGS_SIZE, 1, *cfgfile);
                              fwrite(&areasvar, AREAS_SIZE, 1, *cfgfile);
                              break;
        case MSGAREA_RECORD : fwrite(&msgareavar, MSGAREA_SIZE, 1, *cfgfile);
                              break;
        case MAINT_RECORD   : fwrite(&maintvar, MAINT_SIZE, 1, *cfgfile);
                              break;
    }

    if(ferror(*cfgfile))
    {
        return(FALSE);
    }
    return(TRUE);
}

/************************************************************************/
/* Add a blank record or add a copy of the current record.              */
/************************************************************************/
int Config::add_record(int type, int mode)
{
    FILE **cfgfile;

    if(mode == ADD_NEW_RECORD)
    {
        switch(type)
        {
            case ALLFILE_RECORD : memset(&allnewvar, NULL, ALLNEW_SIZE);
                                  memset(&areasvar, NULL, AREAS_SIZE);
                                  strcpy(allnewvar.list_format, DEFAULT_LISTSTR);
                                  allnewvar.age = 30;
                                  break;
            case ARCHIVE_RECORD : memset(&archivesvar, NULL, ARC_SIZE);
                                  break;
            case ENFORCE_RECORD : memset(&enforcevar, NULL, ENFORCE_SIZE);
                                  break;
            case LOGFILE_RECORD : memset(&logsvar, NULL, LOG_SIZE);
                                  break;
            case FILEAREA_RECORD: memset(&fileareavar, NULL, FILEAREA_SIZE);
                                  break;
            case BULLETIN_RECORD: memset(&bulletinvar, NULL, BULLETIN_SIZE);
                                  memset(&areasvar, NULL, AREAS_SIZE);
                                  bulletinvar.num_listed = 10;
                                  break;
            case USERMSGS_RECORD: memset(&usermsgsvar, NULL, USERMSGS_SIZE);
                                  memset(&areasvar, NULL, AREAS_SIZE);
                                  break;
            case MSGAREA_RECORD : memset(&msgareavar, NULL, MSGAREA_SIZE);
                                  break;
            case MAINT_RECORD   : memset(&maintvar, NULL, MAINT_SIZE);
                                  break;
        }
    }

    cfgfile = NULL;
    switch(type)
    {
        case ALLFILE_RECORD : cfgfile = &LISTScfg;
                              break;
        case ARCHIVE_RECORD : cfgfile = &ARCHIVEScfg;
                              break;
        case ENFORCE_RECORD : cfgfile = &ENFORCEcfg;
                              break;
        case LOGFILE_RECORD : cfgfile = &LOGScfg;
                              break;
        case FILEAREA_RECORD: cfgfile = &FILEAREAcfg;
                              break;
        case BULLETIN_RECORD: cfgfile = &BULLETINcfg;
                              break;
        case USERMSGS_RECORD: cfgfile = &USERMSGScfg;
                              break;
        case MSGAREA_RECORD : cfgfile = &MSGAREAcfg;
                              break;
        case MAINT_RECORD   : cfgfile = &MAINTcfg;
                              break;
    }

    fseek(*cfgfile, 0, SEEK_END);
    switch(type)
    {
        case ALLFILE_RECORD : fwrite(&allnewvar, ALLNEW_SIZE, 1, *cfgfile);
                              fwrite(&areasvar, AREAS_SIZE, 1, *cfgfile);
                              num_lists++;
                              break;
        case ARCHIVE_RECORD : fwrite(&archivesvar, ARC_SIZE, 1, *cfgfile);
                              num_archives++;
                              break;
        case ENFORCE_RECORD : fwrite(&enforcevar, ENFORCE_SIZE, 1, *cfgfile);
                              num_enforce++;
                              break;
        case LOGFILE_RECORD : fwrite(&logsvar, LOG_SIZE, 1, *cfgfile);
                              num_logs++;
                              break;
        case FILEAREA_RECORD: fwrite(&fileareavar, FILEAREA_SIZE, 1, *cfgfile);
                              num_filearea++;
                              break;
        case BULLETIN_RECORD: fwrite(&bulletinvar, BULLETIN_SIZE, 1, *cfgfile);
                              fwrite(&areasvar, AREAS_SIZE, 1, *cfgfile);
                              num_bulletin++;
                              break;
        case USERMSGS_RECORD: fwrite(&usermsgsvar, USERMSGS_SIZE, 1, *cfgfile);
                              fwrite(&areasvar, AREAS_SIZE, 1, *cfgfile);
                              num_usermsgs++;
                              break;
        case MSGAREA_RECORD : fwrite(&msgareavar, MSGAREA_SIZE, 1, *cfgfile);
                              num_msgarea++;
                              break;
        case MAINT_RECORD   : fwrite(&maintvar, MAINT_SIZE, 1, *cfgfile);
                              num_maint++;
                              break;
    }

    if(ferror(*cfgfile))
    {
        return(FALSE);
    }
    return(TRUE);
}

/************************************************************************/
/* Initialize the external programs.                                    */
/************************************************************************/
void Config::init_external(void)
{
        /* DOS programs */
    strcpy(amu_cfgvar.archivers[ARC].dos_arc, "PKPAK.EXE");
    strcpy(amu_cfgvar.archivers[ARC].dos_unarc, "PKUNPAK.EXE");
    strcpy(amu_cfgvar.archivers[ARC].dos_arcparam, "-au");
    strcpy(amu_cfgvar.archivers[ARC].dos_unarcparam, "-xr");

    strcpy(amu_cfgvar.archivers[ARJ].dos_arc, "ARJ.EXE");
    strcpy(amu_cfgvar.archivers[ARJ].dos_unarc, "ARJ.EXE");
    strcpy(amu_cfgvar.archivers[ARJ].dos_arcparam, "a -e -m1 -s -t0 -y");
    strcpy(amu_cfgvar.archivers[ARJ].dos_unarcparam, "e -c -y");

    strcpy(amu_cfgvar.archivers[HAP].dos_arc, "HAP.EXE");
    strcpy(amu_cfgvar.archivers[HAP].dos_unarc, "PAH.EXE");
    strcpy(amu_cfgvar.archivers[HAP].dos_arcparam, "a");
    strcpy(amu_cfgvar.archivers[HAP].dos_unarcparam, "e");

    strcpy(amu_cfgvar.archivers[LZH].dos_arc, "LHA.EXE");
    strcpy(amu_cfgvar.archivers[LZH].dos_unarc, "LHA.EXE");
    strcpy(amu_cfgvar.archivers[LZH].dos_arcparam, "a /mt");
    strcpy(amu_cfgvar.archivers[LZH].dos_unarcparam, "e /cm");

    strcpy(amu_cfgvar.archivers[PAK].dos_arc, "PAK.EXE");
    strcpy(amu_cfgvar.archivers[PAK].dos_unarc, "PAK.EXE");
    strcpy(amu_cfgvar.archivers[PAK].dos_arcparam, "A /L /ST");
    strcpy(amu_cfgvar.archivers[PAK].dos_unarcparam, "E /WA");

    strcpy(amu_cfgvar.archivers[SQZ].dos_arc, "SQZ.EXE");
    strcpy(amu_cfgvar.archivers[SQZ].dos_unarc, "SQZ.EXE");
    strcpy(amu_cfgvar.archivers[SQZ].dos_arcparam, "a /p3q0z3");
    strcpy(amu_cfgvar.archivers[SQZ].dos_unarcparam, "e /o1");

    strcpy(amu_cfgvar.archivers[ZIP].dos_arc, "PKZIP.EXE");
    strcpy(amu_cfgvar.archivers[ZIP].dos_unarc, "PKUNZIP.EXE");
    strcpy(amu_cfgvar.archivers[ZIP].dos_arcparam, "-ao -~");
    strcpy(amu_cfgvar.archivers[ZIP].dos_unarcparam, "-o -ed");

    strcpy(amu_cfgvar.archivers[ZOO].dos_arc, "ZOO.EXE");
    strcpy(amu_cfgvar.archivers[ZOO].dos_unarc, "ZOO.EXE");
    strcpy(amu_cfgvar.archivers[ZOO].dos_arcparam, "a:");
    strcpy(amu_cfgvar.archivers[ZOO].dos_unarcparam, "eO");

    strcpy(amu_cfgvar.archivers[UC2].dos_arc, "UC2.EXE");
    strcpy(amu_cfgvar.archivers[UC2].dos_unarc, "UC2.EXE");
    strcpy(amu_cfgvar.archivers[UC2].dos_arcparam, "A -BFTT");
    strcpy(amu_cfgvar.archivers[UC2].dos_unarcparam, "E -F");

    strcpy(amu_cfgvar.archivers[RAR].dos_arc, "RAR.EXE");
    strcpy(amu_cfgvar.archivers[RAR].dos_unarc, "RAR.EXE");
    strcpy(amu_cfgvar.archivers[RAR].dos_arcparam, "A -Y -STD");
    strcpy(amu_cfgvar.archivers[RAR].dos_unarcparam, "E -Y -STD -EP -C-");

    strcpy(amu_cfgvar.scanners[0].dos_scanner, "SCAN.EXE");
    strcpy(amu_cfgvar.scanners[0].dos_parameters, "/nomem /sub /all");
    strcpy(amu_cfgvar.scanners[0].dos_path, "C:\\UTILITY\\");
    strcpy(amu_cfgvar.scanners[0].dos_viruslevel, "1");


     /* OS/2 programs */
    strcpy(amu_cfgvar.archivers[ARC].os2_arc, "PKPAK.EXE");
    strcpy(amu_cfgvar.archivers[ARC].os2_unarc, "PKUNPAK.EXE");
    strcpy(amu_cfgvar.archivers[ARC].os2_arcparam, "-au");
    strcpy(amu_cfgvar.archivers[ARC].os2_unarcparam, "-xr");

    strcpy(amu_cfgvar.archivers[ARJ].os2_arc, "ARJ.EXE");
    strcpy(amu_cfgvar.archivers[ARJ].os2_unarc, "ARJ.EXE");
    strcpy(amu_cfgvar.archivers[ARJ].os2_arcparam, "a -e -m1 -s -t0 -y");
    strcpy(amu_cfgvar.archivers[ARJ].os2_unarcparam, "e -c -y");

    strcpy(amu_cfgvar.archivers[HAP].os2_arc, "HAP.EXE");
    strcpy(amu_cfgvar.archivers[HAP].os2_unarc, "PAH.EXE");
    strcpy(amu_cfgvar.archivers[HAP].os2_arcparam, "a");
    strcpy(amu_cfgvar.archivers[HAP].os2_unarcparam, "e");

    strcpy(amu_cfgvar.archivers[LZH].os2_arc, "LHA.EXE");
    strcpy(amu_cfgvar.archivers[LZH].os2_unarc, "LHA.EXE");
    strcpy(amu_cfgvar.archivers[LZH].os2_arcparam, "a /mt");
    strcpy(amu_cfgvar.archivers[LZH].os2_unarcparam, "e /cm");

    strcpy(amu_cfgvar.archivers[PAK].os2_arc, "PAK.EXE");
    strcpy(amu_cfgvar.archivers[PAK].os2_unarc, "PAK.EXE");
    strcpy(amu_cfgvar.archivers[PAK].os2_arcparam, "A /L /ST");
    strcpy(amu_cfgvar.archivers[PAK].os2_unarcparam, "E /WA");

    strcpy(amu_cfgvar.archivers[SQZ].os2_arc, "SQZ.EXE");
    strcpy(amu_cfgvar.archivers[SQZ].os2_unarc, "SQZ.EXE");
    strcpy(amu_cfgvar.archivers[SQZ].os2_arcparam, "a /p3q0z3");
    strcpy(amu_cfgvar.archivers[SQZ].os2_unarcparam, "e /o1");

    strcpy(amu_cfgvar.archivers[ZIP].os2_arc, "ZIP.EXE");
    strcpy(amu_cfgvar.archivers[ZIP].os2_unarc, "UNZIP.EXE");
    strcpy(amu_cfgvar.archivers[ZIP].os2_arcparam, "-k -j");
    strcpy(amu_cfgvar.archivers[ZIP].os2_unarcparam, "-C -j -o");

    strcpy(amu_cfgvar.archivers[ZOO].os2_arc, "ZOO.EXE");
    strcpy(amu_cfgvar.archivers[ZOO].os2_unarc, "ZOO.EXE");
    strcpy(amu_cfgvar.archivers[ZOO].os2_arcparam, "a:");
    strcpy(amu_cfgvar.archivers[ZOO].os2_unarcparam, "eO");

    strcpy(amu_cfgvar.archivers[UC2].os2_arc, "UC2.EXE");
    strcpy(amu_cfgvar.archivers[UC2].os2_unarc, "UC2.EXE");
    strcpy(amu_cfgvar.archivers[UC2].os2_arcparam, "A -BFTT");
    strcpy(amu_cfgvar.archivers[UC2].os2_unarcparam, "E -F");

    strcpy(amu_cfgvar.archivers[RAR].os2_arc, "RAR.EXE");
    strcpy(amu_cfgvar.archivers[RAR].os2_unarc, "RAR.EXE");
    strcpy(amu_cfgvar.archivers[RAR].os2_arcparam, "A -Y -STD");
    strcpy(amu_cfgvar.archivers[RAR].os2_unarcparam, "E -Y -STD -EP -C-");

    strcpy(amu_cfgvar.scanners[0].os2_scanner, "SCAN.EXE");
    strcpy(amu_cfgvar.scanners[0].os2_parameters, "/nomem /sub /all");
    strcpy(amu_cfgvar.scanners[0].os2_path, "C:\\UTILITY\\");
    strcpy(amu_cfgvar.scanners[0].os2_viruslevel, "1");
}

/************************************************************************/
/* Save the configuration.  Returns TRUE or FALSE.                      */
/************************************************************************/
void Config::save_cfg(int type)
{
    char oldname[13], newname[13];
    FILE *oldfile, *newfile;

    if(type == SAVE_ALL)
    {
        fseek(AMUcfgfile, 0, SEEK_SET);
        fwrite(&amu_hdrvar, HDR_SIZE, 1, AMUcfgfile);
        fwrite(&amu_cfgvar, CFG_SIZE, 1, AMUcfgfile);
    }
    strcpy(newname, "NEWCFG.$$$");
    strcpy(oldname, LISTS_AMU);
    oldfile = fopen(oldname, "r+b");
    if(oldfile)
    {
        newfile = fopen(newname, "w+b");
        fread(&allnewvar, ALLNEW_SIZE, 1, oldfile);
        fread(&areasvar, AREAS_SIZE, 1, oldfile);
        while(!feof(oldfile) && !ferror(oldfile))
        {
            if(!allnewvar.deleted)
            {
                fwrite(&allnewvar, ALLNEW_SIZE, 1, newfile);
                fwrite(&areasvar, AREAS_SIZE, 1, newfile);
            }
            fread(&allnewvar, ALLNEW_SIZE, 1, oldfile);
            fread(&areasvar, AREAS_SIZE, 1, oldfile);
        }
        fclose(oldfile);
        fclose(newfile);
        Utility_obj.copyfile(newname, oldname);
        remove(newname);
    }

    strcpy(oldname, ARCHIVES_AMU);
    oldfile = fopen(oldname, "r+b");
    if(oldfile)
    {
        newfile = fopen(newname, "w+b");
        fread(&archivesvar, ARC_SIZE, 1, oldfile);
        while(!feof(oldfile) && !ferror(oldfile))
        {
            if(!archivesvar.deleted)
            {
                fwrite(&archivesvar, ARC_SIZE, 1, newfile);
            }
            fread(&archivesvar, ARC_SIZE, 1, oldfile);
        }
        fclose(oldfile);
        fclose(newfile);
        Utility_obj.copyfile(newname, oldname);
        remove(newname);
    }

    strcpy(oldname, ENFORCE_AMU);
    oldfile = fopen(oldname, "r+b");
    if(oldfile)
    {
        newfile = fopen(newname, "w+b");
        fread(&enforcevar, ENFORCE_SIZE, 1, oldfile);
        while(!feof(oldfile) && !ferror(oldfile))
        {
            if(!enforcevar.deleted)
            {
                fwrite(&enforcevar, ENFORCE_SIZE, 1, newfile);
            }
            fread(&enforcevar, ENFORCE_SIZE, 1, oldfile);
        }
        fclose(oldfile);
        fclose(newfile);
        Utility_obj.copyfile(newname, oldname);
        remove(newname);
    }

    strcpy(oldname, LOGS_AMU);
    oldfile = fopen(oldname, "r+b");
    if(oldfile)
    {
        newfile = fopen(newname, "w+b");
        fread(&logsvar, LOG_SIZE, 1, oldfile);
        while(!feof(oldfile) && !ferror(oldfile))
        {
            if(!logsvar.deleted)
            {
                fwrite(&logsvar, LOG_SIZE, 1, newfile);
            }
            fread(&logsvar, LOG_SIZE, 1, oldfile);
        }
        fclose(oldfile);
        fclose(newfile);
        Utility_obj.copyfile(newname, oldname);
        remove(newname);
    }

    strcpy(oldname, FILEAREA_AMU);
    oldfile = fopen(oldname, "r+b");
    if(oldfile)
    {
        newfile = fopen(newname, "w+b");
        fread(&fileareavar, FILEAREA_SIZE, 1, oldfile);
        while(!feof(oldfile) && !ferror(oldfile))
        {
            if(!fileareavar.deleted)
            {
                fwrite(&fileareavar, FILEAREA_SIZE, 1, newfile);
            }
            fread(&fileareavar, FILEAREA_SIZE, 1, oldfile);
        }
        fclose(oldfile);
        fclose(newfile);
        Utility_obj.copyfile(newname, oldname);
        remove(newname);
    }

    strcpy(oldname, BULLETIN_AMU);
    oldfile = fopen(oldname, "r+b");
    if(oldfile)
    {
        newfile = fopen(newname, "w+b");
        fread(&bulletinvar, BULLETIN_SIZE, 1, oldfile);
        fread(&areasvar, AREAS_SIZE, 1, oldfile);
        while(!feof(oldfile) && !ferror(oldfile))
        {
            if(bulletinvar.deleted == FALSE)
            {
                fwrite(&bulletinvar, BULLETIN_SIZE, 1, newfile);
                fwrite(&areasvar, AREAS_SIZE, 1, newfile);
            }
            fread(&bulletinvar, BULLETIN_SIZE, 1, oldfile);
            fread(&areasvar, AREAS_SIZE, 1, oldfile);
        }
        fclose(oldfile);
        fclose(newfile);
        Utility_obj.copyfile(newname, oldname);
        remove(newname);
    }

    strcpy(oldname, USERMSGS_AMU);
    oldfile = fopen(oldname, "r+b");
    if(oldfile)
    {
        newfile = fopen(newname, "w+b");
        fread(&usermsgsvar, USERMSGS_SIZE, 1, oldfile);
        fread(&areasvar, AREAS_SIZE, 1, oldfile);
        while(!feof(oldfile) && !ferror(oldfile))
        {
            if(!usermsgsvar.deleted)
            {
                fwrite(&usermsgsvar, USERMSGS_SIZE, 1, newfile);
                fwrite(&areasvar, AREAS_SIZE, 1, newfile);
            }
            fread(&usermsgsvar, USERMSGS_SIZE, 1, oldfile);
            fread(&areasvar, AREAS_SIZE, 1, oldfile);
        }
        fclose(oldfile);
        fclose(newfile);
        Utility_obj.copyfile(newname, oldname);
        remove(newname);
    }

    strcpy(oldname, MSGAREA_AMU);
    oldfile = fopen(oldname, "r+b");
    if(oldfile)
    {
        newfile = fopen(newname, "w+b");
        fread(&msgareavar, MSGAREA_SIZE, 1, oldfile);
        while(!feof(oldfile) && !ferror(oldfile))
        {
            if(!msgareavar.deleted)
            {
                fwrite(&msgareavar, MSGAREA_SIZE, 1, newfile);
            }
            fread(&msgareavar, MSGAREA_SIZE, 1, oldfile);
        }
        fclose(oldfile);
        fclose(newfile);
        Utility_obj.copyfile(newname, oldname);
        remove(newname);
    }

    strcpy(oldname, MAINT_AMU);
    oldfile = fopen(oldname, "r+b");
    if(oldfile)
    {
        newfile = fopen(newname, "w+b");
        fread(&maintvar, MAINT_SIZE, 1, oldfile);
        while(!feof(oldfile) && !ferror(oldfile))
        {
            if(!maintvar.deleted)
            {
                fwrite(&maintvar, MAINT_SIZE, 1, newfile);
            }
            fread(&maintvar, MAINT_SIZE, 1, oldfile);
        }
        fclose(oldfile);
        fclose(newfile);
        Utility_obj.copyfile(newname, oldname);
        remove(newname);
    }
}

/***********************************************************************/
/* Returns whether or not the given area is active */
int Config::active(long area)
{
    int index, bit;

    index = area / 16;  // Array index
    bit = area % 16;    // Which bit

    if(Utility_obj.bitset(areasvar.status[index], BIT_ARRAY[bit]))
    {
        return(TRUE);
    }
    return(FALSE);
}

/*************************************************************************/
/* Toggles the given area */
void Config::togglearea(long area)
{
    int index, bit;

    index = area / 16;  // Array index
    bit = area % 16;    // Which bit

    Utility_obj.togglebit((unsigned *) &areasvar.status[index], BIT_ARRAY[bit]);
}

/*************************************************************************/
/* Return the BBS software string                                        */
/*************************************************************************/
char *Config::get_bbstype(void)
{
    return(bbs_software);
}

/*************************************************************************/
/* Return the COMSPEC                                                    */
/*************************************************************************/
char *Config::get_comspec(void)
{
    return(comspec);
}

/*************************************************************************/
/* Upgrade the configuration                                             */
/*************************************************************************/
void Config::upgrade_config(void)
{
#if !defined(AMUCFG)
    return;
#else
#include "rev8.h"
    short level;
    FILE *targetfile, *sourcefile;
    char source[MAX_FILESPEC], target[MAX_FILESPEC];
    struct rev8_allnew rev8_allnewvar;
    struct rev8_filearea rev8_fileareavar;
    struct rev8_bulletin rev8_bulletinvar;
    struct rev8_msgarea rev8_msgareavar;

    fseek(AMUcfgfile, 0, SEEK_SET);
    fread(&level, sizeof(level), 1, AMUcfgfile);

    switch(level)
    {
        /* Check to see if this is current */
        case REVISION: printf("\nConfiguration current, no update necessary\n");
                       break;

        case PREVIOUS_REVISION:
                       sprintf(source, "%s", FILEAREA_AMU);
                       sprintf(target, "FILEAREA.$$$");
                       sourcefile = fopen(source, "rb");
                       if(sourcefile)
                       {
                           targetfile = fopen(target, "w+b");
                           memset(&fileareavar, NULL, sizeof(fileareavar));
                           strcpy(fileareavar.areaname, "DEFAULTS FOR NEW AREAS");

                           // Add the default record
                           fwrite(&fileareavar, sizeof(fileareavar), 1, targetfile);

                           // Copy the rest of the data to the new file
                           fread(&fileareavar, sizeof(fileareavar), 1, sourcefile);
                           while(!feof(sourcefile))
                           {
                               fwrite(&fileareavar, sizeof(fileareavar), 1, targetfile);
                               fread(&fileareavar, sizeof(fileareavar), 1, sourcefile);
                           }
                           fclose(sourcefile);
                           fclose(targetfile);
                           Utility_obj.copyfile(target, source);
                           remove(target);

                            // Update the revision level
                           fseek(AMUcfgfile, 0, SEEK_SET);
                           fread(&amu_hdrvar, sizeof(amu_hdrvar), 1, AMUcfgfile);
                           fread(&amu_cfgvar, sizeof(amu_cfgvar), 1, AMUcfgfile);
                           amu_hdrvar.Revision = REVISION;
                           amu_cfgvar.shorten_fnames = FALSE;
                           fseek(AMUcfgfile, 0, SEEK_SET);
                           fwrite(&amu_hdrvar, sizeof(amu_hdrvar), 1, AMUcfgfile);
                           fwrite(&amu_cfgvar, sizeof(amu_cfgvar), 1, AMUcfgfile);
                       }
                       else
                       {
                           textcolor(LIGHTRED);
                           cprintf("Unable to open '%s', configuration not updated", source);
                           textcolor(LIGHTGRAY);
                       }
                       break;

        /* This is the revision of 3.01, 3.10.B1, and 3.10.B2 */
        case 8:
                 level = REVISION;
                 fseek(AMUcfgfile, 0, SEEK_SET);
                 fwrite(&level, sizeof(level), 1, AMUcfgfile);

                 /* Convert the LISTS.AMU file */
                 sprintf(source, "%s", LISTS_AMU);
                 sprintf(target, "%s", "LISTS.$$$");
                 sourcefile = fopen(source, "rb");
                 if(sourcefile)
                 {
                     targetfile = fopen(target, "w+b");
                     fread(&rev8_allnewvar, sizeof(rev8_allnewvar), 1, sourcefile);
                     fread(&areasvar, sizeof(areasvar), 1, sourcefile);
                     while(!feof(sourcefile))
                     {
                         memset(&allnewvar, NULL, sizeof(allnewvar));
                         memcpy(&allnewvar, &rev8_allnewvar, sizeof(rev8_allnewvar));
                         strcpy(allnewvar.list_format, allnewvar.unused);
                         memset(&allnewvar.unused, NULL, sizeof(allnewvar.unused));
                         strcpy(allnewvar.list_format, DEFAULT_LISTSTR);
                         fwrite(&allnewvar, sizeof(allnewvar), 1, targetfile);
                         fwrite(&areasvar, sizeof(areasvar), 1, targetfile);
                         fread(&rev8_allnewvar, sizeof(rev8_allnewvar), 1, sourcefile);
                         fread(&areasvar, sizeof(areasvar), 1, sourcefile);
                     }
                     fclose(sourcefile);
                     fclose(targetfile);
                     Utility_obj.copyfile(source, "LISTS.BAK");
                     Utility_obj.copyfile(target, source);
                     remove(target);
                 }
                 else
                 {
                     textcolor(LIGHTRED);
                     cprintf("\r\n! Unable to open '%s'\r\n", source);
                     textcolor(LIGHTGRAY);
                 }

                 /* Next, convert the FILEAREA.AMU file */
                 sprintf(source, "%s", FILEAREA_AMU);
                 sprintf(target, "%s", "FILEAREA.$$$");
                 sourcefile = fopen(source, "rb");
                 if(sourcefile)
                 {
                     targetfile = fopen(target, "w+b");
                     fread(&rev8_fileareavar, sizeof(rev8_fileareavar), 1, sourcefile);
                     while(!feof(sourcefile))
                     {
                         memset(&fileareavar, NULL, sizeof(fileareavar));
                         memcpy(&fileareavar, &rev8_fileareavar, sizeof(rev8_fileareavar));
                         strcpy(fileareavar.filepath, fileareavar.unused1);
                         strcpy(fileareavar.listpath, fileareavar.unused2);
                         memset(&fileareavar.unused1, NULL, sizeof(fileareavar.unused1));
                         memset(&fileareavar.unused2, NULL, sizeof(fileareavar.unused2));
                         fwrite(&fileareavar, sizeof(fileareavar), 1, targetfile);
                         fread(&rev8_fileareavar, sizeof(rev8_fileareavar), 1, sourcefile);
                     }
                     fclose(sourcefile);
                     fclose(targetfile);
                     Utility_obj.copyfile(source, "FILEAREA.BAK");
                     Utility_obj.copyfile(target, source);
                     remove(target);
                 }
                 else
                 {
                     textcolor(LIGHTRED);
                     cprintf("\r\n! Unable to open '%s'\r\n", source);
                     textcolor(LIGHTGRAY);
                 }

                 /* Next, convert the BULLETIN.AMU file */
                 sprintf(source, "%s", BULLETIN_AMU);
                 sprintf(target, "%s", "BULLETIN.$$$");
                 sourcefile = fopen(source, "rb");
                 if(sourcefile)
                 {
                     targetfile = fopen(target, "w+b");
                     fread(&rev8_bulletinvar, sizeof(rev8_bulletinvar), 1, sourcefile);
                     fread(&areasvar, sizeof(areasvar), 1, sourcefile);
                     while(!feof(sourcefile))
                     {
                         memset(&bulletinvar, NULL, sizeof(bulletinvar));
                         memcpy(&bulletinvar, &rev8_bulletinvar, sizeof(rev8_bulletinvar));
                         fwrite(&bulletinvar, sizeof(bulletinvar), 1, targetfile);
                         fwrite(&areasvar, sizeof(areasvar),1 , targetfile);
                         fread(&rev8_bulletinvar, sizeof(rev8_bulletinvar), 1, sourcefile);
                         fread(&areasvar, sizeof(areasvar), 1, sourcefile);
                     }
                     fclose(sourcefile);
                     fclose(targetfile);
                     Utility_obj.copyfile(source, "BULLETIN.BAK");
                     Utility_obj.copyfile(target, source);
                     remove(target);
                 }
                 else
                 {
                     textcolor(LIGHTRED);
                     cprintf("\r\n! Unable to open '%s'\r\n", source);
                     textcolor(LIGHTGRAY);
                 }

                 /* Next, convert the MSGAREA.AMU file */
                 sprintf(source, "%s", MSGAREA_AMU);
                 sprintf(target, "%s", "MSGAREA.$$$");
                 sourcefile = fopen(source, "rb");
                 if(sourcefile)
                 {
                     targetfile = fopen(target, "w+b");
                     fread(&rev8_msgareavar, sizeof(rev8_msgareavar), 1, sourcefile);
                     while(!feof(sourcefile))
                     {
                         memset(&msgareavar, NULL, sizeof(msgareavar));
                         memcpy(&msgareavar, &rev8_msgareavar, sizeof(rev8_msgareavar));
                         strcpy(msgareavar.basepath, msgareavar.unused1);
                         memset(&msgareavar.unused1, NULL, sizeof(msgareavar.unused1));
                         fwrite(&msgareavar, sizeof(msgareavar), 1, targetfile);
                         fread(&rev8_msgareavar, sizeof(rev8_msgareavar), 1, sourcefile);
                     }
                     fclose(sourcefile);
                     fclose(targetfile);
                     Utility_obj.copyfile(source, "MSGAREA.BAK");
                     Utility_obj.copyfile(target, source);
                     remove(target);
                 }
                 else
                 {
                     textcolor(LIGHTRED);
                     cprintf("\r\n! Unable to open '%s'\r\n", source);
                     textcolor(LIGHTGRAY);
                 }
                 break;

        default: textcolor(LIGHTRED);
                 cprintf("Configuration too old to be updated.\n");
                 textcolor(LIGHTGRAY);
                 break;
    }
    close_config(CFG_RECORD);
#endif
}


/***********************************************************************/
/* Read the BBS file areas into AMU's internal format */
int Config::read_bbsfiles(void)
{
    int x, status;
    struct filearea tmpfileareavar;
    class BBSInfo *BBSInfo_ptr;

    BBSInfo_ptr = new BBSInfo();
    if(!BBSInfo_ptr->open_files())
    {
	delete(BBSInfo_ptr);
	return(FALSE);
    }

    x = 0;
    open_config(FILEAREA_RECORD);
    read_record(FILEAREA_RECORD, DEFAULT_RECORD);
    memcpy(&tmpfileareavar, &fileareavar, FILEAREA_SIZE);
    status = BBSInfo_ptr->read_files(&tmpfileareavar);
    while(status)
    {
        x++;
        memset(&fileareavar, NULL, FILEAREA_SIZE);
        if(x <= num_filearea)
        {
            read_record(FILEAREA_RECORD, x);
            strcpy(fileareavar.areaname, tmpfileareavar.areaname);
            strcpy(fileareavar.filepath, tmpfileareavar.filepath);
            strcpy(fileareavar.listpath, tmpfileareavar.listpath);
            fileareavar.security = tmpfileareavar.security;
            fileareavar.attrib.cdrom = tmpfileareavar.attrib.cdrom;
            fileareavar.attrib.password = tmpfileareavar.attrib.password;
            fileareavar.listtype = tmpfileareavar.listtype;
            fileareavar.areanum = tmpfileareavar.areanum;
            fileareavar.keycrc32 = tmpfileareavar.keycrc32;
            strcpy(fileareavar.lora3key, tmpfileareavar.lora3key);
            update_record(FILEAREA_RECORD, x);
        }
        else
        {
            memcpy(&fileareavar, &tmpfileareavar, FILEAREA_SIZE);
            add_record(FILEAREA_RECORD, COPY_CUR_RECORD);
        }
//      memset(&tmpfileareavar, NULL, FILEAREA_SIZE);
        status = BBSInfo_ptr->read_files(&tmpfileareavar);
    }

    if(x < num_filearea)
    {
//      x++;
        while(x <= num_filearea)
        {
            read_record(FILEAREA_RECORD, x);
            fileareavar.deleted = TRUE;
            update_record(FILEAREA_RECORD, x);
            x++;
        }
    }
    close_config(FILEAREA_RECORD);
    BBSInfo_ptr->close_files();

    amu_hdrvar.files_time = BBSInfo_ptr->get_ftime();
    amu_hdrvar.files_date = BBSInfo_ptr->get_fdate();
    delete(BBSInfo_ptr);
    return(TRUE);
}

/***********************************************************************/
/* Read the BBS msg areas into AMU's internal format */
int Config::read_bbsmsgs(void)
{
    int x, status;
    struct msgarea tmpmsgareavar;
    class BBSInfo *BBSInfo_ptr;

    BBSInfo_ptr = new BBSInfo();
    if(!BBSInfo_ptr->open_msgs())
    {
        delete(BBSInfo_ptr);
        return(FALSE);
    }

    x = 1;
    memset(&tmpmsgareavar, 0, MSGAREA_SIZE);
    status = BBSInfo_ptr->read_msgs(&tmpmsgareavar);
    open_config(MSGAREA_RECORD);
    while(status)
    {
        memset(&msgareavar, 0, MSGAREA_SIZE);
        if(x <= num_msgarea)
        {
            read_record(MSGAREA_RECORD, x);
            strcpy(msgareavar.areaname, tmpmsgareavar.areaname);
            strcpy(msgareavar.basepath, tmpmsgareavar.basepath);
            msgareavar.security = tmpmsgareavar.security;
            msgareavar.areatype = tmpmsgareavar.areatype;
            msgareavar.areanum = tmpmsgareavar.areanum;
            update_record(MSGAREA_RECORD, x);
        }
        else
        {
            memcpy(&msgareavar, &tmpmsgareavar, MSGAREA_SIZE);
            add_record(MSGAREA_RECORD, COPY_CUR_RECORD);
        }
        x++;
        memset(&tmpmsgareavar, 0, MSGAREA_SIZE);
        status = BBSInfo_ptr->read_msgs(&tmpmsgareavar);
    }

    if(x < num_msgarea)
    {
        while(x <= num_msgarea)
        {
            read_record(MSGAREA_RECORD, x);
            msgareavar.deleted = TRUE;
            update_record(MSGAREA_RECORD, x);
            x++;
        }
    }
    num_msgarea = x-1;
    BBSInfo_ptr->close_msgs();

    amu_hdrvar.msgs_time = BBSInfo_ptr->get_mtime();
    amu_hdrvar.msgs_date = BBSInfo_ptr->get_mdate();
    close_config(MSGAREA_RECORD);
    delete(BBSInfo_ptr);
    return(TRUE);
}

/************************************************************************/
/* EOF - CONFIG.CPP */

