/*  Database program for the AMU Beta team */

#include <stdio.h>
#include <dos.h>
#include <conio.h>
#include <ctype.h>
#include <io.h>
#include <string.h>
#include <share.h>
#include <stdlib.h>
#include <dir.h>
#include <mem.h>
#include "global.h"

#define TRUE 1
#define FALSE 0

/* Define values for VALID_TYPE
   When VALID_TYPE is set to SPACES_OK, the getfield() function will
   allow spaces and is case sensitive - the valid() function will ignore the
   input as well
*/
#define PATH 0
#define FILENM 1
#define SPACES_OK 2
#define NO_EXT 3
/*RANGE_1 - 1-200.  RANGE_2  - 1-65535.   RANGE_3 - 1-32766 */
#define RANGE_1 4
#define RANGE_2 5
#define RANGE_3 6

#define BACK 8
#define ESC 27
#define ENTER 13
#define UP 72
#define DOWN 80
#define LEFT 75
#define RIGHT 77
#define HOME 71
#define END 79
#define DEL 83
#define INS 82
#define F1 59
#define F2 60
#define F3 61
#define F4 62
#define F5 63
#define F6 64
#define F7 65
#define HATCH ""

void main_menu(void);
void error(char *instring);
void help(char *str);
int prompt(int mode);
int listbox_2(struct coordinates *cvar);
void make2menu(char *options, char *title, int left, int top, int width, int items, int highlight, int num_high);
void makemenu(char *options, char *title, int left, int top, int width, int items, int highlight, int num_high);
char *getfield(char *infield, int left, int top, int length);
char *make_db_str(void);
int db_sub_menu(void);
void db_list_box(void);
void color(int back, int fore);
int _strlength(char *str);
void toggle(short *i);
void general_help(int i);
char *squeeze(char *instr);
int valid(char *inpath);
void slash(char *str);
void noslash(char *str);
void trimlead(char *instring);
void trimend(char *instring);
void center(char *instring);
void lpad(char *str, int length);
void pad(char *str, int length);

int VALID_TYPE;
char buffer[4000];
char curdir[81];
char DBNAME[13];

struct coordinates
{
    int top,
	left,
	width,
	items,
	high_num,
	uses_blanks,
	position;       // Used to return the position to the calling fn
    char *title,
	 *options;
};

struct
{
    char name[41],
	 fido[41],
	 inet[41];
    short include_inet;
    char bbs[41],
	 os[41],
	 amu_platform[41],
	 link[41];
    short deleted;
    char freespace[200];
}dbvar;

/***************************************************************************/
int main(int argc, char *argv[])
{
    char tmp[81];
    FILE *tmpfile, *infile;

    if(argc == 1)
    {
	strcpy(DBNAME, "BETADB.DAT");
    }
    else
    {
	strcpy(DBNAME, argv[1]);
    }

    _setcursortype(_NOCURSOR);
    getcurdir(0, tmp);
    sprintf(curdir, "%c:\\%s", getdisk()+'A', tmp);
    main_menu();
    db_list_box();
    _setcursortype(_NORMALCURSOR);

    /* Get rid of the deleted records */
    clrscr();
    printf("Removing deleted records... ");
    infile = _fsopen(DBNAME, "r+b", SH_DENYNO);
    tmpfile = _fsopen("DB.$$$", "wb+", SH_DENYNO);
    fseek(infile, 0, SEEK_SET);
    fread(&dbvar, sizeof(dbvar), 1, infile);
    while(!feof(infile))
    {
	if(!dbvar.deleted)
	{
	    fwrite(&dbvar, sizeof(dbvar), 1, tmpfile);
	}
	fread(&dbvar, sizeof(dbvar), 1, infile);
    }
    fclose(infile);
    fclose(tmpfile);
    remove(DBNAME);
    rename("DB.$$$", DBNAME);

    clrscr();
    gotoxy(1, 10);
    strcpy(tmp, "Beta Team Database");
    center(tmp);
    printf("%s\n", tmp);
    strcpy(tmp, "Copyright (C) 1997 by David Hickey");
    center(tmp);
    printf("%s\n", tmp);
    strcpy(tmp, "All Rights Reserved");
    center(tmp);
    printf("%s\n", tmp);
    printf("%s\n\n", tmp);
    strcpy(tmp, "Original look 'n' feel concept designed by Joaquim H. Homrighausen");
    center(tmp);
    printf("%s\n", tmp);
    strcpy(tmp, "Used with permission");
    center(tmp);
    printf("%s\n\n\n\n\n", tmp);
    return(0);
}

/* Append a backslash to the end of the string */
void slash(char *str)
{
    int i = 0;

    while(str[i])
      i++;
    if(str[i-1]  != '\\')
    {
       str[i] = '\\';
       str[i+1] = NULL;
    }
}

int _strlength(char *str)
{
    int i = 0;
    char *ptr;

    ptr = str;
    while(*ptr)
    {
	if(*ptr != '')
	{
	    i++;
	}
	ptr++;
    }
    return(i);
}

/* Strip trailing backslash from the input line */
void noslash(char *str)
{
    int i = 0;

    while(str[i])
      i++;
    if(str[i-1]  == '\\')
    {
       str[i-1] = NULL;
    }
}

/* Determine if the input string is valid */
int valid(char *inpath)
{
    int value;
    long i;
    char *ptr;

    switch(VALID_TYPE)
    {
        case PATH: noslash(inpath);
                   i = chdir(inpath);
                   chdir(curdir);
                   if(i == 0)
		   {
		       value = TRUE;
                       slash(inpath);
		   }
		   else
                   {
                       slash(inpath);
                       if(inpath[0] == NULL)
                       {
                           help("Can not be blank");
                       }
                       else
                       {
                           help("Must be a valid DOS path");
                       }
                       value = FALSE;
                   }
                   break;

        case FILENM: noslash(inpath);
		     value = TRUE;
		     break;

	case SPACES_OK: value = TRUE;
                        break;

	case NO_EXT: ptr = (char *) strchr(inpath, '.');
                     if(ptr)
                     {
                         *ptr = NULL;
                     }
                     value = TRUE;
                     break;
        case RANGE_1: i = atol(inpath);
                      if(i >= 1  &&  i <= 200)
                      {
                          value = TRUE;
                      }
                      else
                      {
			  value = FALSE;
			  help("Must be between 1 and 200");
                      }
                      break;
        case RANGE_2: i = atol(inpath);
                      if(i >= 1  &&  i <= 65535)
		      {
                          value = TRUE;
		      }
                      else
                      {
                          value = FALSE;
                          help("Must be between 1 and 65535");
                      }
                      break;
        case RANGE_3: i = atol(inpath);
                      if(i >= 1  &&  i <= 32766)
                      {
                          value = TRUE;
                      }
		      else
		      {
                          value = FALSE;
                          help("Must be between 1 and 32766");
                      }
                      break;
    }
    return(value);
}

char *squeeze(char *instr)
{
    static char tmpstr[71];
    int i = 0;

    /* Remove the first  you find, append one space to the end, then return the string */
    while(instr[i] != '')
    {
	tmpstr[i] = instr[i];
        i++;
    }
    i++;

    while(instr[i])
    {
	 tmpstr[i-1] = instr[i];
         i++;
    }
    tmpstr[i-1] = '';
    tmpstr[i] = NULL;
    return(tmpstr);
}

void color(int back, int fore)
{
    textcolor(fore);
    textbackground(back);
}

void main_menu(void)
{
    int i;
    char ar[81], logo[81];

    gotoxy(1, 1);
    for(i = 0; i < 80; i++)
      ar[i] = '';
    ar[81] = NULL;
    color(BLACK, LIGHTGRAY);
    for(i = 0; i < 25; i++)
    {
        gotoxy(1, i+1);
        cprintf("%s", ar);
    }
    gotoxy(1,1);
    color(BLACK, CYAN);
    sprintf(logo, "Betateam Database v1.00");
    center(logo);
    cprintf(" %s ", logo);
    gotoxy(1, 2);
    color(BLACK, BLUE);
    for(i = 0; i < 80; i++)
    {
	cprintf("");
    }

    gotoxy(1, 24);
    color(BLACK, BLUE);
    for(i = 0; i < 80; i++)
    {
	cprintf("");
    }
    help("");
}

void help(char *str)
{
    color(BLACK, LIGHTGRAY);
    gotoxy(1, 25);
    delline();
    cprintf("%s", str);
}

int prompt(int mode)
{
    int i, width, max_options, left, top, answer;
    char str[51], title[11], *tmpbuf;

    /* MODE
       0  - Save?
       1  - Delete?
       2  - config not found, continue?
    */

    answer = -1;
    max_options = 3;
    top = 10;

    title[0] = NULL;
    switch(mode)
    {
        case 0: strcpy(str, "  ;Save (Y/n)?  ;  ;");
                help("Save changes?");
                left = 32;
                width = 15;
                break;
        case 1: strcpy(str, "  ;Delete (y/N)?  ;  ;");
                help("Confirm deletion");
		left = 32;
                width = 17;
                break;
        case 2: sprintf(str, "  ;%s not found;Create (Y/n)?  ;  ;", DBNAME);
                width = 22;
                left = 27;
                max_options = 4;
                break;
    }


    tmpbuf = (char *) malloc(4000);
    gettext(1, 1, 80, 25, tmpbuf);
    makemenu(str, title, left, top, width, max_options, 0, 0);

    do
    {
        i = getch();
        if(i == 0)
        {
            i = getch();
        }
        else
        {
            switch(toupper(i))
            {
                case ENTER: if(mode == 0  ||  mode == 2)
                            {
                                answer = TRUE;
                            }
			    else
                            {
				answer = FALSE;
                            }
			    break;
                case 'Y': answer = TRUE;
                          break;
                case 'N': answer = FALSE;
                          break;
            }
        }
    } while(answer < 0);

    puttext(1, 1, 80, 25, tmpbuf);
    free(tmpbuf);
    return(answer);
}

void make2menu(char *options, char *title, int left, int top, int width, int items, int highlight, int num_high)
{
    int i, j, length, column;
    char *ptr, *ptr2, *str, tmpstr[81];

    str = (char *) malloc(1200);

    /* If the number of options is odd, make it even */
    if(items % 2)
    {
        items++;
        strcat(options, " ;");
    }
    length = items / 2;
    width *= 2;                           // The true width is double

    ptr = options;
    gotoxy(left, top);
    color(BLACK, LIGHTGRAY);
    cprintf("");
    if(title)
    {
        color(LIGHTBLUE, YELLOW);
        cprintf("%s", title);
	color(BLACK, LIGHTGRAY);
        i = width - strlen(title) + 1;
    }
    else
    {
        i = width + 1;
    }

    setmem(tmpstr, i, '');
    tmpstr[i] = NULL;
    cprintf("%s", tmpstr);

    for(i = 1; i <= items; i++)
    {
        color(BLACK, LIGHTGRAY);
        /* See if we are at the halfway mark or not */
	if(i <= length)
        {
	    column = 1;
            gotoxy(left, top + i);
	}
        else
        {
	    column = 2;
            gotoxy(left + (width/2) + 1, top + i - length);
        }

        ptr2 = str;
	while(*ptr != ';'  &&  *ptr)
        {
            *ptr2 = *ptr;
            ptr++;
            ptr2++;
        }
        *ptr2 = NULL;
        ptr++;

        while(strlen(str) < (width - 2) / 2)
        {
            strcat(str, " ");
	}

        if(column == 1)
        {
	    cprintf(" ");
        }
        else
        {
	    cprintf("  ");
        }

	if(i == highlight)
	{
	    color(LIGHTGRAY, BLUE);

	    /* Some spots need to have only part of the line highlighted,
	       and num_high will be non-zero (true) if this is the case
	    */
	    ptr2 = str;
	    if(num_high)
	    {
		for(j = 0; j < num_high; j++)
                {
                    tmpstr[j] = *ptr2;
                    ptr2++;
		}
                tmpstr[j] = NULL;
                cprintf("%s", tmpstr);
                color(BLACK, CYAN);
            }
            cprintf("%s", ptr2);
        }
	else
        {
            color(BLACK, CYAN);
            cprintf("%s", str);
        }
        if(column == 2)
	{
            color(BLACK, LIGHTGRAY);
	    cprintf(" ");

            /* Add the shadow */
            color(BLACK, DARKGRAY);
            cprintf("%s", HATCH);
        }
    }


    color(BLACK, LIGHTGRAY);
    gotoxy(left, top + length + 1);
    for(i = 0; i <= width+1; i++)
    {
        tmpstr[i] = '';
    }
    tmpstr[i] = NULL;
    cprintf("%s", tmpstr);


    /* Add the shadow */
    color(BLACK, DARKGRAY);
    cprintf("%s", HATCH);
    gotoxy(left + 1, top + length + 2);
    for(i = 0; i <= width + 3; i++)
    {
        tmpstr[i] = '';
    }
    tmpstr[i] = NULL;
    cprintf("%s", tmpstr);
    color(BLACK, LIGHTGRAY);
    free(str);
}


void makemenu(char *options, char *title, int left, int top, int width, int items, int highlight, int num_high)
{
    int i, j;
    char *ptr, *ptr2, *str, tmpstr[81];

    str = (char *) malloc(1000);
    ptr = options;
    gotoxy(left, top);
    color(BLACK, LIGHTGRAY);
    cprintf("");
    color(LIGHTBLUE, YELLOW);
    cprintf("%s", title);
    color(BLACK, LIGHTGRAY);

    i = width - strlen(title) - 1;
    setmem(tmpstr, i, '');
    tmpstr[i] = NULL;
    cprintf("%s", tmpstr);

    for(i = 1; i <= items; i++)
    {
	color(BLACK, LIGHTGRAY);
	gotoxy(left, top + i);
	j = 0;

	while(*ptr != ';'  &&  *ptr)
	{
	    str[j] = *ptr;
	    ptr++;
	    j++;
	}
	str[j] = NULL;
	ptr++;

	while(strlen(str) < width - 2)
	{
	    strcat(str, " ");
	}
	cprintf(" ");
	if(i == highlight)
	{
	    color(LIGHTGRAY, BLUE);
	    /* Some spots need to have only part of the line highlighted,
	       and num_high will be non-zero (true) if this is the case
	    */
	    ptr2 = str;
	    if(num_high)
	    {
		for(j = 0; j < num_high; j++)
		{
		    tmpstr[j] = *ptr2;
		    ptr2++;
		}
		tmpstr[j] = NULL;
		cprintf("%s", tmpstr);
		color(BLACK, CYAN);
	    }
	    cprintf("%s", ptr2);
	}
	else
	{
	    color(BLACK, CYAN);
	    cprintf("%s", str);
	}
	color(BLACK, LIGHTGRAY);
	cprintf(" ");

	/* Add the shadow */
	color(BLACK, DARKGRAY);
	cprintf("%s", HATCH);
    }


    color(BLACK, LIGHTGRAY);
    gotoxy(left, top + i);
    for(i = 0; i < width; i++)
    {
	tmpstr[i] = '';
    }
    tmpstr[i] = NULL;
    cprintf("%s", tmpstr);


    /* Add the shadow */
    color(BLACK, DARKGRAY);
    cprintf("%s", HATCH);
    gotoxy(left + 1, top + items + 2);
    for(i = 0; i < width + 2; i++)
    {
	tmpstr[i] = '';
    }
    tmpstr[i] = NULL;
    cprintf("%s", tmpstr);
    color(BLACK, LIGHTGRAY);
    free(str);
}

char *getfield(char *infield, int left, int top, int length)
{
    static char field[101], infieldtmp[101];
    static short insert = TRUE;
    unsigned i, k, index, ok;


    /* The input INFIELD parameter must NOT be modified, doing so may result
       in other data being overwritten in memory */


    /* If this looks like it may be a path, reduce the length by one so there
       is enough room for a backslash
    */
    if(length > 25)
    {
	length--;
    }
    ok = FALSE;
    strcpy(infieldtmp, infield);
    gotoxy(left, top);
    color(LIGHTGREEN, BLUE);
    _setcursortype(_NORMALCURSOR);

    while(strlen(infieldtmp) < length)
    {
	strcat(infieldtmp, HATCH);
    }
    cprintf("%s", infieldtmp);


    /* Make a copy of the original string, which we'll use to work on.
       If the user chooses to abort, the original INFIELD will remain intact
       to be restored
    */
    strcpy(field, infieldtmp);

    do
    {

    /* Index is used as an index into the actual string */
	index = 0;
	gotoxy(left, top);
	color(LIGHTGREEN, BLUE);
	i = getch();
    while(i != ESC && i != ENTER)
    {

	 /* An arrow key or something was pressed, handle it */
	 if(i == 0)
	 {
	     i = getch();
	     switch(i)
	     {
		 /* Decrement the index and cursor, IF it's not at position 0 */
		 case LEFT: if(index > 0)
			    {
				gotoxy(wherex() - 1, wherey());
                                index--;
			    }
			    break;

                 /* Increment the index and cursor, IF it's not at the end */
		 case RIGHT: if(index < (length - 1))
			     {
				 gotoxy(wherex() + 1, wherey());
				 index++;
                             }
                             break;

                 /* Go to the end of the line */
		 case END: index = strcspn(field, HATCH) - 1;
			   gotoxy(left + index, wherey());
                           break;

		 /* Go to the beginning of the line */
                 case HOME: index = 0;
			    gotoxy(left, wherey());
			    break;

		 /* If DEL is hit, we need to "squeeze" the string and redisplay it */
		 case DEL: if(length > 1)
			   {
			       field[index] = '';
			       strcpy(field, squeeze(field));
			       gotoxy(left, wherey());
			       cprintf("%s", field);
			       gotoxy(index + left, wherey());
			   }
			   break;
		 case INS: toggle(&insert);
			   if(insert)
			   {
			       _setcursortype(_NORMALCURSOR);
			   }
			   else
			   {
			       _setcursortype(_SOLIDCURSOR);
			   }
			   break;
	     }
	 }
	 else
	 {
	     if(index <= (length)  &&  isprint(i))
	     {

		 if(index == length)
		 {
		     index--;
		 }

		 /* Allow capitalized input with NO spaces */
		 if(!isspace(i)  &&  VALID_TYPE != SPACES_OK)
		 {
		     i = toupper(i);
		     if(insert)
		     {
			 if(_strlength(field) < length)
			 {
			     field[strcspn(field, HATCH)] = NULL;
			     for(k = strlen(field); k > index; k--)
			     {
				 field[k] = field[k-1];
			     }
			     field[index] = i;
			     gotoxy(left, wherey());
			     cprintf("%s", field);
			     gotoxy(left+index+1, wherey());
			     index++;
			 }
		     }
		     else
		     {
			 cprintf("%c", i);
			 field[index] = i;
			 index++;
		     }
		 }

		 /* Allow case sensitive input, spaces ok */
		 if(VALID_TYPE == SPACES_OK)
		 {
		     if(insert)
		     {
			 if(_strlength(field) < length)
			 {
			     field[strcspn(field, HATCH)] = NULL;
			     for(k = strlen(field); k > index; k--)
			     {
				 field[k] = field[k-1];
			     }
			     field[index] = i;
			     gotoxy(left, wherey());
			     cprintf("%s", field);
			     gotoxy(left+index+1, wherey());
			     index++;
			 }
		     }
		     else
		     {
			 cprintf("%c", i);
			 field[index] = i;
			 index++;
		     }
		 }

		 if(index == length)
		 {
		     gotoxy(wherex() - 1, wherey());
		 }

	     }
	     if(i == BACK && index > 0)
	     {
		 index--;
		 field[index] = '';
		 strcpy(field, squeeze(field));
		 gotoxy(left, wherey());
                 cprintf("%s", field);
                 gotoxy(index + left, wherey());
             }
         }
         i = getch();
    }

    field[strcspn(field, HATCH)] = NULL;

    _setcursortype(_NOCURSOR);
    ok = valid(field);
    if(!ok  &&  VALID_TYPE == PATH)
    {
        /* If this is a path, prompt the user to create it */
        if(prompt(3))
        {
	    noslash(field);
            if(mkdir(field) != 0)
	    {
                error("Could not create path");
            }
            else
            {
                slash(field);
            }
        }
        ok = TRUE;
    }
    _setcursortype(_NORMALCURSOR);


    }while(!ok);

    _setcursortype(_NOCURSOR);

    if(i == ESC)
    {
	strcpy(field, infield);
    }
    return(field);
}

int listbox_2(struct coordinates *cvar)
{
    int position, i, amax_options;

    position = cvar->position;
    amax_options = cvar->items;
    if(amax_options % 2)
    {
        amax_options++;
    }

    make2menu(cvar->options, cvar->title, cvar->left, cvar->top, cvar->width, cvar->items, position, cvar->high_num);
    do
    {
	i = getch();
	if(i == 0)
	{
	    i = getch();
	    switch(i)
	    {
                case UP: if(position == 1)
                         {
                             position = cvar->items;
                         }
			 else
                         {
                             position--;
                         }
                         break;

                case DOWN: if(position == cvar->items)
                           {
			       position = 1;
			   }
			   else
                           {
                               position++;
                           }
                           break;
                case RIGHT: if(position <= amax_options / 2)
                            {
                                position += amax_options / 2;
                            }
                            else
                            {
                                position -= amax_options / 2;
                            }
                            break;

                case LEFT: if(position <= amax_options / 2)
                           {
                               position += amax_options / 2;
			   }
                           else
			   {
			       position -= amax_options / 2;
			   }
			   break;
		case F1: cvar->position = position;
			 return(i);
	    }
	    make2menu(cvar->options, cvar->title, cvar->left, cvar->top, cvar->width, cvar->items, position, cvar->high_num);
	}
	else
        {
            if(i == ENTER)
            {
                cvar->position = position;
                return(i);
            }
        }
    } while(i != ESC);
    return(i);
}

char *make_db_str(void)
{
    static char str[750];

    strcpy(str, "  ;");

    strcat(str, "Name            : ");
    strcat(str, dbvar.name);
    strcat(str, ";");

    strcat(str, "FidoNet Address : ");
    strcat(str, dbvar.fido);
    strcat(str, ";");

    strcat(str, "InterNet Address: ");
    strcat(str, dbvar.inet);
    strcat(str, ";");

    strcat(str, "Include Inet?   : ");
    switch(dbvar.include_inet)
    {
	case 0: strcat(str, "No;");
                break;
	case 1: strcat(str, "Yes;");
		break;
    }

    strcat(str, "BBS Software    : ");
    strcat(str, dbvar.bbs);
    strcat(str, ";");

    strcat(str, "Operating System: ");
    strcat(str, dbvar.os);
    strcat(str, ";");

    strcat(str, "AMU Platform    : ");
    strcat(str, dbvar.amu_platform);
    strcat(str, ";");

    strcat(str, "Link            : ");
    strcat(str, dbvar.link);
    strcat(str, ";");

    strcat(str, "  ;");
    return(str);
}

void toggle(short *i)
{
    if(*i)
    {
        *i = FALSE;
    }
    else
    {
        *i = TRUE;
    }
}

int db_sub_menu()
{
    int i, width, max_options, left, top, high_num, offset, position;
    char title[60];

    position = 2;
    max_options = 10;
    width = 60;
    left = 10;
    top = 6;
    high_num = 15;
    offset = 20;

    strcpy(title, "Tester Records");

    makemenu(make_db_str(), title, left, top, width, max_options, position, high_num);

    general_help(position + 9);
    do
    {
	i = getch();
	if(i == 0)
	{
	    i = getch();
	    switch(i)
	    {
		case UP: if(position == 2)
			 {
			     position = max_options - 1;
			 }
			 else
			 {
			     position--;
			 }
			 break;

		case DOWN: if(position == max_options - 1)
                           {
                               position = 2;
                           }
                           else
			   {
                               position++;
                           }
                           break;
            }
	    general_help(position + 9);
	    makemenu(make_db_str(), title, left, top, width, max_options, position, high_num);
	}
	else
	{
	    if(i == ENTER)
	    {
		switch(position)
		{
		    case 2: VALID_TYPE = SPACES_OK;
			    strcpy(dbvar.name,
			    getfield(dbvar.name, left + offset, top + 2, 40));
			    break;
		    case 3: VALID_TYPE = SPACES_OK;
			    strcpy(dbvar.fido,
			    getfield(dbvar.fido, left + offset, top + 3, 40));
			    break;
		    case 4: VALID_TYPE = SPACES_OK;
			    strcpy(dbvar.inet,
			    getfield(dbvar.inet, left + offset, top + 4, 40));
			    break;
		    case 5: toggle(&dbvar.include_inet);
			    break;
		    case 6: VALID_TYPE = SPACES_OK;
			    strcpy(dbvar.bbs,
			    getfield(dbvar.bbs, left + offset, top + 6, 40));
			    break;
		    case 7: VALID_TYPE = SPACES_OK;
			    strcpy(dbvar.os,
			    getfield(dbvar.os, left + offset, top + 7, 40));
			    break;
		    case 8: VALID_TYPE = SPACES_OK;
			    strcpy(dbvar.amu_platform,
			    getfield(dbvar.amu_platform, left + offset, top + 8, 40));
			    break;
		    case 9: VALID_TYPE = SPACES_OK;
			    strcpy(dbvar.link,
			    getfield(dbvar.link, left + offset, top + 9, 40));
			    break;
		}
		general_help(position + 9);
		makemenu(make_db_str(), title, left, top, width, max_options, position, high_num);
	    }
	}
    } while(i != ESC);
    return(prompt(0));
}

void db_list_box(void)
{
    long i, width, max_options, left, top, index, num_recs, found;
    char title[60], *tmpbuf, tmpstr[41], tmpname[41], addr1[90], addr2[90], addr3[90];
    int counter, j;
    FILE *infile, *outfile;

    max_options = 10;
    width = 60;
    left = 10;
    top = 6;
    index = 0;

    /* Initialize a blank DB structure */
    memset(&dbvar, NULL, sizeof(dbvar));

    infile = _fsopen(DBNAME, "r+b", SH_DENYNO);
    if(!infile)
    {
	infile = _fsopen(DBNAME, "w+b", SH_DENYNO);
    }
    else
    {
	fread(&dbvar, sizeof(dbvar), 1, infile);
    }

    strcpy(title, "Record 1/");
    i = fileno(infile);
    num_recs = filelength(i) / sizeof(dbvar);
    if(num_recs == 0)
    {
	/* No records means this is all new, insert a blank record */
	num_recs = 1;
	fwrite(&dbvar, sizeof(dbvar), 1, infile);
    }

    itoa(num_recs, tmpstr, 10);
    strcat(title, tmpstr);
    if(dbvar.deleted)
    {
	strcat(title, " DELETED");
    }

    makemenu(make_db_str(), title, left, top, width, max_options, 0, 0);

    general_help(1);
    do
    {
	i = getch();
        if(i == 0)
	{
	    i = getch();
	    switch(i)
	    {

		/* F1 - Find an account number */
		case F1: rewind(infile);
			 outfile = fopen("BETATEAM.LST", "w+t");
			 while(fread(&dbvar, sizeof(dbvar), 1, infile))
			 {
			     fprintf(outfile, "Name     : %s\n", dbvar.name);
			     fprintf(outfile, "FidoNet  : %s\n", dbvar.fido);
			     fprintf(outfile, "InterNet : %s\n", dbvar.inet);
			     fprintf(outfile, "BBS      : %s\n", dbvar.bbs);
			     fprintf(outfile, "OS       : %s\n", dbvar.os);
			     fprintf(outfile, "Platform : %s\n", dbvar.amu_platform);
			     fprintf(outfile, "Link     : %s\n", dbvar.link);
			     fprintf(outfile, "----------------------\n\n\n");
			 }
			 fclose(outfile);
			 rewind(infile);
			 break;

		case HOME: index = 0;
			   break;
		case END: index = num_recs - 1;
			  break;
		case INS: num_recs++;
			  index = num_recs - 1;

			  /* Initialize a blank db structure */
			  memset(&dbvar, NULL, sizeof(dbvar));
			  fseek(infile, index * sizeof(dbvar), SEEK_SET);
			  fwrite(&dbvar, sizeof(dbvar), 1, infile);
			  break;

		case DEL: toggle(&dbvar.deleted);
			  fseek(infile, index * sizeof(dbvar), SEEK_SET);
			  fwrite(&dbvar, sizeof(dbvar), 1, infile);
			  break;

		case RIGHT: if(index == num_recs - 1)
			    {
				index = 0;
                            }
                            else
                            {
				index++;
                            }
                            break;

                case LEFT: if(index == 0)
                           {
			       index = num_recs - 1;
                           }
			   else
			   {
                               index--;
                           }
			   break;
            }
	    fseek(infile, index * sizeof(dbvar), SEEK_SET);
            fread(&dbvar, sizeof(dbvar), 1, infile);
	    i = fileno(infile);
            num_recs = filelength(i) / sizeof(dbvar);

            /* Remake the TITLE variable since info may have changed */
	    strcpy(title, "Record ");
	    strcat(title, itoa(index + 1, tmpstr, 10));
            strcat(title, "/");
	    strcat(title, itoa(num_recs, tmpstr, 10));
	    if(dbvar.deleted)
	    {
		strcat(title, " DELETED");
	    }

	    makemenu(make_db_str(), title, left, top, width, max_options, 0, 0);
	    general_help(1);
	}
	else
	{
            if(i == ENTER)
            {
                tmpbuf = (char *) malloc(4000);
                gettext(1, 1, 80, 25, tmpbuf);
		fseek(infile, index * sizeof(dbvar), SEEK_SET);
		if(db_sub_menu())
		{
		    fseek(infile, index * sizeof(dbvar), SEEK_SET);
		    fwrite(&dbvar, sizeof(dbvar), 1, infile);
		}
		else
		{
		    fread(&dbvar, sizeof(dbvar), 1, infile);
		}
                puttext(1, 1, 80, 25, tmpbuf);
                free(tmpbuf);
                makemenu(make_db_str(), title, left, top, width, max_options, 0, 0);
                general_help(1);
            }
        }
    } while(i != ESC);
    fclose(infile);
}

void general_help(int i)
{
    switch(i)
    {
	case 1: help("F1 - Raw Listing");
		break;
	case 11: help("Name");
		 break;
	case 12: help("FidoNet Address");
		 break;
	case 13: help("InterNet Address");
		 break;
	case 14: help("Include in BETATEAM list?");
		 break;
	case 15: help("BBS Software");
		 break;
	case 16: help("Operating System");
		 break;
	case 17: help("AMU Platform");
		 break;
	case 18: help("Link");
		 break;
	}
}

void error(char *instring)
{
    int i, width, max_options, left, top;
    char str[21], title[11], *tmpbuf;

    max_options = 1;
    left = 33;
    top = 13;
    width = 7;
    title[0] = NULL;
    strcpy(str, "Error;");

    tmpbuf = (char *) malloc(4000);
    gettext(1, 1, 80, 25, tmpbuf);
    makemenu(str, title, left, top, width, max_options, 0, 0);
    help(instring);

    i = getch();
    if(i == 0)
    {
        i = getch();
    }
    puttext(1, 1, 80, 25, tmpbuf);
    free(tmpbuf);
}

void trimlead(char *instring)
{
    int i;

    strrev(instring);
    i = strlen(instring) - 1;
    while(instring[i] == ' ')
    {
        instring[i] = NULL;
        i--;
    }
    strrev(instring);
}

void trimend(char *instring)
{
    int i;

    i = strlen(instring) - 1;
    while(instring[i] == ' ')
    {
	instring[i] = NULL;
	i--;
    }
}

void center(char *instring)
{
    int i;

    trimend(instring);
    trimlead(instring);
    i = strlen(instring);
    i = (78 - i) / 2;
    lpad(instring, i + strlen(instring));
    pad(instring, 78);
}


/* Pad a string (on the right) with spaces to the specified length */
void pad(char *str, int length)
{
    while(strlen(str) < length)
    {
        strcat(str, " ");
    }
    str[length] = NULL;
}

/* Pad a string (on the left) with spaces to the specified length */
void lpad(char *str, int length)
{
    strrev(str);
    while(strlen(str) < length)
    {
        strcat(str, " ");
    }
    str[length] = NULL;
    strrev(str);
}

/* EOF - BETADB.CPP */
