/**************************************************************************/
/*                   Automated Maintenance Utility (AMU), Y2K             */
/*                                                                        */
/*                                                                        */
/*                            Version 3.16/Beta                           */
/*                      Last revised 23 May 1998                          */
/*                                                                        */
/*                 Copyright (C) 1993-1998 by David L. Hickey             */
/*                            All Rights Reserved                         */
/*                                                                        */
/*               Current platforms supported: DOS, DPMI, OS/2, WinNT      */
/*                                                                        */
/**************************************************************************/

#define SEMAPHORE_CTL "SEMAPHOR.CTL"

#include <stdio.h>
#include <conio.h>
#include <string.h>
#include <stdlib.h>
#include <dos.h>
#include <mem.h>
#include <ctype.h>
#include <malloc.h>
#include <sys\stat.h>
#include <io.h>
#include "extern.h"
#include "config.h"
#include "execute.h"
#include "screen.h"
#include "fileopts.h"
#include "logs.h"
#include "utility.h"
#include "bbsinfo.h"
#include "amustr.h"
#include "archives.h"
#include "users.h"
#include "bulletin.h"
#include "maint.h"
#include "switch.h"
#if defined(__AMU32__)
 #include <graph.h>
#endif

#if defined(__AMU32__) || defined(__AMU95__)

#define BLACK        0
#define BLUE         1
#define GREEN        2
#define CYAN         3
#define RED          4
#define MAGENTA      5
#define BROWN        6
#define LIGHTGRAY    7
#define DARKGRAY     8
#define LIGHTBLUE    9
#define LIGHTGREEN   10
#define LIGHTCYAN    11
#define LIGHTRED     12
#define LIGHTMAGENTA 13
#define YELLOW       14
#define WHITE        15
#define BLINK        128
#endif


/* Global variables */
int use_mono = FALSE;
class Config Config_obj;
class Screen Screen_obj;

    class Archives Archives_obj;
    class Bulletin Bulletin_obj;
    class Maint Maint_obj;
    class Switch Switch_obj;
    class FileOptions FileOptions_obj;
    class Logs Logs_obj;
    class Users Users_obj;
    class BBSInfo BBSInfo_obj;

#define NUM_PARAMETERS 46
#define ARG_LENGTH 25
char Parameters[NUM_PARAMETERS][ARG_LENGTH];

static void display_help(void);
static void write_hlp(char *param, char *desc, int x);
static void create_semaphores(void);
static void remove_semaphores(void);
static void shutdown(long start);
static int validate_parameters(int argc, char *argv[]);

extern unsigned int _stklen = 1024 * 16;
int kill_counterdata;

/************************************************************************/

int main(int argc, char *argv[])
{
    class Amustr Amustr_obj;
    class Utility Utility_obj;
    class Dates Date_Now;
    char str[101], fn[MAX_FILESPEC], tmp_arg[ARG_LENGTH];
    int x, y, errorlevel, cfg_changed, ok, file_opts_present;
    unsigned long start;
    struct screeninfo s;
#if defined(__AMU32__)
    struct videoconfig vc;
    use_mono = FALSE;
    if(vc.monitor == _MONO || vc.monitor == _ANALOGMONO)
    {
	use_mono = TRUE;
    }
#endif
#if defined(__AMU16__) || defined (__AMU2__)
    struct text_info ti;
    use_mono = FALSE;
    gettextinfo(&ti);
    if(ti.currmode == MONO)
    {
	use_mono = TRUE;
    }
#endif


    if(argc > 1 && strstr(argv[1], "?"))
    {
	display_help();
	return(NO_ERRORS);
    }

    Screen_obj.clearscreen();

    /* Check for the AMU and COMSPEC variables
    if(getenv("COMSPEC") == NULL)
    {
	Screen_obj.writestr(1, 5, LIGHTRED, "COMSPEC environment variable not set.  This must be set, example:");
	Screen_obj.writestr(1, 6, LIGHTRED, "SET COMSPEC=C:\\COMMAND.COM");
	return(NO_ENV);
    }
    if(getenv("AMU") == NULL)
    {
	Screen_obj.writestr(1, 5, LIGHTRED, "AMU environment variable not set.  This must be set, example:");
	Screen_obj.writestr(1, 6, LIGHTRED, "SET AMU=C:\\AMU");
	return(NO_ENV);
    }
*/

    start = Date_Now.getSecondsSince1970();

    /* Check for FILE_ID.DIZ/DESC.SDI and remove them */
    if(access("FILE_ID.DIZ", 0) == 0)
    {
	chmod("FILE_ID.DIZ", S_IREAD | S_IWRITE);
	remove("FILE_ID.DIZ");
    }
    if(access("DESC.SDI", 0) == 0)
    {
	chmod("DESC.SDI", S_IREAD | S_IWRITE);
	remove("DESC.SDI");
    }

    errorlevel = NO_ERRORS;
    memset(&s, NULL, sizeof(s));
    if(!Config_obj.open_config(CFG_RECORD))
    {
	printf("Unable to open configuration\n");
	errorlevel = NO_CONFIG;
    }
    else
    {
	Config_obj.read_main_config();
	Screen_obj.screen_init();
	sprintf(str, "---- AMU %s (%s) : %s", VERSION, PLATFORM, __DATE__);
	Utility_obj.logentry(str, LOG_MINIMAL);

#if defined(__AMU16__)
	sprintf(str, "Free memory   : %ldk", coreleft() / 1024);
	Utility_obj.logentry(str, LOG_MINIMAL);
#endif

	sprintf(str, "BBS Software  : %s", Config_obj.get_bbstype());
	Utility_obj.logentry(str, LOG_MINIMAL);

	strcpy(fn, "Command line  : '");
	for(x = 1; x < argc; x++)
	{
	    strcat(fn, argv[x]);
	    strcat(fn, " ");
	}
	Amustr_obj.trimend(fn);
	strcat(fn, "'");
	Utility_obj.logentry(fn, LOG_MINIMAL);

	strcpy(Parameters[0], "ADOPT");
	strcpy(Parameters[1], "BACKUP");
	strcpy(Parameters[2], "BBSDLS");
	strcpy(Parameters[3], "COMMENT");
	strcpy(Parameters[4], "COMPTXT");
	strcpy(Parameters[5], "EXPORTBBS");
	strcpy(Parameters[6], "EXPORTDIZ");
	strcpy(Parameters[7], "HEADERS");
	strcpy(Parameters[8], "IMPORTBBS");
	strcpy(Parameters[9], "IMPORTDIZ");
	strcpy(Parameters[10], "MAILERDLS");
	strcpy(Parameters[11], "NODESC");
	strcpy(Parameters[12], "OLD");
	strcpy(Parameters[13], "ORPHAN");
	strcpy(Parameters[14], "PACK");
	strcpy(Parameters[15], "REARC");
	strcpy(Parameters[16], "SCAN");
	strcpy(Parameters[17], "SORT");
	strcpy(Parameters[18], "LISTS");
	strcpy(Parameters[19], "OKFILE");
	strcpy(Parameters[20], "RESTORE");
	strcpy(Parameters[21], "TOPFILE");
	strcpy(Parameters[22], "FILEOVR");

	strcpy(Parameters[23], "SUBEXP_MSG");
	strcpy(Parameters[24], "UPLOADS_MSG");
	strcpy(Parameters[25], "ENFORCE");
	strcpy(Parameters[26], "BDAYLIST");
	strcpy(Parameters[27], "DAILYDL");
	strcpy(Parameters[28], "DAILYUL");
	strcpy(Parameters[29], "DOORS");
	strcpy(Parameters[30], "NEWUSER_MSG");
	strcpy(Parameters[31], "MSGOVR");
	strcpy(Parameters[32], "SUBLIST");
	strcpy(Parameters[33], "TODAYCALL");
	strcpy(Parameters[34], "CREDIT_MSG");
	strcpy(Parameters[35], "TOPUSERS");
	strcpy(Parameters[36], "USERLIST");
	strcpy(Parameters[37], "ARCHIVES");
	strcpy(Parameters[38], "CLEAN");
	strcpy(Parameters[39], "DAILY_MAINT");
	strcpy(Parameters[40], "INTERVAL_MAINT");
	strcpy(Parameters[41], "MONTHLY_MAINT");
	strcpy(Parameters[42], "WEEKLY_MAINT");
	strcpy(Parameters[43], "SWITCH");
	strcpy(Parameters[44], "BDAY_MSG");
	strcpy(Parameters[45], "CARRIER_MSG");

	// See if any of the parameters are file options
	// If no parameters are passed, we have to scan FILEAREA.AMU anyway
	if(argc == 1)
	{
	    file_opts_present = TRUE;
	}
	else
	{
	    file_opts_present = FALSE;
	}
	for(x = 1; x < argc; x++)
	{
	    strcpy(tmp_arg, argv[x]);
	    strupr(tmp_arg);
	    for(y = 0; y < 22; y++)
	    {
		if(strstr(tmp_arg, Parameters[y]))
		{
		    file_opts_present = TRUE;
		    break;
		}
	    }
	    if(file_opts_present == TRUE)
	    {
		break;
	    }
	}



	/* If the parameters are not valid, shut down */
	if(!validate_parameters(argc, argv))
	{
	    Config_obj.close_config(CFG_RECORD);
	    shutdown(start);
	    errorlevel = BAD_PARAMS;
	    return(errorlevel);
	}

	// Create the semaphore files if SEMAPHOR.CTL exists
	create_semaphores();

	// Analyze the BBS/Mailer logfiles
	Logs_obj.analyze_logs();

	// Read the user file
	Users_obj.process_users(argc, argv);

	strcpy(s.curarea, "Beginning operations");
	Screen_obj.display(&s);

	// See if the File or Message configuration has changed
	cfg_changed = FALSE;
	if(BBSInfo_obj.bbsfiles_changed() || BBSInfo_obj.bbsmsgs_changed())
	{
	    cfg_changed = TRUE;
	}
	if(cfg_changed && Config_obj.amu_cfgvar.old_config == CFG_ABORT)
	{
	    strcpy(s.curoption, "Aborting file area operations");
	    Screen_obj.display(&s);
	    sprintf(str, "! Run AMUCFG to update BBS file/msg area information");
	    Utility_obj.logentry(str, LOG_MINIMAL);
	    Config_obj.close_config(CFG_RECORD);
	}
	else
	{
	    if(cfg_changed)
	    {
		switch(Config_obj.amu_cfgvar.old_config)
		{
		    case CFG_IGNORE: sprintf(str, "[WARNING] Using old data, run AMUCFG to update BBS file/msg area information");
				     Utility_obj.logentry(str, LOG_MINIMAL);
				     break;
		    case CFG_IMPORT: sprintf(str, "Importing new BBS file/msg area information");
				     Utility_obj.logentry(str, LOG_MINIMAL);
				     strcpy(s.curoption, "Importing new BBS file area information");
				     Screen_obj.display(&s);
				     Config_obj.read_bbsfiles();
				     strcpy(s.curoption, "Importing new BBS message area information");
				     Screen_obj.display(&s);
				     Config_obj.read_bbsmsgs();
				     Config_obj.save_cfg(SAVE_ALL);
				     break;
		}
	    }
	    Config_obj.close_config(CFG_RECORD);

	    if(Config_obj.amu_cfgvar.semicolon_ok)
	    {
		strcat(Config_obj.amu_cfgvar.ext_desc_char, ";");
	    }
	    if(Config_obj.amu_cfgvar.ext_offset < 1)
	    {
		Config_obj.amu_cfgvar.ext_offset = 1;
	    }


	    // Do the file area operations if necessary
	    if(file_opts_present == TRUE)
	    {
		Utility_obj.logentry("Beginning file area operations", LOG_MINIMAL);
		Config_obj.open_config(ALLFILE_RECORD);
		Config_obj.open_config(FILEAREA_RECORD);
		if(argc > 1 && Utility_obj.check_param(argc, argv, "OKFILE", 0))
		{
		    FileOptions_obj.okfile(argc, argv);
		}
		ok = TRUE;
		if(Config_obj.amu_cfgvar.BBStype == LORA3)
		{
		    ok = FileOptions_obj.split_LoraFDB();
		}

		if(ok == TRUE)
		{
		    if(Utility_obj.check_param(argc, argv, "LISTS", 0))
		    {
			FileOptions_obj.filelists(ALLNEW_BEGIN);
		    }
		    kill_counterdata = FALSE;
		    for(x = 1; x <= Config_obj.num_filearea; x++)
		    {
			Config_obj.read_record(FILEAREA_RECORD, x);
			FileOptions_obj.process_area(x, &Config_obj.fileareavar, argc, argv);
		    }
		    if(kill_counterdata == TRUE)
		    {
			sprintf(fn, "%s%s", Config_obj.amu_cfgvar.AMUpath, TRANSFER_UPD);
			remove(fn);
		    }

		    if(Utility_obj.check_param(argc, argv, "LISTS", 0))
		    {
			FileOptions_obj.filelists(ALLNEW_END);
		    }

		    if(Config_obj.amu_cfgvar.BBStype == LORA3)
		    {
			FileOptions_obj.merge_LoraFDB();
		    }
		    Config_obj.close_config(FILEAREA_RECORD);
		    Config_obj.close_config(ALLFILE_RECORD);
		}
	    }
	}

	// Process the bulletins
	Bulletin_obj.create_bulletins(argc, argv);

	// Do the maintenance
	Maint_obj.maintenance(argc, argv);

	if(Utility_obj.check_param(argc, argv, "ARCHIVES", 0))
	{
	    // Create the archives
	    Archives_obj.archives();
	}

	if(argc > 1 && Utility_obj.check_param(argc, argv, "SWITCH", 0) ||
	   (argc == 1 && Config_obj.amu_cfgvar._switch) )
	{
	    Switch_obj.switch_();
	}
    }

    remove_semaphores();
    shutdown(start);
    Screen_obj.clearscreen();
    return(errorlevel);
}

void shutdown(long start)
{
    long end;
    char str[151];
    class Dates Date_Now;
    class Utility Utility_obj;

    end = Date_Now.getSecondsSince1970();
    start = end - start;
    sprintf(str, "---- Elapsed time: %02ld:%02ld:%02ld; Failed operations: %d",
	    start / 3600, (start % 3600) / 60, start % 60, Utility_obj.get_errors());
    Utility_obj.logentry(str, LOG_MINIMAL);
    Screen_obj.shutdown();
    Utility_obj.cleanup();
}


/* Write each help line */
void write_hlp(char *param, char *desc, int x)
{
    int Color;

    Color = WHITE;
    Screen_obj.writestr(1, x, Color, param);
    Color = LIGHTGRAY;
    Screen_obj.writestr(strlen(param) + 1, x, Color, desc);
}

/* Display a help screen */
void display_help(void)
{
    int x, Color;
    char str[91];

    x = 1;
    Screen_obj.clearscreen();
    Color = YELLOW;
    sprintf(str, "AMU %s  Automated Maintenance Utility  Help screen (Page 1/3)", VERSION);
    Screen_obj.writestr(1, x++, Color, str);
    Color = CYAN;
    Screen_obj.writestr(1, x++, Color, "Copyright (C) 1994-1998 David Hickey. All rights reserved");
    x++;
    Color = LIGHTGRAY;
    Screen_obj.writestr(1, x++, Color, "Usage: AMU <options> to use selected options");
    Screen_obj.writestr(1, x++, Color, "       AMU to use options in CONFIG.AMU");
    x++;
    write_hlp("ADOPT         ", "Adopt missing files into the file listings", x++);
    write_hlp("BACKUP        ", "Create backups of the file listings", x++);
    write_hlp("BBSDLS        ", "Update download counters - BBS logfiles", x++);
    write_hlp("COMMENT       ", "Add comments to ARJ, ZIP, and RAR archives", x++);
    write_hlp("COMPTXT       ", "Compress text files", x++);
    write_hlp("EXPORTBBS     ", "Export filebase to FILES.BBS", x++);
    write_hlp("EXPORTDIZ     ", "Export descriptions to FILE_ID.DIZ", x++);
    write_hlp("HEADERS       ", "Add headers to file listings", x++);
    write_hlp("IMPORTBBS     ", "Import FILES.BBS into filebase", x++);
    write_hlp("IMPORTDIZ     ", "Import description files for all files", x++);
    write_hlp("MAILERDLS     ", "Update download counters - Mailer logfiles", x++);
    write_hlp("NODESC        ", "Move files with no description or 'Missing Desc'", x++);
    write_hlp("OLD           ", "Remove/move old files", x++);
    write_hlp("ORPHAN        ", "Remove orphaned file entries", x++);
    write_hlp("PACK          ", "Pack file database (RA 2.xx and Concord 0.01 only)", x++);
    write_hlp("REARC         ", "Re-archive files", x++);
    Screen_obj.writestr(1, x++, Color, "");
    Screen_obj.writestr(1, x++, Color, "Press any key for Page 2...");
    getch();

    x = 1;
    Screen_obj.clearscreen();
    Color = YELLOW;
    sprintf(str, "AMU %s  Automated Maintenance Utility  Help screen (Page 2/3)", VERSION);
    Screen_obj.writestr(1, x++, Color, str);
    Color = CYAN;
    Screen_obj.writestr(1, x++, Color, "Copyright (C) 1994-1998 David Hickey. All rights reserved");
    x++;
    Color = LIGHTGRAY;
    Screen_obj.writestr(1, x++, Color, "Usage: AMU <options> to use selected options");
    Screen_obj.writestr(1, x++, Color, "       AMU to use options in CONFIG.AMU");
    x++;
    write_hlp("SCAN          ", "Perform virus scan on files", x++);
    write_hlp("SORT          ", "Sort file listings", x++);
    write_hlp("LISTS         ", "Create Allfiles/Newfiles listings", x++);
    write_hlp("BDAY_MSG      ", "Post message on a user's birthday", x++);
    write_hlp("CARRIER_MSG   ", "Post message to users who drop carrier", x++);
    write_hlp("CREDIT_MSG    ", "Post message when users' credit is low", x++);
    write_hlp("NEWUSER_MSG   ", "Post message to new users", x++);
    write_hlp("SUBEXP_MSG    ", "Post message when a subscription is about to expire", x++);
    write_hlp("UPLOADS_MSG   ", "Post message to users who upload", x++);
    write_hlp("ENFORCE       ", "Enforce ratios, etc", x++);
    write_hlp("BDAYLIST      ", "Today's birthdays bulletin", x++);
    write_hlp("DAILYDL       ", "Daily downloads bulletin", x++);
    write_hlp("DAILYUL       ", "Daily uploads bulletin", x++);
    write_hlp("DOORS         ", "Top doors bulletin", x++);
    write_hlp("FILEOVR       ", "Create overview of the file areas", x++);
    write_hlp("MSGOVR        ", "Create overview of the message base", x++);
    write_hlp("SUBLIST       ", "Subscription expiration bulletin", x++);
    Screen_obj.writestr(1, x++, Color, "");
    Screen_obj.writestr(1, x++, Color, "Press any key for Page 3...");
    getch();

    x = 1;
    Screen_obj.clearscreen();
    Color = YELLOW;
    sprintf(str, "AMU %s  Automated Maintenance Utility  Help screen (Page 3/3)", VERSION);
    Screen_obj.writestr(1, x++, Color, str);
    Color = CYAN;
    Screen_obj.writestr(1, x++, Color, "Copyright (C) 1994-1998 David Hickey. All rights reserved");
    x++;
    Color = LIGHTGRAY;
    Screen_obj.writestr(1, x++, Color, "Usage: AMU <options> to use selected options");
    Screen_obj.writestr(1, x++, Color, "       AMU to use options in CONFIG.AMU");
    x++;
    write_hlp("TODAYCALL     ", "Today's callers bulletin", x++);
    write_hlp("TOPFILE       ", "Create most popular files listing", x++);
    write_hlp("TOPUSERS      ", "Create top user bulletins", x++);
    write_hlp("USERLIST      ", "Create a list of all users", x++);
    write_hlp("ARCHIVES      ", "Create dated archives", x++);
    write_hlp("CLEAN         ", "Clean user records", x++);
    write_hlp("DAILY_MAINT   ", "Perform daily maintenance", x++);
    write_hlp("INTERVAL_MAINT", "Perform interval maintenance", x++);
    write_hlp("MONTHLY_MAINT ", "Perform monthly maintenance", x++);
    write_hlp("WEEKLY_MAINT  ", "Perform weekly maintenance", x++);
    write_hlp("SWITCH        ", "Pick random menus/textfiles", x++);
    write_hlp("OKFILExxxxx   ", "Create list of directories with security <= xxxxx", x++);
    write_hlp("RESTORE       ", "Restore file listings from AMU backups", x++);
    Screen_obj.writestr(1, x++, Color, "");
}

/* Create semaphore files */
void create_semaphores(void)
{
    char fn[151], str[151];
    FILE *semaphore, *ctlfile;
    class Utility Utility_obj;
    class Amustr Amustr_obj;

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.AMUpath, SEMAPHORE_CTL);
    ctlfile = fopen(fn, "rt");
    if(ctlfile)
    {
	sprintf(str, "[STARTUP] Processing '%s'", fn);
	Utility_obj.logentry(str, LOG_NORMAL);
	while(fgets(fn, sizeof(fn)-2, ctlfile))
	{
	    Amustr_obj.strip_n(fn);
            Amustr_obj.trimlead(fn);
            Amustr_obj.trimend(fn);
            Amustr_obj.noslash(fn);
            sprintf(str, "[STARTUP] Creating semaphore '%s'", fn);
            Utility_obj.logentry(str, LOG_EXTENSIVE);
            semaphore = fopen(fn, "w+t");
            if(semaphore)
            {
                fclose(semaphore);
            }
        }
        fclose(ctlfile);
    }
}

/* Remove the semaphores */
void remove_semaphores(void)
{
    char fn[151], str[151];
    FILE *ctlfile;
    class Utility Utility_obj;
    class Amustr Amustr_obj;

    sprintf(fn, "%s%s", Config_obj.amu_cfgvar.AMUpath, SEMAPHORE_CTL);
    ctlfile = fopen(fn, "rt");
    if(ctlfile)
    {
	sprintf(str, "[SHUTDOWN] Processing '%s'", fn);
	Utility_obj.logentry(str, LOG_NORMAL);
	while(fgets(fn, sizeof(fn)-2, ctlfile))
	{
	    Amustr_obj.strip_n(fn);
	    Amustr_obj.trimlead(fn);
	    Amustr_obj.trimend(fn);
	    Amustr_obj.noslash(fn);
	    sprintf(str, "[SHUTDOWN] Removing semaphore '%s'", fn);
	    Utility_obj.logentry(str, LOG_EXTENSIVE);
	    remove(fn);
	}
	fclose(ctlfile);
    }
}

/* See if the parameters are valid or not */
int validate_parameters(int argc, char *argv[])
{
    int status, x, found, index;
    char tmpparam[51], *ptr, str[151], bad_params[151];
    class Utility Utility_obj;
    class Amustr Amustr_obj;

    bad_params[0] = NULL;
    status = TRUE;
    for(x = 1; x < argc; x++)
    {
	/* Copy the parameter to a temp variable and get rid of any
	   'area' specifiers it may contain */
	strcpy(tmpparam, argv[x]);

	ptr = strchr(tmpparam, '[');
	if(ptr)
	{
	    *ptr = NULL;
	}

	/* Check for a '-' or a '/' */
	strrev(tmpparam);
	ptr = strchr(tmpparam, '-');
	if(ptr)
	{
	    *ptr = NULL;
	}
	ptr = strchr(tmpparam, '/');
	if(ptr)
	{
	    *ptr = NULL;
	}
	strrev(tmpparam);

	/* And get rid of any numbers that may be in there too */
	ptr = tmpparam;
	while(*ptr)
	{
	    if(isdigit(*ptr))
	    {
		*ptr = NULL;
		ptr = NULL;
		break;
	    }
	    ptr++;
	}

	strupr(tmpparam);

	found = FALSE;
	for(index = 0; index < NUM_PARAMETERS; index++)
	{
	    if(strcmp(Parameters[index], tmpparam) == 0)
	    {
		found = TRUE;
		break;
	    }
	}

	if(found == FALSE)
	{
	    status = FALSE;
	    sprintf(tmpparam, "'%s' ", argv[x]);
	    strcat(bad_params, tmpparam);
	}
    }

    if(status == FALSE)
    {
	Amustr_obj.trimend(bad_params);
	sprintf(str, "! Incorrect parameters, run AMU -? for help: %s", bad_params);
	Utility_obj.logentry(str, LOG_MINIMAL);
	Screen_obj.clearscreen();
	Screen_obj.writestr(1, 5, LIGHTRED, "Incorrect parameter(s) as follows, run AMU -? for help:");
	Screen_obj.writestr(1, 6, LIGHTRED, bad_params);
    }
    return(status);
}

/***********************************************************************/
/* EOF AMU.CPP */
