/*
 *  XMALLOC.C - Memory allocation routines with internal error checking.
 *  Written 1995-1997 by Andrew Clarke and released to the public domain.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "xmalloc.h"
#include "dlist.h"
#include "unused.h"

static char *msg_null = "";
static char *msg_malloc = "xmalloc";
static char *msg_calloc = "xcalloc";
static char *msg_realloc = "xrealloc";
static char *msg_strdup = "xstrdup";
#ifndef NO_XDEBUG
static char *msg_free = "xfree";
#endif

#ifndef NO_XDEBUG
static char *msg_malloc_zero =
  "%s(%lu) : %s(%s) failed: Attempted to allocate zero bytes of memory.";
static char *msg_strdup_null =
  "%s(%lu) : %s(%s) failed: Attempted to duplicate a NULL pointer.";
static char *msg_free_null =
  "%s(%lu) : %s(%s) failed: Attempted to deallocate a NULL pointer.";
static char *msg_free_nonp =
  "%s(%lu) : %s(%s) failed: Attempted to deallocate a non-allocated pointer.";
static char *msg_free_noalloc =
  "%s(%lu) : %s(%s) failed: No memory allocated to deallocate.";
#endif

static char *msg_malloc_fail =
  "%s(%lu) : %s(%s) failed: %lu (%lXh) byte%s required.";
static char *msg_calloc_fail =
  "%s(%lu) : %s(%s) failed: %lu (%lXh) byte%s (%lu item%s * %lu byte%s) required.";

#ifndef NO_XDEBUG

static PDLIST xm_list;

typedef struct
{
    const void *ptr;
    size_t size;
    int func;
    const char *srce_fn;
    unsigned long srce_ln;
}
*PXMENTRY;

#define XM_FUNC_MALLOC   1
#define XM_FUNC_CALLOC   2
#define XM_FUNC_REALLOC  3
#define XM_FUNC_STRDUP   4

#endif

void error(char *msg,...)
{
    if (msg != NULL)
    {
        va_list ap;
        va_start(ap, msg);
        vprintf(msg, ap);
        va_end(ap);
        putchar('\n');
    }
}

static char *plural(unsigned long n)
{
    return n == 1 ? "" : "s";
}

static void xmallocTerm(void)
{
#ifndef NO_XDEBUG
    PDLISTNODE p_node;
    unsigned long i, total;

    if (xm_list == NULL)
    {
        return;
    }

    i = 0;
    total = 0;
    p_node = dlistTravLast(xm_list);
    while (p_node != NULL)
    {
        PXMENTRY pxm;
        pxm = dlistGetElement(p_node);
        if (pxm != NULL)
        {
            i++;
            total += pxm->size;
            free(pxm);
        }
        p_node = dlistTravPrevious(p_node);
    }

    if (i != 0)
    {
        error("%s(%lu) : Detected %ld unfree'd memory allocation%s in "
          "%lu (%lXh) byte%s.", __FILE__, (unsigned long) __LINE__,
          i, plural(i), total, total, plural(total));
    }

    dlistTerm(xm_list);
    xm_list = NULL;
#endif
}

static void xmallocInit(void)
{
#ifndef NO_XDEBUG
    if (xm_list != NULL)
    {
        return;
    }

    xm_list = dlistInit();
    if (xm_list == NULL)
    {
        return;
    }
#endif

    atexit(xmallocTerm);
}

#ifndef NO_XDEBUG

static void xmallocAddPtr(const void *ptr, size_t size, int func, const char *srce_fn, unsigned long srce_ln)
{
    PDLISTNODE p_node;
    PXMENTRY pxm;

    if (xm_list == NULL)
    {
        return;
    }

    pxm = malloc(sizeof *pxm);
    if (pxm == NULL)
    {
        return;
    }

    pxm->ptr = ptr;
    pxm->size = size;
    pxm->srce_fn = srce_fn;
    pxm->srce_ln = srce_ln;
    pxm->func = func;

    p_node = dlistCreateNode(pxm);
    if (p_node == NULL)
    {
        free(pxm);
        return;
    }

    dlistAddNode(xm_list, p_node);
}

static void xmallocDropPtr(const void *ptr)
{
    PDLISTNODE p_node;

    if (xm_list == NULL)
    {
        return;
    }

    p_node = dlistTravLast(xm_list);
    while (p_node != NULL)
    {
        PXMENTRY pxm;
        pxm = dlistGetElement(p_node);
        if (pxm != NULL && pxm->ptr == ptr)
        {
            free(pxm);
            dlistDropNode(xm_list, p_node);
            return;
        }
        p_node = dlistTravPrevious(p_node);
    }
}

static int xmallocFindPtr(const void *ptr)
{
    PDLISTNODE p_node;

    if (xm_list == NULL)
    {
        return 0;
    }

    p_node = dlistTravLast(xm_list);
    while (p_node != NULL)
    {
        PXMENTRY pxm;
        pxm = dlistGetElement(p_node);
        if (pxm != NULL && pxm->ptr == ptr)
        {
            return 1;
        }
        p_node = dlistTravPrevious(p_node);
    }

    return 0;
}

static char *whichfunc(int func)
{
    switch (func)
    {
    case XM_FUNC_MALLOC:
        return msg_malloc;
    case XM_FUNC_CALLOC:
        return msg_calloc;
    case XM_FUNC_REALLOC:
        return msg_realloc;
    case XM_FUNC_STRDUP:
        return msg_strdup;
    default:
        break;
    }
    return "unknown";
}

static unsigned char hexdump_ch(const unsigned char ch)
{
    if (ch == 0 || (ch >= 7 && ch <= 13) || ch == 255)
    {
        return 250;
    }
    else
    {
        return ch;
    }
}

static void hexdump_ln(const void *ptr)
{
    unsigned short i;
    const char *str;
    str = ptr;
    for (i = 0; i <= 15; i++)
    {
        printf(" %02X", (unsigned char) str[i]);
    }
    printf("   ");
    for (i = 0; i <= 15; i++)
    {
        putchar(hexdump_ch(str[i]));
    }
    putchar('\n');
}

static void hexdump_ptr(const void *ptr, size_t n)
{
    unsigned short i;
    const char *str;
    str = ptr;
    if (n <= 16)
    {
        n = 16;
    }
    for (i = 0; i <= (n / 16) - 1; i++)
    {
        printf(" %06X  ", i * 16);
        hexdump_ln(str + (i * 16));
    }
}

#endif

void xmalloc_list(int hexdump)
{
#ifdef NO_XDEBUG
    unused(hexdump);
#else
    PDLISTNODE p_node;
    unsigned long i, total;

    if (xm_list == NULL)
    {
        return;
    }

    total = dlistTotalNodes(xm_list);
    if (total != 0)
    {
        error("%s(%lu) : xmalloc_list(): %lu total pointer%s allocated.",
          __FILE__, (unsigned long) __LINE__, total, plural(total));
    }

    i = 0;
    p_node = dlistTravFirst(xm_list);
    while (p_node != NULL)
    {
        PXMENTRY pxm;
        pxm = dlistGetElement(p_node);
        if (pxm != NULL)
        {
            i++;
            error("%s(%lu) : %lu (%lXh) byte%s allocated to pointer "
              "#%lu at %p by %s in module %s on line %lu.",
              __FILE__, (unsigned long) __LINE__,
              (unsigned long) pxm->size, (unsigned long) pxm->size,
              plural(pxm->size), i, pxm->ptr, whichfunc(pxm->func),
              pxm->srce_fn, pxm->srce_ln);
            if (hexdump)
            {
                hexdump_ptr(pxm->ptr, pxm->size);
            }
        }
        p_node = dlistTravNext(p_node);
    }
#endif
}

void *pxmalloc(size_t size, char *srce_fn, unsigned long srce_ln)
{
    void *ptr;

    xmallocInit();

#ifndef NO_XDEBUG
    if (size == 0)
    {
        error(msg_malloc_zero, srce_fn, srce_ln, msg_malloc, msg_null);
        exit(0);
    }
#endif

    ptr = malloc(size);
    if (ptr == NULL)
    {
        error(msg_malloc_fail, srce_fn, srce_ln, msg_malloc, msg_null,
          (unsigned long) size, (unsigned long) size, plural(size));
        exit(0);
    }

#ifndef NO_XDEBUG
    xmallocAddPtr(ptr, size, XM_FUNC_MALLOC, srce_fn, srce_ln);
#endif

    return ptr;
}

void *pxcalloc(size_t nitems, size_t size, char *srce_fn, unsigned long srce_ln)
{
    void *ptr;

    xmallocInit();

#ifndef NO_XDEBUG
    if (nitems == 0 || size == 0)
    {
        error(msg_malloc_zero, srce_fn, srce_ln, msg_calloc, msg_null);
        exit(0);
    }
#endif

    ptr = calloc(nitems, size);
    if (ptr == NULL)
    {
        error(msg_calloc_fail, srce_fn, srce_ln, msg_calloc, msg_null,
          (unsigned long) (size * nitems),
          (unsigned long) (size * nitems), plural(size * nitems),
          (unsigned long) nitems, plural(nitems),
          (unsigned long) size, plural(size));
        exit(0);
    }

#ifndef NO_XDEBUG
    xmallocAddPtr(ptr, size, XM_FUNC_CALLOC, srce_fn, srce_ln);
#endif

    return ptr;
}

void *pxrealloc(void *block, size_t size, char *blockn, char *srce_fn, unsigned long srce_ln)
{
    void *new_block;

    xmallocInit();

    if (size == 0)
    {
#ifndef NO_XDEBUG
        if (block == NULL)
        {
            error(msg_free_null, srce_fn, srce_ln, msg_realloc, blockn);
            exit(0);
        }
        else if (dlistIsEmpty(xm_list))
        {
            error(msg_free_noalloc, srce_fn, srce_ln, msg_realloc, blockn);
            exit(0);
        }
        else if (!xmallocFindPtr(block))
        {
            error(msg_free_nonp, srce_fn, srce_ln, msg_realloc, blockn);
            exit(0);
        }
        else
        {
            xmallocDropPtr(block);
#endif
            free(block);
#ifndef NO_XDEBUG
        }
#endif
        return NULL;
    }

    if (block == NULL)
    {
        new_block = malloc(size);
    }
    else
    {
        new_block = realloc(block, size);
    }

    if (new_block == NULL)
    {
        error(msg_malloc_fail, srce_fn, srce_ln, msg_realloc, blockn,
          (unsigned long) size, (unsigned long) size, plural(size));
        exit(0);
    }

#ifndef NO_XDEBUG
    xmallocDropPtr(block);
    xmallocAddPtr(new_block, size, XM_FUNC_REALLOC, srce_fn, srce_ln);
#endif

    return new_block;
}

char *pxstrdup(const char *s, char *srce_fn, unsigned long srce_ln)
{
    char *ptr;
    size_t size;

    xmallocInit();

#ifndef NO_XDEBUG
    if (s == NULL)
    {
        error(msg_strdup_null, srce_fn, srce_ln, msg_strdup, msg_null);
        exit(0);
    }
#endif

    size = strlen(s) + 1;
    ptr = malloc(size);
    if (ptr == NULL)
    {
        error(msg_malloc_fail, srce_fn, srce_ln, msg_strdup, msg_null,
          (unsigned long) size, (unsigned long) size, plural(size));
        exit(0);
    }
    else
    {
        strcpy(ptr, s);
    }

#ifndef NO_XDEBUG
    xmallocAddPtr(ptr, size, XM_FUNC_STRDUP, srce_fn, srce_ln);
#endif
    return ptr;
}

void pxfree(void *block, char *blockn, char *srce_fn, unsigned long srce_ln)
{
#ifdef NO_XDEBUG
    unused(blockn);
    unused(srce_fn);
    unused(srce_ln);
#endif

    xmallocInit();

#ifndef NO_XDEBUG
    if (block == NULL)
    {
        error(msg_free_null, srce_fn, srce_ln, msg_free, blockn);
        exit(0);
    }
    else if (dlistIsEmpty(xm_list))
    {
        error(msg_free_noalloc, srce_fn, srce_ln, msg_free, blockn);
        exit(0);
    }
    else if (!xmallocFindPtr(block))
    {
        error(msg_free_nonp, srce_fn, srce_ln, msg_free, blockn);
        exit(0);
    }
    else
    {
        xmallocDropPtr(block);
#endif
        free(block);
#ifndef NO_XDEBUG
    }
#endif
}
