UNIT PibTimer;

INTERFACE

USES
   Dos, GlobType;

   FUNCTION TimeOfDay : LONGINT;
   FUNCTION TimeDiff( Timer1, Timer2: LONGINT ) : LONGINT;

   FUNCTION TimeOfDayH : LONGINT;
   FUNCTION TimeDiffH( Timer1, Timer2: LONGINT ) : LONGINT;

IMPLEMENTATION

(*--------------------------------------------------------------------------*)
(*                TimeOfDay  --- Get time of day                            *)
(*--------------------------------------------------------------------------*)

FUNCTION TimeOfDay : LONGINT;

(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*     Function:  TimeOfDay                                                 *)
(*                                                                          *)
(*     Purpose:   Gets time of day from internal clock                      *)
(*                                                                          *)
(*     Calling sequence:                                                    *)
(*                                                                          *)
(*        Tod := TimeOfDay : LONGINT;                                       *)
(*                                                                          *)
(*           Tod --- Long integer number which is timer value expressed in  *)
(*                   seconds as:                                            *)
(*                   ( 3600 x hour + 60 x minutes + seconds )               *)
(*                                                                          *)
(*     Calls:  GetTime                                                      *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

VAR
   Hours   : WORD;
   Minutes : WORD;
   Seconds : WORD;
   SecHun  : WORD;

   TimeVal : LONGINT;

BEGIN (* TimeOfDay *)

   GetTime( Hours, Minutes, Seconds, SecHun );

   TimeVal    := Hours;
   TimeOfDay  := TimeVal * 3600 + Minutes * 60 + Seconds;

END   (* TimeOfDay *);

(*--------------------------------------------------------------------------*)
(*        TimeDiff  --- Get difference in time between two timer values     *)
(*--------------------------------------------------------------------------*)

FUNCTION TimeDiff( Timer1, Timer2: LONGINT ) : LONGINT;

(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*     Function:  TimeDiff                                                  *)
(*                                                                          *)
(*     Purpose:   Get difference in time between two timer values in        *)
(*                seconds.                                                  *)
(*                                                                          *)
(*     Calling sequence:                                                    *)
(*                                                                          *)
(*        TDiff := TimeDiff( Timer1, Timer2: LONGINT ) : LONGINT;           *)
(*                                                                          *)
(*           Timer1  --- first timer value (earlier)                        *)
(*           Timer2  --- second timer value (later)                         *)
(*                                                                          *)
(*           TDiff   --- difference between timer values                    *)
(*                                                                          *)
(*     Calls:  None                                                         *)
(*                                                                          *)
(*     Remarks:                                                             *)
(*                                                                          *)
(*        This routine will handle time wrap around midnight.  However, it  *)
(*        only handles timer values <= 24 hours in duration.                *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

CONST
   Secs_Per_Day = 86400    (* Seconds in one day *);

VAR
   TDiff : LONGINT;

BEGIN (* TimeDiff *)

   TDiff := Timer2 - Timer1;

   IF ( TDiff < 0 ) THEN
      TDiff := TDiff + Secs_Per_Day;

   TimeDiff := TDiff;

END   (* TimeDiff *);

(*--------------------------------------------------------------------------*)
(*      TimeOfDayH  --- Get time of day in 1/100 seconds from midnight      *)
(*--------------------------------------------------------------------------*)

FUNCTION TimeOfDayH : LONGINT;

(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*     Function:  TimeOfDayH                                                *)
(*                                                                          *)
(*     Purpose:   Gets time of day from internal clock in 1/100 seconds     *)
(*                                                                          *)
(*     Calling sequence:                                                    *)
(*                                                                          *)
(*        Tod := TimeOfDayH : LONGINT;                                      *)
(*                                                                          *)
(*           Tod --- Real number which is timer value expressed in          *)
(*                   hundredths of seconds as:                              *)
(*                   ( 360000 x hour + 6000 x minutes + 100 x seconds +     *)
(*                     hundredths of seconds ).                             *)
(*                                                                          *)
(*     Calls:  GetTime                                                      *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

VAR
   Hours   : WORD;
   Minutes : WORD;
   Seconds : WORD;
   SecHun  : WORD;

   TimerVal: LONGINT;

BEGIN (* TimeOfDayH *)

   GetTime( Hours, Minutes, Seconds, SecHun );

   TimerVal    := Hours;
   TimeOfDayH  := TimerVal * 360000 + Minutes * 6000 + Seconds * 100 + SecHun;

END   (* TimeOfDayH *);

(*--------------------------------------------------------------------------*)
(*       TimeDiffH  --- Get difference in time between two timer values     *)
(*--------------------------------------------------------------------------*)

FUNCTION TimeDiffH( Timer1, Timer2: LONGINT ) : LONGINT;

(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*     Function:  TimeDiffH                                                 *)
(*                                                                          *)
(*     Purpose:   Get difference in time between two timer values           *)
(*                in hundredths of seconds.                                 *)
(*                                                                          *)
(*     Calling sequence:                                                    *)
(*                                                                          *)
(*        Tdiff := TimeDiffH( Timer1, Timer2: LONGINT ) : REAL;             *)
(*                                                                          *)
(*           Timer1  --- first timer value (earlier)                        *)
(*           Timer2  --- second timer value (later)                         *)
(*                                                                          *)
(*           Tdiff   --- difference between timer values                    *)
(*                                                                          *)
(*     Calls:  None                                                         *)
(*                                                                          *)
(*     Remarks:                                                             *)
(*                                                                          *)
(*        This routine will handle time wrap around midnight.  However, it  *)
(*        only handles timer values <= 24 hours in duration.                *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

CONST
   Hundredths_Secs_Per_Day = 8640000    (* 1/100 Seconds in one day *);

VAR
   TDiff : LONGINT;

BEGIN (* TimeDiffH *)

   TDiff := Timer2 - Timer1;

   IF Tdiff < 0 THEN Tdiff := Tdiff + Hundredths_Secs_Per_Day;

   TimeDiffH := Tdiff;

END   (* TimeDiffH *);

END (* PibTimer *).

