{ rmdoor.pas - Turbo Pascal Door Driver Unit

{
		09/02/94 - Fixed problem with RMDoor displaying the unregistered
                   info if there isn't a config file.

                   Added delay before closing the serial port to be sure
                   that the serial port is empty before closing it.

	    02/12/94 - Added Desqview, Windows, and OS/2 awareness.

                   Added rmredrawstatusline procedure.

                   Fixed a couple of bugs in the display file routines.

        08/04/93 - Added RMDETECTRIP function

                 - Added BEEPUSER procedure.  Will send a beep
                   to the remote user (but NOT local console)
                   if it is NOT a local user.  If it IS a local
                   user, then a beep is sent to the pc-speaker.

                 - Added RMWRITELNREM(s:string) and RMWRITEREM(s:string)
                   procedures.  These will write/writeln to the modem
                   ONLY.  Should be used to write strings to modem if
                   there is actually someone connected (the procedures
                   check for local logon, though.  In which case, the
                   procedure returns without doing anything).

        ---------
	05/04/93 - Added nonstandardirq and appropriate routines to support
               nonstandard IRQs.

        05/25/93 - Changed RMDoor so that, if no configuration file is
                   given on the command line, it wouldn't say "UNREGISTERED"
                   even if they were indeed registered; still displays
                   a copyright notice though.

                   Added procedure RMDISPLAYFILELINES.  This is different
                   from rmdisplayfile in that you can pass the number of
                   lines to pause it at.

                   Added procedure RMREADSSIZE(sizeofstring:integer):string;
                   This will read in a string up to the size given in
                   the SIZEOFSTRING parameter.

                   Left registerrmdoor alone--found that the length of the
                   registration number won't change until we get registration
                   #161....no big deal for now.  :)

    02/19/93 - Added rmkeypressed function.

               No longer displays ending copyright message for registered
               systems.

               Added rmcopyright function.

               Changed LOCAL to just /L

               Documented COMPORT variable as integer

               Added alias, comport and error-correcting connection
               variables.

               Added RMHANGUP procedure

    08/06/92 - Fixed the flow control bug.

    		   Changed support for TRITEL.SYS to TRIBBS.SYS

               Added support for WWIV's CHAIN.TXT.

	12/23/91 - Added support for TRITEL.SYS.

               The chat and drop to DOS functions both restore
               the whole screen now and reposition the cursor
               to it's former position.

    12/29/91  - Moved the unregistered notice from the FIRST
                thing the door does, to the LAST thing the door
                does.

                Added procedure REGISTERRMDOOR(yourname:string;
                registration_no:string); to do the registration system.

	12/30/91  - Fixed the Black on White display problem that RMDoor
                would create on the local side of things.  YUCK!

}

unit rmdoor;
{$L rmdoor}
{$IFNDEF VER55}
{$L slice}
{$ENDIF}

interface

uses dos, crt;

const
    username : string = '';			{ caller's name }
    userfirstname : string = ''; 	{ caller's first name }
    doorname : string = ''; 		{ door name }
    callingfrom : string = ''; { Where caller is from }
    phonenumber : string = ''; { phonenumber }
    ansicolor : boolean = true; { ANSI color graphics? }
    seclevel : integer = 0;	{ caller's security level }
    baudrate : longint = 0; { caller's baud rate }
    minutesleft : integer = 60; { number of minutes the caller has left }
    bbsname : string = '';
    sysopname : string = '';
    lockedrate : longint = 0;
    doorfilepath : string = '';
    useralias : string='';
    errorcorrecting : boolean = false;
    nodenumber : integer = 1;
    nonstandardirq : integer = 0;


var
        configthere:boolean;
	comport : integer;


{ high level door routines }
procedure rmcopyright;
procedure rmhangup;
function rmdetectansi : boolean;
function rmdetectrip : boolean;
function rmtimeon : integer;
function rmreadkey : char;
function rmkeypressed : boolean;
function rmreadi : integer;
function rmreadl : longint;
function rmreads : string;
function rmreadssize(sizeofstring:integer) : string;
function rmtimeleft : integer;
procedure beepuser;
procedure rmwriteln(s:string); (* Write a string to local console/modem *)
procedure rmwrite(s:string);   (* Same as above no C/R or linefeed *)
procedure rmwritelnrem(s:string); (* Write string to modem ONLY w/ CR *)
procedure rmwriterem(s:string);   (* Write string to modem; NO C/R or LF *)
procedure rmwritei(i : integer);
procedure rmwritelni(i : integer);
procedure rmwritel(l : longint);
procedure rmwritelnl(l : longint);
procedure rmsetcolor(f,b:integer);  (* Setcolor routines *)
function rmgetforeground : integer;
function rmgetbackground : integer;
procedure rmclrscr;
procedure rmgotoxy(x, y : integer);
procedure rmdisplayfile(filename:string;pause:boolean);
procedure rmdisplayfilelines(filename:string;lines:integer);
procedure registerrmdoor(yourname:string;registration_no:string);
procedure rmredrawstatusline;


{ low level routines - NOT FOR PUBLIC! }
function carrier : boolean;
procedure close_port;
procedure fifo(n : integer);
function fileexists(filename:string):boolean;
function get_serial : char;
function in_ready : boolean;
function makeup(s:string):string;
procedure open_port(p : integer);
procedure put_serial(c : char);
procedure set_baud(baud : longint);
procedure set_data_format(bits, parity, stopbit : integer);
procedure set_dtr(f : boolean);
procedure set_port(baud : longint; bits, parity, stopbit : integer);
procedure puts_serial(s : string);

implementation

type
	pcboardsys = record
    	p1 : array[1..9] of char;
        errorcorrecting:array[1..2] of char;
        graphics : char;
        p2 : array[13..18] of char;
        baud : array[1..5] of char;
        recnum : integer;
        p3 : array[26..84] of char;
        name : array[1..25] of char;
        left : integer;
        nodenumber:byte;
        p4 : array[113..125] of char;
        port : char;
        p5 : array[127..128] of char;
    end;

    pcboarduser = record
    	p1 : array[1..25] of char;
        citystate : array[1..24] of char;
        p2 : array[50..61] of char;
        phone : array[1..13] of char;
        p3 : array[75..107] of char;
        seclevel : byte;
        p4 : array[109..400] of char;
    end;

const
	NO_PARITY = 0;			{ no parity constant }
	EVEN_PARITY = 1;		{ even parity constant }
	ODD_PARITY = 2;			{ odd parity constant }
	TX = 0;					{ transmit register }
	RX = 0;					{ receive register }
	DLL = 0;				{ divisor latch low }
	IER = 1;				{ interrupt enable register }
	DLH = 1;				{ divisor latch high }
	IIR = 2;				{ interrupt id register }
	LCR = 3;				{ line control register }
	MCR = 4;				{ modem control register }
	LSR = 5;				{ line status register }
	MSR = 6;				{ modem status register }
	RX_INT = 1;				{ received data bit mask }
    INT_MASK = 7;			{ interrupt mask }
	RX_ID = 4;				{ received data interrupt }
    DLAB = $80;				{ DLAB bit mask }
	DTR = 1;				{ data terminal ready bit mask }
	RTS = 2;				{ request to send bit mask }
	MC_INT = 8;				{ GPO2 bit mask }
	RX_RDY = 1;				{ character received bit mask }
	TX_RDY = $20;			{ transmitter buffer empty bit mask }
	CTS = $10;
	DSR = $20;
	DCD = $80;
	IMR = $21;				{ interrupt mask register }
	ICR = $20;				{ interrupt control register }
	IRQ3 = $f7;				{ IRQ 3 interrupt mask }
	IRQ4 = $ef;				{ IRQ 4 interrupt mask }
	EOI = $20;				{ end of interrupt mask }
	XON =  $11;				{ XON character }
	XOFF = $13;				{ XOFF character }
    ilen = 1024;			{ serial input buffer length }
	reg_no : integer = 0;
    port_open : boolean = false;
	crow : integer = 1;		{ ANSI last saved row number }
    ccol : integer = 1;		{ ANSI last saved column number }
	buffer : string = '';	{ ANSI terminal buffer }
    cursorlines : integer = -1; { cursor lines }
    statusline : boolean = TRUE; { display status line? }
    help : boolean = FALSE; { help status line? }
	VideoOffset : word = $B800;
    tx_rts : boolean = true;
    tx_xon : boolean = true;
    tx_xonoff : boolean = false;
    chat : boolean = false;
    atirq : integer = 0;

var
	base, irq, pic_mask, pic_address : integer;
	sibuff, eibuff : integer;
	inbuff : array[0..ilen] of char;
	oldvect : pointer;
    timeonh, timeonm, timeons : word;
    videobuffer : array[1..4000] of char;
    chatbuffer : array[1..4000] of char;
    scrollbuffer : array[1..640] of char;
    {$IFNDEF VER55}
    multitasker : integer;
    {$ENDIF}

procedure door_sys; forward;
procedure read_config; forward;
procedure dorinfo1_def; forward;
procedure callinfo_bbs; forward;
procedure sfdoors_dat; forward;
procedure pcboard; forward;
procedure tritel_sys; forward;
procedure chain_txt; forward;
procedure display_status_line(cflag : boolean); forward;
function kreadkey : char; forward;

{$F+}
function get_serial : char; external;
procedure put_serial(c : char); external;
procedure handler; interrupt; external;
procedure Restore80(col1, row1, col2, row2: integer; var buffer); external;
procedure Save80(col1, row1, col2, row2: integer; var buffer); external;
{$IFNDEF VER55}
function _detect_multitasker : integer; external;
procedure _timeslice; external;
{$ENDIF}
{$F-}

procedure disable; inline($fa);
procedure enable; inline($fb);

{ 16550 FIFO routine }
procedure fifo(n : integer);
var
	i : integer;

begin
	case n of
		1: i := 1;
		4: i := $41;
		8: i := $81;
		14: i := $c1;
		else i := 0;
    end;
	port[base + IIR] := i;	{ set the FIFO buffer }
end;

{ open a serial port}
procedure open_port(p : integer);
var
	i : longint;
    d : integer;

begin
    if not port_open and (p >= 1) and (p <= 4) then
    begin
    	if nonstandardirq > 7 then
        begin
        	atirq := 1;
            pic_address := $a0;
        end
        else
        begin
        	atirq := 0;
            pic_address := $20;
        end;
    	case p of
    		1:
        	begin
				base := $3f8;
                if nonstandardirq = 0 then
            	begin
                	irq := $0c;
                    pic_mask := 1 shl (4 mod 8);
                end
                else
                begin
                	if atirq = 0 then
                    	irq := nonstandardirq + 8
                    else
                    	irq := nonstandardirq + 104;
                    pic_mask := 1 shl (nonstandardirq mod 8);
                end;
        	end;
			2:
        	begin
				base := $2f8;
                if nonstandardirq = 0 then
            	begin
                	irq := $0b;
                    pic_mask := 1 shl (3 mod 8);
                end
                else
                begin
                	if atirq = 0 then
                    	irq := nonstandardirq + 8
                    else
                    	irq := nonstandardirq + 104;
                    pic_mask := 1 shl (nonstandardirq mod 8);
                end;
        	end;
			3:
        	begin
				base := $3e8;
                if nonstandardirq = 0 then
            	begin
                	irq := $0c;
                    pic_mask := 1 shl (4 mod 8);
                end
                else
                begin
                	if atirq = 0 then
                    	irq := nonstandardirq + 8
                    else
                    	irq := nonstandardirq + 104;
                    pic_mask := 1 shl (nonstandardirq mod 8);
                end;
        	end;
			4:
			begin
				base := $2e8;
                if nonstandardirq = 0 then
            	begin
                	irq := $0b;
                    pic_mask := 1 shl (3 mod 8);
                end
                else
                begin
                	if atirq = 0 then
                    	irq := nonstandardirq + 8
                    else
                    	irq := nonstandardirq + 104;
                    pic_mask := 1 shl (nonstandardirq mod 8);
                end;
        	end;
    	end;
        d := port[base];
        d := port[base + IIR];
		disable;                 	{ disable the interupts }
		getintvec(irq, oldvect);	{ save the current vector }
		setintvec(irq, @handler);	{ set the new vector }
		sibuff := 0;				{ set the starting buffer ptr}
		eibuff := 0;				{ set the ending buffer ptr}
        { assert GPO2, DTR, and RTS }
		port[base + MCR] :=  port[base + MCR] or MC_INT or DTR or RTS;
		port[base + IER] := RX_INT;	{ set received data interrupt }
        { set the interrupt }
        port[pic_address + 1] := port[pic_address + 1] and not pic_mask;
		enable;						{ enable the interrupts }
		fifo(14);					{ set FIFO buffer for 14 bytes }
		port_open := TRUE;			{ flag port is open }
    end;
end;

{ close serial port routine }
procedure close_port;
begin
	if port_open then
    begin
		port_open := false;			{ flag port not opened }
        fifo(0);
		disable;					{ disable the interrupts }
        port[pic_address + 1] := port[pic_address + 1] or pic_mask;
		port[base + IER] := 0;		{ clear recceived data int }
        { unassert GPO2 }
		port[base + MCR] := port[base + MCR] and (not MC_INT);
		setintvec(irq, oldvect);	{ reset the old int vector }
		enable;						{ enable the interrupts }
		port[base + MCR] := port[base + MCR] and (not RTS);
    end;
end;

{ set the baud rate }
procedure set_baud(baud : longint);
var
	c, n : integer;

begin
	if baud <> 0 then
    begin
		n := 115200 div baud;				{ figure the divisor }
		disable;								{ disable the interrupts }
		c := port[base + LCR];				{ get line control reg }
		port[base + LCR] := c or DLAB;		{ set divisor latch bit }
		port[base + DLL] := lo(n);		{ set LSB of divisor latch }
		port[base + DLH] := hi(n); { set MSB of divisor latch }
		port[base + LCR] := c;				{ restore line control reg }
		enable;								{ enable the interrupts }
    end;
end;

{ set the data format }
procedure set_data_format(bits, parity, stopbit : integer);
var
	n : integer;

begin
	if (parity < NO_PARITY) or (parity > ODD_PARITY) then
    	exit;
	if (bits < 5) or (bits > 8) then
    	exit;
	if (stopbit < 1) or (stopbit > 2) then
    	exit;
	n := bits - 5;							{ figure the bits value }
    if stopbit <> 1 then
    	n := n or 4;
    case parity of
		ODD_PARITY: n := n or 8;
		EVEN_PARITY: n := n or $18;
    end;
	disable;								{ disable the interrupts }
	port[base + LCR] := n;					{ set the port }
	enable;									{ enable the interrupts }
end;


procedure set_port(baud : longint; bits, parity, stopbit : integer);
begin
	if not port_open then
    	exit;
	set_baud(baud);							{ set the baud rate }
	set_data_format(bits, parity, stopbit); { set the data format }
end;


{ check for byte in input buffer }
function in_ready : boolean;
begin
	if sibuff = eibuff then
	   	in_ready := false
    else
    	in_ready := true;
end;

{ check for carrier routine }
function carrier : boolean;
begin
	if (port[base + MSR] and DCD) = DCD then
    	carrier := true
    else
    	carrier := false;
end;

{ set DTR routine }
procedure set_dtr(f : boolean);
begin
	if f then
    	port[base + MCR] := port[base + MCR] or DTR
    else
    	port[base + MCR] := port[base + MCR] and not DTR;
end;

{ put string out serial port }
procedure puts_serial(s : string);
var
	i : integer;

begin
	i := 1;
    while i <= length(s) do
    begin
    	put_serial(s[i]);
        inc(i);
    end;
end;

{ check for numeric digit }
function bindigit(c : char) : boolean;
begin
	bindigit := (c >= '0') and (c <= '9');
end;

{ convert ASCII to binary }
function ascbin(s : string; var n : integer) : string;
begin
	n := 0;
	while true do
	begin
		if bindigit(s[1]) then
        begin
			n := n * 10;
			n := n + integer(s[1]) - integer('0');
            s := copy(s, 2, length(s) - 1);
        end
        else
        begin
        	ascbin := s;
            exit;
        end;
    end;
end;

{ convert IBM color attribute to ANSI text string }
function ibmtoansi(att : integer) : string;
var
	s : string;

begin
	s := '';
	s := s + #27 + '[0';
	if (att and $80) = $80 then
		s := s + ';5';
	if (att and 8) = 8 then
		s := s + ';1';
    case att and 7 of
		0:
			s := s + ';30';
		1:
			s := s + ';34';
		2:
			s := s + ';32';
		3:
			s := s + ';36';
		4:
			s := s + ';31';
		5:
			s := s + ';35';
		6:
			s := s + ';33';
		7:
			s := s + ';37';
    end;
	case (att shr 4) and 7 of
		0:
			s := s + ';40';
		1:
			s := s + ';44';
		2:
			s := s + ';42';
		3:
			s := s + ';46';
		4:
			s := s + ';41';
		5:
			s := s + ';45';
		6:
			s := s + ';43';
		7:
			s := s + ';47';
    end;
	ibmtoansi := s +  'm';
end;

{ ANSI emulation routine }
procedure ansiout(c : char);
var
	n, row, col : integer;

begin
	if buffer = '' then
	begin
		if c <> #27 then
			case c of
				#0:
                	;
				#8, #127:
					if wherex <> 1 then
						write(#8, ' ', #8);
				#9:
					repeat
                    	write(' ');
                    until (wherex mod 8) <> 1;
				#12:
					clrscr;
                else
                	write(c);
            end
		else
			buffer := c;
        exit;
    end;
	if length(buffer) = 1 then
    begin
		if c = '[' then
			buffer := buffer + c
		else
        begin
        	write(#27);
			if c <> #27 then
            begin
				write(c);
                buffer := '';
            end;
		end;
        exit;
    end;
	if length(buffer) = 2 then
    begin
		case c of
			's':
            begin
				crow := wherey;
				ccol := wherex;
                buffer := '';
                exit;
            end;
			'u':
            begin
				gotoxy(ccol, crow);
				buffer := '';
                exit;
            end;
			'K':
            begin
				clreol;
                buffer := '';
                exit;
            end;
			'H', 'F':
            begin
				gotoxy(1, 1);
                buffer := '';
            	exit;
            end;
			'A':
            begin
				gotoxy(wherex, wherey - 1);
                buffer := '';
				exit;
            end;
			'B':
            begin
				gotoxy(wherex, wherey + 1);
				buffer := '';
                exit;
            end;
			'C':
            begin
				gotoxy(wherex + 1, wherey);
                buffer := '';
                exit;
            end;
			'D':
        	begin
				gotoxy(wherex - 1, wherey);
				buffer := '';
                exit;
            end;
            else
            begin
            	if bindigit(c) then
                	buffer := buffer + c
                else
                	buffer := '';
                exit;
            end;
		end;
	end;
	if bindigit(c) or (c = ';') then
    begin
		buffer := buffer + c;
        exit;
    end;
    delete(buffer, 1, 2);
    buffer := buffer + c;
	case c of
		'H', 'F':
        begin
        	buffer := ascbin(buffer, row);
			if buffer[1] <> ';' then
            begin
				gotoxy(1, row);
                buffer := '';
            	exit;
            end;
            delete(buffer, 1, 1);
			if not bindigit(buffer[1]) then
            begin
            	buffer := '';
                exit;
            end;
			buffer := ascbin(buffer, col);
			gotoxy(col, row);
            buffer := '';
            exit;
        end;
		'A':
        begin
			buffer := ascbin(buffer, n);
			gotoxy(wherex, wherey - n);
            buffer := '';
            exit;
        end;
		'B':
        begin
			buffer := ascbin(buffer, n);
			gotoxy(wherex, wherey + n);
            buffer := '';
            exit;
        end;
		'C':
        begin
			buffer := ascbin(buffer, n);
			gotoxy(wherex + n, wherey);
            buffer := '';
            exit;
        end;
		'D':
        begin
			buffer := ascbin(buffer, n);
			gotoxy(wherex - n, wherey);
            buffer := '';
            exit;
        end;
		'J':
        begin
			buffer := ascbin(buffer, n);
			if n = 2 then
				clrscr;
            buffer := '';
            exit;
        end;
		'm':
			while (TRUE) do
            begin
				buffer := ascbin(buffer, n);
				case n of
					0:
						textattr := 7;
					1:
						textattr := textattr or 8;
					5:
						textattr := textattr or $80;
					30:
						textattr := textattr and $f8;
					31:
						textattr := textattr and $f8 or 4;
					32:
						textattr := textattr and $f8 or 2;
					33:
						textattr := textattr and $f8 or 6;
					34:
						textattr := textattr and $f8 or 1;
					35:
						textattr := textattr and $f8 or 5;
					36:
						textattr := textattr and $f8 or 3;
					37:
						textattr := textattr and $f8 or 7;
					40:
						textattr := textattr and $8f;
					41:
						textattr := textattr and $8f or $40;
					42:
						textattr := textattr and $8f or $20;
					43:
						textattr := textattr and $8f or $60;
					44:
						textattr := textattr and $8f or $10;
					45:
						textattr := textattr and $8f or $50;
					46:
						textattr := textattr and $8f or $30;
					47:
						textattr := textattr and $8f or $70;
                end;
				if buffer[1] = ';' then
                	delete(buffer, 1, 1)
                else
                begin
                	buffer := '';
                    exit;
                end;
			end;
		else
        begin
        	buffer := '';
        	exit;
        end;
    end;
end;

{ display string }
procedure ansistring(s : string);
var
	i : integer;

begin
	i := 0;
	while i < length(s) do
    begin
    	inc(i);
    	ansiout(s[i]);
    end;
end;

procedure setcolor(f, b : integer);
begin
	ansistring(ibmtoansi((b shl 4) or f));
end;

procedure cursoroff;
var
	regs : registers;

begin
	if cursorlines = -1 then
    begin
    	regs.ah := 3;
        regs.bh := 0;
        intr($10, regs);
        cursorlines := regs.cx;
    end;
    regs.ah := 1;
    regs.cx := $2000;
    intr($10, regs);
end;

procedure cursoron;
var
	regs : registers;

begin
	if cursorlines = -1 then
    begin
    	regs.ah := 3;
        regs.bh := 0;
        intr($10, regs);
        cursorlines := regs.cx;
    end;
    regs.ah := 1;
    regs.cx := cursorlines;
    intr($10, regs);
end;

function rmtimeon : integer;
var
	h, m, s, d : word;
    ton, tnow : longint;

begin
	gettime(h, m, s, d);
    ton := longint(timeonh) * longint(3600) + longint(timeonm) * longint(60) + longint(timeons);
    tnow := longint(h) * longint(3600) + longint(m) * 60 + longint(s);
	if tnow >= ton then
    	rmtimeon := (tnow - ton) div 60
    else
    	rmtimeon := (tnow + 86400 - ton) div 60;
end;

function rmtimeleft : integer;
begin
	rmtimeleft := minutesleft - rmtimeon;
end;

function makeup(s:string):string;
var
  lcv:integer;
  tempstring:string;
begin
  tempstring:='';
  for lcv:=1 to length(s) do
    tempstring:=tempstring+upcase(s[lcv]);
  makeup:=tempstring;
end;


procedure read_config;
var
  configfile:text;
  doortype:string;
  good:boolean;
begin
  if paramcount<1 then
    begin
      writeln(#7,#7,' ERROR: Config file name not specified on command line!');
      halt;
    end;
  configthere:=true;
  {$I-};
  assign(configfile,paramstr(1));
  reset(configfile);
  {$I+};
  if ioresult<>0 then
    begin
      writeln(#7,#7,' ERROR: Unable to open file ''',paramstr(1),'''!');
      configthere:=false;
      halt;
    end;
  {$I-};
  readln(configfile,doortype);
  readln(configfile,doorfilepath);
  readln(configfile,bbsname);
  readln(configfile,sysopname);
  readln(configfile,lockedrate);
  {$I+};
  if ioresult<>0 then
    begin
      writeln(#7,#7,' Error in configuration file ''',paramstr(1),'''!');
      configthere:=false;
      halt;
    end;
  close(configfile);
  if paramcount>1 then
    begin
      if makeup(paramstr(2))='/L' then
       begin
        writeln(' Local Logon');
        repeat
          write(' Please enter your name: ');
          readln(username);
        until username<>'';
        exit;
       end;
    end;
  if doorfilepath[length(doorfilepath)]<>'\' then
     doorfilepath:=doorfilepath+'\';
  if makeup(doortype)='GAP' then
     door_sys
  else if makeup(doortype)='PCB' then
     pcboard
  else if makeup(doortype)='SF' then
     sfdoors_dat
  else if makeup(doortype)='RBBS' then
     dorinfo1_def
  else if makeup(doortype)='WC' then
     callinfo_bbs
  else if makeup(doortype) = 'TRIBBS' then
  	tritel_sys
  else if makeup(doortype) = 'WWIV' then
  	chain_txt
  else
    begin
        writeln(#7,#7,' Error in configuration--invalid BBS type ''',makeup(doortype),'''!');
        configthere:=false;
        halt;
    end;
  if lockedrate<>0 then
    writeln(' Serial port locked at: ',lockedrate);
end;


procedure rmwrite(s:string);
begin
  ansistring(s);
  if baudrate<> 0 then    (* NOT a local logon--send to modem too *)
    puts_serial(s);
end;

procedure beepuser;
begin
  if baudrate<>0 then
    puts_serial(#7)
  else
    ansistring(#7);
end;

procedure rmwriterem(s:string);
begin
  if baudrate<> 0 then    (* NOT a local logon--send to modem too *)
    puts_serial(s);
end;

procedure rmwriteln(s:string);
var
	attr : integer;

begin
  rmwrite(s);
  attr := textattr;
  rmsetcolor(7, 0);
  ansistring(#13+#10);
  if baudrate<>0 then      (* NOT a local logon--send to modem too *)
    puts_serial(#13+#10);
  rmsetcolor(attr and $f, attr shr 4);
end;

procedure rmwritelnrem(s:string);
var
	attr : integer;

begin
  attr := textattr;
  rmsetcolor(7, 0);
  if baudrate<>0 then      (* NOT a local logon--send to modem too *)
    puts_serial(s+#13+#10);
  rmsetcolor(attr and $f, attr shr 4);
end;

procedure rmwritei(i : integer);
var
	s : string;

begin
	str(i, s);
    rmwrite(s);
end;

procedure rmwritelni(i : integer);
var
	s : string;

begin
	str(i, s);
    rmwriteln(s);
end;

procedure rmwritel(l : longint);
var
	s : string;

begin
	str(l, s);
    rmwrite(s);
end;

procedure rmwritelnl(l : longint);
var
	s : string;

begin
	str(l, s);
    rmwriteln(s);
end;

procedure rmsetcolor(f,b:integer);
begin
  if ansicolor then
    begin
      setcolor(f,b);    (* Setcolor at local console *)
      if baudrate<>0 then
        puts_serial(ibmtoansi((b shl 4) or f)); (* Send to modem *)
    end;
end;

function rmgetforeground : integer;
begin
	rmgetforeground := textattr and $f;
end;

function rmgetbackground : integer;
begin
	rmgetbackground := textattr shr 4;
end;

procedure rmclrscr;
begin
	if ansicolor then
    	rmwrite(#27 + '[2J')
    else
    	rmwrite(#12);
    display_status_line(FALSE);
end;

procedure rmgotoxy(x, y : integer);
var
	xs, ys : string;

begin
	if not ansicolor then
    	exit;
    str(x, xs);
    str(y, ys);
    rmwrite(#27 + '[' + ys + ';' + xs + 'H');
end;

function fileexists(filename:string):boolean;
var
  cha:file;
begin
  {$I-};
  assign(cha,filename);
  reset(cha);
  close(cha);
  {$I+};
  fileexists:=(ioresult=0) and (filename <> '');
end;

procedure door_sys;                     (* Reads in DOOR.SYS file *)
var
  lcv:integer;                          (* Loop control variable *)
  door_sys_file:text;                   (* variable of DOOR.SYS file *)
  tempstring:string;                    (* Temporary string variable *)
  code:integer;                         (* Req'd for VAL procedure   *)
begin
  if not(fileexists(doorfilepath+'door.sys')) then
    begin
      writeln(#7,#7,' ERROR: ',makeup(doorfilepath),'DOOR.SYS not found!');
      halt;
    end
  else
    begin
      {$I-};
      assign(door_sys_file,doorfilepath+'door.sys');
      reset(door_sys_file);
      {$I+};
      if ioresult<>0 then
	begin
	  writeln(#7,#7,' ERROR opening file:  ',makeup(doorfilepath),'DOOR.SYS!');
	  halt;
	end;
      {$I-};
      readln(door_sys_file,tempstring);
      tempstring:=tempstring[4];
      val(tempstring,comport,code);
      if comport<>0 then
	  readln(door_sys_file,baudrate)
      else
	  begin
	    baudrate:=0;
	    readln(door_sys_file);
	  end;
      readln(door_sys_file);
      readln(door_sys_file,nodenumber);
      for lcv:=1 to 5 do
	    readln(door_sys_file);
      readln(door_sys_file,username);
      readln(door_sys_file,callingfrom);
      readln(door_sys_file,phonenumber);
      for lcv:=1 to 2 do
	readln(door_sys_file);
      readln(door_sys_file,seclevel);
      for lcv:=1 to 3 do
	readln(door_sys_file);
      readln(door_sys_file,minutesleft);
      readln(door_sys_file,tempstring);
      if tempstring='GR' then
	ansicolor:=true
      else
	ansicolor:=false;
    for lcv:=1 to 15 do
      readln(door_sys_file);
    readln(door_sys_file,useralias);
    readln(door_sys_file);
    readln(door_sys_file,tempstring);
    if tempstring='Y' then
      errorcorrecting:=true;
      close(door_sys_file);
      {$I+};
      if ioresult<>0 then
	begin
	  writeln(#7,#7,' ERROR: Could not read from file: ',makeup(doorfilepath),'DOOR.SYS');
	  halt;
	end;
    end;
end;

procedure dorinfo1_def;
var
  lcv:integer;                          (* Loop control variable *)
  dorinfo1_def_file:text;               (* variable of DORINFO1.DEF file *)
  tempstring:string;                    (* Temporary string variable *)
  tempstring2:string;                    (*    "        "       "     *)
  code:integer;                         (* Req'd for VAL procedure   *)
begin
 if not(fileexists(doorfilepath+'dorinfo1.def')) then
    begin
      writeln(#7,#7,' ERROR: ',makeup(doorfilepath),'DORINFO1.DEF not found!');
      halt;
    end
  else
    begin
      {$I-};
      assign(dorinfo1_def_file,doorfilepath+'dorinfo1.def');
      reset(dorinfo1_def_file);
      {$I+};
      if ioresult<>0 then
	begin
	  writeln(#7,#7,' ERROR opening file:  ',makeup(doorfilepath),'DORINFO1.DEF!');
	  halt;
	end;
      {$I-};
      for lcv:=1 to 3 do
	     readln(dorinfo1_def_file);
      readln(dorinfo1_def_file,tempstring);
      tempstring:=tempstring[4];
      val(tempstring,comport,code);
      read(dorinfo1_def_file,baudrate);
      readln(dorinfo1_def_file);
      readln(dorinfo1_def_file);
      readln(dorinfo1_def_file,tempstring);    (* first name *)
      readln(dorinfo1_def_file,tempstring2);   (* last name *)
      username:=tempstring+' '+tempstring2;(* Put first/last name together *)
      useralias:=username;
      readln(dorinfo1_def_file,callingfrom);
      readln(dorinfo1_def_file,tempstring);
      if tempstring='1' then          (* '1' indicates ANSI, else NON-ANSI *)
	    ansicolor:=true
      else
	    ansicolor:=false;
      readln(dorinfo1_def_file,seclevel);
      readln(dorinfo1_def_file,minutesleft);
      close(dorinfo1_def_file);
      {$I+};
      if ioresult<>0 then
    	begin
	      writeln(#7,#7,' ERROR reading from file: ',makeup(doorfilepath),'DORINFO1.DEF');
	      halt;
    	end;
    end;
end;

procedure callinfo_bbs;
var
  lcv:integer;                          (* Loop control variable *)
  callinfo_bbs_file:text;               (* variable of DORINFO1.DEF file *)
  tempstring:string;                    (* Temporary string variable *)
  tempstring2:string;                    (*    "        "       "     *)
  code:integer;                         (* Req'd for VAL procedure   *)
  tempint:integer;
begin
 if not(fileexists(doorfilepath+'callinfo.bbs')) then
    begin
      writeln(#7,#7,' ERROR: ',makeup(doorfilepath),'CALLINFO.BBS not found!');
      halt;
    end
  else
    begin
      {$I-};
      assign(callinfo_bbs_file,doorfilepath+'callinfo.bbs');
      reset(callinfo_bbs_file);
      {$I+};
      if ioresult<>0 then
    	begin
	      writeln(#7,#7,' ERROR opening file:  ',makeup(doorfilepath),'CALLINFO.BBS!');
	      halt;
	    end;
      {$I-};
      readln(callinfo_bbs_file,username);
      useralias:=username;
   	  readln(callinfo_bbs_file,tempstring);
      val(tempstring,tempint,code);
	  case tempint of
        0:baudrate:=2400;
        1:baudrate:=300;
        2:baudrate:=1200;
        3:baudrate:=9600;
        4:baudrate:=19200;
        5:baudrate:=0;
        end;  (* case *)
      readln(callinfo_bbs_file,callingfrom);
      readln(callinfo_bbs_file,seclevel);
      readln(callinfo_bbs_file,minutesleft);
      readln(callinfo_bbs_file,tempstring);
      if tempstring='COLOR' then
        ansicolor:=true
      else
        ansicolor:=false;
      for lcv:=1 to 10 do
        readln(callinfo_bbs_file);
      readln(callinfo_bbs_file,phonenumber);
      for lcv:=1 to 11 do
        readln(callinfo_bbs_file);
      readln(callinfo_bbs_file,tempstring);
      tempstring:=tempstring[4];
      val(tempstring,comport,code);
      for lcv:=1 to 3 do
         readln(callinfo_bbs_file);
      readln(callinfo_bbs_file,tempstring);
      if tempstring<>'Normal Connection' then
        errorcorrecting:=true;
      readln(callinfo_bbs_file);
      readln(callinfo_bbs_file,nodenumber);
      close(callinfo_bbs_file);
      {$I+};
      if ioresult<>0 then
        begin
          writeln(#7,#7,' ERROR reading from file: ',makeup(doorfilepath),'CALLINFO.BBS!');
          halt;
        end;
   end;
end;

procedure sfdoors_dat;
var
  lcv:integer;                          (* Loop control variable *)
  sfdoors_dat_file:text;                (* variable of DORINFO1.DEF file *)
  tempstring:string;                    (* Temporary string variable *)
  code:integer;                         (* Req'd for VAL procedure   *)
begin
 if not(fileexists(doorfilepath+'sfdoors.dat')) then
    begin
      writeln(#7,#7,' ERROR: ',makeup(doorfilepath),'SFDOORS.DAT not found!');
      halt;
    end
  else
    begin
      {$I-};
      assign(sfdoors_dat_file,doorfilepath+'sfdoors.dat');
      reset(sfdoors_dat_file);
      {$I+};
      if ioresult<>0 then
    	begin
	      writeln(#7,#7,' ERROR opening file:  ',makeup(doorfilepath),'SFDOORS.DAT!');
	      halt;
	    end;
      {$I-};
      readln(sfdoors_dat_file);
      readln(sfdoors_dat_file,username);
      useralias:=username;
      for lcv:=1 to 2 do
        readln(sfdoors_dat_file);
      readln(sfdoors_dat_file,baudrate);
      readln(sfdoors_dat_file,comport);
      readln(sfdoors_dat_file,minutesleft);
      for lcv:=1 to 2 do
        readln(sfdoors_dat_file);
      readln(sfdoors_dat_file,tempstring);
      if tempstring='TRUE' then
        ansicolor:=true
      else
        ansicolor:=false;
      readln(sfdoors_dat_file,seclevel);
      for lcv:=1 to 9 do
        readln(sfdoors_dat_file);
      readln(sfdoors_dat_file,tempstring);
      if tempstring='TRUE' then
        errorcorrecting:=true;
      for lcv:=1 to 2 do
        readln(sfdoors_dat_file);
      readln(sfdoors_dat_file,nodenumber);
      close(sfdoors_dat_file);
      {$I+};
      if ioresult<>0 then
        begin
          writeln(#7,#7,' ERROR reading from file: ',makeup(doorfilepath),'SFDOORS.DAT!');
          halt;
        end;
    end;
end;

procedure pcboard;

	procedure stripspaces(var s : string);
    begin
    	while (s <> '') and (s[1] = ' ') do
        	delete(s, 1, 1);
        while (s <> '') and (s[length(s)] = ' ') do
        	delete(s, length(s), 1);
    end;

var
	i : integer;
    ts : string;
  	pcbsys : file of pcboardsys;
  	users : file of pcboarduser;
  	s : pcboardsys;
  	u : pcboarduser;

begin
 	if not fileexists(doorfilepath + 'pcboard.sys') then
    begin
      	writeln(#7, #7, ' ERROR: ', makeup(doorfilepath), 'PCBOARD.SYS not found!');
      	halt;
    end;
 	if not fileexists(doorfilepath + 'USERS') then
    begin
      	writeln(#7, #7, ' ERROR: ', makeup(doorfilepath), 'USERS not found!');
      	halt;
    end;
   	{$I-};
    assign(pcbsys, doorfilepath + 'pcboard.sys');
    reset(pcbsys);
    {$I+};
    if ioresult <> 0 then
	begin
	  	writeln(#7, #7, ' ERROR opening file:  ', makeup(doorfilepath), 'PCBOARD.SYS!');
	  	halt;
	end;
    {$I-};
    read(pcbsys, s);
    close(pcbsys);
    {$I+};
    if ioresult <> 0 then
    begin
		writeln(#7, #7, ' ERROR reading from file: ', makeup(doorfilepath), 'PCBOARD.SYS!');
	    halt;
    end;
   	{$I-};
    assign(users, doorfilepath + 'users');
    reset(users);
    {$I+};
    if ioresult <> 0 then
	begin
	  	writeln(#7, #7, ' ERROR opening file:  ', makeup(doorfilepath), 'USERS!');
	  	halt;
	end;
    {$I-};
    seek(users, s.recnum - 1);
    read(users, u);
    close(users);
    {$I+};
    if ioresult <> 0 then
    begin
		writeln(#7, #7, ' ERROR reading from file: ', makeup(doorfilepath), 'USERS!');
	    halt;
    end;
	comport := integer(s.port) - $30;
    errorcorrecting:=s.errorcorrecting[2]='1';
    nodenumber:=s.nodenumber;
	ts := '';
    for i := 1 to 5 do
    	ts := ts + s.baud[i];
    stripspaces(ts);
    val(ts, baudrate, i);
    username := '';
    for i := 1 to 25 do
    	username := username + s.name[i];
    stripspaces(username);
    useralias:=username;
    callingfrom := '';
    for i := 1 to 24 do
    	callingfrom := callingfrom + u.citystate[i];
    stripspaces(callingfrom);
    if s.graphics = 'Y' then
    	ansicolor := true
    else
    	ansicolor := false;
    seclevel := u.seclevel;
    minutesleft := s.left;
    phonenumber := '';
    for i := 1 to 13 do
    	phonenumber := phonenumber + u.phone[i];
    stripspaces(phonenumber);
end;

procedure chain_txt;
var
  i :integer;                          (* Loop control variable *)
  chain_txt_file:text;                (* variable of DORINFO1.DEF file *)
  tempstring:string;                    (* Temporary string variable *)
begin
 if not(fileexists(doorfilepath+'CHAIN.TXT')) then
    begin
      writeln(#7,#7,' ERROR: ',makeup(doorfilepath),'CHAIN.TXT not found!');
      halt;
    end
  else
    begin
      {$I-};
      assign(chain_txt_file,doorfilepath+'CHAIN.TXT');
      reset(chain_txt_file);
      {$I+};
      if ioresult<>0 then
    	begin
	      writeln(#7,#7,' ERROR opening file:  ',makeup(doorfilepath),'CHAIN.TXT!');
	      halt;
	    end;
      {$I-};
      readln(chain_txt_file);
      readln(chain_txt_file,useralias);
      readln(chain_txt_file, username);
	  for i := 1 to 7 do
      		readln(chain_txt_file);
      readln(chain_txt_file, seclevel);
      for i := 1 to 2 do
	      readln(chain_txt_file);
      readln(chain_txt_file, tempstring);
      if tempstring='1' then
        ansicolor:=true
      else
        ansicolor:=false;
      readln(chain_txt_file);
      readln(chain_txt_file, minutesleft);
      minutesleft := minutesleft div 60;
      for i := 1 to 3 do
      	readln(chain_txt_file);
      readln(chain_txt_file,baudrate);
      readln(chain_txt_file,comport);
      close(chain_txt_file);
      {$I+};
      if ioresult<>0 then
        begin
          writeln(#7,#7,' ERROR reading from file: ',makeup(doorfilepath),'CHAIN.TXT!');
          halt;
        end;
    end;
end;

procedure tritel_sys;
var
  lcv:integer;                          (* Loop control variable *)
  tritel_sys_file:text;                (* variable of DORINFO1.DEF file *)
  tempstring:string;                    (* Temporary string variable *)
begin
 if not(fileexists(doorfilepath+'tribbs.sys')) then
    begin
      writeln(#7,#7,' ERROR: ',makeup(doorfilepath),'TRIBBS.SYS not found!');
      halt;
    end
  else
    begin
      {$I-};
      assign(tritel_sys_file,doorfilepath+'TRIBBS.SYS');
      reset(tritel_sys_file);
      {$I+};
      if ioresult<>0 then
    	begin
	      writeln(#7,#7,' ERROR opening file:  ',makeup(doorfilepath),'TRIBBS.SYS!');
	      halt;
	    end;
      {$I-};
      readln(tritel_sys_file);
      readln(tritel_sys_file, username);
      readln(tritel_sys_file);
      readln(tritel_sys_file, seclevel);
      readln(tritel_sys_file);
      readln(tritel_sys_file, tempstring);
      if tempstring='Y' then
        ansicolor:=true
      else
        ansicolor:=false;
      readln(tritel_sys_file, minutesleft);
      for lcv:=1 to 3 do
        readln(tritel_sys_file);
      readln(tritel_sys_file,nodenumber);
      readln(tritel_sys_file,comport);
      readln(tritel_sys_file,baudrate);
      for lcv:=1 to 2 do
        readln(tritel_sys_file);
      readln(tritel_sys_file,tempstring);
      if tempstring='Y' then
        errorcorrecting:=true;
      for lcv:=1 to 2 do
        readln(tritel_sys_file);
      readln(tritel_sys_file,useralias);
      close(tritel_sys_file);
      {$I+};
      if ioresult<>0 then
        begin
          writeln(#7,#7,' ERROR reading from file: ',makeup(doorfilepath),'TRIBBS.SYS!');
          halt;
        end;
    end;
end;

procedure rmredrawstatusline;
begin
  display_status_line(true);
end;

procedure display_status_line(cflag : boolean);
var
	x, y : integer;
    att : byte;
    bstring : string;

begin
	if statusline then
    begin
    	cursoroff;
        x := wherex;
        y := wherey;
        att := textattr;
        window(1, 24, 80, 25);
        textattr := $70;
        if cflag then
        	clrscr;
        if not help then
        begin
        	gotoxy(2, 1);
            write(username, ' ');
            gotoxy(40 - length(doorname) div 2, 1);
            write(doorname);
            gotoxy(66, 1);
            write('Time On  : ', rmtimeon:3);
            gotoxy(2, 2);
            write('Security Level: ', seclevel);
            gotoxy(33, 2);
            write('[HOME] For Help');
			if baudrate <> 0 then
            begin
                str(baudrate, bstring);
            	bstring := '(' + bstring + ')';
            end
            else
                bstring := '(LOCAL)';
            gotoxy(55 - length(bstring) div 2, 2);
            write(bstring);
            gotoxy(66, 2);
            write('Time Left :', minutesleft - rmtimeon:3);
        end
        else
        begin
        	gotoxy(2, 1);
            write('[F6] Take 5 Minutes');
            gotoxy(30, 1);
            write('[ALT]+[D] Drop To DOS');
            gotoxy(65, 1);
            write('[F9] Quit Door');
            gotoxy(2, 2);
            write('[F7] Give 5 Minutes');
            gotoxy(30, 2);
            write('[F10] Chat Mode');
        end;
        window(1, 1, 80, 23);
        gotoxy(x, y);
        textattr := att;
        cursoron;
    end;
end;

procedure chat_mode;
var
	c, attr : char;
	x, y, col, row, fore, back, v, r1, c1, r2, c2, k, i, color : integer;
	h, m, s, d : word;
    line, line2 : string;
    tstart, tnow, tdiff : longint;

    procedure chatout(c : char);
    var
    	i, j, v : integer;
		line : array[0..159] of char;

    begin
    	if wherex <> 80 then
        begin
        	rmwrite(c);
            exit;
        end;
        save80(1, wherey, 79, wherey, line);
        v := 156;
        while (line[v] <> ' ') and (v <> 0) do
        begin
        	dec(v);
            dec(v);
        end;
        if v = 0 then
        begin
        	rmwrite(c);
            exit;
        end;
        inc(v);
        inc(v);
        i := (158 - v) div 2;
        for j := 0 to i - 1 do
        	rmwrite(#8 + ' ' + #8);
        rmwriteln(' ');
        while v < 158 do
        begin
        	rmwrite(line[v]);
            inc(v);
            inc(v);
        end;
        rmwrite(c);
    end;

    procedure scrollup(c : integer);
    var
    	i, j : integer;

	begin
    	if c = white then
	       	save80(1, 7, 80, 10, scrollbuffer)
        else
        	save80(1, 18, 80, 21, scrollbuffer);
        if c = white then
        	rmgotoxy(1, 1)
        else
			rmgotoxy(1, 12);
       	for i := 1 to 10 do
        	rmwriteln(#27 + '[K');
        if c = white then
        	rmgotoxy(1, 1)
        else
        	rmgotoxy(1, 12);
		rmsetcolor(c, black);
        for i := 1 to 4 do
        	for j := 1 to 80 do
            	rmwrite(scrollbuffer[(i - 1) * 160 + (j - 1) * 2 + 1]);
	end;

	procedure WrapLine(var s1, s2 : string; k : integer);
    var
    	i, j : integer;

    begin
		s2 := '';
		if k = 32 then
    		exit;
    	i := 0;
        for j := 1 to length(s1) do
        	if s1[j] = ' ' then
            	i := j;
		if i = 0 then
    	begin
			rmwriteln('');
			s2 := s2 + char(k);
			exit;
		end;
    	for j := 1 to length(s1) - i + 1 do
    		rmwrite(#8 + ' ' + #8);
        s2 := copy(s1, i + 1, length(s1)) + char(k);
    	s1 := copy(s1, 1, i - 1);
	end;

    label l1;

begin
	chat := true;
	gettime(h, m, s, d);
    tstart := longint(h) * longint(3600) + longint(m) * longint(60) + longint(s);
	x := wherex;
    y := wherey;
    fore := rmgetforeground;
    back := rmgetbackground;
    save80(1, 1, 80, 25, chatbuffer);
	if ansicolor or rmdetectansi then
    begin
		rmwrite(#12);
        rmgotoxy(1, 11);
		rmsetcolor(YELLOW, BLACK);
		for i := 1 to 80 do
			rmwrite(#205);
        rmgotoxy(33, 11);
		rmwrite('[');
		rmsetcolor(LIGHTGREEN, BLACK);
		rmwrite(' Chat Mode ');
		rmsetcolor(YELLOW, BLACK);
		rmwrite(']');
        rmgotoxy(1, 1);
		rmsetcolor(WHITE, BLACK);
		r1 := 1;
		c1 := 1;
		c2 := 1;
		r2 := 12;
l1:		while TRUE do
        begin
			if keypressed then
            begin
				k := integer(rmreadkey);
				if (k = 12) or (k = 0) then
                	goto l1;
				if color <> WHITE then
                begin
					color := WHITE;
					rmsetcolor(WHITE, BLACK);
                    rmgotoxy(c1, r1);
                end;
				if k = 27 then
                begin
                	chat := false;
                    rmsetcolor(fore, back);
                    rmclrscr;
                    attr := #0;
                    v := 1;
                    for row := 1 to 23 do
                    	for col := 1 to 80 do
                        begin
                            if ansicolor and (attr <> chatbuffer[v + 1]) then
                            begin
                            	attr := chatbuffer[v + 1];
                            	rmwrite(ibmtoansi(integer(attr)));
                            end;
                            rmwrite(chatbuffer[v]);
                            inc(v);
                            inc(v);
                            if (row = 23) and (col = 78) then
                            	col := 80;
                        end;
                    rmsetcolor(fore, back);
                	rmwrite(' ' + #8);
                    rmgotoxy(x, y);
                    exit;
                end;
				if (k = 13) or ((k = 32) and (c1 = 79)) then
                begin
					inc(r1);
					c1 := 1;
					if r1 = 11 then
                    begin
						ScrollUp(WHITE);
						r1 := 5;
                    end;
                    rmgotoxy(c1, r1);
					goto l1;
				end;
				if k = 8 then
                begin
					if c1 <> 1 then
                    begin
						rmwrite(#8 + ' ' + #8);
						dec(c1);
					end;
					goto l1;
				end;
				if c1 <> 80 then
                begin
					rmwrite(char(k));
					inc(c1);
					goto l1;
				end;
                save80(1, r1, 80, r1, scrollbuffer);
                line := '';
                for i := 1 to 79 do
                	line := line + scrollbuffer[(i - 1) * 2 + 1];
				WrapLine(line, line2, k);
				inc(r1);
				c1 := 1;
				if r1 = 11 then
                begin
					ScrollUp(WHITE);
					r1 := 5;
				end;
                rmgotoxy(c1, r1);
                rmwrite(line2);
				c1 := length(line2) + 1;
                goto l1;
			end;
			if (baudrate <> 0) and in_ready then
			begin
				k := integer(rmreadkey);
				if k = 12 then
                	goto l1;
				if color <> LIGHTCYAN then
                begin
					color := LIGHTCYAN;
					rmsetcolor(LIGHTCYAN, BLACK);
                    rmgotoxy(c2, r2);
                end;
				if (k = 13) or ((k = 32) and (c2 = 79)) then
            	begin
					inc(r2);
					c2 := 1;
					if r2 = 22 then
                	begin
						ScrollUp(LIGHTCYAN);
						r2 := 16;
					end;
                    rmgotoxy(c2, r2);
                    goto l1;
                end;
				if k = 8 then
                begin
					if c2 <> 1 then
                    begin
						rmwrite(#8 + ' ' + #8);
						dec(c2);
					end;
                    goto l1;
                end;
				if c2 <> 80 then
                begin
					rmwrite(char(k));
					inc(c2);
					goto l1;
				end;
                save80(1, r2, 80, r2, scrollbuffer);
                line := '';
                for i := 1 to 79 do
                	line := line + scrollbuffer[(i - 1) * 2 + 1];
				WrapLine(line, line2, k);
				inc(r2);
				c2 := 1;
				if r2 = 22 then
                begin
					ScrollUp(LIGHTCYAN);
					r2 := 16;
                end;
                rmgotoxy(c2, r2);
                rmwrite(line2);
				c2 := length(line2) + 1;
			end;
        end;
	end;
	rmsetcolor(lightmagenta, black);
    rmwriteln('');
    rmwriteln('Chat mode entered..........');
    rmwriteln('');
    while true do
    begin
    	if (baudrate <> 0) and not carrier then
			halt;
        gettime(h, m, s, d);
    	tnow := longint(h) * longint(3600) + longint(m) * 60 + longint(s);
		if tnow >= tstart then
        	tdiff := tnow - tstart
		else
        	tdiff := tnow + 86400 - tstart;
        if tdiff > 60 then
        begin
        	display_status_line(false);
			gettime(h, m, s, d);
    		tstart := longint(h) * longint(3600) + longint(m) * longint(60) + longint(s);
        end;
        if keypressed then
        begin
        	rmsetcolor(white, black);
            c := kreadkey;
			case c of
            	#0:
                	;
                #13:
                begin
                	rmwriteln('');
                    rmwriteln('');
                end;
                #27:
                begin
                	chat := false;
                    rmsetcolor(lightmagenta, black);
                    rmwriteln('');
                    rmwriteln('Chat mode ended..........');
                    delay(1000);
                    rmsetcolor(fore, back);
                    rmclrscr;
                    attr := #0;
                    v := 1;
                    for row := 1 to 23 do
                    	for col := 1 to 80 do
                        begin
                            if ansicolor and (attr <> chatbuffer[v + 1]) then
                            begin
                            	attr := chatbuffer[v + 1];
                            	rmwrite(ibmtoansi(integer(attr)));
                            end;
                            rmwrite(chatbuffer[v]);
                            inc(v);
                            inc(v);
                            if (row = 23) and (col = 78) then
                            	col := 80;
                        end;
                    rmsetcolor(fore, back);
                	rmwrite(' ' + #8);
                    rmgotoxy(x, y);
                    exit;
                end;
                else
                	if (c = ' ') or (c = #8) then
						rmwrite(c)
					else
                    	chatout(c);
            end;
        end;
        if (baudrate <> 0) and in_ready then
        begin
        	rmsetcolor(lightcyan, black);
            c := get_serial;
			case c of
            	#13:
                begin
                	rmwriteln('');
                    rmwriteln('');
                end;
            	else
                	if (c = ' ') or (c = #8) then
                    	rmwrite(c)
                    else
                    	chatout(c);
            end;
        end;
    end;
end;

function kreadkey : char;
var
	c, attr : char;
    d, x, y, fore, back, v, row, col : integer;
    cd : string;
    regs : registers;

begin
	c := readkey;
    if c <> #0 then
    begin
    	kreadkey := c;
        exit;
    end;
    kreadkey := #0;
    case readkey of
    	#32:
        begin
            x := wherex;
            y := wherey;
    		fore := rmgetforeground;
    		back := rmgetbackground;
            save80(1, 1, 80, 25, videobuffer);
            rmsetcolor(yellow, black);
            rmwriteln('');
            rmwriteln('Your sysop has dropped to DOS.....');
            regs.ah := $19;
            intr($21, regs);
            d := regs.al;
            getdir(0, cd);
            textattr := 7;
            clrscr;
            writeln('Type EXIT to return to the door.....');
            swapvectors;
            exec(getenv('COMSPEC'), '');
            swapvectors;
            regs.ah := $e;
            regs.dl := d;
            intr($21, regs);
            chdir(cd);
            restore80(1, 1, 80, 25, videobuffer);
            gotoxy(x, y);
            rmsetcolor(fore, back);
            rmclrscr;
            attr := #0;
            v := 1;
            for row := 1 to 23 do
            	for col := 1 to 80 do
                begin
                    if ansicolor and (attr <> videobuffer[v + 1]) then
                    begin
                    	attr := videobuffer[v + 1];
                        rmwrite(ibmtoansi(integer(attr)));
                    end;
                    rmwrite(videobuffer[v]);
                    inc(v);
                    inc(v);
                    if (row = 23) and (col = 78) then
                    	col := 80;
                end;
            rmsetcolor(fore, back);
            rmwrite(' ' + #8);
            rmgotoxy(x, y);
            exit;
        end;
    	#64:
        begin
			minutesleft := minutesleft - 5;
            display_status_line(false);
            exit;
        end;
        #65:
        begin
        	minutesleft := minutesleft + 5;
            display_status_line(false);
            exit;
        end;
        #67:
        	halt;
        #68:
        	if not chat then
            	chat_mode;
    	#71:
        begin
        	help := not help;
            display_status_line(true);
            exit;
        end;
    end;
end;

function rmreadkey : char;
var
	key : char;
    nkey : integer;
	h, m, s, d : word;
    tstart, tnow, tdiff : longint;

begin
	while true do
    begin
		gettime(h, m, s, d);
    	tstart := longint(h) * longint(3600) + longint(m) * longint(60) + longint(s);
        display_status_line(false);
        nkey := 0;
        if baudrate = 0 then
        begin
        	while not keypressed do
            begin
            	gettime(h, m, s, d);
    			tnow := longint(h) * longint(3600) + longint(m) * 60 + longint(s);
				if tnow >= tstart then
            		tdiff := tnow - tstart
				else
            		tdiff := tnow + 86400 - tstart;
            	if tdiff > 60 then
            	begin
            		display_status_line(false);
					gettime(h, m, s, d);
    				tstart := longint(h) * longint(3600) + longint(m) * longint(60) + longint(s);
                    inc(nkey);
                    if nkey = 3 then
                    begin
                        rmgotoxy(1,23);
                    	rmsetcolor(yellow, black);
                    	rmwriteln('');
                        rmwriteln('Keyboard input time has expired!');
                        delay(1000);
                        halt;
                    end;
            	end;
            	if rmtimeleft < 1 then
            	begin
                    rmgotoxy(1,23);
            		rmsetcolor(yellow, black);
                	rmwriteln('');
                	rmwriteln('Sorry, your time has expired!');
                    delay(1000);
                	halt;
            	end;
                {$IFNDEF VER55}
                if multitasker <> 0 then
                	_timeslice;
                {$ENDIF}
            end;
            key := kreadkey;
            if key <> #0 then
            begin
            	rmreadkey := key;
            	exit;
            end;
        end
        else
        begin
        	if not carrier then
             begin
               rmgotoxy(1,23);
               rmsetcolor(yellow,black);
               rmwriteln('');
               rmwriteln('Loss of Carrier Detected!  Returning to BBS!');
               halt;
             end;
        	while not keypressed and not in_ready do
            begin
            	gettime(h, m, s, d);
    			tnow := longint(h) * longint(3600) + longint(m) * 60 + longint(s);
				if tnow >= tstart then
            		tdiff := tnow - tstart
				else
            		tdiff := tnow + 86400 - tstart;
            	if tdiff > 60 then
            	begin
            		display_status_line(false);
					gettime(h, m, s, d);
    				tstart := longint(h) * longint(3600) + longint(m) * longint(60) + longint(s);
                    inc(nkey);
                    if nkey = 3 then
                    begin
                        rmgotoxy(1,23);
                    	rmsetcolor(yellow, black);
                    	rmwriteln('');
                        rmwriteln('Keyboard input time has expired!');
                        delay(1000);
                        halt;
                    end;
            	end;
            	if rmtimeleft < 1 then
            	begin
                    rmgotoxy(1,23);
            		rmsetcolor(yellow, black);
                	rmwriteln('');
                	rmwriteln('Sorry, your time has expired!');
                    delay(1000);
                	halt;
            	end;
                {$IFNDEF VER55}
                if multitasker <> 0 then
                	_timeslice;
                {$ENDIF}
            end;
        	if keypressed then
            begin
            	key := kreadkey;
                if key <> #0 then
                begin
                	rmreadkey := key;
                    exit;
                end;
            end
            else
            begin
            	rmreadkey := get_serial;
                exit;
            end;
        end;
    end;
end;

function rmkeypressed : boolean;
begin
    if baudrate=0 then
      rmkeypressed:=keypressed
    else
	rmkeypressed := keypressed or in_ready;
end;

function rmreads : string;
var
	c : char;
	s : string;

begin
	s := '';
    while true do
    begin
    	c := rmreadkey;
        case c of
        	#8:
            	if s <> '' then
                begin
                	delete(s, length(s), 1);
                    rmwrite(c);
                end;
        	#13:
            begin
            	rmreads := s;
                rmwriteln('');
                exit;
            end;
            else
            begin
            	s := s + c;
                rmwrite(c);
            end;
        end;
    end;
end;

function rmreadssize(sizeofstring:integer) : string;
var
	c : char;
	s : string;
        countofsize:integer;

begin
    if sizeofstring<=0 then
      sizeofstring:=80;
    countofsize:=0;
    s := '';
    while true do
    begin
    	c := rmreadkey;
        case c of
           #8:
             if s <> '' then
             begin
               delete(s, length(s), 1);
               rmwrite(c);
               dec(countofsize);
             end;
           #13:
              begin
            	rmreadssize := s;
                rmwriteln('');
                exit;
              end;
            else
            if countofsize<>sizeofstring then
            begin
            	s := s + c;
                rmwrite(c);
                inc(countofsize);
            end;
        end;
    end;
end;

function rmreadi : integer;
var
	n, code : integer;
	c : char;
	s : string;

begin
	s := '';
    while true do
    begin
    	c := rmreadkey;
        case c of
        	#8:
            	if s <> '' then
                begin
                	delete(s, length(s), 1);
                	rmwrite(c);
            	end;
        	#13:
            begin
            	val(s, n, code);
            	rmreadi := n;
                rmwriteln('');
                exit;
            end;
            else
            begin
            	s := s + c;
                rmwrite(c);
            end;
        end;
    end;
end;

function rmreadl : longint;
var
    n : longint;
	code : integer;
	c : char;
	s : string;

begin
	s := '';
    while true do
    begin
    	c := rmreadkey;
        case c of
        	#8:
            	if s <> '' then
                begin
                	delete(s, length(s), 1);
                	rmwrite(c);
            	end;
        	#13:
            begin
            	val(s, n, code);
            	rmreadl := n;
                rmwriteln('');
                exit;
            end;
            else
            begin
            	s := s + c;
                rmwrite(c);
            end;
        end;
    end;
end;

procedure registerrmdoor(yourname:string;registration_no:string);
var
  thisreg:integer;
  tempstr:string;
  tempstr2:string;
  lcv1:integer;
  thiskey:integer;
  code:integer;
  temp:integer;
begin
  registration_no:=makeup(registration_no);
  lcv1:=1;
  tempstr:='';
  while registration_no[lcv1]<>'-' do
   begin
    tempstr:=tempstr+registration_no[lcv1];
    inc(lcv1);
   end;
  val(tempstr,thisreg,code);
  inc(lcv1);
  tempstr2:='';
  repeat
    tempstr2:=tempstr2+registration_no[lcv1];
    inc(lcv1);
    inc(lcv1);
  until lcv1>length(registration_no);
  val(tempstr2,thiskey,code);
  temp:=thiskey div thisreg;
  if temp=62 then
    reg_no:=thisreg;
end;


procedure rmdisplayfile(filename:string;pause:boolean);
var
  filetodisplay:text;
  tempstring:string;
  lcv:integer;
  linecount:integer;
  nonstop:boolean;
  good:boolean;
  thekey:char;
  fore, back : integer;
begin
  linecount:=0;
  {$I-};
  assign(filetodisplay,filename);
  reset(filetodisplay);
  {$I+};
  if ioresult<>0 then
    exit;   (* File not found *)
  nonstop:=false;
  repeat
    readln(filetodisplay,tempstring);
    rmwriteln(tempstring);
    inc(linecount);
    if pause then
     begin
       if (linecount>=24) and (not nonstop) then
         begin
           linecount:=0;
           fore := rmgetforeground;
           back := rmgetbackground;
           rmsetcolor(white, black);
           rmwrite('<S>top, <N>onstop, <ENTER> to continue');
           rmsetcolor(fore, back);
           good:=false;
           repeat
             thekey:=rmreadkey;
             case upcase(thekey) of
               'S':good:=true;
               'N':good:=true;
               #13:good:=true;
             end;  (* Case *)
           until good;
           if upcase(thekey)='S' then
            begin
              close(filetodisplay);
              exit;
            end;
           if upcase(thekey)='N' then
             nonstop:=true;
           for lcv:=1 to 38 do
               rmwrite(#8+' '+#8);
         end;
    end;
  until eof(filetodisplay);
  close(filetodisplay);
end;

procedure rmdisplayfilelines(filename:string;lines:integer);
var
  filetodisplay:text;
  tempstring:string;
  lcv:integer;
  linecount:integer;
  nonstop:boolean;
  good:boolean;
  thekey:char;
  fore, back : integer;
begin
  if (lines<0) then
    exit;
  linecount:=0;
  {$I-};
  assign(filetodisplay,filename);
  reset(filetodisplay);
  {$I+};
  if ioresult<>0 then
    exit;   (* File not found *)
  nonstop:=false;
  if lines=0 then
    nonstop:=true;
  repeat
    readln(filetodisplay,tempstring);
    rmwriteln(tempstring);
    inc(linecount);
    if (lines<>0) then
     begin
       if (linecount>=lines) and (not nonstop) then
         begin
           linecount:=0;
           fore := rmgetforeground;
           back := rmgetbackground;
           rmsetcolor(white, black);
           rmwrite('<S>top, <N>onstop, <ENTER> to continue');
           rmsetcolor(fore, back);
           good:=false;
           repeat
             thekey:=rmreadkey;
             case upcase(thekey) of
               'S':good:=true;
               'N':good:=true;
               #13:good:=true;
             end;  (* Case *)
           until good;
           if upcase(thekey)='S' then
            begin
              close(filetodisplay);
              exit;
            end;
           if upcase(thekey)='N' then
             nonstop:=true;
           for lcv:=1 to 38 do
               rmwrite(#8+' '+#8);
         end;
    end;
  until eof(filetodisplay);
  close(filetodisplay);
end;

procedure sendmodem(s: string);
var
	i : integer;

begin
    i := 1;
    while i <= length(s) do
    begin
    	case s[i] of
        	'~':
            	delay(500);
            '^':
        	begin
            	inc(i);
                if i <= length(s) then
                	put_serial(char(integer(s[i]) - 64));
            end;
            else
            	put_serial(s[i]);
        end;
        inc(i);
        delay(50);
    end;
end;


procedure rmhangup;
var
	i : integer;
    last : word;
    clock : ^word;

begin
    if baudrate=0 then
      exit;
	clock := ptr($00, $46c);
    last := clock^;
    i := 180;
    set_dtr(false);
    while carrier and (i <> 0) do
    begin
    	if last <> clock^ then
        begin
        	dec(i);
            last := clock^;
        end;
    end;
    set_dtr(true);
    if not carrier then
    	exit;
   	sendmodem('~~~+++~~~ATH0^M');
    i := 180;
    while carrier and (i <> 0) do
	begin
    	if last <> clock^ then
        begin
        	dec(i);
            last := clock^;
        end;
    end;
end;

function timedmodemchar(n : integer) : integer;
var
	i : integer;
    last : word;
    clock : ^word;

begin
	clock := ptr($00, $46c);
    last := clock^;
    i := n * 18;
    while true do
	begin
		if not carrier then
			halt;
        if in_ready then
        begin
        	timedmodemchar := integer(get_serial);
            exit;
        end;
    	if last <> clock^ then
        begin
        	dec(i);
            last := clock^;
            if i <= 0 then
            begin
            	timedmodemchar := -1;
                exit;
            end;
        end;
    end;
end;

function rmdetectansi : boolean;
var
	i : integer;
    detect : string;

begin
	detect := #27 + '[6n' + #8+ ' ' + #8 + #8 + ' ' + #8 + #8 + ' ' + #8 + #8 + ' ' + #8;
    if baudrate = 0 then
    begin
    	rmdetectansi := true;
        exit;
    end;
    while carrier and in_ready do
    	i := integer(get_serial);
   	rmwrite(detect);
    if baudrate < 2400 then
    begin
    	i := timedmodemchar(6);
        if i <> 27 then
        begin
        	rmdetectansi := false;
            exit;
        end;
    end
    else
    begin
    	i := timedmodemchar(3);
        if i <> 27 then
        begin
        	rmdetectansi := false;
            exit;
        end;
    end;
    while carrier and (timedmodemchar(1) <> -1) do;
    rmdetectansi := true;
end;

function rmdetectrip : boolean;
var
	i : integer;
    detect : string;

begin
	detect := #27 + '[!' + #8 + ' ' + #8 + #8 + ' ' + #8 + #8 + ' ' + #8;
    if baudrate = 0 then
    begin
    	rmdetectrip := false;
        exit;
    end;
    while carrier and in_ready do
    	i := integer(get_serial);
   	rmwrite(detect);
    if baudrate < 2400 then
    begin
    	i := timedmodemchar(6);
        if i <> 82 then
        begin
        	rmdetectrip := false;
            exit;
        end;
    end
    else
    begin
    	i := timedmodemchar(3);
        if i <> 82 then
        begin
        	rmdetectrip := false;
            exit;
        end;
    end;
    while carrier and (timedmodemchar(1) <> -1) do;
    rmdetectrip := true;
end;

procedure rmcopyright;
begin
   	rmwriteln('');
   	rmsetcolor(lightred, black);
   	rmwrite(#174 + ' ');
   	rmsetcolor(lightmagenta, black);
   	rmwrite('RMDoor ');
   	rmsetcolor(yellow, black);
   	rmwrite('4.5 ');
   	rmsetcolor(lightblue, black);
   	rmwrite('- ');
   	rmsetcolor(lightgreen, black);
   	rmwrite('Reg #');
   	rmwritei(reg_no);
   	rmsetcolor(lightblue, black);
   	rmwrite(' - ');
   	rmsetcolor(yellow, black);
   	rmwrite('Copyright 1991-1993 By ');
   	rmsetcolor(lightcyan, black);
   	rmwrite('Mark Goodwin & Randy Hunt');
   	rmsetcolor(lightred, black);
   	rmwriteln(' ' + #175);
   	rmwriteln(' ');
end;

var
	d : word;
    exitsave : pointer;

{$F+}
procedure rmdoorexit;
begin
  exitproc := exitsave;
  if (baudrate = 0) or carrier then
    if reg_no = 0 then
       begin
      	 rmsetcolor(10,0);
 	 rmwrite('RMDOOR 4.5 ');
         if not configthere then
           rmwriteln('')
         else
           begin
	     rmsetcolor(9,0);
	     rmwrite('- ');
             rmsetcolor(12,8);
	     rmwrite('UNREGISTERED');
	     rmsetcolor(10,0);
	     rmwriteln(' SHAREWARE COPY!');
           end;
    	 rmsetcolor(14,0);
         rmwriteln('Copyright (c) 1991-1993 by Mark Goodwin and Randy Hunt');
         rmsetcolor(7,0);
 	 delay(2000);
       end;
    rmsetcolor(7, 0);
    delay(1000);
    close_port;
    textattr := 7;
    window(1, 24, 80, 25);
    clrscr;
    exitproc := nil;
end;

var
	dummy : integer;

begin
	{$IFNDEF VER55}
    multitasker := _detect_multitasker;
    {$ENDIF}
    configthere:=false;
	if LastMode = Mono then
		VideoOffset := $B000;
    read_config;
    exitsave := exitproc;
    exitproc := @rmdoorexit;
  	if pos(' ', username) <> 0 then
  		userfirstname := copy(username, 1, pos(' ', username) - 1)
  	else
  		userfirstname := username;
  	if paramcount > 1 then
    begin
		val(paramstr(2), nonstandardirq, dummy);
        if (nonstandardirq < 0) or (nonstandardirq > 15) then
        begin
      		writeln(#7,#7,' ERROR: Invalid Nonstandard IRQ!');
      		halt;
        end;
    end;
	open_port(comport);
	if lockedrate = 0 then
    	set_port(baudrate, 8, NO_PARITY, 1)
    else
    	set_port(lockedrate, 8, NO_PARITY, 1);
{    exitsave := exitproc;
    exitproc := @rmdoorexit;}
    gettime(timeonh, timeonm, timeons, d);
    checksnow := FALSE;
    textattr := 7;
    clrscr;
    window(1, 1, 80, 23);
    display_status_line(true);
end.
