/*---------------------------------------------------------------------------*/
/*                    Opus Janus revision 0.22,  1- 9-88                     */
/*                                                                           */
/*                The Opus Computer-Based Conversation System                */
/*           (c) Copyright 1987, Rick Huebner, All Rights Reserved           */
/*---------------------------------------------------------------------------*/
#define  WAZOO_SECTION
#include <sys\types.h>
#include <sys\stat.h>

#include "bbs.h"
#include "janus.h"

#define NothingToSend_msg  (_log[0x3f])


/* External routines needed */
extern void pascal unique_name(byte *);
extern int  pascal JANUS_RawByte(void);
extern int  pascal JANUS_GetCookedByte(void);
extern void pascal xfer_summary(byte *,byte *,byte *,long *,int,int);
extern void pascal update_status(long *,long *,int,int *,long,int *,int,int);
extern void pascal through(long *,long *,int);
#ifdef RICK
extern FILE *f_append(byte *);
#endif

/* External global data needed */
extern byte *ext_flags;
extern byte *Resume_name;
extern byte *Resume_info;
extern byte *Abortlog_name;
extern byte *XFER_str;
extern char *KBD_msg;
extern byte  Resume_WaZOO;
extern byte  isOriginator;
extern int   remote_net;
extern int   remote_node;
extern word  remote_capabilities;
extern long  total_bytes;

/* Private routines */
LOCAL void pascal getfname(word);
LOCAL void pascal sendpkt(byte *,int,int);
LOCAL void pascal txbyte(int);
LOCAL long pascal procfname(void);
LOCAL byte pascal rcvpkt(void);
LOCAL void pascal rxclose(word);
LOCAL void pascal endbatch(void);
LOCAL void cdecl  j_message(byte,byte *,...);
LOCAL void cdecl  j_status(byte *,...);
LOCAL int  pascal j_error(byte *,byte *);
LOCAL void pascal update_y(void);
LOCAL void pascal long_set_timer(long *,word);
LOCAL int  pascal long_time_gone(long *);


/* Private data. I know, proper software design says you shouldn't make data */
/* global unless you really need to.  In this case speed and code size make  */
/* it more important to avoid constantly pushing & popping arguments.        */

static byte *JANUS_err     = "!Janus: %s";
static byte *JANUS_trace   = "#Janus: %s";
static byte *GENERIC_ERROR = "E%s";
static byte *Txbuf;        /* Address of packet transmission buffer          */
static byte *Rxbuf;        /* Address of packet reception buffer             */
static byte *Txfname;      /* Full path of file we're sending                */
static byte *Rxfname;      /* Full path of file we're receiving              */
static byte *Rxbufptr;     /* Current position within packet reception buffer*/
static byte *Rxbufmax;     /* Upper bound of packet reception buffer         */
static byte  Do_after;     /* What to do with file being sent when we're done*/
static int   Txfile;       /* File handle of file we're sending              */
static int   Rxfile;       /* File handle of file we're receiving            */
static word  Rxblklen;     /* Length of data in last data block packet recvd */
static word  Next_y;       /* Number of next available line on screen        */
static word  Tx_y;         /* Line number of file transmission status display*/
static word  Rx_y;         /* Line number of file reception status display   */
static long  Txlen;        /* Total length of file we're sending             */
static long  Rxpos;        /* Current position within file we're receiving   */
static long  Rxlen;        /* Total length of file we're receiving           */
static long  Diskavail;    /* Bytes available in upload directory            */


/*--------------------------------------------------------------------------*/
/* GLOBAL DATA                                                              */
/*--------------------------------------------------------------------------*/
byte  JANUS_WaitFlag;      /*Tells JANUS_RawByte() whether or not to wait   */
word  JANUS_TimeoutSecs;   /* How long to wait for various things           */



/*****************************************************************************/
/* Super-duper neato-whizbang full-duplex streaming ACKless batch file       */
/* transfer protocol for use in Opus-to-Opus mail sessions                   */
/*****************************************************************************/
void pascal Janus()
{
  byte   xstate;           /* Current file transmission state                */
  byte   rstate;           /* Current file reception state                   */
  byte   pkttype;          /* Type of packet last received                   */
  int    txoldlen;         /* Last transmitted data block length displayed   */
  int    txoldeta;         /* Last transmission ETA displayed                */
  int    rxoldlen;         /* Last received data block length displayed      */
  int    rxoldeta;         /* Last reception ETA displayed                   */
  word   blklen;           /* Length of last data block sent                 */
  word   txblklen;         /* Size of data block to try to send this time    */
  word   txblkmax;         /* Max size of data block to send at this speed   */
  word   goodneeded;       /* # good blocks to send before upping txblklen   */
  word   goodblks;         /* Number of good blocks sent at this block size  */
  word   rpos_count;       /* Number of RPOS packets sent at this position   */
  long   xmit_retry;       /* Time to retransmit lost FNAMEPKT or EOF packet */
  long   txpos;            /* Current position within file we're sending     */
  long   lasttx;           /* Position within file of last data block we sent*/
  long   starttime;        /* Time at which we started this Janus session    */
  long   txstpos;          /* Initial data position of file we're sending    */
  long   txsttime;         /* Time at which we started sending current file  */
  long   rxstpos;          /* Initial data position of file we're receiving  */
  long   rxsttime;         /* Time at which we started receiving current file*/
  long   txoldpos;         /* Last transmission file position displayed      */
  long   rxoldpos;         /* Last reception file position displayed         */
  long   rpos_retry;       /* Time at which to retry RPOS packet             */
  long   brain_dead;       /* Time at which to give up on other computer     */
  long   rpos_sttime;      /* Time at which we started current RPOS sequence */
  long   last_rpostime;    /* Timetag of last RPOS which we performed        */
  long   last_blkpos;      /* File position of last out-of-sequence BLKPKT   */


  XON_DISABLE();
  message(NULL);
  bbs_status("-WaZOO method: Janus");
  set_xy(NULL);
  Next_y = locate_y;
  Tx_y = Rx_y = total_bytes = 0;
  time(&starttime);

  /*-------------------------------------------------------------------------*/
  /* Allocate memory                                                         */
  /*-------------------------------------------------------------------------*/
  Txbuf = Rxbuf = Txfname = Rxfname = NULL;
  if (!WZ_InitResume()) goto freemem;
  if ( !(Txbuf=malloc(BUFMAX+8))  || !(Rxbuf=malloc(BUFMAX+8))
   ||  !(Txfname=malloc(PATHLEN)) || !(Rxfname=malloc(PATHLEN)) ) {
    no_memory("J");
    goto freemem;
  }
  Rxbufmax = Rxbuf+BUFMAX+8;

  /*-------------------------------------------------------------------------*/
  /* Initialize file transmission variables                                  */
  /*-------------------------------------------------------------------------*/
  last_rpostime=
  xmit_retry   = 0L;
  if (caller_baud < 300) caller_baud = 300;
  JANUS_TimeoutSecs = 40960/caller_baud;
  if (JANUS_TimeoutSecs < 10) JANUS_TimeoutSecs = 10;
  long_set_timer(&brain_dead,120);
  txblkmax     = caller_baud/300 * 128;
  if (txblkmax > BUFMAX) txblkmax = BUFMAX;
  txblklen     = txblkmax;
  goodblks     = 0;
  goodneeded   = 3;
  Txfile       = -1;
  xstate       = XSENDFNAME;
  getfname(INITIAL_XFER);
  if (!Txfname[0]) j_status(NothingToSend_msg,
                            ctl.alias[0].net, ctl.alias[0].node,
                            remote_net,       remote_node );

  /*-------------------------------------------------------------------------*/
  /* Initialize file reception variables                                     */
  /*-------------------------------------------------------------------------*/
  sprintf(Abortlog_name,"%s%04x%04x.Z",ctl.hold_area,remote_net,remote_node);
  if (ctl.filepath[1]==':')
     Diskavail = freespace(toupper(ctl.filepath[0])-'@');
  else Diskavail = 0x7FFFFFFF;
  Rxbufptr = NULL;
  rpos_retry = rpos_count = 0;
  rstate = RRCVFNAME;

  /*-------------------------------------------------------------------------*/
  /* Send and/or receive stuff until we're done with both                    */
  /*-------------------------------------------------------------------------*/
  do {   /*  while (xstate || rstate)  */

    /*-----------------------------------------------------------------------*/
    /* If nothing useful (i.e. sending or receiving good data block) has     */
    /* happened within the last 2 minutes, give up in disgust                */
    /*-----------------------------------------------------------------------*/
    if (long_time_gone(&brain_dead)) {
      j_status("EOther end died");  /* "He's dead, Jim." */
      goto giveup;
    }

    /*-----------------------------------------------------------------------*/
    /* If we're tired of waiting for an ACK, try again                       */
    /*-----------------------------------------------------------------------*/
    if (xmit_retry) {
      if (long_time_gone(&xmit_retry)) {
        j_message(Tx_y,TIME_msg);
        xmit_retry = 0L;

        switch (xstate) {
          case XRCVFNACK:
            xstate = XSENDFNAME;
            break;
          case XRCVEOFACK:
            dseek(Txfile, txpos=lasttx, SEEK_SET);
            if (j_error(SEEK_msg,Txfname)) goto giveup;
            xstate = XSENDBLK;
            break;
        }
      }
    }

    /*-----------------------------------------------------------------------*/
    /* Transmit next part of file, if any                                    */
    /*-----------------------------------------------------------------------*/
    switch (xstate) {
      case XSENDBLK:
        *((long *)Txbuf) = lasttx = txpos;
        blklen = dread(Txfile, Txbuf+sizeof(txpos), txblklen);
        if (j_error(READ_msg,Txfname)) goto giveup;
        txpos += blklen;
        sendpkt(Txbuf, sizeof(txpos)+blklen, BLKPKT);
        update_status(&txpos, &txoldpos,
                      blklen, &txoldlen,
                      Txlen-txpos, &txoldeta, JANUS_EFFICIENCY, Tx_y);
        if (txpos >= Txlen || blklen < txblklen) {
          long_set_timer(&xmit_retry,JANUS_TimeoutSecs);
          xstate  = XRCVEOFACK;
        } else long_set_timer(&brain_dead,120);

        if (txblklen < txblkmax && ++goodblks > goodneeded) {
          txblklen <<= 1;
          goodblks = 0;
        }
        break;

      case XSENDFNAME:
        blklen = strchr( strchr(Txbuf,'\0')+1, '\0') - Txbuf + 1;
        sendpkt(Txbuf,blklen,FNAMEPKT);
        txoldpos = txoldlen = txoldeta = -1;
        long_set_timer(&xmit_retry,JANUS_TimeoutSecs);
        xstate = XRCVFNACK;
        break;
    }

    /*-----------------------------------------------------------------------*/
    /* Catch up on our reading; receive and handle all outstanding packets   */
    /*-----------------------------------------------------------------------*/
    while (pkttype = rcvpkt()) {
      switch (pkttype) {

        /*-------------------------------------------------------------------*/
        /* File data block or munged block                                   */
        /*-------------------------------------------------------------------*/
        case BADPKT:
        case BLKPKT:
          if (rstate == RRCVBLK) {
            if (pkttype == BADPKT || *((long *)Rxbuf) != Rxpos) {
              if (pkttype == BLKPKT) {
                if (*((long *)Rxbuf) < last_blkpos) rpos_count = 0;
                last_blkpos = *((long *)Rxbuf);
              }
              if (long_time_gone(&rpos_retry))  {
                if (rpos_count > 5) rpos_count = 0;
                if (++rpos_count == 1) time(&rpos_sttime);
                j_message(Rx_y,"Bad packet at %ld", Rxpos);
                *((long *)Rxbuf) = Rxpos;
                *((long *)(Rxbuf+sizeof(Rxpos))) = rpos_sttime;
                sendpkt(Rxbuf, sizeof(Rxpos)+sizeof(rpos_sttime), RPOSPKT);
                long_set_timer(&rpos_retry,JANUS_TimeoutSecs/2);
              }
            } else {
              long_set_timer(&brain_dead,120);
              last_blkpos = Rxpos;
              rpos_retry = rpos_count = 0;
              dwrite(Rxfile, Rxbuf+sizeof(Rxpos), Rxblklen -= sizeof(Rxpos));
              if (j_error(WRITE_msg,Rxfname)) goto giveup;
              Diskavail -= Rxblklen;
              Rxpos += Rxblklen;
              update_status(&Rxpos, &rxoldpos,
                            Rxblklen,&rxoldlen,
                            Rxlen-Rxpos,&rxoldeta,JANUS_EFFICIENCY,Rx_y);
              if (Rxpos >= Rxlen) {
                Rxlen -= rxstpos;
                through(&Rxlen,&rxsttime,Rx_y);
                rxclose(GOOD_XFER);
                j_status("=UL-J %s",Rxfname);
                rstate = RRCVFNAME;
              }
            }
          }
          if (rstate == RRCVFNAME) sendpkt(NULL,0,EOFACKPKT);
          break;

        /*-------------------------------------------------------------------*/
        /* Name and other data for next file to receive                      */
        /*-------------------------------------------------------------------*/
        case FNAMEPKT:
          if (rstate == RRCVFNAME) Rxpos = rxstpos = procfname();
          sendpkt((byte *)&Rxpos,sizeof(Rxpos),FNACKPKT);
          time(&rxsttime);
          rxoldpos = rxoldlen = rxoldeta = -1;
          if (Rxpos > -1) rstate = (Rxfname[0])? RRCVBLK: RDONE;
          else j_status("!Rejecting file: %s",Rxfname);
          if (!(xstate || rstate)) goto breakout;
          break;

        /*-------------------------------------------------------------------*/
        /* ACK to filename packet we just sent                               */
        /*-------------------------------------------------------------------*/
        case FNACKPKT:
          if (xstate == XRCVFNACK) {
            xmit_retry = 0L;
            if (Txfname[0]) {
              if ((txpos = *((long *)Rxbuf)) > -1L) {
                dseek(Txfile, txstpos = txpos, SEEK_SET);
                if (j_error(SEEK_msg,Txfname)) goto giveup;
                time(&txsttime);
                xstate = XSENDBLK;
              } else {
                j_status("!File rejected: %s",Txfname);
                Do_after = NOTHING_AFTER;
                getfname(GOOD_XFER);
                xstate = XSENDFNAME;
              }
            } else xstate = XDONE;
          }
          if (!(xstate || rstate)) goto breakout;
          break;

        /*-------------------------------------------------------------------*/
        /* ACK to last data block in file                                    */
        /*-------------------------------------------------------------------*/
        case EOFACKPKT:
          if (xstate == XRCVEOFACK || xstate == XRCVFNACK) {
            xmit_retry = 0L;
            if (xstate == XRCVEOFACK) {
              Txlen -= txstpos;
              through(&Txlen,&txsttime,Tx_y);
              j_status("=DL-J %s",Txfname);
              getfname(GOOD_XFER);
            }
            xstate = XSENDFNAME;
          }
          break;

        /*-------------------------------------------------------------------*/
        /* Receiver says "let's try that again."                             */
        /*-------------------------------------------------------------------*/
        case RPOSPKT:
          if (xstate == XSENDBLK || xstate == XRCVEOFACK) {
            if (*((long *)(Rxbuf+sizeof(txpos))) != last_rpostime) {
              last_rpostime = *((long *)(Rxbuf+sizeof(txpos)));
              xmit_retry = 0L;
              CLR_OUTBOUND();
              dseek(Txfile, txpos = lasttx = *((long *)Rxbuf), SEEK_SET);
              if (j_error(SEEK_msg,Txfname)) goto giveup;
              j_message(Tx_y,"Resending from %ld", txpos);
              if (txblklen >= 128) txblklen >>= 1;
              goodblks = 0;
              goodneeded = goodneeded<<1 | 1;
              xstate = XSENDBLK;
            }
          }
          break;

        /*-------------------------------------------------------------------*/
        /* Debris from end of previous Janus session; ignore it              */
        /*-------------------------------------------------------------------*/
        case HALTACKPKT:
          break;

        /*-------------------------------------------------------------------*/
        /* Abort the transfer and quit                                       */
        /*-------------------------------------------------------------------*/
        default:
          j_status("EUnknown packet type %d",pkttype);
          /* fallthrough */
        case HALTPKT:
giveup:   j_status("ESession aborted");
          mdm_hangup();
          if (Txfname[0]) getfname(ABORT_XFER);
          if (rstate == RRCVBLK) {
            total_bytes += (Rxpos-rxstpos);
            rxclose(FAILED_XFER);
          }
          goto breakout;

      }  /*  switch (pkttype)  */
    }  /*  while (pkttype)  */
  } while (xstate || rstate);

  /*-------------------------------------------------------------------------*/
  /* All done; make sure other end is also finished (one way or another)     */
  /*-------------------------------------------------------------------------*/
breakout:
  gotoxy(1,Next_y);
  cputs("  Session ");
  through(&total_bytes,&starttime,0);
  update_y();
  endbatch();

  /*-------------------------------------------------------------------------*/
  /* Release allocated memory                                                */
  /*-------------------------------------------------------------------------*/
freemem:
  if (Txbuf)   free(Txbuf);
  if (Rxbuf)   free(Rxbuf);
  if (Txfname) free(Txfname);
  if (Rxfname) free(Rxfname);
  WZ_DisableResume();
}



/*****************************************************************************/
/* Get name and info for next file to be transmitted, if any, and build      */
/* FNAMEPKT.  Packet contents as per ZModem filename info packet, to allow   */
/* use of same method of aborted-transfer recovery.  If there are no more    */
/* files to be sent, build FNAMEPKT with null filename.  Also open file and  */
/* set up for transmission.  Set Txfname, Txfile, Txlen.  Txbuf must not be  */
/* modified until FNACKPKT is received.                                      */
/*****************************************************************************/
LOCAL void pascal getfname(xfer_flag)
word xfer_flag;
{
  static byte floflag, bad_xfers, outboundname[PATHLEN];
  static long floname_pos;
  static FILE *flofile;
  register byte *p;
  int i, hr, mi, se, th;
  long curr_pos;
  struct stat f;

  /*-------------------------------------------------------------------------*/
  /* Initialize static variables on first call of the batch                  */
  /*-------------------------------------------------------------------------*/
  if (xfer_flag == INITIAL_XFER) {
    floflag = outboundname[0] = '\0';
    flofile = NULL;
  /*-------------------------------------------------------------------------*/
  /* If we were already sending a file, close it and clean up                */
  /*-------------------------------------------------------------------------*/
  } else if (Txfile != -1) {
    dclose(Txfile);
    j_error(CLOSE_msg,Txfname);
    Txfile = -1;
    /*-----------------------------------------------------------------------*/
    /* If xfer completed, do post-xfer cleanup                               */
    /*-----------------------------------------------------------------------*/
    if (xfer_flag == GOOD_XFER) {
      /*---------------------------------------------------------------------*/
      /* Perform post-xfer file massaging if neccessary                      */
      /*---------------------------------------------------------------------*/
      switch (Do_after) {
        case DELETE_AFTER:
          j_status("#Delete-after: %s",Txfname);
          unlink(Txfname);
          j_error(UNLINK_msg,Txfname);
          break;
        case TRUNC_AFTER:
          j_status("#Trunc-after: %s",Txfname);
          Txfile = dcreat(Txfname,0);
          j_error(TRUNC_msg,Txfname);
          dclose(Txfile);
          Txfile = -1;
      }
      /*---------------------------------------------------------------------*/
      /* If processing .?LO file, flag filename as sent (name[0] = '~')      */
      /*---------------------------------------------------------------------*/
skipname:
      if (floflag) {
        curr_pos = ftell(flofile);           j_error(SEEK_msg, outboundname);
        fseek(flofile,floname_pos,SEEK_SET); j_error(SEEK_msg, outboundname);
        fputc(Txfname[0] = '~',flofile);     j_error(WRITE_msg,outboundname);
        fseek(flofile,curr_pos,SEEK_SET);    j_error(SEEK_msg, outboundname);
      }
    } else {
abort:
      ++bad_xfers;
    }
  }

  /*-------------------------------------------------------------------------*/
  /* Find next file to be sent and build FNAMEPKT.  If reading .FLO-type     */
  /* file get next entry from it; otherwise check for next .OUT/.FLO file    */
  /*-------------------------------------------------------------------------*/
  if (!floflag) {
    /*-----------------------------------------------------------------------*/
    /* If first getfname() for this batch, init filename to .REQ             */
    /*-----------------------------------------------------------------------*/
    if (!outboundname[0]) {
      sprintf(outboundname, "%s%04x%04x.REQ", ctl.hold_area, remote_net,
       remote_node);
      if (!(remote_capabilities & WZ_FREQ)) {
        if (dexists(outboundname))
         j_status("*F.REQ. declined: %s",outboundname);
        goto nxtout;
      }
    /*-----------------------------------------------------------------------*/
    /* Increment outbound filename until match found or all checked          */
    /* .REQ->.OUT->.DUT->.CUT->.HUT->.FLO->.DLO->.CLO->.HLO->null name       */
    /*-----------------------------------------------------------------------*/
    } else {
nxtout:
      p = strchr(outboundname,'\0')-3;
      if (!strcmp(p,"REQ")) {
        strcpy(p,"OUT");
        *ext_flags = 'O';
      } else {
        for (i=0; i<NUM_FLAGS; ++i) if (ext_flags[i] == *p) break;
        if (i < NUM_FLAGS-1) {
          *p = ext_flags[i+1];
          /* Don't send held mail on our dime */
          if (isOriginator && *p == 'H') goto nxtout;
        } else {
          /*-----------------------------------------------------------------*/
          /* Finished ?,D,C,H sequence; wrap .OUT->.FLO, or .FLO->done       */
          /*-----------------------------------------------------------------*/
          if (!floflag) {
            *p++ = *ext_flags = 'F';
            *p++ = 'L';
            *p = 'O';
            ++floflag;
          } else outboundname[0]=Txfname[0]=Txbuf[0]=Txbuf[1]=floflag='\0';
        }
      }
    }
    /*-----------------------------------------------------------------------*/
    /* Check potential outbound name; if file doesn't exist keep looking     */
    /*-----------------------------------------------------------------------*/
    if (outboundname[0]) {
      if (!dexists(outboundname)) goto nxtout;
      if (floflag) goto rdflo;
      strcpy(Txfname,outboundname);
      p = strchr(outboundname,'\0')-3;
      /*---------------------------------------------------------------------*/
      /* If .REQ file, start FNAMEPKT using simple filename                  */
      /*---------------------------------------------------------------------*/
      if (!strcmp(p,"REQ")) strcpy(Txbuf,outboundname+strlen(ctl.hold_area));
      /*---------------------------------------------------------------------*/
      /* If .?UT-type file, start FNAMEPKT using .PKT alias                  */
      /*---------------------------------------------------------------------*/
      else invent_pkt_name(Txbuf);
      Do_after = DELETE_AFTER;
    }
  /*-------------------------------------------------------------------------*/
  /* Read and process next entry from .?LO-type file                         */
  /*-------------------------------------------------------------------------*/
  } else {
rdflo:
    /*-----------------------------------------------------------------------*/
    /* Open .?LO file for processing if neccessary                           */
    /*-----------------------------------------------------------------------*/
    if (!flofile) {
      errno = bad_xfers = 0;
      _fmode = O_TEXT;           /* Gyrations are to avoid MSC 4.0 "r+t" bug */
      flofile = fopen(outboundname,"r+");
      _fmode = O_BINARY;
      if (j_error(OPEN_msg,outboundname)) goto nxtout;
    }
    floname_pos = ftell(flofile);
    j_error(SEEK_msg,outboundname);
    if (fgets(p = Txfname, PATHLEN, flofile)) {
      /*---------------------------------------------------------------------*/
      /* Got an attached file name; check for handling flags, fix up name    */
      /*---------------------------------------------------------------------*/
      while (*p > ' ') ++p;
      *p = '\0';
      switch (Txfname[0]) {
        case '\0':
        case '~':
        case ';':
          goto rdflo;
        case TRUNC_AFTER:
        case DELETE_AFTER:
          Do_after = Txfname[0];
          strcpy(Txfname,Txfname+1);
          break;
        default:
          Do_after = NOTHING_AFTER;
          break;
      }
      /*---------------------------------------------------------------------*/
      /* Start FNAMEPKT with simple filename                                 */
      /*---------------------------------------------------------------------*/
      while (p>=Txfname && *p!='\\' && *p!=':') --p;
      strcpy(Txbuf,++p);
    } else {
      /*---------------------------------------------------------------------*/
      /* Finished reading this .?LO file; clean up and look for another      */
      /*---------------------------------------------------------------------*/
      errno = 0;
      fclose(flofile);
      j_error(CLOSE_msg,outboundname);
      flofile = NULL;
      if (!bad_xfers) {
        unlink(outboundname);
        j_error(UNLINK_msg,outboundname);
      }
      goto nxtout;
    }
  }

  /*-------------------------------------------------------------------------*/
  /* If we managed to find a valid file to transmit, open it, finish         */
  /* FNAMEPKT, and print nice message for the sysop.                         */
  /*-------------------------------------------------------------------------*/
  if (Txfname[0]) {
    if (xfer_flag == ABORT_XFER) goto abort;
    j_status(":Sending %s",Txfname);
    Txfile = dopen(Txfname,O_RDONLY);
    if (j_error(OPEN_msg,Txfname)) goto skipname;

    stat(Txfname,&f);
    sprintf(p = strchr(Txbuf,'\0')+1, "%lu %lo %o",
     Txlen = f.st_size, f.st_mtime, f.st_mode);

    xfer_summary("J-Send",empty_str,Txfname,&Txlen,JANUS_EFFICIENCY,Next_y);
    update_y();
    cputs(XFER_str);
    update_y();
    Tx_y = Next_y - 1;
  }
}


/*****************************************************************************/
/* Build and send a packet of any type.                                      */
/* Packet structure is: PKTSTRT,contents,packet_type,PKTEND,crc              */
/* CRC is computed from contents and packet_type only; if PKTSTRT or PKTEND  */
/* get munged we'll never even find the CRC.                                 */
/*****************************************************************************/
LOCAL void pascal sendpkt(buf,len,type)
register byte *buf;
int len, type;
{
  register word crc;

  SENDBYTE(DLE);
  SENDBYTE(PKTSTRTCHR^0x40);

  crc=0;
  while (--len >= 0) {
    txbyte(*buf);
    crc = xcrc(crc, ((word)(*buf++)) );
  }

  SENDBYTE(type);
  crc = xcrc(crc,type);

  SENDBYTE(DLE);
  SENDBYTE(PKTENDCHR^0x40);

  txbyte(crc>>8);
  txbyte(crc&0xFF);
}


/*****************************************************************************/
/* Transmit cooked escaped byte(s) corresponding to raw input byte.  Escape  */
/* DLE, XON, and XOFF using DLE prefix byte and ^ 0x40. Also escape          */
/* CR-after-'@' to avoid Telenet/PC-Pursuit problems.                        */
/*****************************************************************************/
LOCAL void pascal txbyte(c)
register int c;
{
  static int lastsent;

  switch (c) {
    case CR:
      if (lastsent != '@') goto sendit;
      /* fallthrough */
    case DLE:
    case XON:
    case XOFF:
      SENDBYTE(DLE);
      c ^= 0x40;
      /* fallthrough */
    default:
sendit: SENDBYTE(lastsent = c);
  }
}


/*****************************************************************************/
/* Process FNAMEPKT of file to be received.  Check for aborted-transfer      */
/* recovery and solve filename collisions.  Check for enough disk space.     */
/* Return initial file data position to start receiving at, or -1 if error   */
/* detected to abort file reception.  Set Rxfname, Rxlen, Rxfile.            */
/*****************************************************************************/
LOCAL long pascal procfname()
{
  register byte *p;
  byte linebuf[64], *fileinfo, *badfname;
  long filestart, bytes;
  FILE *abortlog;
  struct stat f;

  /*-------------------------------------------------------------------------*/
  /* Initialize for file reception                                           */
  /*-------------------------------------------------------------------------*/
  Rxfname[0] = Resume_WaZOO = 0;

  /*-------------------------------------------------------------------------*/
  /* If this is a null FNAMEPKT, return OK immediately                       */
  /*-------------------------------------------------------------------------*/
  if (!Rxbuf[0]) return 0L;

  /*-------------------------------------------------------------------------*/
  /* Save info on WaZOO transfer in case of abort                            */
  /*-------------------------------------------------------------------------*/
  strcpy(Resume_name,fancy_str(Rxbuf));
  strcpy(Resume_info,(fileinfo=strchr(Rxbuf,'\0')+1));

  /*-------------------------------------------------------------------------*/
  /* Keep somebody from uploading a .{B|G}BS file                            */
  /*-------------------------------------------------------------------------*/
  if ( (index(Rxbuf,".Bbs")) || (index(Rxbuf,".Gbs")) ) *(fileinfo-2) = 'X';

  /*-------------------------------------------------------------------------*/
  /* Check if this is a failed WaZOO transfer which should be resumed        */
  /*-------------------------------------------------------------------------*/
  if (dexists(Abortlog_name)) {
    errno = 0;
    abortlog = fopen(Abortlog_name, read_ascii);
    if (!j_error(OPEN_msg,Abortlog_name)) {
      while (!feof(abortlog)) {
        linebuf[0] = '\0';
        if (!fgets(p = linebuf, 64, abortlog)) break;
        while (*p >= ' ') ++p;
        *p = '\0';
        p = strchr(linebuf,' ');
        *p = '\0';
        if (!stricmp(linebuf,Resume_name)) {
          p = strchr( (badfname = ++p), ' ');
          *p = '\0';
          if (!stricmp(++p,Resume_info)) {
            ++Resume_WaZOO;
            break;
          }
        }
      }
      fclose(abortlog);
      j_error(CLOSE_msg,Abortlog_name);
    }
  }

  /*-------------------------------------------------------------------------*/
  /* Open either the old or a new file, as appropriate                       */
  /*-------------------------------------------------------------------------*/
  p = strchr(strcpy(Rxfname,ctl.filepath),'\0');
  if (Resume_WaZOO) {
    strcpy(p,badfname);
    Rxfile = dopen(Rxfname,O_RDWR);
  } else {
    strcpy(p,Rxbuf);
    unique_name(Rxfname);
    if (strcmp(p,Rxbuf)) j_status("+Dupe renamed: %s",p);
    Rxfile = dcreat(Rxfname,0);
  }
  if (j_error(OPEN_msg,Rxfname)) return -1L;

  /*-------------------------------------------------------------------------*/
  /* Determine initial file data position                                    */
  /*-------------------------------------------------------------------------*/
  if (Resume_WaZOO) {
    j_status(":Resuming partial file: %s",Rxbuf);
    stat(Rxfname,&f);
    dseek(Rxfile, filestart = f.st_size, SEEK_SET);
    if (j_error(SEEK_msg,Rxfname)) {
      dclose(Rxfile);
      return -1L;
    }
  } else filestart = 0L;

  /*-------------------------------------------------------------------------*/
  /* Extract and validate filesize                                           */
  /*-------------------------------------------------------------------------*/
  sscanf(fileinfo, "%ld", &Rxlen);
  bytes = Rxlen-filestart+10240;
  if (bytes > Diskavail) {
    j_status("!Disk space: need %ld, have %ld",bytes,Diskavail);
    dclose(Rxfile);
    return -1L;
  }

  /*-------------------------------------------------------------------------*/
  /* Print status message for the sysop                                      */
  /*-------------------------------------------------------------------------*/
  p = check_netfile(Rxbuf);
  if (!p) p = empty_str;
  xfer_summary("J-Recv",p,Rxfname,&Rxlen,JANUS_EFFICIENCY,Next_y);
  update_y();
  cputs(XFER_str);
  update_y();
  Rx_y = Next_y - 1;

  return filestart;
}


/*****************************************************************************/
/* Receive, validate, and extract a packet if available.  If a complete      */
/* packet hasn't been received yet, receive and store as much of the next    */
/* packet as possible.  Each call to rcvpkt() will continue accumulating a   */
/* packet until a complete packet has been received or an error is detected. */
/* Rxbuf must not be modified between calls to rcvpkt() if NOPKT is returned.*/
/* Returns type of packet received, NOPKT, or BADPKT.  Sets Rxblklen.        */
/*****************************************************************************/
LOCAL byte pascal rcvpkt()
{
  static word crc;
  register byte *p;
  register int c;
  word pktcrc;

  /*-------------------------------------------------------------------------*/
  /* Abort transfer if operator pressed ESC                                  */
  /*-------------------------------------------------------------------------*/
  if (KBD_GiveUp()) {
    j_status(GENERIC_ERROR,KBD_msg);
    return HALTPKT;
  }

  /*-------------------------------------------------------------------------*/
  /* If not accumulating packet yet, find start of next packet               */
  /*-------------------------------------------------------------------------*/
  JANUS_WaitFlag = 0;
  if (!(p = Rxbufptr)) {
    do c = JANUS_GetCookedByte();
    while (c >= 0 || c == PKTEND);

    switch (c) {
      case PKTSTRT:
        p = Rxbuf;
        crc = 0;
        break;
      case NOCARRIER:
        j_status(GENERIC_ERROR,CARRIER_msg);
        return HALTPKT;
      default:
        return NOPKT;
    }
  }

  /*-------------------------------------------------------------------------*/
  /* Accumulate packet data until we empty buffer or find packet delimiter   */
  /*-------------------------------------------------------------------------*/
  while ((c=JANUS_GetCookedByte()) >= 0 && p < Rxbufmax)
    crc = xcrc(crc, ((word)(*p++ = c)) );

  /*-------------------------------------------------------------------------*/
  /* Handle whichever end-of-packet condition occurred                       */
  /*-------------------------------------------------------------------------*/
  switch (c) {
    /*-----------------------------------------------------------------------*/
    /* PKTEND found; verify valid CRC                                        */
    /*-----------------------------------------------------------------------*/
    case PKTEND:
      JANUS_WaitFlag = 1;
      if ((c=JANUS_GetCookedByte()) >= 0) {
        pktcrc = c<<8;
        if ((c=JANUS_GetCookedByte()) >= 0) {
          if ((pktcrc|c) == crc) {
            /*---------------------------------------------------------------*/
            /* Good packet verified; compute packet data length and return   */
            /* packet type                                                   */
            /*---------------------------------------------------------------*/
            Rxbufptr = NULL;
            Rxblklen = --p - Rxbuf;
            return *p;
          }
        }
      }
      /* fallthrough */

    /*-----------------------------------------------------------------------*/
    /* Bad CRC, carrier lost, or buffer overflow from munged PKTEND          */
    /*-----------------------------------------------------------------------*/
    default:
      if (c == NOCARRIER) {
        j_status(GENERIC_ERROR,CARRIER_msg);
        return HALTPKT;
      } else {
        Rxbufptr = NULL;
        return BADPKT;
      }

    /*-----------------------------------------------------------------------*/
    /* Emptied buffer; save partial packet and let sender do something       */
    /*-----------------------------------------------------------------------*/
    case BUFEMPTY:
      Rxbufptr = p;
      return NOPKT;

    /*-----------------------------------------------------------------------*/
    /* PKTEND was trashed; discard partial packet and prep for next packet   */
    /*-----------------------------------------------------------------------*/
    case PKTSTRT:
      Rxbufptr = Rxbuf;
      crc = 0;
      return BADPKT;
  }
}



/*****************************************************************************/
/* Close file being received and perform post-reception aborted-transfer     */
/* recovery cleanup if neccessary.                                           */
/*****************************************************************************/
LOCAL void pascal rxclose(xfer_flag)
word xfer_flag;
{
  register byte *p;
  byte namebuf[PATHLEN], linebuf[64], c;
  FILE *abortlog, *newlog;

  /*-------------------------------------------------------------------------*/
  /* Close file we've been receiving                                         */
  /*-------------------------------------------------------------------------*/
  dclose(Rxfile);
  j_error(CLOSE_msg,Rxfname);

  /*-------------------------------------------------------------------------*/
  /* If we completed a previously-aborted transfer, remove log entry & rename*/
  /*-------------------------------------------------------------------------*/
  if (xfer_flag==GOOD_XFER && Resume_WaZOO) {
    abortlog = fopen(Abortlog_name, read_ascii);
    if (!j_error(OPEN_msg,Abortlog_name)) {
      c = 0;
      strcpy(strchr(strcpy(namebuf,Abortlog_name),'\0')-1,"TMP");
      newlog = fopen(namebuf, write_ascii);
      if (!j_error(OPEN_msg,namebuf)) {
        while (!feof(abortlog)) {
          linebuf[0] = '\0';
          if (!fgets(p = linebuf, 64, abortlog)) break;
          while (*p > ' ') ++p;
          *p = '\0';
          if (stricmp(linebuf,Resume_name)) {
            *p = ' ';
            fputs(linebuf,newlog);
            if (j_error(WRITE_msg,namebuf)) break;
            ++c;
          }
        }
        fclose(abortlog);
        j_error(CLOSE_msg,Abortlog_name);
        fclose(newlog);
        j_error(CLOSE_msg,namebuf);
        unlink(Abortlog_name);
        j_error(UNLINK_msg,Abortlog_name);
        if (c) {
          rename(namebuf,Abortlog_name);
          j_error(RENAME_msg,namebuf);
        } else {
          unlink(namebuf);
          j_error(UNLINK_msg,namebuf);
        }
      } else {
        fclose(abortlog);
        j_error(CLOSE_msg,Abortlog_name);
      }
    }
    j_status("*Finished partial file: %s",Resume_name);
    unique_name(strcat(strcpy(namebuf,ctl.filepath),Resume_name));
    rename(Rxfname,namebuf);
    j_error(RENAME_msg,Rxfname);
  /*-------------------------------------------------------------------------*/
  /* If transfer failed and was not an attempted resumption, log it for later*/
  /*-------------------------------------------------------------------------*/
  } else if (xfer_flag==FAILED_XFER && !Resume_WaZOO) {
    j_status("*Saving partial file: %s",Rxfname);
    unique_name(strcat(strcpy(namebuf,ctl.filepath),"BadWaZOO.001"));
    rename(Rxfname,namebuf);
    j_error(RENAME_msg,Rxfname);

    abortlog = f_append(Abortlog_name);
    if (!j_error(OPEN_msg,Abortlog_name)) {
      fprintf(abortlog,"%s %s %s\n", Resume_name, namebuf+strlen(ctl.filepath), Resume_info);
      j_error(WRITE_msg,Abortlog_name);
      fclose(abortlog);
      j_error(CLOSE_msg,Abortlog_name);
    } else {
      unlink(namebuf);
      j_error(UNLINK_msg,namebuf);
    }
  }
}


/*****************************************************************************/
/* Try REAL HARD to disengage batch session cleanly                          */
/*****************************************************************************/
LOCAL void pascal endbatch()
{
  register int done, timeouts;
  long timeval, brain_dead;

  /*-------------------------------------------------------------------------*/
  /* Tell the other end to halt if it hasn't already                         */
  /*-------------------------------------------------------------------------*/
  done = timeouts = 0;
  long_set_timer(&brain_dead,120);
  goto reject;

  /*-------------------------------------------------------------------------*/
  /* Wait for the other end to acknowledge that it's halting                 */
  /*-------------------------------------------------------------------------*/
  while (!done) {
    if (long_time_gone(&brain_dead)) break;

    switch (rcvpkt()) {
      case NOPKT:
      case BADPKT:
        if (long_time_gone(&timeval)) {
          if (++timeouts > 2) ++done;
          else goto reject;
        }
        break;

      case HALTPKT:
      case HALTACKPKT:
        ++done;
        break;

      default:
        timeouts = 0;
reject: sendpkt(NULL,0,HALTPKT);
        long_set_timer(&timeval,JANUS_TimeoutSecs);
        break;
    }
  }

  /*-------------------------------------------------------------------------*/
  /* Announce quite insistently that we're done now                          */
  /*-------------------------------------------------------------------------*/
  for (done=0; done<10; ++done) sendpkt(NULL,0,HALTACKPKT);

  wait_for_clear();
}


/*****************************************************************************/
/* Print a message in the message field of a transfer status line            */
/*****************************************************************************/
LOCAL void cdecl j_message(y,fmt,arg1)
byte  y;
byte *fmt;
int   arg1;
{
  byte  buf[100];
  int  *args;

  if (y) gotoxy(MSG_X,y);
  if (fmt) {
    args = &arg1;
    FORMAT_STRING(buf,fmt,&args,0);
    cputs(buf);
  }
  local_CEOL();
}


/*****************************************************************************/
/* Print & log status message without messing up display                     */
/*****************************************************************************/
LOCAL void cdecl j_status( fmt, arg1 )
byte *fmt;
int   arg1;
{
  byte  buf[100];
  int  *args;
  byte  c;

  args  = &arg1;

  if (((c=fmt[0])=='T') or (c=='E')) fmt++;

  FORMAT_STRING(buf,fmt,&args,0);

  gotoxy(1,Next_y-1);
  bbs_status( (c=='T')? JANUS_trace:
              (c=='E')? JANUS_err:
                        buf,
              buf);

  if (!locate_y) update_y();
}


/*****************************************************************************/
/* Print & log error message without messing up display                      */
/*****************************************************************************/
LOCAL int pascal j_error(msg,fname)
byte *msg, *fname;
{
  register int e;

  if (e = errno) {
    gotoxy(1,Next_y-1);
    bbs_error(msg,fname);
    if (!locate_y) update_y();
  }
  return e;
}


/*****************************************************************************/
/* Update screen position variables after printing a message                 */
/*****************************************************************************/
LOCAL void pascal update_y()
{
  set_xy(NULL);
  if (locate_y == Next_y) {    /* If we didn't go anywhere, screen scrolled; */
    if (--Rx_y < 1) Rx_y = 1;    /* so decrement status line numbers */
    if (--Tx_y < 1) Tx_y = 1;
  } else Next_y = locate_y;
}


/*****************************************************************************/
/* Compute future timehack for later reference                               */
/*****************************************************************************/
LOCAL void pascal long_set_timer(Buffer,Duration)
long *Buffer;
word  Duration;
{
  time(Buffer);
  *Buffer += (long)Duration;
}


/*****************************************************************************/
/* Return TRUE if timehack has been passed, FALSE if not                     */
/*****************************************************************************/
LOCAL int pascal long_time_gone(TimePtr)
long *TimePtr;
{
  return (time(NULL) > *TimePtr);
}




#ifdef xxxNEVER



      The stuff from here to the end of the file is now in J_MISC.ASM...




/*****************************************************************************/
/* Receive cooked escaped byte translated to avoid various problems.         */
/* Returns raw byte, BUFEMPTY, PKTSTRT, PKTEND, or NOCARRIER.                */
/*****************************************************************************/
LOCAL int pascal rcvbyte()
   begin
      register int c, w;

      if ((c = JANUS_RawByte()) == DLE)
         begin
            w = JANUS_WaitFlag++;
            if ((c = JANUS_RawByte()) >= 0)
               begin
                  switch (c ^= 0x40)
                     begin
                        case PKTSTRTCHR:  c = PKTSTRT;
                                          break;
                        case PKTENDCHR:   c = PKTEND;
                                          break;
                     end
               end
            JANUS_WaitFlag = w;
         end
      return c;
   end



/*****************************************************************************/
/* Receive raw non-escaped byte.  Returns byte, BUFEMPTY, or NOCARRIER.      */
/* If waitflag is true, will wait for a byte for Timeoutsecs; otherwise      */
/* will return BUFEMPTY if a byte isn't ready and waiting in inbound buffer. */
/*****************************************************************************/
LOCAL int pascal rcvrawbyte()
{
  register int c;
  long timeval;

  if ((c=READBYTE()) >= 0) return c;

  if (!CARRIER()) return NOCARRIER;
  if (!JANUS_WaitFlag)  return BUFEMPTY;

  timeval = time(NULL) + JANUS_TimeoutSecs;

  while ((c=READBYTE()) < 0) {
    if (!CARRIER())           return NOCARRIER;
    if (time(NULL) > timeval) return BUFEMPTY;
    time_release();
  }

  return c;
}

#endif
