#undef DEBUG
/*
 * COPYRIGHT 1992 SAMUEL H. SMITH
 * ALL RIGHTS RESERVED
 *
 * THIS DOCUMENT CONTAINS CONFIDENTIAL INFORMATION AND TRADE SECRETS
 * PROPRIETARY TO SAMUEL H. SMITH DBA THE TOOL SHOP.
 *
 */


/*
 * hsintr.c - HS/Link low level interrupt handlers
 *
 * This module provides lowest level interrupt handlers for
 * the COM port.
 *
 * This module also implements full xon/xoff handshaking (in both
 * directions) as well as CTS and RTS handshake.
 *
 */

#pragma inline
#undef PROFILE

#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <conio.h>

#include "\hdk\hspriv.h"
#include <hdk.h>
#include "hsl.h"

#include "uart.h"
#include "hsintr.h"

void pascal LL_service_DAV(void);
void pascal LL_service_THRE(void);

/*
 * The following include file defines RESWITCH macro, which is generated by
 * a scan of HSINTR.ASM using FINDJMP.EXE.  The idea is to produce the exact
 * indirect jump used originally in the SWITCH statement.  You may need to
 * rerun FINDJMP whenever logic is changed in this procedure.
 */

#include "hsintr2.h"

/* use the switch() jump table to vector to the proper handler
   for the current IIR register contents */

#define VECTOR_TO_ISR                                   \
        _DX = ComLL.IirBase;                            \
        asm IN AL,DX;                                   \
        _AL &= (IIR_PENDING|IIR_MASK);                  \
        _BX = _AL;                                      \
        asm SHL BX,1;                                   \
        asm RESWITCH

/* ------------------------------------------------------------ */
/*
 * low-level interrupt service routine.  this procedure processes all
 * receive-ready and transmit-ready interrupts from the async chip.
 * DO NOT call this proc directly.
 *
 */

void /*interrupt*/ LL_InterruptHandler(void)
{
        /* enable interrupts only when profiling */
disable();
#ifdef PROFILE
        enable();
#endif
        asm PUSH AX;    /* used everywhere */
        asm PUSH BX;    /* used in enque/deque */
        asm PUSH CX;
        asm PUSH DX;    /* base port pointer */
        asm PUSH DS;    /* must reset data segment */
        asm MOV AX,DGROUP;
        asm MOV DS,AX;

        VECTOR_TO_ISR;

        /* this switch is never executed, but the jump table produced
           by the compiler is used above to vector to the proper ISR */

        switch (_BX)
        {

        /* ------------------------------------------------------------ */
        case IIR_DAV:   /* process DAV interrupts */
                LL_service_DAV();
                VECTOR_TO_ISR;

        /* ------------------------------------------------------------ */
        case IIR_THRE:  /* process THRE interrupts */
                LL_service_THRE();
                VECTOR_TO_ISR;

        /* ------------------------------------------------------------ */
        /* these interrupts are not enabled, but the compiler makes better
           code for the switch if it thinks they are */
        case IIR_MSR:
        case IIR_LSR:
        case 7:
                _DX = ComLL.LsrBase;
                asm IN AL,DX;
                LL_iodelay;             /*???D7*/
                VECTOR_TO_ISR;

        /* ------------------------------------------------------------ */
        default:        /* interrupt service complete- signal EOI and exit */

        #ifdef DEBUG
                ++ComLL.IsrCount;
        #endif
                _AL = IACK_NEOI;        /* non-specific EOI */
                _DX = ComLL.IctlBase+IACKR;
                asm OUT DX,AL;

                asm CMP DX,ICTL1_BASE;
                asm JNE eoi_high_irq;

        cont_eoi_high_irq:
                asm POP DS;
                asm POP DX;
                asm POP CX;
                asm POP BX;
                asm POP AX;
                asm IRET;
                /* return; */

        eoi_high_irq:
                /* send EOI to other interrupt controller for high IRQ interrupts */
                _AL = IACK_NEOI;
                asm OUT ICTL1_BASE+IACKR,AL;
                goto cont_eoi_high_irq;

        } /* end switch */
}

/* ------------------------------------------------------------ */
/* grab foreground cs:ip and store it in ErrorLocation */

#define CAPTURE_ERROR_LOCATION                          \
        asm PUSH BP;                                    \
        asm MOV BP,SP;                                  \
        asm MOV BX,[BP+14];                             \
        asm MOV WORD PTR ComLL.ErrorLocation,BX;        \
        asm MOV BX,[BP+16];                             \
        asm MOV WORD PTR ComLL.ErrorLocation+2,BX;      \
        asm POP BP

/* ------------------------------------------------------------ */
 /*
  * low-level service for DAV interrupt.  Call only from interrupt service.
  *
  */

void pascal LL_service_DAV(void)
{
        _DX = ComLL.LsrBase;
        asm IN AL,DX;
        _AH = _AL;
        if ((_AH & LSR_DAV) == 0)
        {
TRACE('A',_AH);
                asm ret;
        }

        LL_iodelay;             /*???D7*/
        _DX = ComLL.ComBase;
        asm IN AL,DX;           /* get RBR */
TRACE('B',_AL);

        if (_AH & (LSR_OERR|LSR_PERR|LSR_FERR|LSR_BREAK))
                goto got_rxerr;

cont_rxerr:
        /* check for xon/xoff flow control */
        if (WS.Option.XonHandshake != 0)
                goto check_special;

cont_check_special:
        /* assert flow control if past high water mark */
        _BX = RXQUE.qcount;
        if (_BX > QHIGH_WATER)
                goto check_high;

cont_check_high:
        /* place the character into the RXQUE */
        ++_BX;
        if (_BX >= RXQ_SIZE)
                goto rxq_overflow;

cont_rxq_overflow:
        RXQUE.qcount = _BX;

        _BX = RXQUE.qnext_in;
        RXQUE_qdata[_BX] = _AL;
        ++_BX;
        RXQUE.qnext_in = _BX;
        if (_BX >= RXQ_SIZE)
                goto wrap_rxq;

        asm ret;

/* ------------------------------------------------------------ */
/* the following codes handle exceptions and have been moved
   out of the mainline to reduce flushing of the cpu's prefetch que */

got_rxerr:
        ComLL.RxErrorBits |= _AH;
        CAPTURE_ERROR_LOCATION;
        goto cont_rxerr;

rxq_overflow:
        ComLL.RxErrorBits |= RXQ_OVERFLOW_BIT;
        --_BX;
        CAPTURE_ERROR_LOCATION;
        goto cont_rxq_overflow;

check_special:
        /* xon/xoff handshake on transmitter */
        if (_AL == XON_CHR)
                goto got_xon;
        if (_AL == XOFF_CHR)
                goto got_xoff;
        if (WS.Option.AlternateDle == 0)
                goto cont_check_special;

        if (_AL == (XON_CHR+0x80))
                goto got_xon;
        if (_AL == (XOFF_CHR+0x80))
                goto got_xoff;
        goto cont_check_special;

got_xon:
        ComLL.TXoffActive = 0;
        LL_startTransmit();
        asm ret;

got_xoff:
        ComLL.TXoffActive = 1;
        asm ret;

wrap_rxq:
        RXQUE.qnext_in = 0;
        asm ret;

check_high:
        if ((WS.Option.RtsHandshake != 0) && RtsActive)
                LL_lowerRTS();
        if ((WS.Option.XonHandshake != 0) && (ComLL.RXoffActive == 0))
                LL_SendXoff();
        goto cont_check_high;
}


/* ------------------------------------------------------------ */
 /*
  * low-level service for THRE interrupt.  Call only from interrupt
  * service or from LL_startTransmit.
  *
  */

void pascal LL_service_THRE(void)
{
       /* send next byte only if que has data */
        _CX = TXQUE.qcount;
        asm JCXZ stop_transmit;

        /* immediately send any pending priority codes (xon/xoff) */
        if (ComLL.TxPriority != 0)
                goto do_priority;

        /* honor CTS and XON/XOFF flow control for normal byte codes */
        if (ComLL.TXoffActive != 0)
                goto check_hold;

        _DX = ComLL.MsrBase;
        asm IN AL,DX;
        if ((WS.Option.CtsHandshake != 0) && ((_AL & MSR_CTS) == 0))
                goto check_hold;

        /* make sure THRE is really present before queueing the byte */
        LL_iodelay;             /*???D7*/
        _DX = ComLL.LsrBase;
        asm IN AL,DX;
        if ((_AL & (LSR_THRE|LSR_TSRE)) == 0)
                goto ret_THRE;

        /* deque next character and place in uart THR */
        _BX = TXQUE.qnext_out;
        _AL = TXQUE_qdata[_BX];
        _DX = ComLL.ComBase;
        LL_iodelay;             /*ok*/
        asm OUT DX,AL;
TRACE('a',_AL);
        ComLL.XmitActive = 1;

        ++_BX;
        if (_BX >= TXQ_SIZE)
                _BX = 0;

        --_CX;

/* check for multiple-character-transmit uart types */

        asm JCXZ fin_THRE;      /* quit if buffer empty */

        _AH = WS.Option.FifoThresh;
        if (_AH > 15)
                goto THRE_forval;
        if (_AH > 0)
                goto THRE_16550;

fin_THRE:
        TXQUE.qnext_out = _BX;
        TXQUE.qcount = _CX;

ret_THRE:
        asm ret;

/************************************************/

check_hold:
        if (ComLL.XmitActive != 1)
                ++WS.Comstat.TransmitHolds;
        ComLL.XmitHeld = 1;

stop_transmit:
        ComLL.XmitActive = 0;
#ifdef DEBUG
        if (ComLL.XmitActive != 1)
                ++ComLL.StallCount;
#endif
        asm ret;

do_priority:
        _AL = ComLL.TxPriority;
        ComLL.TxPriority = 0;

        _DX = ComLL.ComBase;
        asm OUT DX,AL;
TRACE('b',_AL);

        ComLL.XmitActive = 1;

        --_CX;
        TXQUE.qcount = _CX;
        asm ret;

/* enque additional characters for transmit when ns16550an is detected */

THRE_16550:
next_16550:
        asm DEC AH;
        asm JZ fin_16550;

#ifdef DEBUG
        ++ComLL.MulTxCount;
#endif

        _AL = TXQUE_qdata[_BX];
        asm OUT DX,AL;
TRACE('c',_AL);

        ++_BX;
        if (_BX >= TXQ_SIZE)
                _BX = 0;

        asm LOOP next_16550;

fin_16550:
        TXQUE.qnext_out = _BX;
        TXQUE.qcount = _CX;
        asm ret;

/* enque additional characters for transmit when forval internal is used. */
/* this logic may also work under windows enhanced mode and os/2 or other */
/* protected enivironments that provide a buffered uart emulation */

THRE_forval:
        asm SUB AH,14;          /* make -ft16 = 1 preque and up */

next_forval:
        asm DEC AH;
        asm JZ fin_forval;

wait_forval:
        LL_iodelay;             /*ok*/
        _DX = ComLL.LsrBase;
        asm IN AL,DX;
        if ((_AL & (LSR_THRE|LSR_TSRE)) == 0)
                goto wait_forval;

        LL_iodelay;             /*ok*/
        _AL = TXQUE_qdata[_BX];
        _DX = ComLL.ComBase;
        asm OUT DX,AL;
TRACE('d',_AL);

#ifdef DEBUG
        ++ComLL.MulTxCount;
#endif

        ++_BX;
        if (_BX >= TXQ_SIZE)
                _BX = 0;

        asm LOOP next_forval;

fin_forval:
        TXQUE.qnext_out = _BX;
        TXQUE.qcount = _CX;
        asm ret;
}


/* ------------------------------------------------------------ */

 /*
  * low-level service to kickstart transmit interrupt.  Called only
  * from LL_PollTransmit macro.
  *
  */

void pascal LL_startTransmit(void)
{
        /* this function needed only if xmit is not currently active */
        if (ComLL.XmitActive == 1)
                asm ret;

#ifndef PROFILE
        asm PUSHF;
        disable();
#endif
        asm PUSH AX;
        asm PUSH BX;
        asm PUSH CX;
        asm PUSH DX;
        asm PUSH ES;

        ComLL.XmitHeld = 0;
        LL_service_THRE();

        asm POP ES;
        asm POP DX;
        asm POP CX;
        asm POP BX;
        asm POP AX;
#ifndef PROFILE
        asm POPF;
#endif
}


/* ------------------------------------------------------------ */

/* send XOFF to the remote computer */

void pascal LL_SendXoff(void)
{
        asm PUSHF;
        disable();

        if (ComLL.TxPriority == 0)
        {
                disable();
                ++TXQUE.qcount;
                enable();
        }

        ComLL.TxPriority = XOFF_CHR;

        ComLL.RXoffActive = 1;
        ++WS.Comstat.ReceiveHolds;
        asm POPF;
}


/* ------------------------------------------------------------ */

/* send XON to the remote computer */

void pascal LL_SendXon(void)
{
        asm PUSHF;
        disable();

        if (ComLL.TxPriority == 0)
        {
                disable();
                ++TXQUE.qcount;
                enable();
        }
        ComLL.TxPriority = XON_CHR;

        ComLL.RXoffActive = 0;

        asm POPF;
        LL_startTransmit();
}

/* ------------------------------------------------------------ */

/* lower RTS handshake */

void pascal LL_lowerRTS(void)
{
        asm PUSH DX;
        asm PUSH AX;

        RtsActive = 0;
        _DX = ComLL.ComBase;
        _DX += MCR;
        asm IN AL,DX;

        LL_iodelay;             /*ok*/
        LL_iodelay;             /*ok*/
        LL_iodelay;             /*ok*/
        _AL = _AL & ~MCR_RTS;
        asm OUT DX,AL;

        asm POP AX;
        asm POP DX;
}

/* ------------------------------------------------------------ */

/* raise RTS handshake */

void pascal LL_raiseRTS(void)
{
        asm PUSH DX;
        asm PUSH AX;

        RtsActive = 1;
        _DX = ComLL.ComBase;
        _DX += MCR;
        asm IN AL,DX;

        LL_iodelay;             /*ok*/
        LL_iodelay;             /*ok*/
        LL_iodelay;             /*ok*/
        _AL = _AL | MCR_RTS;
        asm OUT DX,AL;

        asm POP AX;
        asm POP DX;
}


/* ------------------------------------------------------------ */

#ifdef DEBUG
void pascal LL_addtrace(void)
{
        asm PUSH BX;
        Trace[trace_next].event = _DL;
        Trace[trace_next].data = _DH;
        if (trace_next < TRACE_MAX)
                ++trace_next;
        asm POP BX;
}
#endif

