/* mlm.c - preliminary version 0	*/
/*  message library maintenance	*/
/*	Copyright (c) 1991 John W. Valentyn	*/
/*		all rights reserved		*/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <io.h>
#include <string.h>
#include <ctype.h>
#include "opus.h"
#include "nfile.h"
#include "ml.h"
#include "cbtree.h"
#include "sub1.h"
#include "sub2.h"

/* function prototypes */
void		do_update(void);
void		do_build(void);
void		add_rec(byte *, char *, size_t);
void		mod_rec(byte *, char *, long, size_t);
void		del_rec(char *);
int _fastcall	key_valid(byte *, size_t);
void		cmdline(int,byte**);
void		usage(void);

/* global storage */
int     mldat;
int	quiet = FALSE;
int	verbose = FALSE;
int	update_mode = FALSE;
int	cntr = 0;
int	add_cntr = 0;
int	mod_cntr = 0;
int	del_cntr = 0;
Cbtree	*mlidx;
FILE	*infp;
char	*ml_path;
char	*inp_name = NULL;
char	ml_fname[] = "mlfile.";
char	lang_key[] = "en";
byte	mlbuf[ML_SIZE+1];

/* ====================================================================
 * main
 * ====================================================================
 */
void cdecl
main(int argc,char *argv[])
{
    char	*tpath;

    get_sys_path();

    if(argc > 1)
	cmdline(argc,argv);
    else
        usage();

    if(cbinit(0,0) != OK) {
	cbprterr();
	abort();
    }

    tpath = get_cbuf(_MAX_PATH);
    /* MLFILE.EN */
    sprintf(tpath,"%s%s%s",langdir,ml_fname,lang_key);
    if(access(tpath,0) == 0) {		/* if exists */
	mldat = open(tpath,O_BINARY|O_RDWR);
	if(mldat < 0) {
	    printf("Unable to open `%s'\n",tpath);
	    perror(NULL);
	    abort();
	}
	update_mode = TRUE;
    }
    else {		/* doesn't exist */
	mldat = open(tpath,O_BINARY|O_CREAT|O_RDWR,S_IREAD|S_IWRITE);
	if(mldat < 0) {
	    printf("Unable to create %s\n",tpath);
	    perror(NULL);
	    abort();
	}
    }

    /* MLFILE.ENX */
    strcat(tpath,"x");
    if(update_mode) {
	if((mlidx = cbopen(tpath,DEFAULT_COMPARE)) == NULL) {
	    printf("Unable to open `%s'\n",tpath);
	    cbprterr();
	    abort();
	}
    }
    else {
	mlidx = cbcreate(tpath,DEFAULT_COMPARE,DEFAULT_BLKSIZE);
	if(mlidx == NULL) {
	    cbprterr();
	    printf("Unable to create %s\n",tpath);
	    abort();
	}
    }

    if ((infp=fopen(inp_name,"r")) == NULL) {
	printf("unable to open %s\n",inp_name);
	perror(NULL);
	abort();
    }
    if(update_mode)
	do_update();
    else
        do_build();

    if(add_cntr)
	printf("%d records added\n",add_cntr);
    if(mod_cntr)
	printf("%d records changed\n",mod_cntr);
    if(del_cntr)
	printf("%d records deleted\n",del_cntr);

    if(cbflush(mlidx) != OK) {
	cbprterr();
    }
    (void)cbexit();
    (void)close(mldat);
    (void)cbclose(mlidx);
    (void)fclose(infp);
    exit(0);
}	/* main() */

/* ====================================================================
 * update an existing msg library
 * ====================================================================
 */
void
do_update(void)
{
    int		status;
    size_t	len;
    long	dup_pos;
    char	key[KEY_SZ+1];
    byte	ibuf[ML_SIZE+1];

    printf("update mode\n");
    while((fgets(ibuf,ML_SIZE,infp)) != NULL) {
	cntr++;
	stb(ibuf);	/* strip trailing blanks */
	memcpy(key,ibuf,KEY_SZ);
	key[KEY_SZ] = '\0';
	(void)strupr(key);
	len = strlen(ibuf);
	if(key_valid(ibuf,len)) {

	    /* if there's no text in the record, must be a delete */
	    if(len <= TEXT) {
		del_rec(key);
		del_cntr++;
		continue;
	    }

	    status = cbfind(mlidx,key,KEY_SZ,&dup_pos);
	    if(cbflush(mlidx) != OK) {
		cbprterr();
		abort();
	    }
	    if(status == FOUND) {
		mod_rec(ibuf,key,dup_pos,len);
		mod_cntr++;
	    } /* FOUND */
	    else {
		add_rec(ibuf,key,len);
		add_cntr++;
	    }
	}	/* if(key_valid()) */

    }	/* end while(fgets) */
    return;
}	/* do_update() */

/* ====================================================================
 * build a new msg library from scratch
 * ====================================================================
 */
void
do_build(void)
{
    size_t	len;
    long	posit;
    char	key[KEY_SZ+1];
    byte	ibuf[ML_SIZE+1];

    printf("build mode\n");
    while((fgets(ibuf,ML_SIZE,infp)) != NULL) {
	stb(ibuf);	/* strip trailing blanks */
	memcpy(key,ibuf,KEY_SZ);
	key[KEY_SZ] = '\0';
	(void)strupr(key);
	len = strlen(ibuf)+1;
	if(!key_valid(ibuf,len))
	    continue;

	add_cntr++;
	posit = tell(mldat);
	if(write(mldat,ibuf,len) == -1) {
	    perror("write dat file");
	    abort();
	}
	if ((cbinsert(mlidx,key,KEY_SZ,posit,NO)) != OK) {
	    printf("%s\n",ibuf);
	    cbprterr();
	    abort();
	}
	if((add_cntr % 100) == 0) {	/* flush every 100 records */
	    if(cbflush(mlidx) != OK) {
		cbprterr();
		abort();
	    }
	}
    }	/* end while(fgets) */
    return;
}	/* do_build() */

/* ====================================================================
 * add a new record to the msg library
 * ====================================================================
 */
void
add_rec(byte *ibuf, char *key, size_t len)
{
    long	posit;

    if((posit = lseek(mldat,0L,SEEK_END)) == -1L) {
	perror("mlfile.en seek end failed");
	abort();
    }
    if(write(mldat,ibuf,len+1) == -1) {
	perror("write dat file");
	abort();
    }
    if ((cbinsert(mlidx,key,KEY_SZ,posit,NO)) != OK) {
	printf("%s\n",ibuf);
	cbprterr();
	abort();
    }
    if(cbflush(mlidx) != OK) {
	cbprterr();
	abort();
    }
    return;
}	/* add_rec() */

/* ====================================================================
 * modify a record in the msg library
 * ====================================================================
 */
void
mod_rec(byte *ibuf, char *key, long dup_pos, size_t len)
{
    size_t	oldlen;
    long	posit;

    if((lseek(mldat,dup_pos,SEEK_SET)) != dup_pos) {
	perror("mlfile.en seek failed");
	abort();
    }
    if((read(mldat,&mlbuf,ML_SIZE)) < 0) {
	perror("mlfile.en read failed");
	abort();
    }
    oldlen = strlen(mlbuf);
    /* if records are the same, skip update */
    if(oldlen == len) {
	if(strcmp(ibuf,mlbuf) == 0)
	    return;
    }
    /* will new record fit in the old one */
    if(len > oldlen) {		/* won't fit */
	del_rec(key);
	add_rec(ibuf,key,len);
	return;
    }
    posit = dup_pos;
    if((lseek(mldat,dup_pos,SEEK_SET)) != dup_pos) {
	perror("mlfile.en seek failed");
	abort();
    }
    if(write(mldat,ibuf,len+1) == -1) {
	perror("write dat file");
	abort();
    }
    if((cbmodify(mlidx,key,KEY_SZ,dup_pos,key,KEY_SZ,posit,NO)) != OK) {
	printf("failure modifying mlfile.enx - key = %s\n",key);
	cbprterr();
	abort();
    }
    return;
}	/* mod_rec() */

/* ====================================================================
 * delete a record from the msg library
 * ====================================================================
 */
void
del_rec(char *key)
{
    int		status;
    long	posit;

    status = cbfind(mlidx,key,KEY_SZ,&posit);
    if(status == FOUND) {
	if ((cbdelete(mlidx,key,KEY_SZ,posit)) != OK) {
	    printf("%s\n",key);
	    cbprterr();
	    abort();
	}
	return;
    }
    printf("%s not found. Unable to delete.\n",key);
    return;
}	/* del_rec() */

/* ====================================================================
 * test a key to see if it's valid
 * ====================================================================
 */
int _fastcall
key_valid(byte *buf, size_t len)
{
    register int i, j;
    register byte *p;

	if(len < KEY_SZ) {
	    printf("ERROR - invalid key = %s\n",buf);
	    return(FALSE);
	}
    p = buf;
    if(len > TEXT)
	j = TEXT;
    else
	j = len;
    for(i=0; i<j; i++) {
	if(!isxdigit(*p)) {
	    printf("%-12.12s invalid key\n",buf);
	    return(FALSE);
	}
	if(*p >= 'a' && *p <= 'f')	/* iflower */
	    *p += 'A' - 'a';		/* toupper */
	p++;
    }
    /* if short key (delete) - don't check indicators */
    if(len == KEY_SZ)
	return(TRUE);
    if(buf[IND0] != '0') {
	printf("%-12.12s invalid indicator\n",buf);
	return(FALSE);
    }
    i = buf[IND1];
    if(i != '0' && i != '1') {
	printf("%-12.12s invalid indicator\n",buf);
	return(FALSE);
    }
    return(TRUE);
}	/* key_valid() */

/* ====================================================================
 * scan all the command line arguments
 * ====================================================================
 */
void
cmdline(int argc, byte **argv)
{
    register int i;
    register byte *p;
    byte *q;

    for (i = 1; i < argc; i++) {
	p = argv[i];
	if(*p == '?')
	    usage();
	if((*p == '-') || (*p == '/')) {    /* switch dash or slash */
	    ++p;
	    switch (tolower(*p)) {	    /* switch character */
	    case 'l':
		++p;
		q = &lang_key[0];
		*q++ = *p++;
		*q++ = *p++;
		break;
	    case 'q':
		quiet = TRUE;
		break;
	    case 'v':
		verbose = TRUE;
		break;

	    default:
		usage();	/* explain how to use this thing */
	    }
	}	/* End handling '-/' switches */
	else {	/* not a switch */
	    if(inp_name == NULL)
		inp_name = p;
	    else
	        usage();	/* explain how to use this thing */
	}
    }	/* end of args */
    return;
}	/* cmdline() */

/* ====================================================================
 * output usage blurp
 * ====================================================================
 */
void
usage(void)
{
    printf("\nMLM  Message Library Maintenance\n");
    printf("format:\n");
    printf("\tMLM input_file_name [switches]\n");
    printf("   command line switches are:\n");
    printf("    /lXX  language key (EN is default)\n");
    printf("    /q  quiet mode\n");
#ifdef DEBUG
    printf("    /v  verbose gives very detailed output (for debugging)\n");
#endif
    exit(0);
}	/* usage() */


/* mlm.c */
