/*
 * MultiMail offline mail reader
 * 

 Written by Toth Istvan <stoty@vma.bme.hu>
 Modified by William McBrine <wmcbrine@clark.net>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ctype.h>
#include <iostream.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <sys/stat.h>

#include "misc.h"
#include "mmailrc.h"
#include "resource.h"
#include "taglines.h"
#include "../interface/mysystem.h"

resource::resource(mmail *mmA)
{
	mm = mmA;
	char inp;

	//set the names 
	const char *const names[noOfResources + 2] =
	{
		"",
		"homeDir",
		"mmHomeDir",
		"bwPacketDir",
		"bwWorkDir",
		"bwUncompressCommand",
		"bwPacketName",
		"bwUserName",
		"bwBBSName",
		"bwSysOpName",
		"bwEchoTag",
		"bwReplyDir",
		"bwLoginName",
		"bwAliasName",
		"bwCompressCommand",
		"bwUpWorkDir",
		"taglineChooseType",
		"sendConfirm",
		"editor",
		"mono",
		"bwSaveDir",
		"bwReadDir",
		"bwAddressBook",
		"bwTaglineFile",
		"arjUncompressCommand",
		"zipUncompressCommand",
		"lhaUncompressCommand",
		"unknownUncompressCommand",
		"arjCompressCommand",
		"zipCompressCommand",
		"lhaCompressCommand",
		"unknownCompressCommand",
		"qwkBBSID",
		"packMode",
		"signature"
	};

	// read these from rc 
	const int startUp[] =
	{
		homeDir,
		mmHomeDir,
		bwPacketDir,
		bwReplyDir,
		bwSaveDir,
		bwReadDir,
		editor,
		taglineChooseType,
		sendConfirm,
		bwAddressBook,
		bwTaglineFile,
		arjUncompressCommand,
		zipUncompressCommand,
		lhaUncompressCommand,
		unknownUncompressCommand,
		arjCompressCommand,
		zipCompressCommand,
		lhaCompressCommand,
		unknownCompressCommand,
		sigFile
	};

	// no of options in rc
	int startUpLen = 20;	//edit !

	char configFileName[255];
	initinit();

	// fill the name fields
	for (int c = 1; c <= noOfResources; c++) {
		resourceData[c].name = (char *) names[c];
		resourceData[c].fromRc = 0;
		resourceData[c].value = NULL;
	}

	// fill the fromrc fields
	for (int c = 0; c < startUpLen; c++)
		resourceData[startUp[c]].fromRc = 1;

	strcpy(configFileName, getenv("HOME"));
	strcat(configFileName, "/.mmailrc");

	// Make this separate?

	if (!(configFile = fopen(configFileName, "r"))) {
		if (!createNewConfig(configFileName)) {
			cout << "\nWelcome new user!\n\n"
			    "A new (empty) .mmailrc has been created. "
			    "If you continue now, " MM_NAME
			    " will\nuse the default values. If you "
			    "wish to edit your .mmailrc first, say "
			    "'N' at\nthe prompt.\n\nContinue? (y/n) ";
			cin >> inp;
			if (toupper(inp) == 'N')
				exit(1);
			configFile = fopen(configFileName, "r");
		} else
			fatalError("Error opening ~/.mmailrc!");
	}
	parseConfig(configFile);
	fclose(configFile);

	tmpnam(basedir);

	for (int c = 1; c <= noOfResources; c++)
		if ((resourceData[c].init) && (!resourceData[c].value))
			(resourceData[c].init) ();

	if (verifyPaths())
		fatalError("Unable to access data directories\n");
};

resource::~resource()
{
	rmdir(resourceData[bwWorkDir].value);
	rmdir(resourceData[bwUpWorkDir].value);
	rmdir(basedir);
}

int resource::checkPath(const char *onepath, const int show)
{
	if (chdir(onepath)) {
		if (show)
			cout << "Creating " << onepath << "...\n";
		if (mkdir(onepath, S_IRWXU))
			return 1;
	}
	return 0;
}

int resource::verifyPaths(void)
{
	FILE *tagx;
	const char **p;

	// These four lines are the only ones that fit the function name:

	if (checkPath(resourceData[mmHomeDir].value, 1))
		return 1;
	if (checkPath(resourceData[bwPacketDir].value, 1))
		return 1;
	if (checkPath(resourceData[bwReplyDir].value, 1))
		return 1;
	if (checkPath(resourceData[bwSaveDir].value, 1))
		return 1;

	// Create tagline file if it doesn't exist. This should probably
	// be a separate function.

	if ((tagx = fopen(resourceData[bwTaglineFile].value, "r")))
		fclose(tagx);
	else {
		cout << "Creating " << resourceData[bwTaglineFile].value
		    << "...\n";
		if (!(tagx = fopen(resourceData[bwTaglineFile].value, "w")))
			return 1;
		else {
			for (p = defaultTags; **p; p++) {
				fputs(*p, tagx);
				fputc('\n', tagx);
			}
			fclose(tagx);
		}
	}

	return 0;
}

int resource::createNewConfig(const char *configname)
{
	FILE *mmrc;
	const char **p;

	cout << "Creating " << configname << "...\n";
	if (!(mmrc = fopen(configname, "w")))
		return 1;
	else {
		for (p = defaultMmailrc; **p; p++) {
			fputs(*p, mmrc);
			fputc('\n', mmrc);
		}
		fclose(mmrc);
	}

	return 0;
}

void resource::parseConfig(FILE *configFile)
{
	int pos, p1, p2, p3;

	char buffer[255];
	char resName[255];
	char resValue[255];
	while (fgets(buffer, 255, configFile)) {
		if ((buffer[0] == '#') || (buffer[0] == '\n'));
		else {
			pos = 0;
			//trailing spaces
			while (buffer[pos] == ' ' || buffer[pos] == '\t' &&
			       pos < 255)
				pos++;
			p1 = pos;
			//resName
			while (buffer[pos] != ':' && buffer[pos] != ' ' &&
			       buffer[pos] != '\t' && pos < 255)
				resName[pos - p1] = buffer[pos++];
			resName[pos - p1] = '\0';
			//chars between strings
			while (buffer[pos] == ' ' || buffer[pos] == '\t' ||
			       buffer[pos] == ':' && pos < 255)
				pos++;
			p2 = pos;
			//resValue
			while (buffer[pos] != '\n' && pos < 255)
				resValue[pos - p2] = buffer[pos++];
			resValue[pos - p2] = 0;

			p3 = pos;
			//check 
			if (p1 >= p2 || p2 >= p3)
				fatalError("error parsing resource file");
			else
				for (int c = 1; c <= noOfResources; c++)
					if (!strcmp(resourceData[c].name,
					  resName) && (resourceData[c].fromRc))
						resourceData[c].value =
						    strdupplus(resValue);
		};
	};
};

const char *const resource::get(const int ID)
{
	if ((ID <= 0) || (ID > noOfResources))
		return NULL;
	else
		return resourceData[ID].value;
};

const char *resource::set(const int ID, const char *newValue)
{
	if ((ID <= 0) || (ID > noOfResources))
		return NULL;
	if (resourceData[ID].value)
		delete resourceData[ID].value;
	resourceData[ID].value = strdupplus(newValue);
	return newValue;
};

int resource::getValue(const int ID)
{
	if ((ID <= 0) || (ID > noOfResources))
		return 0;
	else
		return resourceData[ID].val;
};

void resource::setValue(const int ID, int newValue)
{
	if ((ID <= 0) || (ID > noOfResources))
		return;
	resourceData[ID].val = newValue;
};

void resource::initinit()
{
	for (int c = 0; c <= noOfResources; c++)
		resourceData[c].init = initNull;

	resourceData[homeDir].init = initHomeDir;
	resourceData[bwPacketDir].init = initBwPacketDir;
	resourceData[bwWorkDir].init = initBwWorkDir;
	resourceData[bwUncompressCommand].init = initBwUncompressCommand;
	resourceData[bwReplyDir].init = initBwReplyDir;
	resourceData[bwReadDir].init = initBwReadDir;
	resourceData[bwCompressCommand].init = initBwCompressCommand;
	resourceData[bwUpWorkDir].init = initBwUpWorkDir;
	resourceData[bwSaveDir].init = initBwSaveDir;
	resourceData[editor].init = initEditor;
	resourceData[mmHomeDir].init = initMmHomeDir;
	resourceData[bwAddressBook].init = initBwAddressBook;
	resourceData[bwTaglineFile].init = initBwTaglineFile;
	resourceData[sigFile].init = initsigFile;
};

//------------------------------------------------------------------------
// The resource initializer functions
//------------------------------------------------------------------------

void resource::initHomeDir()
{
	resourceData[homeDir].value = strdupplus(getenv("HOME"));
};

void resource::initsigFile()
{
	resourceData[sigFile].value = NULL;
};

void resource::initBwWorkDir()
{
	char tmp[255];

	strcpy(tmp, basedir);
	if (checkPath(tmp, 0))
		fatalError("tmp Dir could not be created");
	strcat(tmp, "/work");
	resourceData[bwWorkDir].value = strdupplus(tmp);
	if (checkPath(resourceData[bwWorkDir].value, 0))
		fatalError("tmp Dir could not be created");
};

void resource::initBwUpWorkDir()
{
	char tmp[255];

	strcpy(tmp, basedir);
	if (checkPath(tmp, 0))
		fatalError("tmp Dir could not be created");
	strcat(tmp, "/upwork");
	resourceData[bwUpWorkDir].value = strdupplus(tmp);
	if (checkPath(resourceData[bwUpWorkDir].value, 0))
		fatalError("tmp Dir could not be created");
};

// The default is mmHomeDir/down
void resource::initBwPacketDir()
{
	char tmp[255];

	strcpy(tmp, resourceData[mmHomeDir].value);
	strcat(tmp, "/down");
	resourceData[bwPacketDir].value = strdupplus(tmp);
};

// the default is mmHomeDir/read
void resource::initBwReadDir()
{
	char tmp[255];

	strcpy(tmp, resourceData[mmHomeDir].value);
	strcat(tmp, "/read");
	resourceData[bwReadDir].value = strdupplus(tmp);
};

// the default is mmHomedir/up
void resource::initBwReplyDir()
{
	char tmp[255];

	strcpy(tmp, resourceData[mmHomeDir].value);
	strcat(tmp, "/up");
	resourceData[bwReplyDir].value = strdupplus(tmp);
};

// the default is mmHomedir/save
void resource::initBwSaveDir()
{
	char tmp[255];

	strcpy(tmp, resourceData[mmHomeDir].value);
	strcat(tmp, "/save");
	resourceData[bwSaveDir].value = strdupplus(tmp);
};

// The default is infozip.
void resource::initBwUncompressCommand()
{
	resourceData[bwUncompressCommand].value =
	    strdupplus("/usr/bin/unzip -o -j -L ");
};

void resource::initBwCompressCommand()
{
	resourceData[bwCompressCommand].value =
	    strdupplus("/usr/bin/zip -o -j ");
};

//The default is vi, or the EDITOR enviroment variable (a'la elm)
void resource::initEditor()
{
	if (getenv("EDITOR"))
		resourceData[editor].value = strdupplus(getenv("EDITOR"));
	else
		resourceData[editor].value = strdupplus("vi");	//argh!
};

// the default is ~/mmail
void resource::initMmHomeDir()
{
	char tmp[255];

	strcpy(tmp, resourceData[homeDir].value);
	strcat(tmp, "/mmail");
	resourceData[mmHomeDir].value = strdupplus(tmp);
};

// the default is addressbook
void resource::initBwAddressBook()
{
	char tmp[255];

	strcpy(tmp, resourceData[mmHomeDir].value);
	strcat(tmp, "/addressbook");
	resourceData[bwAddressBook].value = strdupplus(tmp);
};

// the default is taglines
void resource::initBwTaglineFile()
{
	char tmp[255];

	strcpy(tmp, resourceData[mmHomeDir].value);
	strcat(tmp, "/taglines");
	resourceData[bwTaglineFile].value = strdupplus(tmp);
};

void resource::initNull()
{
};
