/*
 * MultiMail offline mail reader
 * 

 Written by Toth Istvan <stoty@vma.bme.hu>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*
	MMail Message Handling System Header file

	The first element of each array is 0
	ID is the universal area or letter identifier.
	num tells the position in the actual list.
*/

#ifndef MMAIL_H 
#define MMAIL_H

#include <time.h>

#include "error.h"
#include "internal.h"
#include "resource.h"

enum areatype { mail, usenet, qwk, bw, bwrep /*, qwkrep -- still as bwrep */ };

// Mail statuses -- bit f.
#define MS_NONE		0
#define MS_READ		1
#define MS_REPLIED	2 
#define MS_MARKED	4

// for letter_list::sort
#define LS_SUBJ     1
#define LS_NAME     2 // not implemented yet
#define LS_MSGNUM   3 // -||- 

class mmail;
class packet_list;
class packet_header;
class thread_list;
class area_list;
class letter_list;
class thread_header;
class area_header;
class letter_header;
class specific_driver;
class reply_driver;
class driver_list;
class error;
class resource;
class read_class;

struct net_address
{
	int zone;
	int net;
	int node;
	int point;
};
 
class mmail
{
	packet_list *packetList;
public:
	error *errorObject;
	resource *resourceObject;
	driver_list *driverList;
	area_list *areaList;
	read_class *readObject;
	
	mmail();
	~mmail();
	packet_list *getPacketList();
	void selectPacket(char *packetName);
	void saveRead();
};	

class packet_list
{
	mmail *mm;
	int noOfPackets;
	packet_header **packets;
	int activePacket;
	
	void initPackets(char *bwDirNameA);
	void pswap(int a, int b);
	void packqsort(int left, int right);
	void psort();
public:
	packet_list(mmail *mmA);
	~packet_list();
	int	getNoOfPackets();
	int	gotoPacket(int packetNo);
	char	*getName();
	time_t	getDate();
	off_t	getSize();
	int	getNoOfPersonal();
	int	getNoOfLetters();
	int	getNoOfUnread();
};

class area_list
{
	int		no;	
	int		current;
	area_header	**areaHeader;
	specific_driver	*actDriver;
	letter_list	*actLetterList;
	mmail		*mm;

	void		refreshArea(int areaNo);

public:
	area_list(mmail *mmA);
	~area_list();
	const char	*getShortName();
	const char	*getName();
	const char	*getName(int areaNo);
	const char	*getXName(int area);
	const int	getXNum(int area);
	const char	*getDescription();
	const char	*getDescription(int area);
	const int	getNoOfLetters();
	const int	getNoOfUnread();
	const int	getNoOfPersonal();
	const enum 	areatype getType();
	int		getID();
	thread_list	*getThreadList();
	letter_list	*getLetterList();
	
	char		*getOperatorName();
	char		*getProviderName();	
	
	void		enterLetter(int area, char *from, char *to,
				char *subject, int replyTo, int privat,
				net_address *netAddress, char *filename);
	void		killLetter(int letterNo);
	void		makeReply(int area);

	int		gotoArea(int num);
	int		getAreaNo();
	int		noOfAreas();	
	int		getUseAlias();
};

class thread_list
{
public:
	const char	*getSubject();
	const int	getNoOfLetters();
	const int	getNoOfUnread();
	const char	*getStarterName();
	const letter_list	*getLetterList();

	int		gotoThread();
	int		noOfThreads();
};

class letter_list
{ 
	specific_driver *driver;
	int noOfLetters;
	int areaNumber;
	letter_header **letterHeader;
	int currentLetter;
	mmail	*mm;
	
	void init();
	void cleanup();
	void letterswap( int a, int b );
	void lettqsort( int left, int right );

public:
	letter_list(mmail *mmA, int areaNumberA); 
	~letter_list();
	const char	*getSubject();
	const char	*getTo();
	const char	*getFrom();
	const char	*getDate();
	const int	getReplyTo();
	const int	getReplyAt();
	const int	getMsgNum();
	const int	getAreaID();
	const int	getRead();
	const int	getRecieved();
	const int	getPrivate();
	const int	getReplied();
	const int	getKilled();
	const int	getOriginalID();
	const char	*getBody();
	const int	getLength();
	
	int		killLetter();
	void		rrefresh();
	void		sort( int keyord );

	const int	getStatus();
	void 		setStatus( int stat );
	
	void		setRead();
	void		setUnread();
	int		gotoLetter(int newLetter);
	int		noOfLetter();
	const int	getCurrent();
	void		setBody( char *btext ); // csak reply-nel van!
};

class area_header
{
	mmail *mm;
	specific_driver *driver;
	char *shortName;
	char *name;
	char *description;
	areatype type;
	int noOfLetters;
	int noOfUnread;
	int useAlias;
	int noOfPersonal;
	int num;
	int ID;
public:
	area_header(mmail *mmA,int num, const char *shortA,
		const char *nameA, const char* descriptionA,
		areatype typeA, int noOfLettersA, 
		int noOfUnreadA, int noOfPersonal, int useAliasA);

	~area_header();
	const char	*getShortName();
	const char	*getName();
	const char	*getDescription();
	const enum areatype 	getType();
 	const int	getNoOfLetters();
	const int	getNoOfUnread();
	const int	getNoOfPersonal();
	const int	getID();
	const int	getNum();
	const int	getUseAlias();
};
	
class thread_header
{
	char	*subject;
	int	noOfLetters;
	int	noOfUnread;
	char	*starterzzName;
	letter_list	*Letterlist;	
public:
	const char	*getSubject();
	const int	getNoOfLetters();
	const int	getNoOfUnread();
	const char	*getStarterName();
	const letter_list	*getLetterList();
};

class packet_header
{
	mmail	*mm;
	char	*name;
	int	noOfLetters;
	int	noOfUnread;
	int	noOfPersonal;
	time_t	date;
	off_t	size;
public:
	packet_header(mmail *mmA, char *nameA, int noOfLettersA,
		int noOfUnreadA, int noOfPersonalA, time_t dateA,
		off_t sizeA);

	~packet_header();
 	char	*getName();
	int	getNoOfLetters();
	int	getNoOfUnread();
	int	getNoOfPersonal();
	time_t	getDate();
	off_t	getSize();
};

class letter_header
{
	mmail *mm;
	char *subject;
	char *to;
	char *from;
	char *date;
	int replyTo;
	int replyAt;
	int LetterID;
	int AreaID;
	int read;
	int recieved;
	int privat;		//!
	int replied;
	int killed;
	int originalID; // <- now used to store area# of replies
	int length;
//	int stat; // hasznalni kene ezt is ??
	int msgNum;
	specific_driver *driver;

public:
	letter_header(mmail *mmA, const char * subjectA, const char *toA, 
		const char *fromA, const char* dateA, const int replyToA, 
		const int replyAtA, const int LetterIDA, const int AreaIDA, 
		const int readA, const int recievedA, const int privatA, 
		const int repliedA, const int killedA, const int originalIDA, 
		const int lengthA, specific_driver *driverA);

	~letter_header();				
	const char	*getSubject();
	const char	*getTo();
	const char	*getFrom();
	const char	*getDate();
	const int	getReplyTo();
	const int	getReplyAt();
	const int	getLetterID();
	const int	getAreaID();
	const int	getRead();
	const int	getRecieved();
	const int	getPrivate();
	const int	getReplied();
	const int	getKilled();
	const int	getOriginalID();
	const int	getLength();
//	const char	*getBody();
//	int		setBody( char *btext ); //reply-nel csak
	specific_driver	*getDriver();
	void		setRead();
	void		setUnread();
	const int	getMsgNum();
	void		setMsgNum( int mn );

	const int	getStatus();
	void		setStatus( int stat );
};

class driver_list
{
	mmail *mm;
	struct driver_struct 
	{
		specific_driver *driver;
		read_class	*read;
		areatype	type;
		int		offset;
		int		noOfAreas;
	};

	int noOfDrivers;
	
	struct driver_struct driverList[3];
	
public:
	driver_list(mmail *mmA);
	~driver_list();
	
	int getNoOfDrivers();
	
	specific_driver *getDriver(int areaNo);
	reply_driver *getReplyDriver(int areaNo);
	
	read_class *getReadObject( specific_driver *driver);
	int getOffset( specific_driver *driver);
};

class read_class
{
public:
	read_class(){};
	virtual ~read_class(){};
	virtual void setRead(int area, int letter, int value)=0;
	virtual int getRead(int area, int letter)=0;
	virtual void setStatus(int area, int letter, int value)=0;
	virtual int getStatus(int area, int letter)=0;
	virtual int getNoOfUnread(int area)=0;
	virtual void saveAll()=0;
};

class bw_read_class : public read_class
{
	mmail *mm;
	specific_driver *driver;
	int **readStore;
	int *noOfLetters;

	int init(mmail *mmA);
	int cleanup();
	void init();
	void cleanUp();
	
public:
	bw_read_class(mmail *mmA, specific_driver *driverA);
	~bw_read_class();
	void setRead(int area, int letter, int value);
	int getRead(int area, int letter);
	void setStatus(int area, int letter, int value);
	int getStatus(int area, int letter);
	int getNoOfUnread(int area);
	void saveAll();
	char *readFilePath(const char *bwFile);
};

class dummy_read_class: public read_class
{
public:
	dummy_read_class(mmail *mmA ,specific_driver *driverA)
	{
		mmA=(mmail*)driverA;
	};
	~dummy_read_class(){};
	void setRead(int area, int letter, int value)
	{
		area=letter=value;
	};
	int getRead(int area, int letter)
	{
		area=letter;
		return 1;
	};
	void setStatus(int area, int letter, int value)
	{
		area=letter=value;
	};
	int getStatus(int area, int letter)
	{
		area=letter;
		return 1;
	};
	int getNoOfUnread(int area)
	{
		area=0;
		return 0;
	};
	void saveAll()
	{
	};
};	

#endif
