/*
 * MultiMail offline mail reader
 * 

 Written by Toth Istvan <stoty@vma.bme.hu>
 Modified by William McBrine <wmcbrine@clark.net>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <string.h>

#include "bw.h"
#include "bwreply.h"
#include "misc.h"
#include "mmail.h"

// ---------------------------------------------------------------------------
// Letter header methods
// ---------------------------------------------------------------------------

letter_header::letter_header(mmail *mmA, const char *subjectA,
	      const char *toA, const char *fromA, const char *dateA,
	      const int replyToA, const int replyAtA, const int LetterIDA,
	      const int AreaIDA, const int readA, const int recievedA,
	      const int privatA, const int repliedA, const int killedA,
     const int originalIDA, const int lengthA, specific_driver *driverA)
{
	mm = mmA;
	subject = strdupplus(subjectA);
	to = strdupplus(toA);
	from = strdupplus(fromA);
	date = strdupplus(dateA);
	replyTo = replyToA;
	replyAt = replyAtA;
	LetterID = LetterIDA;
	AreaID = AreaIDA;
	recieved = recievedA;
	privat = privatA;
	replied = repliedA;
	killed = killedA;
	originalID = originalIDA;
	length = lengthA;
	driver = driverA;
	read = readA;		//suppress warning

	read = mm->driverList->getReadObject(driver)->getRead(AreaID, LetterID);
	msgNum = LetterIDA;
};

letter_header::~letter_header()
{
	delete subject;
	delete to;
	delete from;
	delete date;
};

const char *letter_header::getSubject()
{
	return subject;
};

const char *letter_header::getTo()
{
	return to;
};

const char *letter_header::getFrom()
{
	return from;
};

const char *letter_header::getDate()
{
	return date;
};

const int letter_header::getReplyTo()
{
	return replyTo;
};

const int letter_header::getReplyAt()
{
	return replyAt;
};

const int letter_header::getLetterID()
{
	return LetterID;
};

const int letter_header::getAreaID()
{
	return AreaID + mm->driverList->getOffset(driver);
};

const int letter_header::getRead()
{
	return read;
};

const int letter_header::getRecieved()
{
	return recieved;
};

const int letter_header::getPrivate()
{
	return privat;
};

const int letter_header::getReplied()
{
	return replied;
};

const int letter_header::getKilled()
{
	return killed;
};

const int letter_header::getOriginalID()
{
	return originalID;
};

const int letter_header::getLength()
{
	return length;
};

specific_driver *letter_header::getDriver()
{
	return driver;
};

void letter_header::setRead()
{
	read = 1;
	mm->driverList->getReadObject(driver)->setRead(AreaID,
						       LetterID, read);
};

void letter_header::setStatus(int stat)
{
	mm->driverList->getReadObject(driver)->setStatus(AreaID,
							 LetterID, stat);
};

const int letter_header::getStatus()
{
	return mm->driverList->getReadObject(driver)->getStatus(AreaID,
								LetterID);
};

void letter_header::setUnread()
{
	read = 0;
};

const int letter_header::getMsgNum()
{
	return msgNum;
};

void letter_header::setMsgNum(int mn)
{
	msgNum = mn;
};

// ---------------------------------------------------------------------------
// Letterlist methods
// ---------------------------------------------------------------------------

letter_list::letter_list(mmail *mmA, int areaNumberA)
{
	mm = mmA;
	areaNumber = areaNumberA;
	init();
}

void letter_list::init()
{
	driver = mm->driverList->getDriver(areaNumber);
	driver->selectArea(areaNumber - mm->driverList->getOffset(driver));
	noOfLetters = driver->getNoOfLetters();
	letterHeader = new letter_header *[noOfLetters + 1];
	driver->resetLetters();
	for (int c = 1; c <= noOfLetters; c++)
		letterHeader[c] = driver->getNextLetter();
	currentLetter = 1;
	if (noOfLetters > 1)
		sort(LS_SUBJ);	// test/default sort key should be here
	// not yet! there's a bug
};

letter_list::~letter_list()
{
	cleanup();
};

void letter_list::cleanup()
{
	for (int c = 1; c <= noOfLetters; c++)
		delete letterHeader[c];
	delete letterHeader;
};

void letter_list::letterswap(int a, int b)
{
	letter_header *lh;
	lh = letterHeader[a];
	letterHeader[a] = letterHeader[b];
	letterHeader[b] = lh;
}

void letter_list::lettqsort(int left, int right)
{
	int lower, upper, d, t;
	letter_header *pivot;

	lower = left;
	upper = right;
	pivot = letterHeader[((left + right) >> 1)];
	do {
		do {
			d = strcasecmp(letterHeader[lower]->getSubject(),
				   pivot->getSubject());
			t = ((d < 0) || ((d == 0) && ((letterHeader[lower]->
			      getLetterID()) < (pivot->getLetterID()))));
			if (t)
				lower++;
		} while (t);
		do {
			d = strcasecmp(letterHeader[upper]->getSubject(),
				   pivot->getSubject());
			t = ((d > 0) || ((d == 0) && ((letterHeader[upper]->
			      getLetterID()) > (pivot->getLetterID()))));
			if (t)
				upper--;
		} while (t);
		if (lower <= upper) {
			if (lower != upper)
				letterswap(lower, upper);
			lower++;
			upper--;
		}
	} while (lower < upper);
	if (left < upper)
		lettqsort(left, upper);
	if (lower < right)
		lettqsort(lower, right);
}

void letter_list::sort(int keyord)	// lame, not really impl'd etc...
{
	switch (keyord) {
	case LS_SUBJ:
		lettqsort(1, noOfLetters);
	}
}

int letter_list::noOfLetter()
{
	return noOfLetters;
};

int letter_list::gotoLetter(int newLetter)
{
	if ((newLetter > 0) && (newLetter <= noOfLetters))
		currentLetter = newLetter;
	return newLetter;
};

const char *letter_list::getSubject()
{
	return letterHeader[currentLetter]->getSubject();
};

const char *letter_list::getFrom()
{
	return letterHeader[currentLetter]->getFrom();
};

const int letter_list::getMsgNum()
{
	return letterHeader[currentLetter]->getMsgNum();
};

const char *letter_list::getTo()
{
	return letterHeader[currentLetter]->getTo();
};

const char *letter_list::getDate()
{
	return letterHeader[currentLetter]->getDate();
};

const int letter_list::getReplyTo()
{
	return letterHeader[currentLetter]->getReplyTo();
};

const int letter_list::getReplyAt()
{
	return letterHeader[currentLetter]->getReplyAt();
};

/* Same as getMsgNum()
   const int    letter_list::getLetterID()
   {
   return letterHeader[currentLetter]->getLetterID();
   };
 */

const int letter_list::getCurrent()
{
	return currentLetter;
};

const int letter_list::getAreaID()
{
	return letterHeader[currentLetter]->getAreaID();
};

const int letter_list::getRead()
{
	return letterHeader[currentLetter]->getRead();
};

const int letter_list::getRecieved()
{
	return letterHeader[currentLetter]->getRecieved();
};

const int letter_list::getPrivate()
{
	return letterHeader[currentLetter]->getPrivate();
};

const int letter_list::getReplied()
{
	return letterHeader[currentLetter]->getReplied();
};

const int letter_list::getKilled()
{
	return letterHeader[currentLetter]->getKilled();
};

const int letter_list::getOriginalID()
{
	return letterHeader[currentLetter]->getOriginalID();
};

const char *letter_list::getBody()
{
	return driver->getBody(letterHeader[currentLetter]->getLetterID());
};

void letter_list::setBody(char *btext)
{
	driver->setBody(letterHeader[currentLetter]->getLetterID(), btext);
};

const int letter_list::getLength()
{
	return letterHeader[currentLetter]->getLength();
};

const int letter_list::getStatus()
{
	return letterHeader[currentLetter]->getStatus();
};

void letter_list::setStatus(int stat)
{
	letterHeader[currentLetter]->setStatus(stat);
};

void letter_list::setUnread()
{
	letterHeader[currentLetter]->setUnread();
};

void letter_list::setRead()
{
	letterHeader[currentLetter]->setRead();
};

void letter_list::rrefresh()
{
	cleanup();
	init();
};
