/*
 * MultiMail offline mail reader
 * 

 Written by Toth Istvan <stoty@vma.bme.hu>
 Modified by William McBrine <wmcbrine@clark.net>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <strings.h>
#include <unistd.h>

#include "bw.h"
#include "compress.h"
#include "error.h"
#include "misc.h"
#include "resource.h"
#include "../interface/mysystem.h"

// ---------------------------------------------------------------------------
// The bluewave methods
// ---------------------------------------------------------------------------

// the constructor
bluewave::bluewave(mmail *mmA)
{
	mm = mmA;
	init();
};

void bluewave::init()
{
	areaSerial = 1;
	ID = 0;

	/* DEBUG!!! 
	   FILE *f=fopen("/root/alignt.dat","wb");
	   MIX_REC  mr;
	   memset(&mr,'-',sizeof(MIX_REC));
	   memset(mr.areanum,0x0a,6);
	   mr.totmsgs=0x0b0b;
	   mr.numpers=0x0c0c;
	   mr.msghptr=0x0d0d0d0d;
	   fwrite(&mr,1,sizeof(MIX_REC),f);
	   fclose(f);
	 */

	// getBody handles this, but for the first time it needs sg to delete  
	bodyString = new char[1];

	infoHeader = new INF_HEADER;

	bwDirName = strdupplus(mm->resourceObject->get(bwPacketDir));
	packetName = strdupplus(mm->resourceObject->get(bwPacketName));

	//clean up
	if (chdir(mm->resourceObject->get(bwWorkDir)))
		fatalError("bw::cleanup: could not cd to tmp");
	mysystem("rm *");

	uncompressPacket(packetName);

	packetBaseName = findBaseName(packetName);

	initInf(packetBaseName);
	readMixRec(packetBaseName);

	body = new struct bodytype *[noOfMixRecs + 1];

	initFti(packetBaseName);
	initDat(packetBaseName);

	currentAreaID = 1;
	currentLetter = 1;
};

// the desctructor
bluewave::~bluewave()
{
	cleanup();
};

void bluewave::cleanup()
{
	delete bodyString;
	delete infoHeader;
	delete bwDirName;
	delete packetName;
	delete packetBaseName;

	for (int c = 1; c <= noOfMixRecs; c++)
		delete body[c];
	delete mixRecord;
	delete body;

	fclose(infFile);
	fclose(mixFile);
	fclose(ftiFile);
	fclose(datFile);

	if (chdir(mm->resourceObject->get(bwWorkDir)))
		fatalError("bw::cleanup: could not cd to tmp");
	mysystem("rm *");
};

// returns the no Of Active areas in the packet
int bluewave::getNoOfAreas()
{
	return noOfMixRecs;
};

void bluewave::resetAll()
{
	cleanup();
	init();
};

// returns the next area in an area_header object.
area_header *bluewave::getNextArea()
{
	area_header *tmp;
	INF_AREA_INFO infAreainfo;
	char num[7], name[22], description[51];
	int c, mixIndex;

	do {
		mixIndex = 0;

		// read the Area from the .INF file     
		fseek(infFile, (infoHeaderLen +
			  (areaSerial - 1) * infoAreainfoLen), SEEK_SET);
		if (!fread(&infAreainfo, sizeof(INF_AREA_INFO), 1, infFile)) {
			mm->errorObject->setErrorMsg(
		" Internal: premature EOF in bluewave::getNextArea\n\n");
			mm->errorObject->setErrorType(fatal);
			return NULL;
		};
		// find the corresponding record in the MixRecord array
		for (c = 1; c <= noOfMixRecs; c++)
			if (!numcmp((char *) infAreainfo.areanum,
				    (char *) mixRecord[c].areanum)) {
				mixIndex = c;
				break;
			}
		areaSerial++;

	} while (!mixIndex);	//stop if found an active area

	body[++ID] = new struct bodytype[mixRecord[mixIndex].totmsgs + 1];

	strncpy(num, (char *) infAreainfo.areanum, 6);
	strncpy(name, (char *) infAreainfo.echotag, 21);
	strncpy(description, (char *) infAreainfo.title, 50);
	tmp = new area_header(mm, ID + mm->driverList->getOffset(this),
		 num, name, description, bw, mixRecord[mixIndex].totmsgs,
			0, mixRecord[mixIndex].numpers,
				(infAreainfo.area_flags & INF_ALIAS_NAME));
	return tmp;
};

// sets the current area
void bluewave::selectArea(int ID)
{
	currentAreaID = ID;
	resetLetters();
};

// sets the current Letter to 1
void bluewave::resetLetters()
{
	currentLetter = 1;
};

int bluewave::getNoOfLetters()
{
	return mixRecord[currentAreaID].totmsgs;
};

// returns the next letterheader in a letter_header object
letter_header *bluewave::getNextLetter()
{
	letter_header *letter;
	FTI_REC ftiRec;
	char from[37], to[37], subject[73], date[21];

	if (currentLetter > mixRecord[currentAreaID].totmsgs)
		return NULL;

	fseek(ftiFile, mixRecord[currentAreaID].msghptr + (currentLetter - 1) *
		ftiStructLen, SEEK_SET);

	if (!(fread(&ftiRec, sizeof(FTI_REC), 1, ftiFile))) {
		mm->errorObject->setErrorMsg("Errror reading .FTI file \n\n");
		mm->errorObject->setErrorType(fatal);
		return NULL;
	};

	(*(body + currentAreaID))[currentLetter].pointer = ftiRec.msgptr;
	(*(body + currentAreaID))[currentLetter].msgLength = ftiRec.msglength;

	strncpy(from, (char *) ftiRec.from, 36);
	strncpy(to, (char *) ftiRec.to, 36);
	strncpy(subject, (char *) ftiRec.subject, 72);
	strncpy(date, (char *) ftiRec.date, 21);

	letter = new letter_header(mm, stripre(subject), to, from, date,
	    ftiRec.replyto, ftiRec.replyat, currentLetter, currentAreaID,
				   0, (ftiRec.flags & MSG_NET_RECEIVED),
	     (ftiRec.flags & MSG_NET_PRIVATE), 0, 0, 0, ftiRec.msglength,
				   this);

	letter->setMsgNum(ftiRec.msgnum);

	currentLetter++;

	return letter;
};

// returns the body of the requested letter in the active area
char *bluewave::getBody(int ID)
{
	int c, kar, lfig;		// ^A flag
	char *p;

	// no check, should do it in areaheader
	fseek(datFile, (*(body + currentAreaID))[ID].pointer, SEEK_SET);

	if (fgetc(datFile) != ' ') {
		mm->errorObject->setErrorMsg("Errror reading .DAT file \n\n");
		mm->errorObject->setErrorType(fatal);
		return NULL;
	};

	delete bodyString;
	bodyString = new char[(*(body + currentAreaID))[ID].msgLength + 1];

	lfig = 0;
	for (c = 0, p = bodyString;
	     c < (*(body + currentAreaID))[ID].msgLength; c++) {
		kar = fgetc(datFile);

		//Ctrl-A line filtering

		// More should be done with the hidden lines (they shouldn't
		// just be discarded like this).

		if (kar == '\n') {
			*(p++) = '\n';
			if (lfig)
				p--;
			lfig = 0;
		} else if (kar == 1)
			lfig = 1;
		else if (!lfig)
			*(p++) = kar;
	}
	*p = '\0';

	return bodyString;
};

void bluewave::setBody(int ID, char *btext)
{
	// THIS FUNCTION SHOULD BE ELIMINATED

	ID = ID;		// warning suppression
	btext = btext;
}

// finds the latest file in the given directory, and returns its name

// clears the working directory and uncompresses the bw packet into it.
int bluewave::uncompressPacket(const char *packetName)
{
	char fname[255];

	strcpy(fname, mm->resourceObject->get(bwPacketDir));
	strcat(fname, "/");
	strcat(fname, packetName);

	return uncompressFile(mm, fname,
			      (char *) mm->resourceObject->get(bwWorkDir),
			      "*.FTI *.DAT *.MIX *.INF");
};


// opens the .INF file, fills the infoHeader struct and sets the 
// recordlengths. Reads the misc data from the Header
void bluewave::initInf(const char *packetBaseName)
{
	char infName[255];

	chdir(mm->resourceObject->get(bwWorkDir));
	strcpy(infName, packetBaseName);
	strcat(infName, ".INF");

	if (!(infFile = ftryopen(infName, "r"))) {
		mm->errorObject->setErrorMsg("Error opening .INF file\n\n");
		mm->errorObject->setErrorType(fatal);
		return;
	}
	if (!fread(infoHeader, sizeof(INF_HEADER), 1, infFile)) {
		mm->errorObject->setErrorMsg("Error reading .INF file\n\n");
		mm->errorObject->setErrorType(fatal);
		return;
	};

	if (infoHeader->inf_header_len < ORIGINAL_INF_HEADER_LEN)
		infoHeaderLen = ORIGINAL_INF_HEADER_LEN;
	else
		infoHeaderLen = infoHeader->inf_header_len;

	if (infoHeader->inf_areainfo_len < ORIGINAL_INF_AREA_LEN)
		infoAreainfoLen = ORIGINAL_INF_AREA_LEN;
	else
		infoAreainfoLen = infoHeader->inf_areainfo_len;

	if (infoHeader->mix_structlen < ORIGINAL_MIX_STRUCT_LEN)
		mixStructLen = ORIGINAL_MIX_STRUCT_LEN;
	else
		mixStructLen = infoHeader->mix_structlen;

	if (infoHeader->fti_structlen < ORIGINAL_FTI_STRUCT_LEN)
		ftiStructLen = ORIGINAL_FTI_STRUCT_LEN;
	else
		ftiStructLen = infoHeader->fti_structlen;

	mm->resourceObject->set(bwLoginName, (char *) infoHeader->loginname);
	mm->resourceObject->set(bwAliasName, (char *) infoHeader->aliasname);
	mm->resourceObject->set(bwSysOpName, (char *) infoHeader->sysop);
	mm->resourceObject->set(bwBBSName, (char *) infoHeader->systemname); 
};

// Reads the .MIX file into the MixRecord Array
void bluewave::readMixRec(const char *packetBaseName)
{
	char mixName[255];
	long tmp;
	int c;

	chdir(mm->resourceObject->get(bwWorkDir));
	strcpy(mixName, packetBaseName);
	strcat(mixName, ".MIX");

	if (!(mixFile = ftryopen(mixName, "rb"))) {
		mm->errorObject->setErrorMsg("Error opening .MIX file\n\n");
		mm->errorObject->setErrorType(fatal);
		return;
	}
	fseek(mixFile, 0, SEEK_END);
	tmp = ftell(mixFile);
	rewind(mixFile);
	noOfMixRecs = (int) (tmp / mixStructLen);

	mixRecord = new MIX_REC[noOfMixRecs + 1];

	memset(&(mixRecord[0]), 0, sizeof(MIX_REC));

	for (c = 1; c <= noOfMixRecs; c++) {
		fseek(mixFile, (c - 1) * mixStructLen, SEEK_SET);
		fread(&mixRecord[c], mixStructLen, 1, mixFile);
	}
};

// Opens the .FTI file
void bluewave::initFti(const char *packetBaseName)
{
	char ftiName[255];

	chdir(mm->resourceObject->get(bwWorkDir));
	strcpy(ftiName, packetBaseName);
	strcat(ftiName, ".FTI");

	if (!(ftiFile = ftryopen(ftiName, "r"))) {
		mm->errorObject->setErrorMsg("could not open .FTI file");
		mm->errorObject->setErrorType(fatal);
	}
};

// Opens the .DAT file
void bluewave::initDat(const char *packetBaseName)
{
	char datName[255];

	chdir(mm->resourceObject->get(bwWorkDir));
	strcpy(datName, packetBaseName);
	strcat(datName, ".DAT");

	if (!(datFile = ftryopen(datName, "r"))) {
		mm->errorObject->setErrorMsg("could not open .DAT file");
		mm->errorObject->setErrorType(fatal);
	};
};
