/*
 * MultiMail offline mail reader
 * 

 Written by Kolossvary Tamas <thomas@vma.bme.hu>
 Modified by William McBrine <wmcbrine@clark.net>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdlib.h>
#include <time.h>

#include "interface.h"

extern mmail		mm;
extern Interface	*interface;
extern AreaListWindow	areas;
extern LetterListWindow	letters;
extern LetterWindow	letterwindow;
extern HelpWindow	helpwindow;
extern int		getstring(WINDOW *, int, int, char *, int, int, int);

tagline::tagline(void)
{
	next = NULL;
}

TaglineWindow::TaglineWindow(int no_draw)
{
	int c;
	int end = 0;

	nodraw = no_draw;
	if (!nodraw) {
		srand((unsigned) time(NULL));

		list_max_y = LINES - 14;
		list = makeWindow(LINES - 10, COLS - 2, 5, 1, C_TBBACK);
		mvwaddch(list, 0, 2, ACS_RTEE);
		mvwaddch(list, 0, 17, ACS_LTEE);

		wattrset(list, C_TTEXT);
		mvwaddstr(list, 0, 3, "Select tagline");
		mvwaddstr(list, LINES - 13, 59, " - Apply tagline");
		mvwaddstr(list, LINES - 13, 3, " - Don't apply tagline");
		mvwaddstr(list, LINES - 12, 3, " - Reread tagline file");
		mvwaddstr(list, LINES - 12, 28, " - Manual enter tagline");
		mvwaddstr(list, LINES - 13, 28, " - Random select tagline");
		wattrset(list, C_TKEYSTEXT);
		wmove(list, LINES - 14, 1);
		whline(list, ACS_HLINE, COLS - 4);
		mvwaddstr(list, LINES - 13, 54, "Enter");
		mvwaddch(list, LINES - 13, 2, 'Q');
		mvwaddch(list, LINES - 12, 2, 'F');
		mvwaddch(list, LINES - 12, 27, 'E');
		mvwaddch(list, LINES - 13, 27, 'R');
	}
	ReadFile();
	if (!nodraw) {
		// keyboard handling
		do {
			Draw();
			c = wgetch(screen);
			switch (c) {
			case 'q':
			case 'Q':
			case '\033':
			case '\n':
				end = 1;
				break;
			case 'e':
			case 'E':
				EnterTagline();
				break;
			case 'f':
			case 'F':
				ReadFile();
				break;
			case 'r':
			case 'R':
				RandomTagline();
				break;
			case KEY_DOWN:
			case KEY_RIGHT:
				Move(DOWN);
				break;
			case KEY_UP:
			case KEY_LEFT:
				Move(UP);
				break;
			case KEY_PPAGE:
				Move(PGUP);
				break;
			case ' ':
			case KEY_NPAGE:
				Move(PGDN);
				break;
			case KEY_HOME:
				Move(HOME);
				break;
			case KEY_END:
				Move(END);
				break;
			}
		} while (!end);
		if ((c == '\n') && (NumOfTaglines > 0))
			letterwindow.set_Tagline(getTextOf(active));
		//DestroyChain();
		//delwin(list);
		this->~TaglineWindow();
	}
}

TaglineWindow::~TaglineWindow(void)
{
	DestroyChain();
	delwin(list);
	touchwin(screen);
	wnoutrefresh(screen);
	switch (interface->active()) {
	case packetlist:
		break;
	case arealist:
		areas.ReDraw();
		helpwindow.redraw();
		break;
	case threadlist:
		break;
	case letterlist:
		letters.ReDraw();
		helpwindow.redraw();
		break;
	case letter:
		letterwindow.ReDraw();
		break;
	case letter_help:
		break;
	case littlearealist:
		break;
	}
}

void TaglineWindow::RandomTagline(void)
{
	int i, j;

	i = rand() / (RAND_MAX / NumOfTaglines);
	Move(HOME);
	for (j = 1; j <= i; j++)
		Move(DOWN);
	Draw();
}

int TaglineWindow::EnterTagline(char *tag)
{
	int y;
	char newtagline[TAGLINE_LENGTH + 1];
	FILE *fd;

	Move(END);
	if (NumOfTaglines >= list_max_y - 1) {
		y = list_max_y - 1;
		position++;
		active++;
	} else {
		y = NumOfTaglines + 1;
		active++;
	}
	if (!nodraw)
		Draw();
	else {
		list = makeWindow(5, COLS - 2, 10, 1, C_TBBACK);
		wattrset(list, C_TTEXT);
		mvwaddstr(list, 1, 1, "Enter new tagline:");
		wrefresh(list);
	}

	if (tag)
		strcpy(newtagline, tag);
	else
		newtagline[0] = '\0';

	if (getstring(list, (nodraw) ? 2 : y, 1, newtagline, TAGLINE_LENGTH,
		      C_TENTER, C_TENTERGET)) {

		char *p;
		for (p = &newtagline[strlen(newtagline) - 1];
		     (p > newtagline) && (*p == ' '); p--);	//crop spaces

		*++p = '\0';

		if (newtagline[0]) {
			tagline *curr = &head;
			//check dupes
			while (curr && strcmp(newtagline, curr->text))
				curr = curr->next;
			if (!curr)
				if ((fd = fopen(mm.resourceObject->
				get(bwTaglineFile), "a"))) {
					fputs(newtagline, fd);
					fputc('\n', fd);
					fclose(fd);
				}
			ReadFile();
		}
	}
	Move(END);
	if (!nodraw)
		Draw();
	else
		wrefresh(list);
	return 1;
}

void TaglineWindow::ReadFile(void)
{
	FILE *fd;
	tagline *curr;

	NumOfTaglines = 0;
	DestroyChain();
	if ((fd = fopen(mm.resourceObject->get(bwTaglineFile), "r"))) {
		position = 0;
		active = 0;
		curr = &head;
		while (fgets(curr->text, TAGLINE_LENGTH, fd)) {
			//cut the \n down
			if (curr->text[strlen(curr->text) - 1] == '\n')
				curr->text[strlen(curr->text) - 1] = '\0';
			curr->next = new tagline;
			curr = curr->next;
			NumOfTaglines++;
		}
	}
}

void TaglineWindow::DestroyChain(void)
{
	tagline *curr;
	tagline *tmp;

	curr = head.next;
	while (curr) {
		tmp = curr;
		curr = curr->next;
		delete tmp;
	}
	head.next = NULL;
}

void TaglineWindow::Draw(void)
{
	int i, j;

	if (NumOfTaglines > 0) {
		for (i = 1; i < list_max_y; i++) {
			wattrset(list, C_TLINES);
			if (position + i - 1 == active)
				wattron(list, A_REVERSE);
			if (position + i > NumOfTaglines)
				for (j = 1; j <= TAGLINE_LENGTH; j++)
					mvwaddch(list, i, j, ' ');
			else
				mvwprintw(list, i, 1, "%-76s",
					getTextOf(position + i - 1));
		}
		wrefresh(list);
	}
}

int TaglineWindow::NumOfItems(void)
{
	return NumOfTaglines;
}

char *TaglineWindow::getTextOf(int num)
{
	int i;
	tagline *curr;

	curr = &head;
	for (i = 1; (i <= num) && (i < NumOfTaglines); i++)
		curr = curr->next;
	return curr->text;
}
