/*
 * MultiMail offline mail reader
 * 

 Written by Kolossvary Tamas <thomas@tvnet.hu>
 Modified by John Zero <john@graphisoft.hu>,
             William McBrine <wmcbrine@clark.net>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdlib.h>
#include <unistd.h>
#include <sys/utsname.h>

#include "interface.h"
#include "mysystem.h"

extern mmail		mm;
extern AreaListWindow	areas;
extern LetterListWindow	letters;
extern LetterWindow	letterwindow;
extern HelpWindow	helpwindow;
extern letter_list	*letterList;
extern Interface	*interface;
extern int		getstring(WINDOW *, int, int, char *, int, int, int);

//*************         ListWindow              *****************

ListWindow::ListWindow(void)
{
	position = 0;
	active = 0;
}

void ListWindow::Move(direction dir)
{
	switch (dir) {
	case UP:
		if (position + active > 0) {
			active--;
			if (active == position - 1)
				position--;
		}
		break;
	case DOWN:
		if (active + 1 < NumOfItems()) {
			active++;
			if (active - position + 1 >= list_max_y)
				position++;
		}
		break;
	case PGUP:
		position -= list_max_y - 1;
		active -= list_max_y - 1;
		if (active < 0)
			active = 0;
		if (position < 0)
			position = 0;
		break;
	case PGDN:
		position += list_max_y - 1;
		active += list_max_y - 1;
		if (active >= NumOfItems())
			active = NumOfItems() - 1;
		if (position > NumOfItems() - list_max_y) {
			position = NumOfItems() - list_max_y + 1;
			if (position < 0)
				position = 0;
		}
		break;
	case HOME:
		active = 0;
		position = 0;
		break;
	case END:
		active = NumOfItems() - 1;
		position = NumOfItems() - list_max_y + 1;
		if (position < 0)
			position = 0;
		break;
	}
}

void ListWindow::ReDraw(void)
{
	touchwin(list);
	wnoutrefresh(list);
}

//****************         LETTTERLIST        ****************

LetterListWindow::LetterListWindow(void)
{
	position = 0;
	active = 0;
}

void LetterListWindow::Save(void)
{
	char filename[255];
	FILE *fd;
	WINDOW *question;
	int i;

	question = makeWindow(5, 60, 10, 10, C_LLSAVETEXT);
	mvwaddstr(question, 1, 2, "Save to file:");
	mvwaddch(question, 2, 2, '<');
	mvwaddch(question, 2, 57, '>');
	mvwaddstr(question, 3, 30, "<ESC> (pause) to cancel");
	sprintf(filename, "%s/%s", mm.resourceObject->get(bwSaveDir),
		mm.areaList->getName());
	if (getstring(question, 2, 3, filename, 54, C_LLSAVETEXT, C_LLSAVEGET))
	{
		chdir(mm.resourceObject->get(bwSaveDir));
		if ((fd = fopen(filename, "a"))) {
			for (i = 1; i <= mm.areaList->getNoOfLetters(); i++) {
				letterList->gotoLetter(i);
				letterwindow.write_to_file(fd);
			}
			fclose(fd);
		}
	}
	delwin(question);
	touchwin(screen);
	wnoutrefresh(screen);
	touchwin(list);
	Draw();
	helpwindow.redraw();
}

void LetterListWindow::FirstUnread(void)
{
	int i;

	position = 0;
	active = 0;
	for (i = 0; i < NumOfItems(); i++) {
		letterList->gotoLetter(i + 1);
		if (letterList->getRead()) {
			Move(DOWN);
			// These lines make it scroll when rereading
			// a packet. Looks nice, can be very slow.
			//Draw();
			//doupdate();
		} else
			i = NumOfItems();
	}
	Draw();
}

void LetterListWindow::NextUnread(void)
{
	do {
		Move(DOWN);
		letterList->gotoLetter(active + 1);
	} while (letterList->getRead() && ((active + 1) < NumOfItems()));
	Draw();
}

void LetterListWindow::PrevUnread(void)
{
	do {
		Move(UP);
		letterList->gotoLetter(active + 1);
	} while (letterList->getRead() && (position + active > 0));
	Draw();
}

int LetterListWindow::NumOfItems(void)
{
	return mm.areaList->getNoOfLetters();
}

void LetterListWindow::Draw(void)
{
	int i;
	char *tmp;
	char stat[10];
	int st;

	tmp = new char[COLS - 47];

	tmp[COLS - 48] = '\0';
	for (i = 1; i < list_max_y; i++) {
		int x, y;
		char tmp[100];

		letterList->gotoLetter(position + i);
		strcpy(stat, "   ");
		st = letterList->getStatus();
		if (st & MS_READ)
			stat[2] = '*';
		if (st & MS_REPLIED)
			stat[1] = '~';
		if (st & MS_MARKED)
			stat[0] = 'M';

		wattrset(list, COLOR_PAIR(26) | A_NORMAL);
		mvwprintw(list, i + 1, 1, "%s", stat);
		if (!(st & MS_READ))
			wattron(list, A_BOLD);
		if (position + i - 1 == active)
			wstandout(list);

		mvwprintw(list, i + 1, 1 + 3, " %5d", letterList->getMsgNum());
		strcpy(tmp, letterList->getFrom());
		charconv_in(tmp);
		mvwprintw(list, i + 1, 7 + 3, "  %-16s", tmp);
		strcpy(tmp, letterList->getTo());
		charconv_in(tmp);
		mvwprintw(list, i + 1, 23 + 3, " %-16s", tmp);
		strncpy(tmp, letterList->getSubject(), COLS - 50);
		tmp[COLS - 50] = 0;
		charconv_in(tmp);
		mvwprintw(list, i + 1, 38 + 3, " %s", tmp);
		getyx(list, y, x);

		//Double kludge!
		if ((mm.areaList->getType() == bwrep) ||
		    ((mm.resourceObject->getValue(packMode) == PM_QWK) &&
		      !strcmp("PERS", mm.areaList->getShortName()))) {
			while (x < COLS - 20)
				mvwaddch(list, y, x++, ' ');
			// Kludge for QWK, ugly, change it:
			if (mm.resourceObject->getValue(packMode) == PM_QWK)
				strncpy(tmp, mm.areaList->getXName(letterList->
					getOriginalID()), 13);
			else
				strncpy(tmp, mm.areaList->
					getDescription(letterList->
						getAreaID()), 13);
			tmp[13] = 0;
			mvwprintw(list, i + 1, COLS - 20, " %-14s", tmp);
		} else {
			while (x < COLS - 5)
				mvwaddch(list, y, x++, ' ');
		}
		if ((position + i) == NumOfItems())
			i = list_max_y;
	}
	delete tmp;
	wnoutrefresh(list);
}

void LetterListWindow::MakeActive(void)
{
	areas.Reset_areaList();
	if (NumOfItems() < LINES - 11)
		list_max_y = NumOfItems() + 1;
	else
		list_max_y = LINES - 10;

	list = makeWindow(list_max_y + 2, COLS - 4, 2, 2, C_LLBBORD);
	mvwaddch(list, 0, 2, ACS_RTEE);
	wattrset(list, C_LLTOPTEXT1);
	waddstr(list, "Letters in ");
	wattrset(list, C_LLTOPTEXT2);
	wprintw(list, "%s", mm.areaList->getDescription());
	wattrset(list, C_LLBBORD);
	waddch(list, ACS_LTEE);
	wattrset(list, C_LLHEAD);
	mvwaddstr(list, 1, 3, "   Msg#  From           To             Subject");
	//Double kludge!
	if ((mm.areaList->getType() == bwrep) ||
	    ((mm.resourceObject->getValue(packMode) == PM_QWK) &&
	      !strcmp("PERS", mm.areaList->getShortName())))
		mvwaddstr(list, 1, COLS - 19, "Area");
	wnoutrefresh(list);

	if (mm.areaList->getType() == bwrep)
		helpwindow.reply_area_letterlist();
	else
		helpwindow.letterlist();
}

void LetterListWindow::SetCurrent(void)
{
	letterList->gotoLetter(active + 1);
}

void LetterListWindow::Delete(void)
{
	delwin(list);
	helpwindow.Delete();
	touchwin(screen);
	wnoutrefresh(screen);
}

//****************************************************************************
//*                                                                          *
//*                                LETTERWINDOW                              *
//*                                                                          *
//****************************************************************************

Line::Line(void)
{
	next = NULL;
}

net_address LetterWindow::PickNetAddr(void)
{
	unsigned int i;
	Line *line;
	net_address result =
	{0, 0, 0, 0};

	line = head.next;
	while (strncmp(" * Origin:", line->text, 10) && line->next)
		line = line->next;
	//we have the Origin line

	i = strlen(line->text);
	while (((line->text[i - 1]) != '(') && i > 0)
		i--;
	//we have the opening bracket

	while ((i < strlen(line->text)) && ((line->text[i] < '0') ||
					    (line->text[i] > '9')))
		i++;
	//we have the begining of the address

	if (sscanf(&line->text[i], "%d:%d/%d.%d", &result.zone,
		   &result.net,
		   &result.node,
		   &result.point) == 3)
		result.point = 0;

	return result;
}

void LetterWindow::set_columns(int arg)
{
	columns = arg;
}

void LetterWindow::ReDraw(void)
{
	touchwin(header);
	wnoutrefresh(header);
	touchwin(text);
	wnoutrefresh(text);
}

void LetterWindow::DestroyChain(void)
{
	Line *tmp;

	curr = head.next;
	while (curr) {
		delete curr->text;
		tmp = curr;
		curr = curr->next;
		delete tmp;
	}
	head.next = NULL;
	letter_in_chain = 0;
	NumOfLines = 0;
}

void LetterWindow::MakeChain(void)
{
	char *message;
	int end = 0;
	int j = 0;		//pointer in the text
	int k;			//pointer in the line 
	int begin;		//pointer to the begining of the last word

	tagline[0] = 0;		// reset tagline to zero

	DestroyChain();
	message = (char *) letterList->getBody();
	charconv_in(message);
	letter_in_chain = letterList->getCurrent();
	curr = &head;

	while (!end) {
		//new line
		curr->next = new Line;
		curr = curr->next;
		curr->text = new char[columns + 1];	//+1 for the '\0'

		NumOfLines++;

		begin = 0;
		k = 0;
		while ((message[j] != 13) && (message[j] != 10) && (!end)
		&& (k < columns)) {
			if (!begin && (message[j] != ' '))
				begin = j;
			if (message[j] == ' ')
				begin = 0;
			//put character to its place
			if (message[j])
				curr->text[k++] = message[j++];
			else
				end = 1;
		}
		if (k == columns && begin && (j - begin) < columns) {
			//we must go back to 'begin'
			k -= j - begin;
			j = begin;
		}
		if ((message[j] == 10) || (message[j] == 13))
			j++;	//to omit the '\n'

		if ((message[j - 1] == 13) && (message[j] == 10))
			j++;
		curr->text[k] = '\0';
	}
}

void LetterWindow::MarkToggle()
{
	int stat;
	stat = letterList->getStatus();
	stat ^= MS_MARKED;
	letterList->setStatus(stat);
	DrawHeader();
}

void LetterWindow::ReadToggle()
{
	int stat;
	stat = letterList->getStatus();
	stat ^= MS_READ;
	letterList->setStatus(stat);
	DrawHeader();
}

void LetterWindow::Draw()
{
	letterList->setRead();	// nem ide kene? de.

	DrawHeader();
	DrawBody();
}

void LetterWindow::DrawHeader()
{
	int stat;

	wmove(header, 1, 0);
	wbkgdset(header, C_LHEADTEXT);
	wclrtobot(header);
	wbkgdset(header, 0);
	wattrset(header, C_LHEADTEXT);
	mvwaddstr(header, 1, 2, "Msg#:");
	mvwaddstr(header, 2, 2, "From:");
	mvwaddstr(header, 3, 2, "  To:");
	mvwaddstr(header, 4, 2, "Subj:");
	mvwaddstr(header, 1, COLS - 33, "Date:");
	mvwaddstr(header, 2, COLS - 33, "Line:");
	mvwaddstr(header, 3, COLS - 33, "Stat:");
	stat = letterList->getStatus();

	wattrset(header, C_LHFLAGS);
	wattrset(header, (letterList->getPrivate()) ? C_LHFLAGSHI : C_LHFLAGS);
	mvwaddstr(header, 3, COLS - 27, "Pvt");
	wattrset(header, (stat & MS_READ) ? C_LHFLAGSHI : C_LHFLAGS);
	mvwaddstr(header, 3, COLS - 23, "Read");
	wattrset(header, (stat & MS_REPLIED) ? C_LHFLAGSHI : C_LHFLAGS);
	mvwaddstr(header, 3, COLS - 18, "Replied");
	wattrset(header, (stat & MS_MARKED) ? C_LHFLAGSHI : C_LHFLAGS);
	mvwaddstr(header, 3, COLS - 10, "Marked");

	wattrset(header, C_LHMSGNUM);
	mvwprintw(header, 1, 8, "%d (%d of %d)   ", letterList->getMsgNum(),
		letterList->getCurrent(), mm.areaList->getNoOfLetters());
	mvwprintw(header, 2, COLS-27, "%5d/%-21d", position + 1, NumOfLines);

	char *tmp = new char[COLS];

	wattrset(header, C_LHFROM);
	//sprintf(tmp, "%-71s", letterList->getFrom());
	strcpy(tmp, letterList->getFrom());
	charconv_in(tmp);
	mvwaddstr(header, 2, 8, tmp);
	//for (int i = 0; i < COLS - 79; i++)
	//	waddch(header, ' ');
	wattrset(header, C_LHTO);
	strcpy(tmp, letterList->getTo());
	charconv_in(tmp);
	mvwaddstr(header, 3, 8, tmp);
	sprintf(tmp, "%-71s", letterList->getSubject());
	charconv_in(tmp);
	wattrset(header, C_LHSUBJ);
	mvwaddstr(header, 4, 8, tmp);
	for (int i = 0; i < COLS - 79; i++)
		waddch(header, ' ');
	strcpy(tmp, letterList->getDate());
	charconv_in(tmp);
	wattrset(header, C_LHDATE);
	mvwaddstr(header, 1, COLS - 27, tmp);

	delete tmp;

	wnoutrefresh(header);
}

void LetterWindow::DrawBody()
{
	int i, j, c;
	char test, *t, *r;

	if (letter_in_chain != letterList->getCurrent())
		MakeChain();

	// Although this is part of the header window, I draw it here
	// because it needs to be updated whenever the position of the
	// text window is:

	wattrset(header, C_LHMSGNUM);
	mvwprintw(header, 2, COLS-27, "%5d/%-21d", position + 1, NumOfLines);
	wnoutrefresh(header);

	touchwin(text);

	//find positionth Line
	curr = head.next;
	for (i = 0; i < (int) position; i++)
		curr = curr->next;
	for (i = 1; i <= y; i++)
		if (curr) {
			wattrset(text, C_LTEXT);
			for (int m = 0; m < 5; m++)
				if (curr->text[m] == '>')	//quoting
					wattrset(text, C_LQTEXT);
			test = curr->text[0];
			if ((curr->text[1] == test) &&
			    (curr->text[2] == test) && ((curr->text[3] == ' ')
				 || (curr->text[3] == '\r')))
				switch (test) {
				case '.':
					wattrset(text, C_LTAGLINE); //tagline
					t = &curr->text[4];
					r = tagline;
					for (c = 0; (*t) && (*t != '\r') &&
					     (*t != '\n') && c < 79;
						t++, r++, c++)
						   *r = *t;
					*r = 0;
					break;
				case '-':
					wattrset(text, C_LTEAR);
					//mailer prog
					break;
				case '~':
					wattrset(text, C_LTEAR);
					//mailer prog
					break;
			} else if (!strncmp(curr->text, " * Origin:", 10))
				wattrset(text, C_LORIGIN);	//origin line

			mvwaddstr(text, i, 0, curr->text);
			for (j = 0; j < (columns - (int) strlen(curr->text));
			 j++)
				waddch(text, ' ');

			curr = curr->next;
		} else
			for (j = 0; j < x; j++)
				mvwaddch(text, i, j, ' ');

	wattrset(text, C_LBOTTSTAT);
	mvwprintw(text, LINES - 6, 0, " %-71s", mm.areaList->getDescription());

	for (i = 0; i < (columns - 86); i++)
		waddch(text, ' ');
	mvwaddstr(text, LINES - 6, COLS - 15, "F1 or ? - Help ");

	wnoutrefresh(text);
}

void LetterWindow::MakeActive(void)
{
	DestroyChain();
	header = newwin(4 + 1, COLS, 0, 0);
	text = newwin(LINES - 5, COLS, 5, 0);
	wattrset(header, C_LBOTTSTAT);

	wprintw(header, " " MM_TOPHEADER, MM_NAME, MM_MAJOR, MM_MINOR);
	for (int i = 0; i < columns-29; i++)
		waddch(header, ' ');

	wbkgdset(text, C_LHBORDER);
	werase(text);
	wbkgdset(text, 0);
	// mvwaddch(text, 0, 0, ACS_LLCORNER);
	// whline(text, ACS_HLINE, COLS-2);
	// mvwaddch(text, 0, COLS-1, ACS_LRCORNER);

	getmaxyx(text, y, x);
	y -= 2;			//the border and the status line

	position = 0;
	Draw();
}

void LetterWindow::Next(void)
{
	if (letterList->getCurrent() < mm.areaList->getNoOfLetters()) {
		letters.Move(DOWN);
		letterList->gotoLetter(letterList->getCurrent() + 1);
		position = 0;
		Draw();
	} else {
		interface->back();
		doupdate();
		interface->back();
		doupdate();
		interface->RIGHT_ARROW();
	}
}

void LetterWindow::Previous(void)
{
	if (letterList->getCurrent() > 1) {
		letters.Move(UP);
		letterList->gotoLetter(letterList->getCurrent() - 1);
		position = 0;
		Draw();
	} else {
		interface->back();
		doupdate();
		interface->back();
		doupdate();
		interface->ikeypad(UP);
	}
}

void LetterWindow::Move(direction dir)
{
	switch (dir) {
	case UP:
		if (position > 0) {
			position--;
			DrawBody();
		}
		break;
	case DOWN:
		if (position < NumOfLines - y) {
			position++;
			DrawBody();
		}
		break;
	case HOME:
		position = 0;
		DrawBody();
		break;
	case END:
		position = NumOfLines - y;
		DrawBody();
		break;
	case PGUP:
		if (y < position)
			position -= y;
		else
			position = 0;
		DrawBody();
		break;
	case PGDN:
		if (position < NumOfLines - y) {
			position += y;
			if (position > NumOfLines - y)
				position = NumOfLines - y;
			DrawBody();
		}
		break;
	}
}

void LetterWindow::NextDown(void)
{
	position += y;
	if (position >= NumOfLines)
		Next();
	else
		DrawBody();
}

void LetterWindow::Delete(void)
{
	if (header)
		delwin(header);
	if (text)
		delwin(text);
	touchwin(screen);
	wnoutrefresh(screen);
}

void LetterWindow::Save(void)
{
	char filename[255];
	WINDOW *question;
	FILE *fd;

	question = makeWindow(5, 60, 10, 10, C_LLSAVETEXT);
	mvwaddstr(question, 1, 2, "Save to file:");
	mvwaddch(question, 2, 2, '<');
	mvwaddch(question, 2, 57, '>');
	mvwaddstr(question, 3, 30, "<ESC> (pause) to cancel");
	sprintf(filename, "%s/%s.%d", mm.resourceObject->get(bwSaveDir),
		mm.areaList->getName(), letterList->getCurrent());
	// getLetterID()); -- majd msgnum kell!

	if (getstring(question, 2, 3, filename, 54, C_LLSAVETEXT, C_LLSAVEGET))
	{
		chdir(mm.resourceObject->get(bwSaveDir));
		if (!(fd = fopen(filename, "a"))) {
#ifdef SOUND_EFFECTS
			mysystem("zcat /var/lib/mmail/ohno.wav >/dev/dsp &");
#endif
		} else {
			write_to_file(fd);
			fclose(fd);
		}
	}
	delwin(question);
	Draw();
}

void LetterWindow::set_Letter_Params(net_address * nm, char to[30])
{
	NM.zone = nm->zone;
	NM.net = nm->net;
	NM.node = nm->node;
	NM.point = nm->point;
	if (to)
		strncpy(To, to, 29);
	else
		To[0] = '\0';
}

void LetterWindow::set_Letter_Params(int area, char param_key)
{
	key = param_key;
	if (key == 'N') {
		//find netmail area
		int i = 1;
		while ((i <= mm.areaList->noOfAreas()) &&
		       strcmp(mm.areaList->getShortName(), "NET"))
			mm.areaList->gotoArea(i++);
		if (!strcmp(mm.areaList->getShortName(), "NET"))
			replyto_area = mm.areaList->getAreaNo();
		else
			//netmail area not found, we should warn the user
			replyto_area = area;
		areas.Reset_areaList();
	} else
		replyto_area = area;
}

void LetterWindow::EnterLetter(void)
{
	WINDOW *rep_header;
	FILE *tmp, *reply;
	char tmp_filename[255], reply_filename[255], command[255];
	char FROM[26], TO[26], SUBJ[72], TMP[26], TMP2[26], mg[4];
	int end, i;
	TaglineWindow *taglines;

	rep_header = makeWindow(5, COLS - 2, (LINES / 2) - 3, 1, C_LETEXT);
	mvwaddstr(rep_header, 1, 2, "From:");
	mvwaddstr(rep_header, 2, 2, "  To:");
	mvwaddstr(rep_header, 3, 2, "Subj:");
	wrefresh(rep_header);

	mm.areaList->gotoArea(replyto_area);

	int usealias = mm.areaList->getUseAlias();
	if (usealias) {
		strcpy(FROM, mm.resourceObject->get(bwAliasName));
		if (!FROM[0])
			usealias = 0;
	}
	if (!usealias)
		strcpy(FROM, mm.resourceObject->get(bwLoginName));

	charconv_in(FROM);
	if (strcmp(mm.areaList->getShortName(), "NET"))
		getstring(rep_header, 1, 8, FROM, 20, C_LEGET2, C_LEGET2);
	else
		mvwaddstr(rep_header, 1, 8, FROM);
	if ((key == 'R') || (key == 'N'))
		strcpy(TO, letterList->getFrom());

	if (key == 'R')
		letterList->setStatus(letterList->getStatus() | MS_REPLIED);

	if (key == 'O')
		strcpy(TO, letterList->getTo());

	if (key == 'E') {
		if (!To[0])
			strcpy(TO, "All");
		else
			strcpy(TO, To);
		SUBJ[0] = '\0';	//we don't have subject yet

	} else
		strcpy(SUBJ, letterList->getSubject());

	charconv_in(TO);
	if (strcmp(mm.areaList->getShortName(), "NET"))
		getstring(rep_header, 2, 8, TO, 20, C_LEGET1, C_LEGET1);
	else if (NM.point)
		mvwprintw(rep_header, 2, 8, "%s (%d:%d/%d.%d)", TO,
			  NM.zone,
			  NM.net,
			  NM.node,
			  NM.point);
	else
		mvwprintw(rep_header, 2, 8, "%s (%d:%d/%d)", TO,
			  NM.zone,
			  NM.net,
			  NM.node);
	charconv_in(SUBJ);
	getstring(rep_header, 3, 8, SUBJ, 69, C_LEGET2, C_LEGET2);
	delwin(rep_header);

	int privat;

	if (key == 'N')
		privat = 1;
	else {
		if (key == 'E')
			privat = 0;
		else
			privat = letterList->getPrivate();
		if (!WarningWindow(
			"Make letter private?", privat ? "Yes" : "No",
				   privat ? "No" : "Yes"))
			privat = !privat;
	}

	tmpnam(reply_filename);
	reply = fopen(reply_filename, "w");
	if (key == 'E')
		fclose(reply);	//this was only delete the file
	else {
		strcpy(TMP, letterList->getFrom());
		strcpy(TMP2, letterList->getTo());
		charconv_in(TMP);
		charconv_in(TMP2);
		fprintf(reply, "-=> %s wrote to %s <=-\n", TMP, TMP2);
		fclose(reply);
		tmpnam(tmp_filename);
		if ((tmp = fopen(tmp_filename, "w"))) {
			columns = 75;
			write_to_file(tmp, 0);
			columns = COLS;
			fclose(tmp);
		}
		strncpy(mg, letterList->getFrom(), 2);
		mg[2] = '\0';
		mg[3] = '\0';
		strcpy(TMP, letterList->getFrom());
		i = 1;
		for (int j = 1; j < 3; j++) {
			end = 0;
			while (TMP[i] && !end) {
				if ((TMP[i - 1] == ' ') && (TMP[i] != ' ')) {
					mg[j] = TMP[i];
					if (j == 1)
						end = 1;
				}
				i++;
			}
		}
		charconv_in(mg);
		sprintf(command, "%s%s%s%s%s%s", "sed 's/^/ ", mg, "> /' ",
			tmp_filename, " >> ", reply_filename);
		mysystem(command);
		remove(tmp_filename);
	}
	sprintf(command, "%s %s", mm.resourceObject->get(editor),
		reply_filename);
	mysystem(command);
	switch (interface->active()) {
	case packetlist:
		break;
	case arealist:
		areas.ReDraw();
		break;
	case letterlist:
		letters.ReDraw();
		break;
	case threadlist:
		break;
	case letter:
		letterwindow.DrawHeader();	// needed. but why?
		letterwindow.ReDraw();
		break;
	case letter_help:
		break;
	case littlearealist:
		break;
	}

	reply = fopen(reply_filename, "r");
	fseek(reply, -1, SEEK_END);
	int addlf = (fgetc(reply) != '\n');
	fclose(reply);

	reply = fopen(reply_filename, "a+");
	if (addlf)
		fputc('\n', reply);

	// Signature

	char *sg = (char *) mm.resourceObject->get(sigFile);
	FILE *s;
	char bufx[82];
	if (sg && sg[0]) {
		s = fopen(sg, "rt");
		if (s) {
			while (fgets(bufx, 79, s)) {
				fputs(bufx, reply);
				addlf = (bufx[strlen(bufx) - 1] != '\n');
			}
			if (addlf)
				fputc('\n', reply);
			fclose(s);
		}
	}
	// Tagline

	tagline[0] = '\0';
	taglines = new TaglineWindow;
	if (tagline[0])
		fprintf(reply, "... %s", tagline);

	// Tearline (not for Blue Wave -- it does its own)

	if (mm.resourceObject->getValue(packMode) == PM_QWK) {
		struct utsname buf;
		uname(&buf);
		fprintf(reply, "\n--- %s/%s v%d.%d", MM_NAME, buf.sysname,
			MM_MAJOR, MM_MINOR);
	}
	fclose(reply);

	// Reconvert the text

	reply = fopen(reply_filename, "r");
	fseek(reply, 0, SEEK_END);
	long replen = ftell(reply);
	rewind(reply);
	char *body = new char[replen + 1];
	fread(body, 1, replen, reply);
	fclose(reply);
	body[replen] = '\0';
	charconv_out(body);
	reply = fopen(reply_filename, "w");
	fwrite(body, 1, replen, reply);
	fclose(reply);

	switch (interface->active()) {	// ide is kell. ?
	case packetlist:
		break;
	case arealist:
		areas.ReDraw();
		break;
	case letterlist:
		letters.ReDraw();
		break;
	case threadlist:
		break;
	case letter:
		letterwindow.DrawHeader();	// needed. but why?
		letterwindow.ReDraw();
		break;
	case letter_help:
		break;
	case littlearealist:
		break;
	}

	charconv_out(FROM);
	charconv_out(TO);
	charconv_out(SUBJ);

	net_address *NM2;

	if (!strcmp(mm.areaList->getShortName(), "NET"))  // Fix this!
		NM2 = &NM;
	else
		NM2 = NULL;

	int replyto_num;
	if (key == 'E')
		replyto_num = 0;
	else
		replyto_num = letterList->getMsgNum();

	mm.areaList->enterLetter(replyto_area, FROM, TO, SUBJ, replyto_num,
		privat, NM2, reply_filename);

	areas.Reset_areaList();
	To[0] = '\0';
}

void LetterWindow::EditBody(void)
{
	char reply_filename[255];
	char command[255];
	FILE *reply;
	char *body;
	long siz;

	DestroyChain();		// current letter's chain reset

	mm.areaList->gotoArea(replyto_area);

	tmpnam(reply_filename);
	body = (char *) letterList->getBody();
	charconv_in(body);
	reply = fopen(reply_filename, "w");
	fwrite(body, strlen(body), 1, reply);
	fclose(reply);
	body = NULL;		// it will be auto-dealloc'd by next getBody

	sprintf(command, "%s %s", mm.resourceObject->get(editor),
		reply_filename);
	mysystem(command);

	reply = fopen(reply_filename, "r");
	fseek(reply, 0, SEEK_END);
	siz = ftell(reply);
	rewind(reply);
	body = new char[siz + 1];
	fread(body, siz, 1, reply);
	body[siz] = '\0';
	fclose(reply);
	if (body[siz - 1] == '\n')
		body[siz - 1] = '\0';
	charconv_out(body);
	letterList->setBody(body);
	delete body;
	remove(reply_filename);

	switch (interface->active()) {
	case packetlist:
		break;
	case arealist:
		areas.ReDraw();
		break;
	case letterlist:
		letters.ReDraw();
		break;
	case threadlist:
		break;
	case letter:
		letterwindow.DrawBody();
		letterwindow.ReDraw();
		break;
	case letter_help:
		break;
	case littlearealist:
		break;
	}
	areas.Reset_areaList();
	// kene? mm.areaList->refreshArea(mm.areaList->getAreaNo());
	// To[0] = '\0';
}


void LetterWindow::set_Tagline(char *tl)
{
	strncpy(tagline, tl, 76);
}

int LetterWindow::GetTagline(void)
{
	TaglineWindow *tags;

	tags = new TaglineWindow(1);
	tags->EnterTagline(tagline);
	delete tags;

	return 1;
}

void LetterWindow::write_to_file(FILE *fd, int NEED_HEADER)
{
	int j;
	char FROM[26], TO[26], SUBJ[72];

	if (NEED_HEADER == 1) {	//write header to file
		fputc('\n', fd);
		// Should this be COLS, or should it be hard-wired?
		for (j = 0; j < COLS; j++)
			fputc('=', fd);
		strcpy(FROM, letterList->getFrom());
		strcpy(TO, letterList->getTo());
		strcpy(SUBJ, letterList->getSubject());
		charconv_in(FROM);
		charconv_in(TO);
		charconv_in(SUBJ);
		fprintf(fd, "\n   From: %s\n     To: %s\n   Subj: %s\n", FROM,
			TO, SUBJ);
		for (j = 0; j < COLS; j++)
			fputc('-', fd);
		fputc('\n', fd);
	}
	//write chain to file
	MakeChain();
	curr = head.next;
	while (curr) {
		if (NEED_HEADER == 2)
			for (char *p = curr->text; *p; p++)
				if (*p == '`') *p = 27;
		fprintf(fd, "%s\n", curr->text);
		curr = curr->next;
	}
}

void LetterWindow::ansi_dump(void)
{
	FILE *temptext;
	char tempname[255], command[255];

	strcpy(tempname, mm.resourceObject->get(bwWorkDir));
	strcat(tempname, "/body.txt");
	temptext = fopen(tempname, "w");
	if (temptext) {
		write_to_file(temptext, 2);
		fclose(temptext);
		sprintf(command, "less -C -r %s", tempname);
		//sprintf(command, "more %s", tempname);
		mysystem(command);
		remove(tempname);
		//cout << "Press enter to continue:";
		//cin >> command;
	}
}
