/*
 * MultiMail offline mail reader
 * 

 Written by Kolossvary Tamas <thomas@tvnet.hu>
 Modified by John Zero <john@graphisoft.hu>, 
             William McBrine <wmcbrine@clark.net>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <unistd.h>

#include "interface.h"

extern Welcome		welcome;
extern PacketListWindow	packets;
extern AreaListWindow	areas;
extern LetterListWindow	letters;
extern LetterWindow	letterwindow;
extern HelpWindow	helpwindow;
extern mmail		mm;
extern packet_list	*packetList;
extern area_list	*areaList;
extern letter_list	*letterList;
extern Interface	*interface;
extern LittleAreaListWindow *littleareas;
extern AddressBook	addresses;

Interface::Interface(int mode)
{
	screen_init(mode);
	letterwindow.set_columns(COLS);
	welcome.MakeActive();
	packets.MakeActive();
	state = packetlist;
	dontcallEnterLetter = 0;
	doupdate();
}

statetype Interface::active(void)
{
	return state;
}

void Interface::select(void)
{
	switch (state) {
	case packetlist:
		if (mm.resourceObject->getValue(packMode) != PM_UNDEF) {
			unsaved_reply = 0;
			any_read = 0;
			packets.Delete();
			welcome.Delete();
			mm.selectPacket(packetList->getName());
			touchwin(screen);
			wnoutrefresh(screen);
			areas.AreaListWindow();
			areas.MakeActive();
			areas.FirstUnread();
			state = arealist;
		}
		break;
	case arealist:
		if (mm.areaList->getNoOfLetters() > 0) {
			letterList = mm.areaList->getLetterList();
			areas.Delete();
			letters.MakeActive();
			letters.FirstUnread();
			state = letterlist;
		}
		break;
	case letterlist:
		letters.SetCurrent();
		letters.Delete();
		letterwindow.MakeActive();
		any_read = 1;
		state = letter;
		break;
	case threadlist:
		break;
	case letter:
		letterwindow.Next();
		break;
	case letter_help:
		helpwindow.Delete();
		letterwindow.Draw();
		state = letter;
		break;
	case littlearealist:
		delete littleareas;
		if (!strcmp(mm.areaList->getShortName(), "NET"))
			if ((Key == 'N') || (Key == 'E'))
				addresses.MakeActive();
			else {
				net_address nm;

				nm = letterwindow.PickNetAddr();
				letterwindow.set_Letter_Params(&nm,
							       NULL);
			}
		Key = '\0';
		areas.Reset_areaList();
		letterwindow.ReDraw();
		state = letter;
		if (!dontcallEnterLetter) {
			letterwindow.EnterLetter();
			unsaved_reply = 1;
		} else
			dontcallEnterLetter = 0;
		break;
	}
}

int Interface::back(void)
{
	switch (state) {
	case packetlist:
		if (WarningWindow("Do you really want to quit?",
				  "Yes", "No")) {
			packets.Delete();
			welcome.Delete();
			return 1;
		} else {
			touchwin(screen);
			wnoutrefresh(screen);
			welcome.ReDraw();
			packets.ReDraw();
			helpwindow.redraw();
			return 0;
		}
	case arealist:
		if (any_read) {
			if (WarningWindow("Save lastread pointers?",
					  "Yes", "No"))
				mm.saveRead();
			any_read = 0;
		}
		if (unsaved_reply)
			if (WarningWindow(
		"The content of the reply area has changed. Create new packet?",
						 "Yes", "No"))
				create_reply_packet();
		areas.Delete();
		packets.MakeActive();
		welcome.MakeActive();
		state = packetlist;
		break;
	case letterlist:
		letters.Delete();
		areas.MakeActive();
		state = arealist;
		break;
	case threadlist:
		break;
	case letter:
		letterwindow.Delete();
		letters.MakeActive();
		letters.Draw();
		state = letterlist;
		break;
	case letter_help:
		helpwindow.Delete();
		letterwindow.Draw();
		state = letter;
		break;
	case littlearealist:
		delete littleareas;
		areas.Reset_areaList();
		state = letter;
		break;
	}
	return 0;
}

void Interface::ReplyPacket(void)
{
	if (state == arealist) {
		if (WarningWindow(
			"This will overwrite existing reply packet. Continue?",
				  "Yes", "No")) {
			create_reply_packet();
		} else
			areas.ReDraw();
	}
}

void Interface::create_reply_packet(void)
{
	mm.areaList->makeReply(REPLY_AREA);
	unsaved_reply = 0;
	touchwin(screen);
	wnoutrefresh(screen);
	areas.ReDraw();
	helpwindow.redraw();
}

void Interface::RIGHT_ARROW(void)
{
	switch (state) {
	case packetlist:
		packets.Move(DOWN);
		packets.Draw();
		break;
	case arealist:
		do {
			mm.areaList->gotoArea(mm.areaList->getAreaNo() + 1);
			areas.Move(DOWN);
		} while (!mm.areaList->getNoOfLetters() &&
			 (mm.areaList->getAreaNo() <
			  mm.areaList->noOfAreas()));
		areas.Draw();
		break;
	case letterlist:
		letters.NextUnread();
		break;
	case threadlist:
		break;
	case letter:
		letterwindow.Next();
		break;
	case letter_help:
		helpwindow.Delete();
		letterwindow.Draw();
		state = letter;
		break;
	case littlearealist:
		littleareas->Move(DOWN);
		littleareas->Draw();
		break;
	}
}

void Interface::LEFT_ARROW(void)
{
	switch (state) {
	case packetlist:
		packets.Move(UP);
		packets.Draw();
		break;
	case arealist:
		do {
			mm.areaList->gotoArea(mm.areaList->getAreaNo() - 1);
			areas.Move(UP);
		} while (!mm.areaList->getNoOfLetters() &&
			 (mm.areaList->getAreaNo() > 1));
		areas.Draw();
		break;
	case letterlist:
		letters.PrevUnread();
		break;
	case threadlist:
		break;
	case letter:
		letterwindow.Previous();
		break;
	case letter_help:
		helpwindow.Delete();
		letterwindow.Draw();
		state = letter;
		break;
	case littlearealist:
		littleareas->Move(UP);
		littleareas->Draw();
		break;
	}
}

void Interface::ikeypad(direction dir)
{
	switch (state) {
	case packetlist:
		packets.Move(dir);
		packets.Draw();
		break;
	case arealist:
		areas.Move(dir);
		areas.Draw();
		break;
	case letterlist:
		letters.Move(dir);
		letters.Draw();
		break;
	case threadlist:
		break;
	case letter:
		letterwindow.Move(dir);
		break;
	case letter_help:
		helpwindow.Delete();
		letterwindow.Draw();
		state = letter;
		break;
	case littlearealist:
		littleareas->Move(dir);
		littleareas->Draw();
		break;
	}
}

void Interface::redraw(void)
{
	switch (state) {
	case packetlist:
		break;
	case arealist:
		areas.Draw();
		break;
	case letterlist:
		letters.Draw();
		break;
	case letter:
		letterwindow.DrawHeader();
		letterwindow.MakeChain();
		letterwindow.DrawBody();
	case threadlist:
	case letter_help:
	case littlearealist:
		break;
	}
}

void Interface::help(void)
{
	switch (state) {
	case packetlist:
		break;
	case arealist:
		break;
	case letterlist:
		break;
	case threadlist:
		break;
	case letter:
		if (mm.areaList->getType() != bwrep)
			helpwindow.letter();
		else
			helpwindow.reply_area_letter();
		state = letter_help;
		break;
	case letter_help:
		helpwindow.Delete();
		letterwindow.Draw();
		state = letter;
		break;
	case littlearealist:
		break;
	}
}

void Interface::enterletter(char key)
{
	int stat;

	switch (state) {
	case packetlist:
		if (key == ' ') {
			ikeypad(PGDN);
			break;
		}
		if (key == 'M')
			packets.switchMode();
		break;
	case arealist:
		if (key == ' ') {
			ikeypad(PGDN);
			break;
		}
		if ((mm.areaList->getType() != bwrep) && (key == 'E')) {
			if (strcmp(mm.areaList->getShortName(), "NET")) {
				letterwindow.set_Letter_Params(
					  mm.areaList->getAreaNo(), 'E');
				letterwindow.EnterLetter();
				unsaved_reply = 1;
			} else {
				addresses.MakeActive();
				letterwindow.set_Letter_Params(
					  mm.areaList->getAreaNo(), 'E');
				if (!dontcallEnterLetter) {
					letterwindow.EnterLetter();
					unsaved_reply = 1;
				} else
					dontcallEnterLetter = 0;
			}
		}
		break;
	case letterlist:
		switch (key) {
		case ' ':
			ikeypad(PGDN);
			break;

			// Toggle read/unread and marked from letterlist

		case 'U':
		case 'M':
			letters.SetCurrent();
			stat = letterList->getStatus();
			if (key == 'U')
				stat ^= MS_READ;
			else
				stat ^= MS_MARKED;
			letterList->setStatus(stat);
			any_read = 1;
			ikeypad(DOWN);
			break;

		case 'E':
			if (mm.areaList->getType() != bwrep) {
				if (strcmp(mm.areaList->getShortName(),
				"NET")) {
					letterwindow.set_Letter_Params(
					  mm.areaList->getAreaNo(), 'E');
					letterwindow.EnterLetter();
					unsaved_reply = 1;
				} else {
					addresses.MakeActive();
					letterwindow.set_Letter_Params(
					  mm.areaList->getAreaNo(), 'E');
					if (!dontcallEnterLetter) {
						letterwindow.EnterLetter();
						unsaved_reply = 1;
					} else
						dontcallEnterLetter = 0;
				}
			}
			break;
		}
		break;
	case threadlist:
		break;
	case letter:
		if (key == ' ') {
			letterwindow.NextDown();
			break;
		}
		if (mm.areaList->getType() == bwrep) {
			if (key == 'E') {
				letterwindow.EditBody();
				unsaved_reply = 1;
			}
		} else {
			switch (key) {
			case 'M':
				letterwindow.MarkToggle();
				break;
			case 'U':
				letterwindow.ReadToggle();
				break;
			case 'R':
			case 'O':
			case 'E':
				Key = key;
				littleareas = new LittleAreaListWindow(key);
				state = littlearealist;
				break;
			case 'N':
				net_address nm;
				Key = key;
				nm = letterwindow.PickNetAddr();
				letterwindow.set_Letter_Params(
					  mm.areaList->getAreaNo(), 'N');
				letterwindow.set_Letter_Params(&nm, NULL);
				letterwindow.EnterLetter();
				unsaved_reply = 1;
				break;
			case 'T':
				letterwindow.GetTagline();
				break;
			case 1:	// CTRL-A

				letterwindow.ansi_dump();
				break;
			}
		}
		break;
	case letter_help:
		helpwindow.Delete();
		letterwindow.Draw();
		state = letter;
		break;
	case littlearealist:
		break;
	}
}

void Interface::save(void)
{
	switch (state) {
	case packetlist:
		break;
	case arealist:
		break;
	case letterlist:
		letters.Save();
		break;
	case threadlist:
		break;
	case letter:
		letterwindow.Save();
		break;
	case letter_help:
		helpwindow.Delete();
		letterwindow.Draw();
		state = letter;
		break;
	case littlearealist:
		break;
	}
}

void Interface::search(void)
{
}

void Interface::ANY_KEY(void)
{
	switch (state) {
	case packetlist:
	case arealist:
	case letterlist:
	case threadlist:
	case letter:
	case littlearealist:
		break;
	case letter_help:
		helpwindow.Delete();
		letterwindow.Draw();
		state = letter;
		break;
	}
}

void Interface::kill_letter(void)
{
	if (mm.areaList->getType() == bwrep)
		if (WarningWindow(
			"Are you sure you want to delete this letter?",
				  "Yes", "No")) {
			if (state == letterlist) {
				letters.SetCurrent();
				letters.Delete();
			}
			mm.areaList->killLetter(letterList->getMsgNum());
			unsaved_reply = 1;
			if (state == letter)
				back();
			else {
				letters.MakeActive();
				letters.Draw();
			}
			if (!mm.areaList->getNoOfLetters())
				back();
			else
				ikeypad(UP);

		} else if (state == letter)
			letterwindow.ReDraw();
		else {
			letters.Delete();
			letters.MakeActive();
			letters.Draw();
		}
}

void Interface::kill_packet(void)
{
	char tmp[128];

	sprintf(tmp, "Do you really want to delete %s?",
		packetList->getName());
	if (WarningWindow(tmp, "Yes", "No")) {
		chdir(mm.resourceObject->get(bwPacketDir));
		remove(packetList->getName());
		packets.ReList();
	}
	touchwin(screen);
	wnoutrefresh(screen);
	welcome.ReDraw();
	packets.ReDraw();
	helpwindow.redraw();
}

void Interface::kill(void)
{
	switch (state) {
	case packetlist:
		kill_packet();
		break;
	case letterlist:
	case letter:
		kill_letter();
		break;
	case littlearealist:
	case letter_help:
	case threadlist:
	case arealist:
		break;
	}
}

void Interface::addressbook(void)
{
	if (state != littlearealist)
		addresses.MakeActive(1);	//1 means NOENTER 
}

void Interface::set_dontcallEnterLetter(void)
{
	dontcallEnterLetter = 1;
}
