/*
 * LISTSERV
 * 
 * Sample program that takes a GIGO function request file (FUNCTION.REQ), checks
 * it, and then modifies the appropriate .ML files (for mailing lists).
 * 
 * Jason Fesler  December, 1990    jfesler@wmeonlin.sacbbx.com
 */

/*
 * Modification History:
 * 
 * Date     By               Description 
 * 2/21/95  David Gibbs      Added recognition of SET & REVIEW command, both commands inform requester that commands are not implemented.
 * 2/3/95   David Gibbs      Changed command recognition from 1 character to 3.
 * 4/25/94  David Gibbs      Added basic activity logging
 * 12/30/95 Jason Fesler     Added in a "confirmation" process
 * 
 */

/*
 * For simplicity sake, I include everything, so that I don't have any
 * problems with the compiler knowing what I am talking about.
 */
#include <time.h>
#include <io.h>
#include <fcntl.h>
#include <sys\stat.h>
#include <process.h>
#include <share.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <malloc.h>
#include <conio.h>
#include <ctype.h>
#include <direct.h>
#include <dos.h>
#include <stdarg.h>


/*
 * Global variables are usually a bad programming practice. However, I don't
 * give a damn.. :-)
 * 
 */

extern unsigned int _stklen = 10000;
//Borland style stack declaration,
//ymmv

struct headers {
	char            Apparently_To[256];
	char            Apparently_From[256];
	char            To[256];
	char            From[256];
	char            Subject[256];
}               header;

char            wholeparm[512] = "";

FILE           *infile, *outfile, *file;
int             amihere = 0;
int             amiheremsg = 1;

int             useconf = 0;

enum commands {
	HELP, INDEX, FAQ, SUBSCRIBE, UNSUBSCRIBE, QUERY, QUIT,
	DISCONNECT, SET, SIG, REVIEW, ADDRESS, CONFIRM, UNKNOWN
};

void logproblem(char *cmd, char *parm);


/*
 * cmpcopy looks to see if string at <source> compares to the string at
 * <token>.  If it does, the parameters after <token> are going to be copied
 * to <dest>.
 */


void
cmpcopy(char *token, char *source, char *dest)
{
	if (!strncmp(source, token, strlen(token))) {
		source += strlen(token);
		//get past the token itself
			while (*source == ' ')
			source++;
		//get past the spaces
			strcpy(dest, source);
	}
}
/*
 * readheaders reads in the top part of the file until it reaches a newline
 * character.
 * 
 */


void
readheaders(void)
{
	char            line[1024] = " ";
	memset(&header, 0, sizeof(headers));
	while ((*line) && (!feof(infile))) {

		/* remember, GIGO stands for garbage in, garbage out. */
		memset(line, 0, sizeof(line));
		fgets(line, sizeof(line) - 1, infile);

		/* fgets includes the line terminator; we need to remove it. */
		if (*line)
			if (line[strlen(line) - 1] == '\n')
				line[strlen(line) - 1] = 0;
		if (*line)
			if (line[strlen(line) - 1] == '\r')
				line[strlen(line) - 1] = 0;
		if (!*line)
			continue;	/* We got a blank line, or an eof */
		cmpcopy("Apparently-To:", line, header.Apparently_To);
		cmpcopy("Apparently-From:", line, header.Apparently_From);
		cmpcopy("To:", line, header.To);
		cmpcopy("From:", line, header.From);
		cmpcopy("Subject:", line, header.Subject);
	}
}


void
show_internal_help()
{
	fprintf(outfile,
	      "Simple help instructions for the GIGO listserv processor..\n"
	      "----------------------------------------------------------\n"
                "HELp (or ?)           This help information\n"
                "CONFIRM number        Submits confirmation number\n"
                "INDex                 Shows the list of available lists\n"
     	        "FAQ listname          Shows the FAQ (info) file for listname\n"
		"SUBscribe listname    Subscribes to listname\n"
		"UNSubscribe listname  Unsubscribes to listname\n"
		"QUEry                 Query the various lists to see which you are on\n"
		"\n"
	        "DISconnect            Disconnects from ALL lists on this server\n"
		"\n"
		"Commands may be abbreviated down to 3 characters.\n"
		"\n"
	);
}
/* show_file dumps the file to the reply.  If it does not exist, returns a 1. */

int
show_file(char *filename)
{
	//1 = error, 0 = noerror
    FILE * hfile;
    char buf[256];
	int             c;
	hfile = fopen(filename, "rt");
        if (!hfile) {
            sprintf(buf,"Unable to show_file(\"%s\"):  %s",
                    filename,strerror(errno));
            logproblem(buf,"");
            return 1;
        }
	while (!feof(hfile)) {
		c = fgetc(hfile);
		fputc(c, outfile);
	}
	fclose(file);
	return 0;
}
/*
 * Shows the faq file for filename  (filename.FAQ).  If it does not exist, it
 * will print a message to that effect.
 */

void
show_faq(char *filename)
{
	char            buf[256];
	sprintf(buf, "%s.faq", filename);
	if (show_file(buf))
		fprintf(outfile, "Sorry, no FAQ file for %s.\n", filename);
}
/*
 * Attempts to dump LISTSERV.HLP to the reply; if that doesn't work, then
 * shows the internal help file instead.
 */

void
process_help(void)
{
	if (show_file("LISTSERV.HLP"))
		show_internal_help();
}

unsigned long   getcrc(char *buf);

void            subscribe(char *filename, int unsub);

void
startsubscribe(char *filename)
{
    char            bigbuf[1024];
    char buf[1024];
    FILE *oldfile;
	FILE            *newfile;
	int             found = 0;
	unsigned long   c;
	time_t          t;
	errno = 0;
	if (!strchr(header.Apparently_From, '@')) {
		fprintf(outfile, "SUB/UNSUB commands require a valid internet style address.\n"
			"Your address appears to be %s  .. \n", header.Apparently_From);
		return;
	}
	errno = 0;
	newfile = fopen("listserv.con", "at");
	if (!newfile)
		newfile = fopen("listserv.con", "wt");
	if (!newfile) {
		fprintf(outfile, "Could not create confirmation file\n", filename);
		return;
	}

/* make sure that we even have such a list */
	sprintf(buf, "%s.ml", filename);
        oldfile = fopen(buf, "rt");
       if (!oldfile) {
         fprintf(outfile,"There is no such list called \"%s\".\n",filename);
        return;
       } else fclose(oldfile);

	fprintf(outfile, "-------------------------------------------------------------\n");
	fprintf(outfile, "You are in the process of being subscribed to list %s.\n", filename);
	fprintf(outfile, "Following will be a confirmation number; you will need\n");
	fprintf(outfile, "to send back the following to finish subscribing yourself\n");
	fprintf(outfile, "to the list.\n");
	fprintf(outfile, "\nSend back the following:\n\n");

	time(&t);
	sprintf(bigbuf, "%s %s %lu", filename, header.Apparently_From, t);
	c = getcrc(bigbuf);
	fprintf(outfile, "  confirm %08lx\n\n", c);
	fprintf(newfile, "%s %08lx\n", bigbuf, c);
	fprintf(outfile, "If you did not send a message to subscribe, you don't have to\n");
	fprintf(outfile, "do anything.  Otherwise, please send this message back as soon\n");
	fprintf(outfile, "as possible, in order to finish subscribing you to list %s.\n", filename);
	fprintf(outfile, "-------------------------------------------------------------\n");
	fclose(newfile);
}

void
finishconfirm(char *text)
{
	FILE           *confile;
	char            filename[256] = "";
	char            whofrom[256] = "";
	char            crcval[256] = "";
	char            timestarted[256] = "";
	char            oldfrom[1024] = "";
	char            line[1024];
	char           *p, *p1, *p2, *p3, *p4;

	/* text should be just the confirmation number */

	if (!text)
		text = "";
	if (text[0] == 0) {
		fprintf(outfile, "You need to specify a confirmation number.\n");
		return;
	}
	fprintf(outfile, "Looking for confirmation number %s\n", text);
	strcpy(oldfrom, header.Apparently_From);
	confile = fopen("listserv.con", "rt");
	if (!confile) {
notfound:
		fprintf(outfile, "You are not currently in the confirmation file.\n");
		fprintf(outfile, "Please send a SUBSCRIBE listname  message to start\n");
		fprintf(outfile, "the subscription process.\n");
		return;
	}
	while (!feof(confile)) {
		memset(line, 0, sizeof(line));
		fgets(line, sizeof(line), confile);
		if (p = strpbrk(line, "\r\n"))
			*p = 0;
		p1 = strtok(line, " ");
		p2 = strtok(NULL, " ");
		p3 = strtok(NULL, " ");
		p4 = strtok(NULL, " ");
		if (!p4)
			continue;
		if (stricmp(p4, text) == 0) {
			strcpy(filename, p1);
			strcpy(whofrom, p2);
			strcpy(timestarted, p3);
			strcpy(crcval, p4);
		}
	}
	fclose(confile);
	if (filename[0] == 0) {
		goto notfound;
	}
	fprintf(outfile, "Found subscription for list %s, at address %s.\n",
		filename, whofrom);
	strcpy(header.Apparently_From, whofrom);
	subscribe(filename, 0);
	strcpy(header.Apparently_From, oldfrom);
}



/*
 * Subscribes to filename.ML... (routine should only specify the 8 character
 * name, and not the extension).  If UNSUB==1, then the user is unsubscribing
 * instead.  We get to reuse most of the code this way.
 */

void
subscribe(char *filename, int unsub)
{
	char            buf[256];
	char            line[512];
	FILE           *oldfile, *newfile;
	int             found = 0;
	errno = 0;
	if (!strchr(header.Apparently_From, '@')) {
		fprintf(outfile, "SUB/UNSUB commands require a valid internet style address.\n"
			"Your address appears to be %s  .. \n", header.Apparently_From);
		return;
	}
	unlink("LISTSERV.$$$");
	errno = 0;
        sprintf(buf, "%s.ml", filename);
        
        oldfile = fopen(buf,"rt");
        if (!oldfile) {
            logproblem("Could not open list file: ",strerror(errno));
            fprintf(outfile,"There is no such list.\n");
                    return;
        }
        fclose(oldfile);
        
	if (rename(buf, "LISTSERV.$$$")!=0) {
            logproblem("Could not rename list file:",strerror(errno));
            fprintf(outfile,"We experienced an error, please tell the operator!\n");
            fprintf(outfile,"Could not rename list file:%s\n",strerror(errno));
            rename("LISTSERV.$$$",buf);
            return;
        }
        errno = 0;
        oldfile = fopen("LISTSERV.$$$", "rt");
        if (!oldfile) {
            logproblem("Could not open listserv.$$$:",strerror(errno));
            fprintf(outfile,"We experienced an error, please tell the operator!\n");
            fprintf(outfile,"Could not open listserv.$$$:%s\n",strerror(errno));
            rename("LISTSERV.$$$",buf);
            return;
        }
	newfile = fopen(buf, "wt");
        if (!newfile) {
            logproblem("Could not create new file:",strerror(errno));
            fprintf(outfile,"We experienced an error, please tell the operator!\n");
            fprintf(outfile,"Could not create new list file:%s\n",strerror(errno));
            fprintf(outfile, "Could not access mailing list \"%s\".\n", filename);
            fclose(oldfile);
            rename("LISTSERV.$$$",buf);
		return;
	}
	if (!unsub)
		fprintf(newfile, "%s\n", header.Apparently_From);

	while (!feof(oldfile)) {
		memset(line, 0, sizeof(line));
		fgets(line, sizeof(line), oldfile);
		if (line[0])
			if (line[strlen(line) - 1] == '\n')
				line[strlen(line) - 1] = 0;
		if (line[0])
			if (line[strlen(line) - 1] == '\r')
				line[strlen(line) - 1] = 0;
		if (!line[0])
			continue;
		if (stricmp(header.Apparently_From, line) != NULL)
			fprintf(newfile, "%s\n", line);
		else
			found = 1;
	}
	if (unsub) {
		if (found)
			fprintf(outfile, "List %s:  Unsubscribed.\n", filename);
		else
			fprintf(outfile, "List %s:  Already unsubscribed.\n", filename);
	} else {
		if (found)
			fprintf(outfile, "List %s:  Already subscribed.\n", filename);
		else
			fprintf(outfile, "List %s:  Subscribed.\n", filename);
		show_faq(filename);
	}
	fclose(oldfile);
	fclose(newfile);
}


/*
 * All of the answer_*() functions are used with the check_multiple(function
 * pointer) command.  check_multiple scans the directory for all .ML files,
 * and runs the routine specified passing the parameter of the list name in
 * question.
 * 
 */

/* answer_unsubscribe works with check_multiple to globally unsubscribe. */

void
answer_unsubscribe(char *filename)
{
	subscribe(filename, 1);
}
/*
 * answer_amihere works with check_multiple to globally find out if a person
 * is in each list.  A message is displayed for each area that the user is
 * in; a global integer "amihere" is also updated.
 */

void
answer_amihere(char *filename)
{
	char            buf[256];
	char            line[512];
	sprintf(buf, "%s.ml", filename);
	file = fopen(buf, "rt");
	if (!file)
		return;
	while (!feof(file)) {
		memset(line, 0, sizeof(line));
		fgets(line, sizeof(line), file);
		if (line[0])
			if (line[strlen(line) - 1] == '\n')
				line[strlen(line) - 1] = 0;
		if (line[0])
			if (line[strlen(line) - 1] == '\r')
				line[strlen(line) - 1] = 0;
		if (!line[0])
			continue;
		if (stricmp(header.Apparently_From, line) == NULL) {
			if (amiheremsg)
				fprintf(outfile, "You are in the %s list.\n", filename);
			amihere++;
			break;
		}
	}
	fclose(file);
}
/*
 * answer_index works with check_multiple to tell the user what mailing lists
 * there are.  This is only used if the file LISTSERV.IDX is not available
 * (hence the manual check).
 */

void
answer_index(char *name)
{
	fprintf(outfile, "  %s\n", name);
}
/*
 * check_multiple is a special function..
 * 
 * It takes as it's parameter the address to another function(char*).
 * 
 * All answer_*(char*) functions use this procedure.
 * 
 * check_multiple will run that function once for every mailing list in the
 * current directory.  It has multiple uses; it can make an index of the
 * lists, globally desubscribe, and check to see which lists a user is in.
 * This method of handling it means that I don't have to have 3-4 (maybe more
 * in the future) copies of the directory-reading code.
 * 
 */

void
check_multiple(void func(char *))
{

#if defined(__TURBOC__)
	struct ffblk    ffblk;
	int             done;
	char            buf[256];

	done = findfirst("*.*", &ffblk, 0);
	while (!done) {
            strcpy(buf, ffblk.ff_name);
            strupr(buf);
		if (strstr(buf, ".ML")) {
			if (strchr(buf, '.'))
				*strchr(buf, '.') = 0;
			func(buf);
		}
		done = findnext(&ffblk);
	}
#else

	DIR            *dir;
	struct dirent  *ent;
	char            buf[256];
	errno = 0;
	if ((dir = opendir("*.*")) == NULL) {
		fprintf(stdout, "Unable to open directory\n");
		return;
	}
	while ((ent = readdir(dir)) != NULL) {
            strcpy(buf, ent->d_name);
            strupr(buf);
		if (strstr(buf, ".ML")) {
			if (strchr(buf, '.'))
				*strchr(buf, '.') = 0;
			func(buf);
		}
	}
#endif
}

void
process_query(void)
{
	amiheremsg = 1;
	amihere = 0;
	fprintf(outfile, "Checking to see which mailing lists you are in...\n");
	check_multiple(answer_amihere);
	if (amihere)
		fprintf(outfile, "-----\n"
			"Total of %u lists.\n\n", amihere);
	else
		fprintf(outfile, "You are in no lists with your present email address.\n");
}
void
process_index(void)
{
	if (!show_file("LISTSERV.IDX"))
		return;
	//we had an ascii one handy

		fprintf(outfile, "Checking to see which mailing lists are available...\n");
	check_multiple(answer_index);
	fprintf(outfile, "End of lists.\n");
}

commands
test_command(char *cmd)
{
	if (strnicmp(cmd, "HEL", 3) == 0)
		return HELP;
	if (strnicmp(cmd, "?", 1) == 0)
		return HELP;
	if (strnicmp(cmd, "IND", 3) == 0)
		return INDEX;
	if (strnicmp(cmd, "FAQ", 3) == 0)
		return FAQ;
	if (strnicmp(cmd, "SUB", 3) == 0)
		return SUBSCRIBE;
	if (strnicmp(cmd, "UNS", 3) == 0)
		return UNSUBSCRIBE;
	if (strnicmp(cmd, "CON", 3) == 0)
		return CONFIRM;
	if (strnicmp(cmd, "QUE", 3) == 0)
		return QUERY;
	if (strnicmp(cmd, "QUI", 3) == 0)
		return QUIT;
	if (strnicmp(cmd, "DIS", 3) == 0)
		return DISCONNECT;
	if (strnicmp(cmd, "-", 1) == 0)
		return SIG;
	if (strnicmp(cmd, "SET", 3) == 0)
		return SET;
	if (strnicmp(cmd, "REV", 3) == 0)
		return REVIEW;
	if (strnicmp(cmd, "ADD", 3) == 0)
		return ADDRESS;

	return UNKNOWN;
}

void logproblem(char *cmd, char *parm)
{
	char            buf[256];
	time_t          time_of_day;

        FILE           *logfile;
    
    	logfile = fopen("LISTSERV.LOG", "at");

	tzset();
	time(&time_of_day);
	strftime(buf, sizeof(buf), "%m-%d-%y  %H:%M:%S", localtime(&time_of_day));
	fprintf(logfile, "%s  Userid: %s, command: %s %s\n",
		buf, header.Apparently_From, cmd, parm);
	fclose(logfile);
	strupr(cmd);
	strupr(parm);

}

void
Process_Line(char *cmd, char *parm)
{
    char buf[1024];
        logproblem(cmd,parm);
	switch (test_command(cmd)) {
		//dmg 2 / 3 / 95
	case HELP:
		//dmg 2 / 3 / 95
			process_help();
		break;
	case INDEX:
		process_index();
		break;
	case FAQ:
		show_faq(parm);
		break;
	case SUBSCRIBE:
		if (useconf)
			startsubscribe(parm);
		else
			subscribe(parm, 0);
		break;
	case UNSUBSCRIBE:
		subscribe(parm, 1);
		break;
	case CONFIRM:
		finishconfirm(parm);
		break;
	case QUERY:
		process_query();
		break;
	case QUIT:
		fprintf(outfile, "QUIT: End of listserv request from user; all other input ignored.\n");
		goto sigfound;
	case DISCONNECT:
		fprintf(outfile, "Disconnecting you from all lists.\n");
		check_multiple(answer_unsubscribe);
		break;
	case SIG:
		fprintf(outfile, "Signature or tearline detected, end of process.\n");
sigfound:
		while (!feof(infile))
			fgets(buf, sizeof(buf), infile);
		return;
	case SET:
	case REVIEW:
		fprintf(outfile, "Sorry, the requested command not implemented in this LISTSERV");
		break;
	case ADDRESS:
		strcpy(header.Apparently_From, wholeparm);
		fprintf(outfile, "ADDRESS %s accepted for following commands.\n", header.Apparently_From);
		break;
	default:
		fprintf(outfile, "I don't recognize that command.  HELP will give you more information.\n");
	}
}

void
Process_Lines(void)
{
	char            line[256];
	char           *cmd;
	char           *parm;
	char           *genpurpose;
	while (!feof(infile)) {
		memset(line, 0, sizeof(line));
		fgets(line, sizeof(line), infile);
		if (line[0])
			if (line[strlen(line) - 1] == '\n')
				line[strlen(line) - 1] = 0;
		if (line[0])
			if (line[strlen(line) - 1] == '\r')
				line[strlen(line) - 1] = 0;
		if (!line[0])
			continue;
		fprintf(outfile, "\n>%s\n", line);
		printf(">%s\n", line);
		cmd = strtok(line, " \x09");
		parm = strtok(NULL, " \x09");
		if (!cmd)
			cmd = "";
		if (!parm)
			parm = "";
		strcpy(wholeparm, parm);
		while ((genpurpose = strpbrk(parm, ":/\\")) != NULL)
			parm = genpurpose + 1;
		while ((genpurpose = strpbrk(parm, "?*")) != NULL)
			*genpurpose = '_';
		if ((genpurpose = strchr(parm, '.')) != NULL)
			*genpurpose = 0;
		//truncate
			Process_Line(cmd, parm);
	}
}

void
needhelp(char *errmsg)
{
	fprintf(stderr, "Error! %s\n", errmsg);
	fprintf(stderr, "LISTSERV is meant to be run as a function request under GIGO.\n"
		"Please see the LISTSERV documentation for proper usage.\n\n\n(Pausing 2 seconds)\n");

	delay(2000);
	exit(1);
}

void checkenv(void) {
  char buf[1024];
  char *p;
  p=getenv("LISTSERV");
  if (!p) return;
  strcpy(buf,p);
  p=strtok(buf," \t\r\n");
  while(p) {
   if (stricmp(p,"CONFIRM")==NULL)  useconf=1;
   else {
     fprintf(stderr,"Unknown LISTSERV option in environment: %s\n",p);
   }
   p=strtok(NULL," \t\r\n");
  }

}

void
main(int _argc, char *_argv[])
{
	int             i;
	fprintf(stderr, "LISTSERV processor for GIGO compile " __DATE__ ".\n");

	infile = fopen("FUNCTION.REQ", "rt");
	if (!infile)
		needhelp("FUNCTION.REQ not available for input.");
	outfile = fopen("FUNCTION.REP", "wt");

	if (!outfile)
		needhelp("FUNCTION.REP not available for output.");
        checkenv();
	readheaders();

	if (header.Apparently_From[0] == 0)
		needhelp("Apparently-From: line missing from FUNCTION.REQ file");
	printf("Function request for: %s\n", header.Apparently_From);
	if ((strstr(header.Apparently_From, "listserv")) ||
	    (strstr(header.Apparently_From, "LISTSERV"))) {
		fclose(outfile);
		fclose(infile);
		unlink("FUNCTION.REP");
		fprintf(stderr, "We are NOT Going to reply to other LISTSERV devices!");
		exit(1);
	}
	if (show_file("LISTSERV.TXT"))
		fprintf(outfile,
			"Thank you for using GIGO's listserv processor. Your commands are being quoted\n"
			"below, followed by the processor's responses.  For this process, we are using\n"
		"your email address of \"%s\".\n\n", header.Apparently_From);
	else
		fprintf(outfile, "\nFor this process, we are using\n"
		"your email address of \"%s\".\n\n", header.Apparently_From);


	if (_argc > 2) {
		switch (test_command(_argv[1])) {
		case SUBSCRIBE:
			for (i = 2; i < _argc; i++)
				Process_Line("SUB", _argv[i]);
			break;
		case UNSUBSCRIBE:
			for (i = 2; i < _argc; i++)
				Process_Line("UNS", _argv[i]);
			break;
		default:
			needhelp("Invalid command line option (rtfm).");
		}
	} else if (_argc == 2)
		needhelp("Not enough parameters or invalid command line option.");
	else
		Process_Lines();
	fclose(infile);
	fclose(outfile);
}
