/*
 * WAV2GSM.C, V1.00
 *
 * Converts WAV files to USRobotics GSM format
 *
 * Public Domain by Tamas Vincze
 *
 * The original GSM codec is made by Jutta Degener and Carsten Bormann.
 * The GSM library used here is a port of GSM 06.10, patchlevel 4, to
 * MS-DOS by Richard Elofsson.
 *
 * Notes:
 *
 * IMPORTANT: for best results, the input file should be a mono WAV,
 * sampled at 8 kHz using 16-bit samples. Otherwise some basic conversions
 * will be done before sending the sample to the GSM coder:
 *  - the left and right channel of stereo WAVs are mixed;
 *  - 8-bit samples are extended to 16-bit by zeroing the low-order byte;
 *  - sampling rates greater than 8 kHz will be downsampled to 8 kHz by
 *    a very simple algorithm, resulting in poor quality.
 *
 * This little utility is made for converting WAV files to GSM format,
 * which can be played back by GVoice programs.
 *
 * The output file is written in raw format, containing the GSM data stream
 * which can be sent directly to the modem.
 *
 * USRobotics modems use 8 kHz sampling rate which is identical to the
 * sampling rate used by the GSM codec. Other modem manufacturers
 * (e.g. Rockwell) use 7.2 kHz sampling rate which is incompatible with
 * the GSM codec. I have a USR so it works fine for me. I haven't tested
 * it with other modems yet.
 *
 * Compile with Borland C++ 3.1:
 * bcc -c wav2gsm.c
 * tlink -Tde c0s wav2gsm , wav2gsm , NUL , cs libgsm
 *
 * THERE IS ABSOLUTELY NO WARRANTY FOR THIS SOFTWARE
 *
 * Any improvements that may be of general interest are welcome.
 *
 * Tamas Vincze
 *
 * e-mail: geza@polghiv.szeged.hu
 *         tamas_vincze@electric.fido.hu
 * WWW: http://gvoice.home.ml.org
 *      http://bubu.teleki.jgytf.u-szeged.hu/ell/gvoice
 * FidoNet netmail: Tamas Vincze, 2:370/36
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gsm.h"

// WAV header

struct
{
	char		riff_id[4];		// "RIFF"
	unsigned long   riff_size;  		// = file size - 8
	char		wave_id[4];		// "WAVE"
	char		fmt_id[4];		// "fmt "
	unsigned long	fmt_size;    		// = 16
	unsigned int	format;			// 1=PCM
	unsigned int	channels;		// 1=mono, 2=stereo
	unsigned long	sampling_rate;
	unsigned long	bytes_per_sec;		// = sampling_rate * channels
	unsigned int	reserved;		// block alignment???
	unsigned int	bits_per_sample;	// e.g. 8, 12, 16
	char		data_id[4];		// "data"
	unsigned long	data_size;		// remaining bytes in the file
} w;

int main(int argc, char *argv[])
{
	FILE		*infile, *outfile;
	char		infname[_MAX_PATH], outfname[_MAX_PATH];
	unsigned long	cnt, s;
	int		i, j, x, x2, ch;

	gsm		gsmhand;
	gsm_frame	gsmbuf;
	gsm_signal	sample[160];
	unsigned char	usrbuf[19*2];		// USRobotics frame type!
	int		percent;
	long		pstep, px;

	printf("WAV => GSM converter V1.00 by Tamas Vincze.\n"
	       "Original GSM codec by Jutta Degener and Carsten Bormann.\n"
	       "MS-DOS port of the GSM library by Richard Elofsson.\n\n");
	if(argc < 2 || argc > 3 || !strcmp(argv[1], "/?"))
	{
		printf("Usage: WAV2GSM infile[.wav] [outfile[.gsm]]\n\n");
		return 1;
	}

	for(i=j=0; (infname[i]=*(argv[1]+i)) != 0; i++)
		if(infname[i]=='.')
			j=i;

	if(!j)
	{
		j = strlen(infname);
		strcat(infname, ".WAV");
	}

	if(argc == 3)
	{
		for(i=j=0; (outfname[i]=*(argv[2]+i)) != 0; i++)
			if(outfname[i]=='.')
				j=1;
		if(!j)
			strcat(outfname, ".GSM");
	}
	else
	{
		strncpy(outfname, infname, j);
		outfname[j] = 0;
		strcat(outfname, ".GSM");
	}

/* ====================================================================== */

	printf("Converting: %s\n        to: %s\n\n", infname, outfname);

	if( (infile = fopen(infname, "rb")) == NULL )
	{
		printf("Can't open infile!\n\n");
		return 1;
	}

	fseek(infile, 0L, SEEK_END);
	pstep = ftell(infile) / 100;
	fseek(infile, 0L, SEEK_SET);

	if( (outfile = fopen(outfname, "wb")) == NULL )
	{
		printf("Can't create outfile!\n\n");
		fclose(infile);
		return 1;
	}

/* ====================================================================== */

	if ( fread(&w, sizeof(w), 1, infile) < 1 )
	{
		printf("Can't read WAV header!\n\n");
		fclose(infile);
		fclose(outfile);
		remove(outfname);
		return 1;
	}

	px = sizeof(w);

	if ( w.riff_id[0]!='R' || w.riff_id[1]!='I' || w.riff_id[2]!='F' || w.riff_id[3]!='F' ||
	     w.wave_id[0]!='W' || w.wave_id[1]!='A' || w.wave_id[2]!='V' || w.wave_id[3]!='E' ||
	      w.fmt_id[0]!='f' ||  w.fmt_id[1]!='m' ||  w.fmt_id[2]!='t' ||  w.fmt_id[3]!=' ' ||
	      w.fmt_size!=16 )
	{
		printf("Unknown WAV format!\n\n");
		fclose(infile);
		fclose(outfile);
		remove(outfname);
		return 1;
	}

	if ( w.format != 1 )
	{
		printf("WAV isn't in PCM format!\n\n");
		fclose(infile);
		fclose(outfile);
		remove(outfname);
		return 1;
	}

	printf("Input file is a %u-channel, %lu Hz, %u-bit WAV.\n\n", w.channels, w.sampling_rate, w.bits_per_sample);

	if(w.bits_per_sample != 8 && w.bits_per_sample != 16)
	{
		printf("The number of bits per sample must be 8 or 16!\n\n");
		fclose(infile);
		fclose(outfile);
		remove(outfname);
		return 1;
	}

	if(w.channels != 1 && w.channels != 2)
	{
		printf("The number of channels must be 1 or 2!\n\n");
		fclose(infile);
		fclose(outfile);
		remove(outfname);
		return 1;
	}

	if(w.sampling_rate < 8000)
	{
		printf("Sampling rate of the WAV must be 8 kHz or greater!\n\n");
		fclose(infile);
		fclose(outfile);
		remove(outfname);
		return 1;
	}

	if(w.sampling_rate > 8000)
		printf("WARNING: downsampling to 8 kHz results in poor quality!\n\n");

	if( !(gsmhand = gsm_create()) )
	{
		printf("Not enough memory!\n\n");
		fclose(infile);
		fclose(outfile);
		remove(outfname);
		return 1;
	}

	i = 0;				// sample counter
	s = cnt = 0;
	percent = 0;
	printf("%3d%%\r", 0);

	while( cnt < w.data_size )
	{
		ch = fgetc(infile);
		cnt++, px++;

		if( w.bits_per_sample > 8 )
		{
			x = (fgetc(infile)<<8) | ch;
			cnt++, px++;
		}
		else
			x = (ch-128) * 256;

		if( w.channels == 2 )
		{
			ch = fgetc(infile);
			cnt++, px++;
			if( w.bits_per_sample > 8 )
			{
				x2 = (fgetc(infile)<<8) | ch;
				cnt++, px++;
			}
			else
				x2 = (ch-128) * 256;
			x = (x/2) + (x2/2);
		}

		s += 8000;
		if( s >= w.sampling_rate )
		{
			s -= w.sampling_rate;
			sample[i++] = x;
			if( i == 160 )		// sample buffer full?
			{
				i = 0;

				gsm_encode(gsmhand, sample, gsmbuf);

				// convert frame to USRobotics format
				usrbuf[0] = 0xFE;	// "begin frame" word
				usrbuf[1] = 0xFE;
				for(j=0; j<33; j++)
					usrbuf[j+2] = gsmbuf[j];
				usrbuf[35] = 0;
				usrbuf[36] = 0xA5;	// "end frame" word
				usrbuf[37] = 0xA5;

				if ( fwrite(usrbuf, sizeof(usrbuf), 1, outfile) != 1 )
				{
					printf("Can't write to outfile!\n\n");
					gsm_destroy(gsmhand);
					fclose(infile);
					fclose(outfile);
					remove(outfname);
					return 1;
				}
			}
		}
		if ( px >= pstep )
		{
			px -= pstep;
			printf("%3d%%\r", ++percent);
		}
	}

	if( cnt > w.data_size )
	{
		printf("Incorrect WAV file size!\n\n");
		fclose(infile);
		fclose(outfile);
		remove(outfname);
		return 1;
	}

/* ====================================================================== */

	gsm_destroy(gsmhand);
	fclose(infile);
	fclose(outfile);
	printf("Done!\n\n");
	return 0;
}

