/*
 * GSM2WAV.C, V1.01
 *
 * Converts USRobotics format GSM files to WAV format
 *
 * Public Domain by Tamas Vincze
 *
 * The original GSM codec is made by Jutta Degener and Carsten Bormann.
 * The GSM library used here is a port of GSM 06.10, patchlevel 4, to
 * MS-DOS by Richard Elofsson.
 *
 * Notes:
 *
 * This little utility is made for converting voice messages (recorded by
 * the GVoice answering machine software) to WAV format, so you can listen
 * them via your sound card.
 *
 * The input file is expected to be in raw format, containing the GSM data
 * stream as it came from the modem.
 *
 * USRobotics modems use 8 kHz sampling rate which is identical to the
 * sampling rate used by the GSM codec. Other modem manufacturers
 * (e.g. Rockwell) use 7.2 kHz sampling rate which is incompatible with
 * the GSM codec. I have a USR so it works fine for me. I haven't tested
 * it with other modems yet.
 *
 * Compile with Borland C++ 3.1:
 * bcc -c gsm2wav.c
 * tlink -Tde c0s gsm2wav , gsm2wav , NUL , cs libgsm
 *
 * THERE IS ABSOLUTELY NO WARRANTY FOR THIS SOFTWARE
 *
 * Any improvements that may be of general interest are welcome.
 *
 * Tamas Vincze
 *
 * e-mail: geza@polghiv.szeged.hu
 *         tamas_vincze@electric.fido.hu
 * WWW: http://gvoice.home.ml.org
 *      http://bubu.teleki.jgytf.u-szeged.hu/ell/gvoice
 * FidoNet netmail: Tamas Vincze, 2:370/36
 *
 * HISTORY:
 * V1.00 (1997 Jul 28) - initial release
 * V1.01 (1997 Jul 31) - skips last frame of infile if it's incomplete
 *                     - displays warning if USR frame's "begin frame word" or
 *                       "end frame word" isn't correct
 *                     - dumps GSM frame if gsm_decode() fails on it
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gsm.h"

// WAV header

struct
{
	char		riff_id[4];		// "RIFF"
	unsigned long   riff_size;  		// = file size - 8
						// = data_size + 36
	char		wave_id[4];		// "WAVE"
	char		fmt_id[4];		// "fmt "
	unsigned long	fmt_size;    		// = 16
	unsigned int	format;			// 1=PCM
	unsigned int	channels;		// 1=mono, 2=stereo
	unsigned long	sampling_rate;
	unsigned long	bytes_per_sec;		// = sampling_rate * channels
	unsigned int	reserved;		// block alignment???
	unsigned int	bits_per_sample;	// e.g. 8, 12, 16
	char		data_id[4];		// "data"
	unsigned long	data_size;		// remaining bytes in the file
} w = {
	"RIFF",
	0,
	"WAVE",
	"fmt ",
	16,
	1,
	1,
	8000,
	16000,
	2,
	16,
	"data",
	0
};


int main(int argc, char *argv[])
{
	FILE		*infile, *outfile;
	char		infname[_MAX_PATH], outfname[_MAX_PATH];
	char		*inf, *outf;
	gsm		gsmhand;
	gsm_frame	gsmbuf;
	gsm_signal	sample[160];
	unsigned char	usrbuf[19*2];		// USRobotics frame type!
	int		i, j, k, percent, peak, amplify=0, maxamp, ampstep=4;
	long		pstep, px;

	printf("GSM => WAV converter V1.01 by Tamas Vincze.\n"
	       "Original GSM codec by Jutta Degener and Carsten Bormann.\n"
	       "MS-DOS port of the GSM library by Richard Elofsson.\n\n");

	if(argc < 2 || argc > 4 || !strcmp(argv[1], "/?"))
	{
		printf( "Usage: GSM2WAV infile[.gsm] [outfile[.wav]] [/A]\n\n"
			"/A - enable automatic gain control (AGC)\n\n");
		return 1;
	}

	inf = outf = NULL;
	for(i=1; i<argc; i++)
		if(!stricmp(argv[i], "/A"))
			amplify = 1;
		else if(inf == NULL)
			inf = argv[i];
		else
			outf = argv[i];

	for(i=j=0; (infname[i]=*(inf+i)) != 0; i++)
		if(infname[i]=='.')
			j=i;

	if(!j)
	{
		j = strlen(infname);
		strcat(infname, ".GSM");
	}

	if(outf != NULL)
	{
		for(i=j=0; (outfname[i]=*(outf+i)) != 0; i++)
			if(outfname[i]=='.')
				j=1;
		if(!j)
			strcat(outfname, ".WAV");
	}
	else
	{
		strncpy(outfname, infname, j);
		outfname[j] = 0;
		strcat(outfname, ".WAV");
	}

/* ====================================================================== */

	printf("Converting: %s\n        to: %s\n", infname, outfname);
	if( amplify )
		printf("Using AGC\n\n");
	else
		printf("\n");

	if( (infile = fopen(infname, "rb")) == NULL )
	{
		printf("Can't open infile!\n\n");
		return 1;
	}

	fseek(infile, 0L, SEEK_END);
	pstep = ftell(infile) / 100;
	fseek(infile, 0L, SEEK_SET);

	if( (outfile = fopen(outfname, "wb")) == NULL )
	{
		printf("Can't create outfile!\n\n");
		fclose(infile);
		return 1;
	}

	fseek(outfile, sizeof(w), SEEK_SET);	// leave space for WAV header

/* ====================================================================== */

	if( !(gsmhand = gsm_create()) )
	{
		printf("Not enough memory!\n\n");
		fclose(infile);
		fclose(outfile);
		remove(outfname);
		return 1;
	}

	percent = 0;
	px = 0;
	printf("%3d%%\r", 0);

	while ( fread(usrbuf, 1, sizeof(usrbuf), infile) == sizeof(usrbuf) )
	{
		px += sizeof(usrbuf);
		if ( px >= pstep )
		{
			px -= pstep;
			printf("%3d%%\r", ++percent);
		}

		for(i=0; i<33; i++)
			gsmbuf[i] = usrbuf[i+2];   // skip "begin frame" word
		gsmbuf[0] |= 0xD0;		   // set GSM "magic" bits

		if( ( (usrbuf[0]!=0xFE || usrbuf[1]!=0xFE) && (usrbuf[0]!=0xB6 || usrbuf[1]!=0xB6) ) ||
		    ( usrbuf[35]!=0x00 || usrbuf[36]!=0xA5 || usrbuf[37]!=0xA5 ) )
		{
			printf("WARNING: bad USR frame at 0x%08lX\n", ftell(infile)-sizeof(usrbuf));
		}

		if ( gsm_decode(gsmhand, gsmbuf, sample) )
		{
			printf("ERROR: gsm_decode() failed on this frame:\n");
			printf("0x%08lX: ", ftell(infile)-sizeof(usrbuf)+2);
			for(i=0; i<16; i++)
				printf("%02X ", gsmbuf[i]);
			printf("\n0x%08lX: ", ftell(infile)-sizeof(usrbuf)+18);
			for(i=16; i<32; i++)
				printf("%02X ", gsmbuf[i]);
			printf("\n0x%08lX: %02X\n\n", ftell(infile)-sizeof(usrbuf)+34, gsmbuf[32]);
			gsm_destroy(gsmhand);
			fclose(infile);
			fclose(outfile);
			remove(outfname);
			return 1;
		}

		// AGC

		if( amplify )
		{
			for(i=0,peak=1; i<160; i++)
				if( abs(sample[i]) > peak )
					peak = abs(sample[i]);

			maxamp = 32000 / peak;
			if(maxamp == 0)
				maxamp = 1;

			if(peak*amplify > 32000)	// overdrive?
				amplify = maxamp;
			else
			{
				amplify += ampstep;
				if(amplify > maxamp)
					amplify = maxamp;
			}

			for(i=0; i<160; i++)
				sample[i] *= amplify;
		}

		if ( fwrite(sample, sizeof(sample), 1, outfile) != 1 )
		{
			printf("Can't write to outfile!\n\n");
			gsm_destroy(gsmhand);
			fclose(infile);
			fclose(outfile);
			remove(outfname);
			return 1;
		}
		w.data_size += sizeof(sample);
	}

/* ====================================================================== */

	// write WAV header

	w.riff_size = w.data_size + 36;

	fseek(outfile, 0, SEEK_SET);
	if ( fwrite(&w, sizeof(w), 1, outfile) != 1 )
	{
		printf("Can't write WAV header!\n\n");
		fclose(infile);
		fclose(outfile);
		remove(outfname);
		return 1;
	}

	gsm_destroy(gsmhand);
	fclose(infile);
	fclose(outfile);
	printf("Done!\n\n");
	return 0;
}

