/*
	term.c
	A very simple terminal driver implimented using FOSSIL
	communications functions

	Written by David Nugent, Unique Computing Pty Ltd
	as an example of how to interface to FOSSIL

	This program was compiled using Turbo C v2.00 but there are no
	library dependancies so should compile and work with most other
	C compilers under MS-DOS.

	For the timer() and timerset() routines, see BinkleyTerm sources.
*/

#include <stdio.h>
#include <keydef.h>
#include <ctype.h>
#include <stdarg.h>
#include "fossil.h"
#include "term.h"

	/* Function prototypes */

static int strcpyf (char far *dst, char far *src);
static int getline (char *ptr, int max);
static void cls (void);
static void setcolor (unsigned col);
static int cdecl pprintf (int port, int chk, char *fmt, ...);
static int pputs (int port, int chk, register char *str);
static int cdecl aprintf (char *fmt, ...);
static int aputs (register char *str);
static void chgrate (void);
static void chgport (void);
static int lookup (unsigned *arr, unsigned f, int no);

	/* Static/global data */

int port = 0;					/* Default port: 0=COM1 */
int baud = 2400;				/* Default baud: 2400 */
int mask = (CP_PARNONE|CP_STOP1|CP_CHLEN8);	/* Default comm parameters */

	/* Functions */


int main (int argc, char *argv[])
{
int k;
unsigned ps;
struct tinfo T;
struct finfo F;
char fdesc[256];
struct fossildata FD;

if (ComPortInit (port, 0, &F) != FSIG) {	/* Check for FOSSIL available */
	fputs ("\r\n\x7No FOSSIL loaded or port unavailable!\r\n", stdout);
	return 1;
	}
if (F.maxfunc < 0x1bH) {			/* Make sure we can use it */
	fputs ("\r\n\x7Incompatible FOSSIL: must use Rev 5 for later\r\n");
	printf ("Loaded FOSSIL is Revision %d, Maximum supported function is 0x%2x\n", F.revision, F.maxfunc);
	return 1;
	}
setcolor (bgrnd(BLUE)+light(WHITE));	/* Change screen attributtes & clear */
cls();
aputs ("FOSSIL Terminal Test Program V1.00\r\n"
       "Copyright (c) 1989, David Nugent & Unique Computing Ptd Ltd\r\n");
ComTimer (port, &T);			/* Get/display timer parameters */
aprintf ("Timer interrupt = %02x, Ticks/second = %d, Miliseconds/tick = %d\r\n", T.timerint, T.ticksecs, T.milltick);
ComDrvInfo (port, &FD, sizeof(FD));	/* Get/display port/FOSSIL information */
strcopyf (fdesc, FD.drvid);
aprintf ("FOSSIL: %s\r\n", fdesc);
aprintf ("Revision %d Ver %d  RX = %db(%d) TX = %db(%d) Screen = %dx%d\r\n",
	FD.specver, FD.drvlvl, FD.rxsize, FD.rxavail, FD.txsize, FD.txavail,
	FD.scnwid, FD.scnlen);
ComFlowCtl (port, FC_LOCXON|FC_REMXON);	/* Enable XON/XOFF flow control */
for (;;) {
	ps = ComPortStat (port);
	if (ps & PS_RXCHARS) ComWrAnsi (ComRxChar (port));
	if (ComKbPeek () != -1) {
		k = ComKbChar ();
		switch (k) {
			case ALTP:
				chgport ();
				continue;
			case ALTB:
				chgrate ();
				continue;
			case ALTF:
				ComTxPurge (port);
				ComFlowCtl (port, 0);
				ComRxPurge (port);
				ComFlowCtl (port, FC_LOCXON|FC_REMXON);
				continue;
			case ALTH:
				aputs ("Hanging up ...");
				ComSetDtr (port, 0);
				timer (50);
				ComSetDtr (port, 1);
				aputs ("\r\n");
				continue;
			case ALTX: break;
			default:
				if (k & 0xff) ComTxChar (port, k);
				continue;
			}
		}
	else continue;
	break;
	}
setcolor (WHITE);
ComPortDeInit (port);
return 0;
}


static void chgport (void)
{
char newport[25];

aprintf ("Change to port number (currently %d): ", port);
if (getline (newport, 20) <= 0 || !isdigit (*newport)) return;
ComPortDeInit (port);
sscanf (newport, "%d", &port);
ComPortInit (port, 0, 0);
aprintf ("Port %d active\r\n", port);
}

unsigned speeds[] = {
	300U, 1200U, 2400U, 4800U, 9600U, 19200U, 38400U
	};

unsigned codes[] = { CP_B300, CP_B1200, CP_B2400, CP_B4800,
			CP_B9600, CP_B19200, CP_B38400 };

static void chgrate (void)
{
char newrate[20];
unsigned xrate, rateidx;

aprintf ("Change baud rate to (currently %u): ", baud);
if (getline (newrate, 20) <= 0 || !isdigit (*newrate)) return;
sscanf (newrate, "%u", &xrate);
rateidx = lookup (speeds, xrate, 7);
if (rateidx == (unsigned)-1) aputs ("Invalid baud rate\r\n");
else {
	ComPortSet (port, codes[rateidx] | mask);
	aprintf ("Baud rate set to %u\r\n", xrate);
	baud = xrate;
	}
}


static int aputs (register char *str)
{
register int i;
for (i = 0; str[i]; ++i) ComWrAnsi (str[i]);
return i;
}


static int cdecl aprintf (char *fmt, ...)
{
va_list arg;
char buf[512];

va_start (arg, fmt);
vsprintf (buf, fmt, arg);
va_end (arg);
return aputs (buf);
}


static int pputs (int port, int chk, register char *str)
{
register int i;
unsigned s;
long t;

for (i = 0; str[i]; ++i) {
	for (t = timerset (10); !((s=ComPortStat (port)) & PS_TXEMPTY); ) {
		if (chk && !(s && PS_CARRIER)) return NO_CARRIER;
		if (timeup(t)) return TIMEOUT;
		}
	ComTxChar (port, str[i]);
	}
return i;
}

static int cdecl pprintf (int port, int chk, char *fmt, ...)
{
va_list arg;
char buf[512];

va_start (arg, fmt);
vsprintf (buf, fmt, arg);
va_end (arg);
return pputs (port, chk, buf);
}


/* --- Screen functions --- */

static int col_fg[8] = {
	30, 34, 32, 36, 31, 35, 33, 37
	};
static int col_bg[8] = {
	40, 44, 42, 46, 41, 45, 43, 47
	};

static void setcolor (unsigned col)
{
int intense, blink, fg, bg;

blink = ((col & 128) != 0);
intense = ((col & 8) != 0);
fg = col & 7;
bg = (col & 127) >> 4;
aprintf ("\x1B[%d;%d;%d;%dm", (blink) ? 5: 0, (intense) ? 1 : 0,
	col_bg[bg], col_fg[fg]);
}

static void cls (void)
{
aputs ("\x1B[2J");
}


static int getline (char *ptr, int max)
{
unsigned k;
int i;

*ptr = '\0';
for (i=0;;) {
	k = ComKbChar ();
	switch (k & 0xff) {
		case 27: i = -1; goto quit;
		case 13:
		case 10: goto quit;
		case 0:  continue;
		case 8:
			ptr[i] = '/\0';
			if (--i < 0) i = 0;
			else aputs ("\b \b");
			break;
		default:
			ptr[i] = (unsigned char)k;
			ptr[i+1] = '\0';
			if (i < max) { ++i; ComWrAnsi (k); }
			break;
		}
	}
quit:
aputs ("\r\n");
return i;
}


static int strcpyf (char far *dst, char far *src)
{
register int i;
for (i=0; *src;++i) *(src++) = *(dst++);
return i;
}

static int lookup (unsigned *arr, unsigned f, int no)
{
register int i, r = -1;
for (i = 0; i < no; ++i) if (f == arr[i]) { r = i; break; }
return r;
}

