// file7.cpp : implementation file
//
// Class:		InterBbsNodeFile
//
// Created:		20/05/98
//
// Author:		S.D.Holme
//
// Version:		1.00
//
// Modifications:
//
//	1.00	20/05/98	S.D.Holme	Created.
//			27/05/98	S.D.Holme	Add m_bModified to InterBbsNodeFile class.
//									Added the following functions to
//									InterBbsNodeFile class:
//										AddNode();
//										RemoveNode();
//									Converted InterBbsNode from a structure
//									into a class.
//			08/06/98	S.D.Holme	Fixed bug in AddNode() and RemoveNode()
//									where m_wTotalSystems wasn't being
//									updated.
//									Fixed bug in AddNode() which meant a
//									system could be added multiple times.
//			09/06/98	S.D.Holme	Made g_szLocationSeperator a public
//									member of InterBbsNodeFile as m_sz...
//									Added GetNode() const function.
//									Fixed bug in writing routing info.
//			10/06/98	S.D.Holme	Fixed bug in writing location info.
//			19/06/98	S.D.Holme	Public release.
//	1.01	16/08/98	S.D.Holme	Added support for crash and hold flags in
//									the InterBBS node file
//
// File Library. Copyright (c) 1995-98 Fonty Technologies, Inc.
//
#include <assert.h>
#include <mem.h>
#include <stdlib.h>
#include <string.h>

#include "file.h"

char InterBbsNodeFile::m_szLocationSeperator[] = ", ";

/////////////////////////////////////////////////////////////////////////////
// InterBbsNode Class

InterBbsNode::InterBbsNode()
{
	// Initialise member attributes
	Empty();
}

InterBbsNode::InterBbsNode(const InterBbsNode& rSource)
{
	// Call the copy function
	Copy(rSource);
}

InterBbsNode::InterBbsNode(const char* lpszName, const char* lpszAddress,
						   const char* lpszLocation, INT16 iRouteTo,
						   BOOL bCrash, BOOL bHold)
{
	// Initialise the member attributes
	SetName(lpszName);
	SetAddress(lpszAddress);
	SetLocation(lpszLocation);
	SetRouteTo(iRouteTo);
	SetCrash(bCrash);
	SetHold(bHold);
}

InterBbsNode::~InterBbsNode()
{
}

BOOL InterBbsNode::SetName(const char* lpszName)
{
	// Check for valid parameters
	if(strlen(lpszName) > F7_MAX_NODE_NAME)
	{
		// Return failure
		return FALSE;
	}

	// Set the member attribute
	strcpy(m_szName, lpszName);

	// Set the modified flag
	m_bModified = TRUE;

	// Return success
	return TRUE;
}

BOOL InterBbsNode::SetAddress(const char* lpszAddress)
{
	// Check for valid parameters
	if(strlen(lpszAddress) > F7_MAX_NODE_ADDRESS)
	{
		// Return failure
		return FALSE;
	}

	// Set the member attribute
	strcpy(m_szAddress, lpszAddress);

	// Set the modified flag
	m_bModified = TRUE;

	// Return success
	return TRUE;
}

BOOL InterBbsNode::SetLocation(const char* lpszLocation)
{
	// Check for valid parameters
	if(strlen(lpszLocation) > F7_MAX_NODE_NAME)
	{
		// Return failure
		return FALSE;
	}

	// Set the member attribute
	strcpy(m_szLocation, lpszLocation);

	// Set the modified flag
	m_bModified = TRUE;

	// Return success
	return TRUE;
}

BOOL InterBbsNode::SetRouteTo(INT16 iRouteTo)
{
	// Set the member attribute
	m_iRouteTo = iRouteTo;

	// Set the modified flag
	m_bModified = TRUE;

	// Return success
	return TRUE;
}

BOOL InterBbsNode::SetCrash(BOOL bCrash)
{
	// Set the member attribute
	m_bCrash = bCrash;

	// Set the modified flag
	m_bModified = TRUE;

	// Return success
	return TRUE;
}

BOOL InterBbsNode::SetHold(BOOL bHold)
{
	// Set the member attribute
	m_bHold = bHold;

	// Set the modified flag
	m_bModified = TRUE;

	// Return success
	return TRUE;
}

const char* InterBbsNode::GetName(void) const
{
	// Return the member attribute
	return m_szName;
}

const char* InterBbsNode::GetAddress(void) const
{
	// Return the member attribute
	return m_szAddress;
}

const char* InterBbsNode::GetLocation(void) const
{
	// Return the member attribute
	return m_szLocation;
}

INT16 InterBbsNode::GetRouteTo(void) const
{
	// Return the member attribute
	return m_iRouteTo;
}

BOOL InterBbsNode::GetCrash(void) const
{
	// Return the member attribute
	return m_bCrash;
}

BOOL InterBbsNode::GetHold(void) const
{
	// Return the member attribute
	return m_bHold;
}

BOOL InterBbsNode::IsModified(void) const
{
	// Return the modified flag member attribute
	return m_bModified;
}

BOOL InterBbsNode::Empty(void)
{
	// Empty the member attributes
	m_szName[0] = '\0';
	m_szAddress[0] = '\0';
	m_szLocation[0] = '\0';
	m_iRouteTo = -1;
	m_bCrash = FALSE;
	m_bHold = FALSE;

	m_bModified = FALSE;

	// Return success
	return TRUE;
}

BOOL InterBbsNode::IsEmpty(void) const
{
	// Test if the name is empty
	if(m_szName[0] == '\0')
	{
		// Return yes
		return TRUE;
	}

	// Return no
	return FALSE;
}

void InterBbsNode::Copy(const InterBbsNode& rSource)
{
	// Call the copy operator
	operator=(rSource);
}

InterBbsNode& InterBbsNode::operator=(const InterBbsNode& rSource)
{
	// Copy the member attributes
	SetName(rSource.m_szName);
	SetAddress(rSource.m_szAddress);
	SetLocation(rSource.m_szLocation);
	SetRouteTo(rSource.m_iRouteTo);
	SetCrash(rSource.m_bCrash);
	SetHold(rSource.m_bHold);

	m_bModified = rSource.m_bModified;

	// Return a reference to this object
	return *this;
}

/////////////////////////////////////////////////////////////////////////////
// InterBbsNodeFile Class

// Constructor
InterBbsNodeFile::InterBbsNodeFile(const char* pszFilename, int nMode)
	: TextFile(pszFilename, nMode)
{
	// Initialise member attributes
	m_bModified = FALSE;
}

// Destructor
InterBbsNodeFile::~InterBbsNodeFile()
{
	if(m_pNode != NULL)
	{
		// Deallocate the node array
		delete[] m_pNode;
		m_pNode = NULL;
	}
}

// Initialisation
BOOL InterBbsNodeFile::Initialise(const char* lpszHostNodeName, const char* lpszHostNodeAddress)
{
	assert(lpszHostNodeName != NULL);
	assert(lpszHostNodeAddress != NULL);

	if(lpszHostNodeName == NULL || lpszHostNodeAddress == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Call the base function
	if(! TextFile::Initialise())
	{
		// Return failure
		return FALSE;
	}

	// Initialise member attributes
	if(strlen(lpszHostNodeName) >= F7_MAX_NODE_NAME)
	{
		// Return failure
		return FALSE;
	}

	if(strlen(lpszHostNodeAddress) >= F7_MAX_NODE_ADDRESS)
	{
		// Return failure
		return FALSE;
	}
	strcpy(m_szHostNodeName, lpszHostNodeName);
	strcpy(m_szHostNodeAddress, lpszHostNodeAddress);

	// Allocate the node structure
	m_pNode = new InterBbsNode[F7_MAX_NODES];

	// Check memory allocation
	if(m_pNode == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Initialise the memory to zero
	for(int i = 0; i < F7_MAX_NODES; i++)
	{
		// Empty the node (and set the RouteTo to -1)
		m_pNode[i].Empty();
	}

	// Return success
	return TRUE;
}

// Functions

BOOL InterBbsNodeFile::Read(void)
{
	WORD nNode;
	char szLine[256 + 1];
	char szNodeName[F7_MAX_NODE_NAME + 1];
	char szNodeAddress[F7_MAX_NODE_ADDRESS + 1];
	char szNodeLocation[F7_MAX_LOCATION + 1];
	char szNodeState[F7_MAX_STATE + 1];
	char szNodeCountry[F7_MAX_COUNTRY + 1];
	char szNodeHosting[256 + 1];
	BOOL bDone = FALSE;
	BOOL bCrash;
	BOOL bHold;

	assert(m_pNode != NULL);

	// Check array is valid
	if(m_pNode == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Check file is open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	m_wTotalSystems = 0;

	while(! bDone)
	{
		// Read the first line of the configuration file
		if(ReadNot(szLine) && strchr(szLine, '-') == NULL)
		{
			// Convert line to upper case for ease of use
			strupr(szLine);

			// Look for CRASH and HOLD flags before HOST as the
			// HOST routine modifies the szLine variable

			// Locate the CRASH keyword
			char* pszCrashKeyword = strstr(szLine, " CRASH");
			bCrash = pszCrashKeyword == NULL ? FALSE : TRUE;

			// Locate the HOLD keyword
			char* pszHoldKeyword = strstr(szLine, " HOLD");
			bHold = pszHoldKeyword == NULL ? FALSE : TRUE;

			// Copy line to a temporary storage area
			char* pszHostKeyword = strstr(szLine, " HOST ");
			if(pszHostKeyword == NULL)
			{
				nNode = atoi(szLine) - 1;
				strcpy(szNodeHosting, "");
			}
			else
			{
				// Null terminate where the space was found
				*pszHostKeyword = '\0';

				// So we can get the node number safely
				nNode = atoi(szLine) - 1;

				// Adance the pointer to where the node numbers start
				pszHostKeyword += 6;

				// Get the node numbers safely
				if(*pszHostKeyword == NULL)
				{
					strcpy(szNodeHosting, "");
				}
				else
				{
					strcpy(szNodeHosting, pszHostKeyword);
				}
			}

			// Read the second line of the configuration file
			if(ReadNot(szLine))
			{
				// Copy line to a temporary storage area
				memset(szNodeName, '\0', sizeof(szNodeName));
				strncpy(szNodeName, szLine, sizeof(szNodeName) - 1);

				// Read the third line of the configuration file
				if(ReadNot(szLine))
				{
					// Copy line to a temporary storage area
					memset(szNodeAddress, '\0', sizeof(szNodeAddress));
					strncpy(szNodeAddress, szLine, sizeof(szNodeAddress) - 1);

					// Read the forth line of the configuration file
					if(ReadNot(szLine))
					{
						// Copy line to a temporary storage area
						memset(szNodeLocation, '\0', sizeof(szNodeLocation));
						strncpy(szNodeLocation, szLine, sizeof(szNodeLocation) - 1);

						// Read the fifth line of the configuration file
						if(Readln(szLine))
						{
							// Copy line to a temporary storage area
							memset(szNodeState, '\0', sizeof(szNodeState));
							strncpy(szNodeState, szLine, sizeof(szNodeState) - 1);

							// Read the sixth line of the configuration file
							if(Readln(szLine))
							{
								// Copy line to a temporary storage area
								memset(szNodeCountry, '\0', sizeof(szNodeCountry));
								strncpy(szNodeCountry, szLine, sizeof(szNodeCountry) - 1);
							}
							else
							{
								bDone = TRUE;
							}
						}
						else
						{
							bDone = TRUE;
						}
					}
					else
					{
						bDone = TRUE;
					}
				}
				else
				{
					bDone = TRUE;
				}
			}
			else
			{
				bDone = TRUE;
			}
		}
		else
		{
			bDone = TRUE;
		}

		// If not done
		if(! bDone && nNode < F7_MAX_NODES)
		{
			// Copy the important information
			m_pNode[nNode].SetName(szNodeName);
			m_pNode[nNode].SetAddress(szNodeAddress);

			if(strcmp(szNodeState, "") != 0)
			{
				strcat(szNodeLocation, m_szLocationSeperator);
				strcat(szNodeLocation, szNodeState);
			}
			if(strcmp(szNodeCountry, "") != 0)
			{
				strcat(szNodeLocation, m_szLocationSeperator);
				strcat(szNodeLocation, szNodeCountry);
			}

			m_pNode[nNode].SetLocation(szNodeLocation);

			// Set crash / hold flags
			m_pNode[nNode].SetCrash(bCrash);
			m_pNode[nNode].SetHold(bHold);

			// Store hosting as host against each node
			// as opposed to all nodes against each hub
			char* pszNode = strtok(szNodeHosting, " ");
			while(pszNode != NULL)
			{
				int nHostedNode;

				nHostedNode = atoi(pszNode) - 1;
				pszNode = strtok(NULL, " ");
				if(nHostedNode >= 0)
				{
					m_pNode[nHostedNode].SetRouteTo(nNode);
				}
			}

			// Increment the number of systems
			m_wTotalSystems++;
		}

		// Check for end of file
		if(Eof() || m_wTotalSystems >= F7_MAX_NODES)
		{
			bDone = TRUE;
		}
	}

	// Check number of nodes
	if(m_wTotalSystems == 0)
	{
		// Return error code
		return FALSE;
	}

	// Reset the modified flag
	m_bModified = FALSE;

	// Return success
	return TRUE;
}

BOOL InterBbsNodeFile::Write(void)
{
	char szCity[F7_MAX_LOCATION + 1] = { 0, };
	char szState[F7_MAX_LOCATION + 1] = { 0, };
	char szCountry[F7_MAX_LOCATION + 1] = { 0, };
	char* psz;

	assert(m_pNode != NULL);

	// Check array is valid
	if(m_pNode == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Check file is open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	// Write out all the nodes
	WORD wNode1 = 0;
	for(WORD i = 0; i < F7_MAX_NODES && wNode1 < m_wTotalSystems; i++)
	{
		// Except any missing nodes
		if(! m_pNode[i].IsEmpty())
		{
			// Write the item number out
			if(! Writef("%u", i + 1))
			{
				// Return failure
				return FALSE;
			}

			// Write the systems that we HOST for out
			BOOL bFound = FALSE;
			WORD wNode2 = 0;
			for(WORD j = 0; j < F7_MAX_NODES && wNode2 < m_wTotalSystems; j++)
			{
				// Except any missing nodes
				if(! m_pNode[j].IsEmpty())
				{
					if(m_pNode[j].GetRouteTo() == i && j != i)
					{
						if(! bFound)
						{
							if(! TextFile::Write(" HOST"))
							{
								// Return failure
								return FALSE;
							}

							bFound = TRUE;
						}

						// Write the item number out
						if(! Writef(" %u", j + 1))
						{
							// Return failure
							return FALSE;
						}
					}

					// Increment counter
					wNode2++;
				}
			}

			// Write out the CRASH flag
			if(m_pNode[i].GetCrash())
			{
				if(! TextFile::Write(" CRASH"))
				{
					// Return failure
					return FALSE;
				}
			}

			// Write out the HOLD flag
			if(m_pNode[i].GetHold())
			{
				if(! TextFile::Write(" HOLD"))
				{
					// Return failure
					return FALSE;
				}
			}

			// Line feed
			if(! Writeln())
			{
				// Return failure
				return FALSE;
			}

			// Write the node name out
			if(! Writeln(m_pNode[i].GetName()))
			{
				// Return failure
				return FALSE;
			}

			// Write the node address out
			if(! Writeln(m_pNode[i].GetAddress()))
			{
				// Return failure
				return FALSE;
			}
			// Copy location into temp city string
			strcpy(szCity, m_pNode[i].GetLocation());
			// Find seperator in the temp city string
			psz = strstr(szCity, m_szLocationSeperator);
			if(psz != NULL)
			{
				// Found - So null terminate string at the comma
				*psz = '\0';

				// Copy after the seperator to the temp state string
				strcpy(szState, psz + strlen(m_szLocationSeperator));

				// Find seperator in the temp state string
				psz = strstr(szState + 1, m_szLocationSeperator);
				if(psz != NULL)
				{
					// Found - So null terminate the string at the comma
					*psz = '\0';

					// Copy after the seperator to the temp country string
					strcpy(szCountry, psz + strlen(m_szLocationSeperator));
				}
				else
				{
					// No state specified
					strcpy(szCountry, szState);
					szState[0] = '\0';
				}
			}

			// Write the node city out
			if(! Writeln(szCity))
			{
				// Return failure
				return FALSE;
			}

			// Write the node state out
			if(! Writeln(szState))
			{
				// Return failure
				return FALSE;
			}

			// Write the node country out
			if(! Writeln(szCountry))
			{
				// Return failure
				return FALSE;
			}

			// Write a blank line seperator
			Writeln();

			// Increment counter
			wNode1++;
		}
	}

	// Reset the modified flag
	m_bModified = FALSE;

	// Return success
	return TRUE;
}

WORD InterBbsNodeFile::GetTotalSystems(void) const
{
	// Return the total systems member attribute
	return m_wTotalSystems;
}

INT16 InterBbsNodeFile::Find(const char* lpszNodeName, const char* lpszNodeAddress) const
{
	assert(lpszNodeName != NULL);
	assert(lpszNodeAddress != NULL);

	if(lpszNodeName == NULL || lpszNodeAddress == NULL)
	{
		// Return failure
		return -1;
	}

	// Find the node that matches the name and address
	WORD wNode = 0;
	for(WORD i = 0; i < F7_MAX_NODES && wNode < m_wTotalSystems; i++)
	{
		if(! m_pNode[i].IsEmpty())
		{
			if(stricmp(m_pNode[i].GetAddress(), lpszNodeAddress) == 0
			&& stricmp(m_pNode[i].GetName(), lpszNodeName) == 0)
			{
				return i;
			}

			wNode++;
		}
	}

	// Return the node index
	return -1;
}

INT16 InterBbsNodeFile::GetRouteTo(const char* lpszNodeName, const char* lpszNodeAddress) const
{
	INT16 iNode;

	// Find the node
	iNode = Find(lpszNodeName, lpszNodeAddress);

	if(iNode < 0)
	{
		// Return failure
		return iNode;
	}

	// Call overloaded GetRouteTo() function
	return GetRouteTo(iNode);
}

INT16 InterBbsNodeFile::GetRouteTo(WORD wNode) const
{
	INT16 iRouteTo = wNode;
	INT16 iSystem = wNode;
	INT16 iHost;
	BOOL bDone = FALSE;

	// Find the program's host node
	iHost = Find(m_szHostNodeName, m_szHostNodeAddress);

	// Calculate who I should send the message to, using routing
	do
	{
		// Get the system that we are routing to
		iRouteTo = m_pNode[iRouteTo].GetRouteTo();

		// Check the route is valid
		if(iRouteTo != -1)
		{
			// Check if the node is me
			if(iRouteTo == iHost)
			{
				// So we route directly
				bDone = TRUE;
			}
			else
			{
				iSystem = iRouteTo;
			}
		}
		// Route is not valid
		else
		{
			// Get our boss node (as all other routing should go to it)
			iRouteTo = m_pNode[iHost].GetRouteTo();

			// Check the route is valid
			if(iRouteTo != -1)
			{
				// Route to it
				iSystem = iRouteTo;

				// And we have finished looking
				bDone = TRUE;
			}
			// Route is not valid
			else
			{
				// So we are finished looking
				bDone = TRUE;
			}
		}
	} while(!bDone);

	return iSystem;
}

InterBbsNode* InterBbsNodeFile::GetNode(WORD wNode)
{
	assert(wNode < F7_MAX_NODES);

	if(wNode >= F7_MAX_NODES)
	{
		return NULL;
	}

	return &m_pNode[wNode];
}

const InterBbsNode& InterBbsNodeFile::GetNode(WORD wNode) const
{
	static InterBbsNode emptyNode;

	assert(wNode < F7_MAX_NODES);

	if(wNode >= F7_MAX_NODES)
	{
		return emptyNode;
	}

	return m_pNode[wNode];
}

BOOL InterBbsNodeFile::IsModified(void) const
{
	for(int i = 0; i < F7_MAX_NODES; i++)
	{
		// Check if a node has been modified
		if(m_pNode[i].IsModified())
		{
			return TRUE;
		}
	}

	// Return the member attribute
	return m_bModified;
}

INT16 InterBbsNodeFile::AddNode(const InterBbsNode& rNode)
{
	// Make sure node doesn't already exist
	if(Find(rNode.GetName(), rNode.GetAddress()))
	{
		// Return failure
		return FALSE;
	}

	for(WORD i = 0; i < F7_MAX_NODES; i++)
	{
		// Test for an empty node
		if(m_pNode[i].IsEmpty())
		{
			// We found one - so copy data
			*m_pNode = rNode;

			// Update the modified flag
			m_bModified = TRUE;

			// Increment the number of systems
			m_wTotalSystems++;

			// Return success
			return i;
		}
	}

	// Return failure
	return -1;
}

BOOL InterBbsNodeFile::RemoveNode(WORD wNode)
{
	assert(wNode < F7_MAX_NODES);

	if(wNode >= F7_MAX_NODES)
	{
		// Return failure
		return FALSE;
	}

	// Set node's data to blank
	m_pNode[wNode].Empty();

	// Decrement the number of systems
	m_wTotalSystems--;

	// Update the modified flag
	m_bModified = TRUE;

	// Return success
	return TRUE;
}