// file0.cpp : implementation file
//
// Class:		File
//
// Created:		19/09/95
//
// Author:		S.D.Holme
//
// Version:		1.01
//
// Modifications:
//
//	1.00	19/09/95	S.D.Holme	Created.
//			20/03/96	S.D.Holme	Added Seek() function.
//			02/04/96	S.D.Holme   Changed Seek() to acccept a long
//									rather than an int.
//			31/10/96	S.D.Holme	Added the following functions:
//										Exists()
//										Remove()
//										Rename()
//										GetLength()
//										Trunc()
//										SetFTime()
//										GetFTime()
//									Renamed eof() to Eof().
//									Class no longer uses fstream as the base
//									class.
//			21/06/97	S.D.Holme	Changed retry timeout to be time
//									dependent.
//			12/07/97	S.D.Holme	Public release.
//
//	1.01	20/05/98	S.D.Holme	Tidied up the class implementation.
//									Added DirExists() function.
//									Changed SetFilename() return type.
//									Changed GetFilename() return type.
//									Renamed file to file0.cpp
//			21/05/98	S.D.Holme	Added m_bInitialised member attribute.
//
// File Library. Copyright (c) 1995-98 Fonty Technologies, Inc.
//
#include <assert.h>
#include <dos.h>
#include <errno.h>
#include <io.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if !defined(WIN32) && !defined(__WIN32__)
	#include <dir.h>
	#include <time.h>
#endif

#include "file.h"

/////////////////////////////////////////////////////////////////////////////
// BinaryFile Class

// Constructor
File::File(const char* pszFilename, int nMode)
{
	// Initialise member attributes
	m_dwRetry = 1000;
	m_nHandle = -1;
	m_bInitialised = FALSE;

	SetFilename(pszFilename);
	SetMode(nMode);
}

// Destructor
File::~File()
{
	// Close the file
	Close();
}

// Initialisation
BOOL File::Initialise(void)
{
	// Initialise member attributes
	m_bInitialised = TRUE;

	return TRUE;
}

// Functions
BOOL File::Open(int nAccess)
{
	unsigned nMode;

	// Check file is already open
	if(IsOpen())
	{
		// Return failure
		return FALSE;
	}

	if(nAccess >= 0)
	{
		SetMode(nAccess);
	}

	// Prepare the mode for O_CREATE access
	if(m_nAccess & O_RDONLY)
	{
		nMode = S_IREAD;
	}
	else if(m_nAccess & O_WRONLY)
	{
		nMode = S_IWRITE;
	}
	else if(m_nAccess & O_RDWR)
	{
		nMode = S_IREAD | S_IWRITE;
	}
	else
	{
		return FALSE;
	}

	#if defined(WIN32) || defined(__WIN32__)
		// Calculate the start time and duration using Win SDK functions
		DWORD dwStart = GetCurrentTime();
		DWORD dwDuration = m_dwRetry;
		DWORD dwNow;
	#else
		// Calculate the start time and durating using DOS functions
		clock_t dwStart = clock();
		clock_t dwDuration = m_dwRetry / (1000 / CLK_TCK);
		clock_t dwNow;
	#endif

	do
	{
		// Open the file
		if(m_nAccess & O_CREAT)
		{
			m_nHandle = open(m_szFilename, m_nAccess, nMode);
		}
		else
		{
			m_nHandle = open(m_szFilename, m_nAccess);
		}

		// If no errno or not EACCES then break the loop
		if(m_nHandle != -1 || errno != EACCES)
		{
			break;
		}

		#if defined(WIN32) || defined(__WIN32__)
			// Get the current time using the Win SDK function
			dwNow = GetCurrentTime();
		#else
			// Get the current time using the DOS functin
			dwNow = clock();
		#endif

	}while(dwStart + dwDuration < dwNow);

	// Check for valid file handle
	if(m_nHandle == -1)
	{
		// Return failure
		return FALSE;
	}

	// Return success
	return TRUE;
}

BOOL File::Close(void)
{
	// Check file is open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	// Close the file
	if(close(m_nHandle) == -1)
	{
		// Return failure
		return FALSE;
	}

	// Reset the handle
	m_nHandle = -1;

	// Return success
	return TRUE;
}

BOOL File::SetMode(int nAccess)
{
	// Not allowed to change the access mode whilst a file is open
	if(IsOpen())
	{
		// Return failure
		return FALSE;
	}

	// Set the access mode
	m_nAccess = nAccess;

	// Return success
	return TRUE;
}

int File::GetMode(void)
{
	// Return the access mode
	return m_nAccess;
}

BOOL File::SetFilename(const char* pszFilename)
{
	assert(pszFilename != NULL);

	// Check for valid parameters
	if(pszFilename == NULL || strlen(pszFilename) >= sizeof(m_szFilename))
	{
		// Return failure
		return FALSE;
	}

	// Set the filename member attribute
	strcpy(m_szFilename, pszFilename);

	// Return success
	return TRUE;
}

BOOL File::GetFilename(char* pszFilename)
{
	assert(pszFilename != NULL);

	// Check for valid parameters
	if(pszFilename == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Return the filename member attribute
	strcpy(pszFilename, m_szFilename);

	// Return success
    return TRUE;
}

BOOL File::IsOpen()
{
	return m_nHandle == -1 ? FALSE : TRUE;
}

BOOL File::Seek(long lPosition, int nFromWhere)
{
	// Check file is open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	if(lseek(m_nHandle, lPosition, nFromWhere) == -1L)
	{
		// Return failure
		return FALSE;
	}

	// Return success
	return TRUE;
}

// Output

BOOL File::Write(void* pBuffer, unsigned nBytesToWrite)
{
	int nBytesWritten;

	assert(pBuffer != NULL);

	// Check for valid parameters
	if(pBuffer == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Check file is open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	// Check file mode is OUTput
	if(! (GetMode() & O_WRONLY || GetMode() & O_RDWR))
	{
		// Return failure
		return FALSE;
	}

	// Write
	nBytesWritten = write(m_nHandle, pBuffer, nBytesToWrite);

	// Check the number of bytes written
	if(nBytesWritten == -1)
	{
		// Return failure
		return FALSE;
	}

	if((unsigned)nBytesWritten < nBytesToWrite)
	{
		// Return failure
		return FALSE;
	}

	// Return success
	return TRUE;
}

BOOL File::Write(const char* pszText)
{
	assert(pszText != NULL);

	// Check for valid parameters
	if(pszText == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Check file is open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	if(strlen(pszText) == 0)
	{
		// Return failure
		return FALSE;
	}

	return Write((void*)pszText, strlen(pszText));
}

BOOL File::Writeln(const char* pszText)
{
	assert(pszText != NULL);

	// Check for valid parameters
	if(pszText == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Check file is open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	if(strlen(pszText) == 0)
	{
		// Return failure
		return FALSE;
	}

	if(! Write(pszText))
	{
		// Return failure
		return FALSE;
	}

	return Write("\n\r");
}

BOOL File::Writef(const char* pszFormat, ...)
{
	va_list vaList;
	char szBuffer[512];

	assert(pszFormat != NULL);

	// Check for valid parameters
	if(pszFormat == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Check file is open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	va_start(vaList, pszFormat);
	vsprintf(szBuffer, pszFormat, vaList);
	va_end(vaList);

	return Write(szBuffer);
}

BOOL File::Putch(char chCharacter)
{
	// Check file is Open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	// Write
	return Write(&chCharacter, 1);
}

BOOL File::Repeat(char chCharacter, int nTimes)
{
	BOOL bReturn = TRUE;

	// Check file is Open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	for(int i = 0; i < nTimes; i++)
	{
		if(! Putch(chCharacter))
			bReturn = FALSE;
	}

	return bReturn;
}

// Input

BOOL File::Read(void* pBuffer, unsigned nBytesToRead)
{
	int nBytesRead;

	assert(pBuffer != NULL);

	// Check for valid parameters
	if(pBuffer == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Check file is open
	if(! IsOpen())
	{
		// Return failure
		return FALSE;
	}

	// Check file mode is INput
	if(! (GetMode() & O_RDONLY || GetMode() & O_RDWR))
	{
		// Return failure
		return FALSE;
	}

	// Read
	nBytesRead = read(m_nHandle, pBuffer, nBytesToRead);

	// Check the number of bytes read
	if(nBytesRead == -1)
	{
		// Return failure
		return FALSE;
	}

	// Return success
	return TRUE;
}

// Misc

BOOL File::Exists(void)
{
	// Test if the file exists
	if(access(m_szFilename, 0) == -1)
	{
		// Return failure
		return FALSE;
	}

	// Return success
	return TRUE;
}

BOOL File::Remove(void)
{
	// Remove the file
	if(unlink(m_szFilename) == -1)
	{
		// Return failure
		return FALSE;
	}

	// Return success
	return TRUE;
}

BOOL File::Rename(const char* pszNewName)
{
	assert(pszNewName != NULL);

	// Check for valid parameters
	if(pszNewName == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Rename the file
	if(rename(m_szFilename, pszNewName) == -1)
	{
		// Return failure
		return FALSE;
	}

	// Update the filename
	SetFilename(pszNewName);

	// Return succes
	return TRUE;
}

BOOL File::Eof(void)
{
	if(!IsOpen())
	{
		// Return failure
		return FALSE;
	}

	// Test for EOF
	if(eof(m_nHandle) == 0)
	{
		// Return failure
		return FALSE;
	}

	// Return success
	return TRUE;
}

long File::GetLength(void)
{
	if(!IsOpen())
	{
		// Return failure
		return -1;
	}

	return filelength(m_nHandle);
}

BOOL File::Trunc(long nNewSize)
{
	if(chsize(m_nHandle, nNewSize) == -1)
	{
		// Return failure
		return FALSE;
	}

	// Return success
	return TRUE;
}

#ifndef WIN32
BOOL File::SetFTime(struct ftime* pFileTime)
{
	assert(pFileTime != NULL);

	// Check for valid parameters
	if(pFileTime == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Set the file time
	if(setftime(m_nHandle, pFileTime) == -1)
	{
		// Return failure
		return FALSE;
	}

	// Return success
	return TRUE;
}

BOOL File::GetFTime(struct ftime* pFileTime)
{
	assert(pFileTime != NULL);

	// Check for valid parameters
	if(pFileTime == NULL)
	{
		// Return failure
		return FALSE;
	}

	// Get the file time
	if(getftime(m_nHandle, pFileTime) == -1)
	{
		// Return failure
		return FALSE;
	}

	// Return success
	return TRUE;
}
#endif

BOOL File::DirExists(const char *pszDirName)
{
	char szDirFileName[_MAX_PATH];
#ifndef WIN32
	struct ffblk dirEntry;
#else
	struct _finddata_t dirEntry;
#endif

	assert(pszDirName != NULL);
	assert(strlen(pszDirName) < (size_t)_MAX_PATH);

	// Check for valid parameters
	if(pszDirName == NULL || strlen(pszDirName) >= (size_t)_MAX_PATH)
	{
		// Return failure
		return FALSE;
	}

	strcpy(szDirFileName, pszDirName);

	// Remove any trailing backslash from directory name
	if(szDirFileName[strlen(szDirFileName) - 1] == '\\')
	{
		szDirFileName[strlen(szDirFileName) - 1] = '\0';
	}

#ifndef WIN32
	// Return true iff file exists and it is a directory
	return(findfirst(szDirFileName, &dirEntry, FA_ARCH|FA_DIREC) == 0 &&
			 (dirEntry.ff_attrib & FA_DIREC));
#else
	long hFile;

	dirEntry.attrib = _A_ARCH|_A_SUBDIR;

	hFile = _findfirst(szDirFileName, &dirEntry);
		
	if(hFile == -1)
	{
		_findclose(hFile);

		return FALSE;
	}
	else
	{
		_findclose(hFile);

		return TRUE;
	}
#endif
}

