/*****************************************************************************
 *
 * File ..................: mbftpd/glob.c
 * Purpose ...............: MBSE BBS Ftp Daemon
 * Last modification date : 21-Feb-2000
 * Original author .......: Rene' Seindal
 *
 *****************************************************************************
 * Copyright (C) 1997-2000
 *   
 * Michiel Broek		FIDO:		2:280/2802
 * Beekmansbos 10
 * 1971 BV IJmuiden
 * the Netherlands
 *
 * This file is part of MBSE BBS.
 *
 * This BBS is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * MBSE BBS is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with MBSE BBS; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/
/*
 * Copyright (c) 1990 Rene' Seindal
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#define MAXNAMLEN  FILENAME_MAX

#include "../config.h"
#include "../lib/libs.h"
#include "../lib/structs.h"
#include "../lib/common.h"
#include "../lib/clcomm.h"
#include "ftpd.h"
#include "ftpcmd.h"
#include "glob.h"


/*
 * glob_match matches pattern against string according to the normal
 * rules for shell globbing.  It returns GL_SUCCES if string matches
 * pattern, GL_FAIL if it doesn't, and GL_ERROR if pattern is illformed.
 *
 * To be more specific, a pattern can be:
 *	?	which matches any single character
 *	*	which matches zero or more character
 *	[...]	which matches any single character in ...
 *	[^...]	which matches any single character not in ...
 *	\x	where x is any character, matches that character literally
 *	x	where x is any character except ? * [ and \, matches x
 * 
 * Character within [...] can include single characters and ranges of
 * the form x-y, which matches any characters in the ranges x - y
 * inclusive.  It is considered an error is y < x.  An ] can be included
 * as the very first character in ..., and a - as the first (after a
 * possibly [) or last character in ...
 */


#define GL_SUCCES	1
#define GL_FAIL		0
#define GL_ERROR	0

#define NEXTP {if ( (p = *++pattern) == '\0') return GL_ERROR;}


int glob_match(char *pattern, char *string)
{
	register char	p;
	char		seenstar = '\0';

	for (; (p = *pattern); pattern++) {

		if ( p == '\\' ) {
			NEXTP;		/* matches next char literally (but not \0) */
			if ( p != *string++ )
				return GL_FAIL;	/* string too short */
			continue;
		} else if ( p == '?' ) {
			if ( *string++ )	/* matches any character */
				continue;
			else
				return GL_FAIL;	/* string too short */
		} else if ( p == '*' ) {
			seenstar = '\1';
			continue;
		} else {
			if ( seenstar ) {
				int tmp;
				while ( *string ) {
					tmp = glob_match( pattern, string );
					if ( tmp != GL_FAIL )
						return tmp;
					string++;
				}
				return GL_FAIL;
				/* NOTREACHED */
			}

			if ( p == '\0' )
				return GL_FAIL;

			if ( p == '[' ) {
				register char s = *string++;
				char reverse = '\0';
				char first, last;
				char gotcha = '\0';

				NEXTP;
				if ( p == '^' ) {
					reverse = '\1';
					NEXTP;
				}
				if ( p == ']' ) { /* special case */
					gotcha = (s==p);
					NEXTP;
				}

				while (  p != ']' && !gotcha ) {
					first = p;
					NEXTP;
					if ( p == '-' && pattern[1] != ']' ) {
						NEXTP;
						last = p;
						NEXTP;
					} else
						last = first;
					if ( first > last )
						return GL_ERROR;
					gotcha = (first <= s && s <= last );
				}
				while ( p != ']' )
					NEXTP;

				if ( reverse ? gotcha : !gotcha )
					return GL_FAIL;
			} else if ( p != *string )
				return GL_FAIL;
			else
				string++;
		}
	}

	if (seenstar)
		return GL_SUCCES;

	if (*string)
		return GL_FAIL;
	return GL_SUCCES;
}



static char	*main_path;	/* ptr to scratchpad */
static int	offset;		/* no of leading char in main_path to ignore */
static char	**namelist;	/* name list buildup */
static int	nnames;		/* no of names found */
static int	left;		/* no of slots allocated but not used yet */

#define MAXSEG	50		/* max segments in pattern */
#define CHUNK	20		/* no of slots to allocate at a time */



int glob_path(char *pattern, char *(*names[]))
{
	char		mpath[ MAXPATHLEN + MAXNAMLEN + 1 ];
	char		*gpat[MAXSEG];
	register char	*pat;

	if ( pattern == 0)
		return -1;

	if ( (pat = malloc( strlen(pattern) + 1 )) == NULL )
		return -1;

	strcpy( pat, pattern );

	if ( split_pat( pat, gpat ) < 0 ) {
		free( pat );
		return -1;
	}
    
	main_path = mpath;		/* initalisation of static storage */
	namelist = 0;
	nnames = left = 0;

	if ( *gpat && **gpat == '/' ) {
		main_path[0] = '/';
		main_path[1] = '\0';
		offset = 0;
		do_glob( main_path, gpat+1 );
	} else {
		offset = 2;
		strcpy( main_path, "." );
		do_glob( main_path + 1, gpat );
	}

	free( pat );

	if (namelist == 0)
		*names = (char **)malloc( sizeof(char *) );
	else
		*names = (char **)realloc( namelist, (nnames+1)*sizeof(char *) );

	if ( *names == 0 )
		return -1;
	(*names)[nnames] = 0;
	return nnames;
}



int split_pat(char *pattern, char **table )
{
	register char	*pp = pattern;
	int		size = MAXSEG;

	if ( *pattern == '/' ) {
		*table++ = (char *)"/";
		--size;
	}
	do {
		while ( *pp == '/' ) *pp++ = '\0';
			if ( *pp == '\0' )
				break;
		if ( --size < 0 )
			return -1;
		*table++ = pp;
		while ( *pp && *pp != '/' ) 
			pp++;
	} while ( *pp );

	*table = 0;
	return 0;
}


#define ISGLOB(x) ((x)=='*' || (x)=='?' || (x)=='[')


int no_glob(char *pat)
{
	while ( *pat && !ISGLOB(*pat) )
		pat++;
	return (*pat == '\0');
}



void do_glob(char *path_end, char **gpat )
{
	char		*saved_end = path_end;	/* saved to be resored */
	register char	*pat;			/* current pattern segment */
	struct stat	st;			/* to check if file exists */

	for ( ; (pat = *gpat) != 0 && no_glob(pat); gpat++ ) {
		*path_end = '/';
		strcpy( path_end+1, pat );
		path_end += strlen(pat) + 1;

		if ( stat( main_path, &st ) != 0 ) {
			*saved_end = '\0';
			return;
		}
	}
	if ( pat )
		matchdir( path_end, gpat );
	else
		add_name();

	*saved_end = '\0';
	return;
}



void matchdir(char *path_end, char **gpat )
{
	register char	*x = NULL;	/* scratch */
	DIR		*dirp;		/* for directory reading */
	struct dirent	*dp;		/* directory entry */
	struct stat	st;		/* to determine files type */

	if ( (dirp = opendir( main_path )) == NULL )
		return;

	*path_end = '/';
    
	while ((dp = readdir(dirp)) != NULL ) {
		x = dp->d_name;
		if (*x == '.' && (*++x == '\0' || (*x == '.' && *++x == '\0')))
			continue;
		if (*dp->d_name == '.' && **gpat != '.')
			continue;

		strcpy( path_end+1, dp->d_name );

		if ( glob_match(*gpat, dp->d_name) ) {	/* this is a match */
			if ( *(gpat+1) == 0 ) {		/* and it is the last */
				add_name();		/* so eat it */
				continue;
			}
			if (stat(main_path, &st ) == 0 && (st.st_mode & S_IFMT) == S_IFDIR)
				do_glob( path_end + strlen(dp->d_name) + 1, gpat+1);
		} 
	}
	*path_end = '\0';
}



void add_name()
{
	register char *np;

	if ( --left <= 0 ) {
		if (namelist == 0)
			namelist = (char **)malloc( CHUNK * sizeof(char *) );
		else
			namelist = (char **)realloc( namelist, (nnames+CHUNK)*sizeof(char *) );

		if ( namelist == NULL )
			return;
	
		left = CHUNK;
	}
	np = malloc( strlen(main_path) - offset + 1 );
	if ( np == 0 )
		return;
	strcpy( np, main_path+offset );
	namelist[nnames++] = np;
	return;
}



/* the following is not general purpose code. */
char	*globerr;
char	globerr_buf[1024];
char	*home = NULL;



int name_cmp(char **s1, char **s2)
{
	return strcmp(*s1, *s2);
}



char *expand_tilde(char *pat)
{
	static char	buf[BUFSIZ];

	struct passwd	*pw;
	register char	*tmp;
	register char	*bp;

	if (*pat != '~')
		return 0;

	pat++;
	if (*pat && *pat != '/') {
		bp = buf;
		for (tmp = pat; *tmp && *tmp != '/'; )
			*bp++ = *tmp++;
		*bp = '\0';

		pw = getpwnam(buf);
		if (pw == 0) {
			sprintf(globerr_buf, "%s: Unknown user.", pat);
			globerr = globerr_buf;
			return 0;
		}
		pat = tmp ? tmp : (char *)"";
		tmp = pw->pw_dir;
	} else {
		if (*pat)
			pat++;
		tmp = home;
	}
	for (bp = buf; *tmp;)
		*bp++ = *tmp++;
	*bp++ = '/';

	while (*pat)
		*bp++ = *pat++;
	*bp = '\0';

	return buf;
}



char **glob(char *pat)
{
	char	**names;
	int	nnnames;

	if (*pat == '~') {
		pat = expand_tilde(pat);
		if (pat == 0)
			return 0;
	}

	nnnames = glob_path(pat, &names);

	switch (nnnames) {
		case -1:
			globerr = (char *)sys_errlist[errno];
			return 0;
		case 0:
			globerr = (char *)"No match.";
			return 0;
		default:
			qsort(names, nnnames, sizeof(char *), (int(*)(const void *, const void *))name_cmp);
			return names;
	}
}



/* following added by Rene' Seindal Wed Jan 24 15:56:13 1990 */

char **copyblk(char **v)
{
	int	n;
	char	**vv;
	char	**svv;

	n = 0; 
	for (vv = v; *vv != 0; vv++)
		n++;

	vv = (char **)malloc((n+1) * sizeof(char *));
	if (vv == 0)
		fatal((char *)"Out of memory.");

	svv = vv;
	while (--n >= 0)
		*vv++ = copy(*v++);
	*vv = 0;
	return svv;
}



void blkfree(char **v)
{
	char **sv = v;

	while (*v)
		free(*v++);
	free(sv);
}


