/*****************************************************************************
 *
 * File ..................: mbftpd/ftpd.c
 * Purpose ...............: MBSE BBS Ftp Daemon
 * Last modification date : 12-Mar-2000
 *
 *****************************************************************************
 * Copyright (C) 1997-2000
 *   
 * Michiel Broek		FIDO:		2:280/2802
 * Beekmansbos 10
 * 1971 BV IJmuiden
 * the Netherlands
 *
 * This file is part of MBSE BBS.
 *
 * This BBS is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * MBSE BBS is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with MBSE BBS; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/

#ifndef _XOPEN_SOURCE
#define	_XOPEN_SOURCE
#endif

#include "../config.h"
#include "../lib/libs.h"
#include "../lib/structs.h"
#include "../lib/records.h"
#include "../lib/common.h"
#include "../lib/clcomm.h"
#include "../lib/dbcfg.h"
#include "extensions.h"
#include "glob.h"
#include "access.h"
#include "conversions.h"
#include "logwtmp.h"
#include "popen.h"
#include "util.h"
#include "ftpcmd.h"
#include "ftpd.h"

#ifndef NBBY
#define	NBBY 8
#endif


jmp_buf		errcatch, urgcatch;
char		envptr[128];
char		version[256];
int		logged_in;
struct passwd	*pw;
time_t		t_start, t_end;
int		timeout = 900;
int		maxtimeout = 7200;
int		logged_in = FALSE;
int		anonymous = 0;
int		administrator = 0;
int		data;
int		type;
int		form;
int		stru;
int		mode;
int		usedefault = 1;		/* for data transfers		*/
int		pdata = -1;		/* for passive mode		*/
int		restart_point = 0;
int		transflag;
int		login_attempts;		/* number of failed login attempts */
int		askpasswd;		/* had user command, ask for passwd */
int		mangleopts;
off_t		filesize;
off_t		bytecount;
int		defumask = 027;		/* default umask value		*/
char		tmpline[7];
char		hostname[MAXHOSTNAMELEN];
char		remotehost[MAXHOSTNAMELEN];
char		remoteaddr[MAXHOSTNAMELEN];
char		guestpw[MAXHOSTNAMELEN];
char		privatepw[MAXHOSTNAMELEN];
int		nameserved;
char		*typenames[] = {(char *)"0", (char *)"ASCII", (char *)"EBCDIC", (char *)"Image", (char *)"Local" };
char		*formnames[] = {(char *)"0", (char *)"Nonprint", (char *)"Telnet", (char *)"Carriage-control" };
char		*strunames[] = {(char *)"0", (char *)"File", (char *)"Record", (char *)"Page" };
char		*modenames[] = {(char *)"0", (char *)"Stream", (char *)"Block", (char *)"Compressed" };
static char	ttyline[20];
uid_t		sysuid;			/* mbse uid			*/
gid_t		sysgid;			/* bbs gid			*/



extern fil_areas	*are;
extern unsigned long	Downloads;
extern unsigned long	DownloadK;
extern unsigned long	Uploads;
extern unsigned long	UploadK;



/*
 * Allow use of lreply(); this is here since some older FTP clients don't
 * support continuation messages.  In violation of the RFCs...
 */
int		dolreplies = 1;



/*
 * Spontaneous reply text.  To be sent along with next reply to user
 */
char		*autospout = NULL;
int		autospout_free = 0;



/*
 * Timeout intervals for retrying connections
 * to hosts that don't accept PORT cmds.  This
 * is a kludge, but given the problems with TCP...
 */
#define SWAITMAX	90		/* wait at most 90 seconds */
#define SWAITINT	5		/* interval between retries */

int		swaitmax = SWAITMAX;
int		swaitint = SWAITINT;

extern char	cbuf[];
extern char	*home;

struct sockaddr_in	ctrl_addr;
struct sockaddr_in	data_source;
struct sockaddr_in	data_dest;
struct sockaddr_in	his_addr;
struct sockaddr_in	pasv_addr;



void die(int onsig)
{
	signal(onsig, SIG_IGN);

	/*
	 * Regain root privileges and chroot to the real root so system update
	 * can be done. This doesn't work yet!
	 */
	Syslog('f', "uid now %d.%d %d.%d", getuid(), getgid(), geteuid(), getegid());

	if (getegid()) {
		if  (setegid((gid_t)0) == -1)
			WriteError("$Can't set gid %d", (int)sysgid);

		if (geteuid() && (seteuid((uid_t)0) == -1))
				WriteError("$Can't set uid %d", (int)sysuid);

		if ((chroot("/") == -1) || (chdir("/") == -1))
			WriteError("$Can't chroot(\"/\") again");
		else
			Syslog('f', "chroot oke");

        	if (logged_in)
	                Logwtmp(ttyline, (char *)"", (char *)"");
	}

        if (pw->pw_name)
                free(pw->pw_name);
	if (pw->pw_passwd)
		free(pw->pw_passwd);
        if (pw->pw_gecos)
                free(pw->pw_gecos);
        if (pw->pw_dir)
                free(pw->pw_dir);
        if (pw->pw_shell)
                free(pw->pw_shell);

	if (home)
		free(home);
	exit_bbs();

	if (onsig)
		WriteError("$Terminated on signal %d", onsig);

	time(&t_end);
	Syslog(' ', "MBFTPD finished in %s", t_elapsed(t_start, t_end));

	ExitClient(onsig);
}



void lostconn(void)
{
	Syslog('+', "Lost connection to %s [%s]", remotehost, remoteaddr);
	die(0);
}



void myoob(void)
{
	char	*cp;

	/*
	 * only process if transfer occurring
	 */
	if (!transflag)
		return;
	cp = tmpline;
	if (Getline(cp, 7, stdin) == NULL) {
		reply(221, "You could at least say goodbye.");
		die(0);
	}
	upper(cp);
	if (strcmp(cp, "ABOR\r\n") == 0) {
		tmpline[0] = '\0';
		reply(426, "Transfer aborted. Data connection closed.");
		reply(226, "Abort successful");
		longjmp(urgcatch, 1);
	}
	if (strcmp(cp, "STAT\r\n") == 0) {
		if (filesize != (off_t) -1)
			reply(213, "Status: %lu of %lu bytes transferred", bytecount, filesize);
		else
			reply(213, "Status: %lu bytes transferred", bytecount);
	}
}



void reply(int n, const char *fmt, ...)
{
	va_list		va_ptr;
	char		*outstr;

	outstr = calloc(2048, sizeof(char));

	va_start(va_ptr, fmt);
	vsprintf(outstr, fmt, va_ptr);
	va_end(va_ptr);

	/*
	 * First handle sponntaneous reply text if any.
	 */
	if (autospout != NULL) {
		char *ptr = autospout;
  
		printf("%d-", n);
		while (*ptr) {
			if (*ptr == '\n') {
				printf("\r\n");
				if (*(++ptr))
					printf("%d-", n);
			} else {
				putchar(*ptr++);
			}
		}
		if (*(--ptr) != '\n')
			printf("\r\n");
		if (autospout_free) {
			(void)free(autospout);
			autospout_free = 0;
		}
		autospout = 0;
	}

	printf("%d %s\r\n", n, outstr);
	(void)fflush(stdout);
	if (CFG.ftp_log_cmds)
		Syslog('+', "%d %s", n, outstr);
	free(outstr);
}



void lreply(int n, const char *fmt, ...)
{
	va_list         va_ptr;
	char            *outstr;

	if (!dolreplies)
		return;

	outstr = calloc(2048, sizeof(char));

	va_start(va_ptr, fmt);
	vsprintf(outstr, fmt, va_ptr);
	va_end(va_ptr);

	printf("%d-%s\r\n", n, outstr);
	(void)fflush(stdout);
	free(outstr);
}



int main (int argc, char **argv)
{
	struct utsname	u;
	int		addrlen, i, on = 1;
	char		*cmd;
	struct passwd	*ppw;

#ifdef MEMWATCH
	mwInit();
#endif

	/*
	 *  First we supply a fake environment variable MBSE_ROOT
	 *  because this program is called by inetd as root.
	 */
	ppw = getpwnam((char *)"mbse");
	sysuid = ppw->pw_uid;
	sysgid = ppw->pw_gid;
	sprintf(envptr, "MBSE_ROOT=%s", ppw->pw_dir);
	putenv(envptr);
	InitConfig();
	time(&t_start);

	cmd = xstrcpy((char *)"Cmd: mbftpd");

	for (i = 1; i < argc; i++) {
		cmd = xstrcat(cmd, (char *)" ");
		cmd = xstrcat(cmd, tl(argv[i]));
	}

	InitClient(ppw->pw_name, (char *)"mbftpd", CFG.location, CFG.logfile, CFG.util_loglevel, CFG.error_log);
	IsDoing("Login");

	Syslog(' ', " ");
	Syslog(' ', "MBFTPD v%s", MBSEVersion);
	Syslog(' ', "WARNING: DO NOT USE THIS PROGRAM");
	Syslog(' ', cmd);
	free(cmd);

	uname(&u);
	sprintf(version, "%s %s %s %s %s", u.sysname, u.nodename, u.release, u.version, u.machine);

	addrlen = sizeof(his_addr);
	if (getpeername(0, (struct sockaddr *)&his_addr, &addrlen) < 0) {
		WriteError("getpeername(%s): %m", argv[0]);
		ExitClient(1);
	}

	addrlen = sizeof(ctrl_addr);
	if (getsockname(0, (struct sockaddr *)&ctrl_addr, &addrlen) < 0) {
		WriteError("getsockname(%s): %m", argv[0]);
		ExitClient(1);
	}

	data_source.sin_port = htons(ntohs(ctrl_addr.sin_port) - 1);

	freopen("/dev/null", "w", stderr);
	signal(SIGCHLD, SIG_IGN);
	if ((int)signal(SIGURG, (void (*))myoob) < 0)
		WriteError("signal: %m");
	
	/* Try to handle urgent data inline */
#ifdef SO_OOBINLINE
	if (setsockopt(0, SOL_SOCKET, SO_OOBINLINE, (char *)&on, sizeof(on)) < 0)
		WriteError("setsockopt: %m");
#endif

	if (fcntl(fileno(stdin), F_SETOWN, getpid()) == -1)
		WriteError("fcntl F_SETOWN: %m");

	dolog(&his_addr);
	/*
	 * Set up default state
	 */
	data = -1;
	type = TYPE_A;
	form = FORM_N;
	stru = STRU_F;
	mode = MODE_S;
	tmpline[0] = '\0';
	(void)gethostname(hostname, sizeof (hostname));

	if (is_shutdown() != 0) {
		Syslog('+', "Connection refused (server shut down) from %s [%s]", remotehost, remoteaddr);
		reply(500, "%s FTP server shut down -- please try again later.", hostname);
		die(0);
	}

	show_banner(220);
	signal(SIGPIPE, (void (*))lostconn);
	reply(220, "%s FTP server (%s) ready.", hostname, version);
	(void)setjmp(errcatch);

	for (;;)
		(void)yyparse();
	/* NOTREACHED */

	return 0;
}



void dolog(struct sockaddr_in *ssin)
{
	struct hostent *hp = gethostbyaddr((char *)&ssin->sin_addr, sizeof(struct in_addr), AF_INET);

	strncpy(remoteaddr, inet_ntoa(ssin->sin_addr), sizeof (remoteaddr));
	if (hp) {
		nameserved = TRUE;
		strncpy(remotehost, hp->h_name, sizeof (remotehost));
	} else {
		nameserved = FALSE;
		strncpy(remotehost, remoteaddr, sizeof (remotehost));
	}

	Syslog('+', "Connection from %s [%s]", remotehost, remoteaddr);
}



/*
 * Helper function for sgetpwnam().
 */
char *sgetsave(char *);
char *sgetsave(char *s)
{
	char	*new;
    
	new = (char *) malloc(strlen(s) + 1);

	if (new == NULL) {
		perror_reply(421, (char *)"Local resource failure: malloc");
		die(1);
	}
	(void) strcpy(new, s);
	return (new);
}



/*
 * Save the result of a getpwnam.  Used for USER command, since the data
 * returned must not be clobbered by any other command (e.g., globbing).
 */
struct passwd *sgetpwnam(char *name)
{
	static struct passwd	save;
	register struct passwd	*p;

	if ((p = getpwnam(name)) == NULL)
		return (p);

	if (save.pw_name)   
		free(save.pw_name);
	if (save.pw_passwd)
		free(save.pw_passwd);
	if (save.pw_gecos)  
		free(save.pw_gecos);
	if (save.pw_dir)    
		free(save.pw_dir);
	if (save.pw_shell)  
		free(save.pw_shell);

	save = *p;
	save.pw_name = sgetsave(p->pw_name);
	save.pw_passwd = sgetsave(p->pw_passwd);
	save.pw_gecos = sgetsave(p->pw_gecos);
	save.pw_dir = sgetsave(p->pw_dir);
	save.pw_shell = sgetsave(p->pw_shell);
	return(&save);
}



/*
 * USER command. Sets global passwd pointer pw if named account exists and is
 * acceptable; sets askpasswd if a PASS command is expected.  If logged in
 * previously, need to reset state.  If name is "ftp" or "anonymous", the
 * name is not in the bbs userfile, and ftp account exists, set anonymous
 * and pw, then just return.  If account doesn't exist, ask for passwd
 * anyway.  Otherwise, check user requesting login privileges.  Disallow
 * anyone who does not have a shell defined in /etc/passwd.
 * Disallow anyone not in the bbs userfile.
 */
void user(char *name)
{
	char	*shell;
	int	why, l_compress = 0, l_tar = 0;

	if (logged_in) {
		if (anonymous) {
			reply(530, "Can't change user from guest login.");
			return;
		}
		end_login();
	}

	anonymous = 0;
	administrator = 0;
	mangleopts = 0;
	if (CFG.ftp_compress) {
		l_compress = 1;
		mangleopts |= l_compress * (O_COMPRESS | O_UNCOMPRESS);
	}
	if (CFG.ftp_tar) {
		l_tar = 1;
		mangleopts |= l_tar * O_TAR;
	}

	/*
	 * Anonymous users check
	 */
	if (!strcasecmp(name, "ftp") || !strcasecmp(name, "anonymous")) {

		if ((pw = sgetpwnam((char *)"ftp")) != NULL) {
			anonymous = 1;
			why = access_ok(pw->pw_name);
			switch (why) {
			case 1:
				askpasswd = 1;
				reply(331, "Guest login ok, send e-mail address as password.");
				break;
			case 0:
				reply(530, "User %s access denied.", name);
				Syslog('+', "FTP LOGIN REFUSED (access denied) FROM %s [%s], %s", remotehost, remoteaddr, name);
				die(0);
			case -1:
				reply(530, "Server access denied, too many users, try again later.");
				Syslog('+', "FTP LOGIN REFUSED (too many users) FROM %s [%s], %s", remotehost, remoteaddr, name);
				die(0);
			}
		} else {
			reply(530, "User %s unknown.", name);
			Syslog('+', "FTP LOGIN REFUSED (ftp not in /etc/passwd) FROM %s [%s], %s", remotehost, remoteaddr, name);
			die(0);
		}
		return;
	}

	/*
	 * Check if user has some kind of shell
	 */
	if ((pw = sgetpwnam(name)) != NULL) {
		/*
		 * Check if user has some kind of shell
		 */
		if ((shell = pw->pw_shell) == NULL || *shell == 0) {
			reply(530, "User %s access denied.", name);
			Syslog('+', "FTP LOGIN REFUSED (no shell) FROM %s [%s], %s", remotehost, remoteaddr, name);
			pw = (struct passwd *) NULL;
			return;
		}

		if (strcmp(pw->pw_name, "mbse") == 0)
			administrator = 1;

		/*
		 * Regular users check
		 */
		why = access_ok(pw->pw_name);
		switch (why) {
		case 0:
			reply(530, "User %s access denied.", name);
			Syslog('+', "FTP LOGIN REFUSED (access denied) FROM %s [%s], %s", remotehost, remoteaddr, name);
			die(0);
		case -1:
			reply(530, "Server access denied, too many users, try again later.");
			Syslog('+', "FTP LOGIN REFUSED (too many users) FROM %s [%s], %s", remotehost, remoteaddr, name);
			die(0);
		}
	}

	reply(331, "Password required for %s.", name);
	askpasswd = 1;

	/*
	 * Delay before reading passwd after first failed
	 * attempt to slow down passwd-guessing programs.
	 */
	if (login_attempts)
		sleep((unsigned) login_attempts);
}



/*
 * Terminate login as previous user, if any, resetting state;
 * used when USER command is given or login fails.
 */
void end_login()
{
	(void) seteuid((uid_t)0);
	if (logged_in)
		Logwtmp(ttyline, (char *)"", (char *)"");
	pw = NULL;
	logged_in = 0;
	anonymous = 0;
	administrator = 0;
}



int validate_eaddr(char *);
int validate_eaddr(char *eaddr)
{
	int i, host, state;

	for (i = host = state = 0; eaddr[i] != '\0'; i++) {
		switch (eaddr[i]) {
		case '.':
			if (!host)
				return 0;
			if (state == 2)
				state = 3;
			host = 0;
			break;
		case '@':
			if (!host || state > 1 || !strncasecmp("ftp", eaddr + i - host, host))
				return 0;
			state = 2;
			host = 0;
			break;
		case '!':
		case '%':
			if (!host || state > 1)
				return 0;
			state = 1;
			host = 0;
			break;
		case '-':
			break;
		default:
			host++;
		}
	}
	if (((state == 3) && host > 1) || ((state == 2) && !host) || ((state == 1) && host > 1))
		return 1;
	else
		return 0;
}



void pass(char *passwd)
{
	char		*xpasswd, *salt, *the_user;

	if (logged_in || askpasswd == 0) {
		reply(503, "Login with USER first.");
		return;
	}
	askpasswd = 0;

	/* 
	 * Disable lreply() if the first character of the password is '-' since
	 * some hosts don't understand continuation messages and hang...
	 */
	if (*passwd == '-') 
		dolreplies = 0; 
	else 
		dolreplies = 1;

	if (!anonymous) {               /* "ftp" is only account allowed no password */
		if (*passwd == '-') 
			passwd++;
#ifdef SHADOW_PASSWORD
		if (pw) {
			struct spwd *spw = getspnam(pw->pw_name);
			if (!spw) { 
				pw->pw_passwd = (char *)"";
			} else { 
				pw->pw_passwd = spw->sp_pwdp;
			}
		}
#endif
		*guestpw = (char) NULL;
		if (pw == NULL)
			salt = (char *)"xx";
		else {
			salt = pw->pw_passwd;
		}
		xpasswd = crypt(passwd, salt);
		/*
		 * The strcmp does not catch null passwords!
		 */
		if (pw == NULL || *pw->pw_passwd == '\0' || strcmp(xpasswd, pw->pw_passwd)) {
			reply(530, "Login incorrect.");
			if (pw)
				the_user = pw->pw_name;
			else
				the_user = (char *)"unknown";
			pw = NULL;
			Syslog('+', "Failed login from %s [%s], %s", remotehost, remoteaddr, SS(the_user));
			if (++login_attempts >= CFG.ftp_loginfails) {
				Syslog('+', "Repeated login failures from %s [%s]", remotehost, remoteaddr);
				die(0);
			}
			return;
		}
	} else {
		char	*pwin, *pwout = guestpw;

		if (!*passwd) {
			strcpy(guestpw, "[none_given]");
		} else {
			int	cnt = sizeof(guestpw) - 2;
			for (pwin = passwd; *pwin && cnt--; pwin++)
				if (!isgraph(*pwin))
					*pwout++ = '_';
				else
					*pwout++ = *pwin;
		}
	}

	login_attempts = 0;             /* this time successful */

	/*
	 * Tell mbsed who we are and what we are doing
	 */
	UserCity(getpid(), pw->pw_name, remotehost);
	IsDoing("Browse");

	/* open wtmp before chroot */
	(void) sprintf(ttyline, "ftp%d", getpid());
	Logwtmp(ttyline, pw->pw_name, remotehost);
	logged_in = 1;

	Syslog('f', "logged in: anonymous=%s administrator = %s", anonymous ? (char *)"True" : (char *)"False", 
					   administrator ? (char *)"True" : (char *)"False");

	/*
	 * Before chroot we must load all relevant bbs data because
	 * all users except the administrator can't access the 
	 * $MBSE_ROOT anymore.
	 */
	if (!init_bbs(pw->pw_name, anonymous)) {
		reply(550, "Can't initialize server database");
		goto bad;
	}

	if (anonymous) {
		/*
		 * We MUST do a chdir() after the chroot. Otherwise
		 * the old current directory will be accessible as "."
		 * outside the new root!
		 */
		if (chroot(pw->pw_dir) < 0 || chdir("/") < 0) {
			reply(550, "Can't set guest privileges.");
			goto bad;
		}
		home = xstrcpy(pw->pw_dir);
	} else if (administrator) {
		if (chdir(pw->pw_dir) < 0) {
			if (chdir("/") < 0) {
				reply(530, "User %s: can't change directory to %s.", pw->pw_name, pw->pw_dir);
				goto bad;
			} else
				lreply(230, "No directory! Logging in with home=/");
		}
		home = xstrcpy(pw->pw_dir);
	} else {
		/*
		 * Regular bbs users
		 */
		if (chroot(CFG.ftp_base) < 0 || chdir("/") < 0) {
			reply(550, "User %s: can't change directory to %s.", pw->pw_name, CFG.ftp_base);
			goto bad;
		}
		home = xstrcpy(CFG.ftp_base);
	}

	if (setegid(sysgid) == -1) {
		WriteError("$Can't set gid %d", (int)sysgid);
		reply(550, "Can't set gid.");
		goto bad;
	}

	if (seteuid(sysuid)  == -1) {
		WriteError("$Can't set uid %d", (int)sysuid);
		reply(550, "Can't set uid.");
		goto bad;
	}

	if (anonymous) {
		if (validate_eaddr(passwd) == 0) {
			lreply(230, "Next time please use your e-mail address as your password");
			lreply(230, "for example: %s@%s", "joe", remotehost);
		}
		show_message(230, E_LOGIN);
		show_readme(230, E_LOGIN);
		(void)is_shutdown(); /* display any shutdown messages now */

		reply(230, "Guest login ok, access restrictions apply.");
		Syslog('+', "ANONYMOUS FTP LOGIN FROM %s [%s], %s", remotehost, remoteaddr, passwd);
	} else if (!administrator) {
		/*
		 * regular bbs users are also in the ftp base directory
		 */
		show_message(230, E_LOGIN);
		show_readme(230, E_LOGIN);
		(void) is_shutdown(); /* display any shutdown messages now */
		reply(230, "BBS access restrictions apply.");
		Syslog('+', "BBS USER LOGIN FROM %s [%s]", remotehost, remoteaddr);
	} else {
		(void) is_shutdown(); /* display any shutdown messages now */
		reply(230, "BBS Administrator logged in");
		Syslog('+', "ADMINISTRATOR LOGIN FROM %s [%s]", remotehost, remoteaddr);
	}

	(void) umask(defumask);
	return;
bad:
	/* Forget all about it... */
	end_login();
}



char *opt_string(int options)
{
	static char	buf[100];
	char		*ptr = buf;

	if ((options & O_COMPRESS) != 0) 
		*ptr++ = 'C';
	if ((options & O_TAR) != 0) 
		*ptr++ = 'T';
	if ((options & O_UNCOMPRESS) != 0) 
		*ptr++ = 'U';
	if (options == 0) 
		*ptr++ = '_';
	*ptr++ = '\0';
	return(buf);
}



void retrieve(char *cmd, char *name)
{
	FILE		*fin = NULL, *dout;
	struct stat	st, junk;
	int		(*closefunc)(FILE *) = NULL;
	int		options = 0;
	time_t		start_time = time(NULL);
	static char	*logname;
	char		namebuf[MAXPATHLEN];

	Syslog('f', "Retrieve(%s, %s)", SS(cmd), SS(name));

	if (!cmd && stat(name, &st)) {
		char    fnbuf[MAXPATHLEN], *ptr;

		Syslog('f', "1");
		do {
			if (!(mangleopts & O_COMPRESS) && (cvtptr->options & O_COMPRESS))
				continue;
			if (!(mangleopts & O_UNCOMPRESS)&&(cvtptr->options & O_UNCOMPRESS))
				continue;
			if (!(mangleopts & O_TAR) && (cvtptr->options & O_TAR))
				continue;

			if (cvtptr->postfix) {
				int pfxlen = strlen(cvtptr->postfix);
				int namelen = strlen(name);

				if (namelen <= pfxlen) 
					continue;
				(void) strcpy(fnbuf, name);
				if (strcmp(fnbuf + namelen - pfxlen, cvtptr->postfix)) 
					continue;
				*(fnbuf + namelen - pfxlen) = (char) NULL;
				if (stat(fnbuf, &st)) 
					continue;
			}

			if (cvtptr->stripfix) {
				(void) strcpy(fnbuf, name);
				(void) strcat(fnbuf, cvtptr->stripfix);
				if (stat(fnbuf, &st)) 
					continue;
			}

			if (S_ISDIR(st.st_mode)) {
				if (!(cvtptr->types & T_DIR)) {
					(void) reply(550, "Cannot %s directories.", cvtptr->name);
					return;
				}
				if (cvtptr->options & O_TAR) { 
					strcpy(namebuf, fnbuf);
					strcat(namebuf, "/.notar");
					if (!stat(namebuf, &junk)) {
						(void) reply(550, "Sorry, you may not TAR that directory.");
						return;
					}
				}
			}

			if (S_ISREG(st.st_mode) && !(cvtptr->types & T_REG)) {
				(void) reply(550, "Cannot %s plain files.", cvtptr->name);
				return;
			}

			if (!S_ISREG(st.st_mode) && !S_ISDIR(st.st_mode)) {
				(void) reply(550, "Cannot %s special files.", cvtptr->name);
				return;
			}

			if (!(cvtptr->types & T_ASCII) && deny_badasciixfer(550, (char *)""))
				return;

			logname = fnbuf;
			options |= cvtptr->options;

			strcpy(namebuf, cvtptr->external_cmd);
			if ((ptr = strchr(namebuf, ' ')) != NULL)
				*ptr = '\0';
			if (stat(namebuf, &st) != 0) {
				syslog(LOG_ERR, "external command %s not found", namebuf);
				(void) reply(550, "Local error: conversion program not found. Cannot %s file.", cvtptr->name);
				return;
			}
			(void) retrieve(cvtptr->external_cmd, fnbuf);

			goto dolog;
		} while ((++cvtptr)->name);
	} else {
		Syslog('f', "2");
		logname = (char *) NULL;
	}

	if (cmd == 0) {
		fin = fopen(name, "r"), closefunc = fclose;
		st.st_size = 0;
	} else {
		char line[BUFSIZ];

		(void) sprintf(line, cmd, name), name = line;
		fin = ftpd_popen(line, (char *)"r"), closefunc = ftpd_pclose;
		st.st_size = -1;
		st.st_blksize = BUFSIZ;
	}
	if (fin == NULL) {
		if (errno != 0)
			perror_reply(550, name);
		return;
	}
	if (cmd == 0 && (fstat(fileno(fin), &st) < 0 || (st.st_mode&S_IFMT) != S_IFREG)) {
		reply(550, "%s: not a plain file.", name);
		goto done;
	}
	if (restart_point) {
		if (type == TYPE_A) {
			register int i, n, c;

			n = restart_point;
			i = 0;
			while (i++ < n) {
				if ((c=getc(fin)) == EOF) {
					perror_reply(550, name);
					goto done;
				}
				if (c == '\n')
					i++;
			}       
		} else if (lseek(fileno(fin), restart_point, L_SET) < 0) {
			perror_reply(550, name);
			goto done;
		}
	}
	dout = dataconn(name, st.st_size, (char *)"w");
	if (dout == NULL)
		goto done;
	send_data(fin, dout, st.st_blksize);
	(void) fclose(dout);

dolog:
	if ((cmd == 0)) {
		int     xfertime = time(NULL) - start_time;

		/*
		 * Mark download and update user counters
		 */
		fill_down(GetArea(name), name);
		Downloads++;
		DownloadK += (bytecount / 1024);
		if (!xfertime) 
			xfertime++;
		Syslog('+', "Sent %s %d bytes in %d seconds (%d cps) %s", name, bytecount, xfertime, 
				bytecount / xfertime, (type == TYPE_A) ? (char *)"Ascii" : (char *)"Binary");
	}

	data = -1;
	pdata = -1;
done:
	if (closefunc) 
		(*closefunc)(fin);
}



void store(char *name, char *mmode, int unique)
{
	FILE		*fout, *din;
	struct stat	st;
	int		(*closefunc)(FILE *);
	time_t		start_time = time(NULL), xfertime;
	register int	n = 0, i, c;
	int		match_value = -1;
	uid_t		uid;
	gid_t		gid;
	int		valid = 0, f_mode = -1;

	Syslog('f', "store(%s, %s, int)", name, mmode, unique);

	if (unique && stat(name, &st) == 0 && (name = gunique(name)) == NULL)
		return;

	/*
	 * Check filename, is it legal?
	 */
	if ((fn_check(name)) <= 0)
		return;

	/*
	 * Overwrite check, not permitted.
	 */
	if (!stat(name, &st)) {
		reply(553, "%s: Permission denied. (Overwrite)", name);
		return;
	}


	if ((match_value = upl_check(name, &uid, &gid, &f_mode, &valid)) < 0)
		return;

	/*
	 * If we have a uid and gid, then use them.
	 */
//	if (valid > 0) {
//		oldid = getuid();
//		(void) seteuid((uid_t)0);
//		if ((fchown(fdout, uid, gid)) < 0) {
//			(void) seteuid(oldid);
//			perror_reply(550, (char *)"fchown");
//			return;
//		}
//		(void) seteuid(oldid);
//	}

	if (restart_point)
		mmode = (char *)"r+w";
	fout = fopen(name, mmode);
	closefunc = fclose;
	if (fout == NULL) {
		perror_reply(553, name);
		return;
	}
	if (restart_point) {
		if (type == TYPE_A) {
			n = restart_point;
			i = 0;
			while (i++ < n) {
				if ((c=getc(fout)) == EOF) {
					perror_reply(550, name);
					goto done;
				}
				if (c == '\n')
					i++;
			}       
			/*
			 * We must do this seek to "current" position
			 * because we are changing from reading to
			 * writing.
			 */
			if (fseek(fout, 0L, L_INCR) < 0) {
				perror_reply(550, name);
				goto done;
			}
		} else if (lseek(fileno(fout), restart_point, L_SET) < 0) {
			perror_reply(550, name);
			goto done;
		}
	}
	din = dataconn(name, (off_t)-1, (char *)"r");
	if (din == NULL)
		goto done;
	if (receive_data(din, fout) == 0) {
		if (unique)
			reply(226, "Transfer complete (unique file name:%s).", name);
		else
			reply(226, "Transfer complete.");
	}
	(void) fclose(din);

	xfertime = time(NULL) - start_time;
	if (!xfertime) 
		xfertime++;

	Syslog('+', "Received %s %d bytes in %d seconds (%d cps) %s", name, bytecount, xfertime,
		bytecount / xfertime, (type == TYPE_A) ? (char *)"Ascii" : (char *)"Binary");
	data = -1;
	pdata = -1;
done:
	(*closefunc)(fout);
}



FILE *getdatasock(char *mmode)
{
	int	s, on = 1, tries;

	if (data >= 0)
		return (fdopen(data, mmode));
	(void)setuid((uid_t)0);
	s = socket(AF_INET, SOCK_STREAM, 0);
	if (s < 0)
		goto bad;
	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *) &on, sizeof (on)) < 0) {
		Syslog('f', "setsockopt failed");
		goto bad;
	}
	/* anchor socket to avoid multi-homing problems */
	data_source.sin_family = AF_INET;
	data_source.sin_addr = ctrl_addr.sin_addr;
	for (tries = 1; ; tries++) {
		if (bind(s, (struct sockaddr *)&data_source, sizeof (data_source)) >= 0)
			break;
		if (errno != EADDRINUSE || tries > 10)
			goto bad;
		sleep(tries);
	}
	(void)seteuid((uid_t)pw->pw_uid);
#ifdef IP_TOS
//	on = IPTOS_THROUGHPUT;
//	if (setsockopt(s, IPPROTO_IP, IP_TOS, (char *)&on, sizeof(int)) < 0)
//		Syslog('+', "setsockopt (IP_TOS): %m");
#endif
	return (fdopen(s, mmode));
bad:
	(void) seteuid((uid_t)pw->pw_uid);
	(void) close(s);
	return (NULL);
}



FILE *dataconn(char *name, off_t size, char *mmode)
{
	char	sizebuf[32];
	FILE	*ffile;
	int	retry = 0;
//	int	tos;

	filesize = size;
	bytecount = 0;
	if (size != (off_t) -1)
		(void) sprintf (sizebuf, " (%ld bytes)", size);
	else
		(void) strcpy(sizebuf, "");
	if (pdata >= 0) {
		struct sockaddr_in from;
		int s, fromlen = sizeof(from);

		s = accept(pdata, (struct sockaddr *)&from, &fromlen);
		if (s < 0) {
			reply(425, "Can't open data connection.");
			(void) close(pdata);
			pdata = -1;
			return(NULL);
		}
		(void) close(pdata);
		pdata = s;
#ifdef IP_TOS
//		tos = IPTOS_LOWDELAY;
//		(void) setsockopt(s, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(int));
#endif
		reply(150, "Opening %s mode data connection for %s%s.", type == TYPE_A ? "ASCII" : "BINARY", name, sizebuf);
		return(fdopen(pdata, mmode));
	}
	if (data >= 0) {
		reply(125, "Using existing data connection for %s%s.", name, sizebuf);
		usedefault = 1;
		return (fdopen(data, mmode));
	}
	if (usedefault)
		data_dest = his_addr;
	usedefault = 1;
	ffile = getdatasock(mmode);
	if (ffile == NULL) {
		reply(425, "Can't create data socket (%s,%d): %s.", inet_ntoa(data_source.sin_addr),
		ntohs(data_source.sin_port), strerror(errno));
		return (NULL);
	}
	data = fileno(ffile);
	while (connect(data, (struct sockaddr *) &data_dest, sizeof(data_dest)) < 0) {
		if ((errno == EADDRINUSE || errno == EINTR) && retry < swaitmax) {
			sleep((unsigned) swaitint);
			retry += swaitint;
			continue;
		}
		perror_reply(425, (char *)"Can't build data connection");
		(void) fclose(ffile);
		data = -1;
		return (NULL);
	}
	reply(150, "Opening %s mode data connection for %s%s.", type == TYPE_A ? "ASCII" : "BINARY", name, sizebuf);
	return (ffile);
}



/*
 * Tranfer the contents of "instr" to
 * "outstr" peer using the appropriate
 * encapsulation of the data subject
 * to Mode, Structure, and Type.
 *
 * NB: Form isn't handled.
 */
void send_data(FILE *instr, FILE *outstr, off_t blksize)
{
	register int	c, cnt, outcnt, newcnt;
	register char	*buf;
	int		netfd, filefd;

	transflag++;
	if (setjmp(urgcatch)) {
		transflag = 0;
		return;
	}
	switch (type) {

	case TYPE_A:
		while ((c = getc(instr)) != EOF) {
			bytecount++;
			if (c == '\n') {
				if (ferror(outstr))
					goto data_err;
				(void) putc('\r', outstr);
				Nopper();
			}
			(void) putc(c, outstr);
		}
		fflush(outstr);
		transflag = 0;
		if (ferror(instr))
			goto file_err;
		if (ferror(outstr))
			goto data_err;
		reply(226, "Transfer complete.");
		return;

	case TYPE_I:
	case TYPE_L:
		if ((buf = malloc((u_int)blksize)) == NULL) {
			transflag = 0;
			perror_reply(451, (char *)"Local resource failure: malloc");
			return;
		}
		netfd = fileno(outstr);
		filefd = fileno(instr);
		while ((cnt = read(filefd, buf, (u_int)blksize)) > 0) {
			outcnt=newcnt=0;
			while ((outcnt=write(netfd, buf+newcnt, cnt-newcnt)) != cnt-newcnt)
				newcnt+=outcnt;
			bytecount += cnt;
			Nopper();
		}
		transflag = 0;
		(void)free(buf);
		if (cnt != 0) {
			if (cnt < 0)
				goto file_err;
			goto data_err;
		}
		reply(226, "Transfer complete.");
		return;
        default:
		transflag = 0;
		reply(550, "Unimplemented TYPE %d in send_data", type);
		return;
	}

data_err:
	transflag = 0;
	perror_reply(426, (char *)"Data connection");
	return;

file_err:
	transflag = 0;
	perror_reply(551, (char *)"Error on input file");
}



/*
 * Transfer data from peer to
 * "outstr" using the appropriate
 * encapulation of the data subject
 * to Mode, Structure, and Type.
 *
 * N.B.: Form isn't handled.
 */
int receive_data(FILE *instr, FILE *outstr)
{
	register int	c;
	int		cnt; 
	static int	bare_lfs = 0;
	char		buf[BUFSIZ];

	transflag++;
	if (setjmp(urgcatch)) {
		transflag = 0;
		return (-1);
	}
	switch (type) {

	case TYPE_I:
	case TYPE_L:
		while ((cnt = read(fileno(instr), buf, sizeof buf)) > 0) {
			if (write(fileno(outstr), buf, cnt) != cnt)
				goto file_err;
			bytecount += cnt;
			Nopper();
		}
		if (cnt < 0)
			goto data_err;
		transflag = 0;
		return (0);

	case TYPE_E:
		reply(553, "TYPE E not implemented.");
		transflag = 0;
		return (-1);

	case TYPE_A:
		while ((c = getc(instr)) != EOF) {
			bytecount++;
			if (c == '\n') {
				bare_lfs++;
				Nopper();
			}
			while (c == '\r') {
				if (ferror(outstr))
					goto data_err;
				if ((c = getc(instr)) != '\n') {
					(void) putc ('\r', outstr);
					if (c == '\0' || c == EOF)
						goto contin2;
				}
			}
			(void) putc(c, outstr);
        contin2:        ;
		}
		fflush(outstr);
		if (ferror(instr))
			goto data_err;
		if (ferror(outstr))
			goto file_err;
		transflag = 0;
		if (bare_lfs) {
			lreply(230, "WARNING! %d bare linefeeds received in ASCII mode", bare_lfs);
			printf("   File may not have transferred correctly.\r\n");
		}
		return (0);
        default:
		reply(550, "Unimplemented TYPE %d in receive_data", type);
		transflag = 0;
		return (-1);
        }

data_err:
	transflag = 0;
	perror_reply(426, (char *)"Data Connection");
	return (-1);

file_err:
	transflag = 0;
	perror_reply(452, (char *)"Error writing file");
	return (-1);
}



void statfilecmd(char *filename)
{
	char	line[BUFSIZ];
	FILE	*fin;
	int	c;

	Syslog('f', "statfilecmd(%s)", filename);

	(void) sprintf(line, "/bin/ls -lgA %s", filename);
	fin = ftpd_popen(line, (char *)"r");
	lreply(211, "status of %s:", filename);
	while ((c = getc(fin)) != EOF) {
		if (c == '\n') {
			if (ferror(stdout)){
				perror_reply(421, (char *)"control connection");
				(void) ftpd_pclose(fin);
				die(1);
				/* NOTREACHED */
			}
			if (ferror(fin)) {
				perror_reply(551, filename);
				(void) ftpd_pclose(fin);
				return;
			}
			(void) putc('\r', stdout);
                }
		(void) putc(c, stdout);
	}
	(void) ftpd_pclose(fin);
	reply(211, "End of Status");
}



void statcmd(void)
{
	struct sockaddr_in	*ssin;
	u_char			*a, *p;

	Syslog('f', "statcmd()");

	lreply(211, "%s FTP server status:", hostname, version);
	printf("     %s\r\n", version);
	printf("     Connected to %s", remotehost);
	if (!isdigit(remotehost[0]))
		printf(" (%s)", inet_ntoa(his_addr.sin_addr));
	printf("\r\n");
	if (logged_in) {
		if (anonymous)
			printf("     Logged in anonymously\r\n");
		else
			printf("     Logged in as %s\r\n", pw->pw_name);
	} else if (askpasswd)
		printf("     Waiting for password\r\n");
	else
		printf("     Waiting for user name\r\n");
	printf("     TYPE: %s", typenames[type]);
	if (type == TYPE_A || type == TYPE_E)
		printf(", FORM: %s", formnames[form]);
	if (type == TYPE_L)
#if NBBY == 8
		printf(" %d", NBBY);
#else
		printf(" %d", bytesize);        /* need definition! */
#endif
	printf("; STRUcture: %s; transfer MODE: %s\r\n",
	strunames[stru], modenames[mode]);
	if (data != -1)
		printf("     Data connection open\r\n");
	else if (pdata != -1) {
		printf("     in Passive mode");
		ssin = &pasv_addr;
goto printaddr;
	} else if (usedefault == 0) {
		printf("     PORT");
		ssin = &data_dest;
printaddr:
		a = (u_char *) &ssin->sin_addr;
		p = (u_char *) &ssin->sin_port;
#define UC(b) (((int) b) & 0xff)
		printf(" (%d,%d,%d,%d,%d,%d)\r\n", UC(a[0]),
			UC(a[1]), UC(a[2]), UC(a[3]), UC(p[0]), UC(p[1]));
#undef UC
	} else
		printf("     No data connection\r\n");
	reply(211, "End of status");
}



void fatal(char *s)
{
	reply(451, "Error in server: %s\n", s);
	reply(221, "Closing connection due to server error.");
	die(0);
}



void ack(char *s)
{
	reply(250, "%s command successful.", s);
}



void nack(char *s)
{
	reply(502, "%s command not implemented.", s);
}



void yyerror(char *s)
{
	char	*cp;

	if ((cp = strchr(cbuf, '\n')) != NULL)
		*cp = '\0';
	reply(500, "'%s' : command not understood.", cbuf);
}



void delete(char *name)
{
	struct stat	st;
	char		path[MAXPATHLEN];

	Syslog('f', "delete(%s)", name);

	if ((del_check(name)) == 0)
		return;

	if (stat(name, &st) < 0) {
		perror_reply(550, name);
		return;
	}
	if ((st.st_mode & S_IFMT) == S_IFDIR) {
		if (rmdir(name) < 0) {
			perror_reply(550, name);
			return;
		}
		goto done;
	}
	if (unlink(name) < 0) {
		perror_reply(550, name);
		return;
	}
done:
	realpath(name, path);
	Syslog('+', "%s of %s [%s] deleted %s", pw->pw_name, remotehost, remoteaddr, path);
	ack((char *)"DELE");
}



/*
 * Change directory, check access rights. We also forbid access
 * to password protected directories. The FTP protocol does not
 * support this so these are errors for the users. Pity that
 * browser users never see error messages.
 */
void cwd(char *path)
{
	char		*temp, *new, *test;
	fil_areas	*tmp;
	int		ok = 0;

	temp = calloc(PATH_MAX, sizeof(char));
	new  = calloc(PATH_MAX, sizeof(char));
	test = calloc(PATH_MAX, sizeof(char));

	(void)getcwd(temp, PATH_MAX);

	if (chdir(path) < 0)
		perror_reply(550, path);
	else {
		if (!administrator) {
			/*
			 * If not an administrator test if we may access this
			 * directory. If not, go back to were we came from.
			 */
			(void)getcwd(new, PATH_MAX);
			for (tmp = are; tmp; tmp = tmp->next) {
				sprintf(test, "%s%s", home, new);
				if (strcmp(tmp->Path, test) == 0) {
					if (!Access(tmp->DNsec))
						ok = 1;
					if (!ok && strlen(tmp->Password))
						ok = 2;
					break;
				}
			}
			if (ok)
				chdir(temp);
		}
		switch (ok) {
			case 0:
				show_message(250, E_CWD);
				show_readme(250, E_CWD);
				ack((char *)"CWD");
				break;
			case 1:
				reply(550, "No directory access, not authorized");
				break;
			case 2:
				reply(550, "Password protected directory, use the BBS");
				break;
		}
	}

	(void)getcwd(new, PATH_MAX);
	Syslog('f', "cwd(%s) -> %s", path, new);

	free(temp);
	free(new);
	free(test);
}



void makedir(char *name)
{
	uid_t	uid;
	gid_t	gid;
	int	valid = 0;

	Syslog('f', "makedir(%s)", name);

	/*
	 * Check the directory, can we mkdir here>
	 */
	if ((dir_check(name, &uid, &gid, &valid)) <= 0)
		return;

	/*
	 * Check the filename, is it legal?
	 */
	if ((fn_check(name)) <= 0)
		return;
	
	if (mkdir(name, 0777) < 0)
		perror_reply(550, name);
	else
		reply(257, (char *)"MKD command successful.");
}



void removedir(char *name)
{
	uid_t	uid;
	gid_t	gid;
	int	valid = 0;

	/*
	 * Check the directory, can we rmdir here?
	 */
	if ((dir_check(name, &uid, &gid, &valid)) <= 0)
		return;

	/*
	 * Delete permission?
	 */
	if ((del_check(name)) == 0)
		return;

	Syslog('f', "removedir(%s)", name);
	if (rmdir(name) < 0)
		perror_reply(550, name);
	else
		ack((char *)"RMD");
}



void pwd(void)
{
	char path[MAXPATHLEN + 1];

	Syslog('f', "pwd()");

	if (getcwd(path, PATH_MAX) == (char *)NULL)
		reply(550, "%s.", path);
	else
		reply(257, "\"%s\" is current directory.", path);
}



char *renamefrom(char *name)
{
	struct stat	st;

	Syslog('f', "renamefrom(%s)", name);

	if (stat(name, &st) < 0) {
		perror_reply(550, name);
		return ((char *)0);
	}
	reply(350, "File exists, ready for destination name");
	return (name);
}



void renamecmd(char *from, char *to)
{
	Syslog('f', "renamecmd(%s, %s)", from, to);

	/*
	 * Check the filename, is it legal?
	 */
	if ((fn_check(to)) == 0)
		return;

	if (rename(from, to) < 0)
		perror_reply(550, (char *)"rename");
	else
		ack((char *)"RNTO");
}



/*
 * Note: a response of 425 is not mentioned as a possible response to
 *      the PASV command in RFC959. However, it has been blessed as
 *      a legitimate response by Jon Postel in a telephone conversation
 *      with Rick Adams on 25 Jan 89.
 */
void passive(void)
{
	int		len;
	register char	*p, *a;

	Syslog('f', "passive()");
	pdata = socket(AF_INET, SOCK_STREAM, 0);
	if (pdata < 0) {
		perror_reply(425, (char *)"Can't open passive connection");
		return;
	}
	pasv_addr = ctrl_addr;
	pasv_addr.sin_port = 0;
	(void) seteuid((uid_t)0);
	if (bind(pdata, (struct sockaddr *)&pasv_addr, sizeof(pasv_addr)) < 0) {
		(void) seteuid((uid_t)pw->pw_uid);
		goto pasv_error;
	}
	(void) seteuid((uid_t)pw->pw_uid);
	len = sizeof(pasv_addr);
	if (getsockname(pdata, (struct sockaddr *) &pasv_addr, &len) < 0)
		goto pasv_error;
	if (listen(pdata, 1) < 0)
		goto pasv_error;
	a = (char *) &pasv_addr.sin_addr;
	p = (char *) &pasv_addr.sin_port;

#define UC(b) (((int) b) & 0xff)

	reply(227, "Entering Passive Mode (%d,%d,%d,%d,%d,%d)", UC(a[0]),
		UC(a[1]), UC(a[2]), UC(a[3]), UC(p[0]), UC(p[1]));
	return;

pasv_error:
	(void) close(pdata);
	pdata = -1;
	perror_reply(425, (char *)"Can't open passive connection");
	return;
}



/*
 * Generate unique name for file with basename "local".
 * The file named "local" is already known to exist.
 * Generates failure reply on error.
 */
char *gunique(char *local)
{
	static char	new[MAXPATHLEN];
	struct stat	st;
	char		*cp = strrchr(local, '/');
	int		count = 0;

	if (cp)
		*cp = '\0';
	if (stat(cp ? local : ".", &st) < 0) {
		perror_reply(553, cp ? local : (char *)".");
		return((char *) 0);
	}
	if (cp)
		*cp = '/';
	(void) strcpy(new, local);
	cp = new + strlen(new);
	*cp++ = '.';
	for (count = 1; count < 100; count++) {
		(void) sprintf(cp, "%d", count);
		if (stat(new, &st) < 0)
			return(new);
	}
	reply(452, "Unique file name cannot be created.");
	return((char *) 0);
}



void perror_reply(int code, char *string)
{
	reply(code, "%s: %s.", string, strerror(errno));
}


static char *onefile[] = { (char *)"", 0 };

void send_file_list(char *whichfiles)
{
	struct stat	st;
	DIR		*dirp = NULL;
	struct dirent	*dir;
	static FILE	*dout = NULL;
	static char	**dirlist; 
	char		*dirname;
	int		simple = 0;

	Syslog('f', "send_file_list(%s)", whichfiles);

	if (strpbrk(whichfiles, "~{[*?") != NULL) {
		extern char *globerr;

		globerr = NULL;
		dirlist = glob(whichfiles);
		if (globerr != NULL) {
			reply(550, globerr);
			return;
		} else if (dirlist == NULL) {
			errno = ENOENT;
			perror_reply(550, whichfiles);
			return;
		}
	} else {
		onefile[0] = whichfiles;
		dirlist = onefile;
		simple = 1;
	}

	if (setjmp(urgcatch)) {
		transflag = 0;
		return;
	}

	while ((dirname = *dirlist++) != NULL) {
		if (stat(dirname, &st) < 0) {
			/*
			 * If user typed "ls -l", etc, and the client
			 * used NLST, do what the user meant.
			 */
			if (dirname[0] == '-' && *dirlist == NULL && transflag == 0) {
				retrieve((char *)"/bin/ls %s", dirname);
				return;
			}
			perror_reply(550, whichfiles);
			if (dout != NULL) {
				(void) fclose(dout);
				transflag = 0;
				data = -1;
				pdata = -1;
			}
			return;
                }
		if ((st.st_mode&S_IFMT) == S_IFREG) {
			if (dout == NULL) {
				dout = dataconn((char *)"file list", (off_t)-1, (char *)"w");
				if (dout == NULL)
					return;
				transflag++;
			}
			fprintf(dout, "%s%s\n", dirname, type == TYPE_A ? "\r" : "");
			bytecount += strlen(dirname) + 1;
			continue;
		} else if ((st.st_mode&S_IFMT) != S_IFDIR)
			continue;

		if ((dirp = opendir(dirname)) == NULL)
			continue;
		while ((dir = readdir(dirp)) != NULL) {
			char nbuf[MAXPATHLEN];

			if ((dir->d_name[0] == '.') && (dir->d_reclen == 1))
				retrieve((char *)"/bin/ls %s", dirname);
			return;
			if ((dir->d_reclen == 2) && (dir->d_name[0] == '.') && (dir->d_name[1] == '.')) 
				retrieve((char *)"/bin/ls %s", dirname);
			continue;

			sprintf(nbuf, "%s/%s", dirname, dir->d_name);

			/*
			 * We have to do a stat to insure it's
			 * not a directory or special file.
			 */
			if (simple || (stat(nbuf, &st) == 0 && (st.st_mode&S_IFMT) == S_IFREG)) {
				if (dout == NULL) {
					dout = dataconn((char *)"file list", (off_t)-1, (char *)"w");
					if (dout == NULL)
						return;
					transflag++;
				}
				if (nbuf[0] == '.' && nbuf[1] == '/')
					fprintf(dout, "%s%s\n", &nbuf[2], type == TYPE_A ? "\r" : "");
				else
					fprintf(dout, "%s%s\n", nbuf, type == TYPE_A ? "\r" : "");
				bytecount += strlen(nbuf) + 1;
			}
		}
                (void) closedir(dirp);
	}

	if (dout == NULL)
		reply(550, "No files found.");
	else if (ferror(dout) != 0)
		perror_reply(550, (char *)"Data connection");
	else
		reply(226, "Transfer complete.");

	transflag = 0;
	if (dout != NULL)
		(void) fclose(dout);
	data = -1;
	pdata = -1;
}



