/*****************************************************************************
 *
 * File ..................: mbfido/scan.h
 * Purpose ...............: Scan for outgoing mail.
 * Last modification date : 15-Sep-2000
 *
 *****************************************************************************
 * Copyright (C) 1997-2000
 *   
 * Michiel Broek		FIDO:		2:280/2802
 * Beekmansbos 10
 * 1971 BV IJmuiden
 * the Netherlands
 *
 * This file is part of MBSE BBS.
 *
 * This BBS is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * MBSE BBS is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with MBSE BBS; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/

#include "../lib/libs.h"
#include "../lib/structs.h"
#include "../lib/records.h"
#include "../lib/common.h"
#include "../lib/msg.h"
#include "../lib/clcomm.h"
#include "../lib/msgtext.h"
#include "../lib/dbnode.h"
#include "../lib/mbinet.h"
#include "addpkt.h"
#include "pack.h"
#include "tracker.h"
#include "scan.h"


extern	int	do_quiet;
extern	int	net_out;
extern	int	net_bad;
extern	int	email_out;
extern	int	echo_out;
int		scanned;

#define	MAXSEEN 70


/*
 * Internal prototypes
 */
void ScanFull(void);
void ScanOne(char *, unsigned long);
void ExportEcho(sysconnect, unsigned long, fa_list **);
void ExportNet(unsigned long, int);
void ExportEmail(unsigned long, char *);



/*
 *  Scan for outgoing mail. If using the files $MBSE_ROOT/tmp/echomail.jam
 *  or netmail.jam not all mail is scanned a full mailscan will be 
 *  performed.
 */
void ScanMail(int DoAll)
{
	int		DoFull = FALSE, i = 0;
	unsigned long	msg;
	char		*Fname = NULL, *temp, *path;
	FILE		*fp;

	if (DoAll) {
		DoFull = TRUE;
	} else {
		scanned = 0;
		Fname = calloc(128, sizeof(char));
		temp  = calloc(128, sizeof(char));

		sprintf(Fname, "%s/tmp/echomail.jam", getenv("MBSE_ROOT"));
		if ((fp = fopen(Fname, "r")) != NULL) {
			while ((fgets(temp, 128, fp)) != NULL) {
				path = strtok(temp, " ");
				msg = atol(strtok(NULL, "\n"));
				Syslog('+', "Export message %lu from %s", msg, path);
				ScanOne(path, msg);
				i++;
			}
			fclose(fp);
			unlink(Fname); 
		}

		sprintf(Fname, "%s/tmp/netmail.jam", getenv("MBSE_ROOT"));
		if ((fp = fopen(Fname, "r")) != NULL) {
			while ((fgets(temp, 128, fp)) != NULL) {
				path = strtok(temp, " ");
				msg = atol(strtok(NULL, "\n"));
				Syslog('+', "Export message %lu from %s", msg, path);
				ScanOne(path, msg);
				i++;
			}
			fclose(fp);
			unlink(Fname);
		}

		if ((i != scanned) || (i == 0)) {
			Syslog('+', "Not all messages exported, forcing full mail scan");
			Syslog('+', "i=%d scanned=%d", i, scanned);
			DoFull = TRUE;
		}
		free(Fname);
		free(temp);
	}

	if (DoFull)
		ScanFull();

	if (echo_out || net_out)
		packmail();
	RemoveSema((char *)"mailout");
}



void ScanFull()
{
	char		sAreas[81];
	FILE		*pAreas;
	long		arearec = 0, sysstart, nextstart;
	unsigned long	Total, Number;
	int		i;
	sysconnect	Link;
	fa_list		*sbl = NULL;

	Syslog('+', "Full mailscan");
	IsDoing("Scanning mail");

	if (!do_quiet) {
		colour(9, 0);
		printf("Scanning mail\n");
		colour(3, 0);
		fflush(stdout);
	}

	sprintf(sAreas, "%s/etc/users.data", getenv("MBSE_ROOT"));
	if ((pAreas = fopen(sAreas, "r")) != NULL) {
		fread(&usrconfighdr, sizeof(usrconfighdr), 1, pAreas);

		while (fread(&usrconfig, usrconfighdr.recsize, 1, pAreas) == 1) {
			if (usrconfig.Email && strlen(usrconfig.Name)) {
				Syslog('m', "Scanning email from %s", usrconfig.Name);
				if (!do_quiet) {
					colour(3, 0);
					printf("\r%8s %-40s", usrconfig.Name, usrconfig.sUserName);
					colour(13, 0);
					fflush(stdout);
				}

				sprintf(sAreas, "%s/%s/mailbox", CFG.bbs_usersdir, usrconfig.Name);
				if (Msg_Open(sAreas)) {
					if ((Total = Msg_Number()) != 0L) {
						Number = Msg_Lowest();

						do {
							if (CFG.slow_util && do_quiet)
								usleep(1);

							if (((Number % 10) == 0) && (!do_quiet)) {
								printf("%6lu\b\b\b\b\b\b", Number);
								fflush(stdout);
							}

							Msg_ReadHeader(Number);
							if (Msg.Local) {
								if (Msg_Lock(15L)) {
									Syslog('m', "Export %lu email from %s", Number, usrconfig.Name);
									ExportEmail(Number, usrconfig.Name);
									Msg.Local = FALSE;
									Msg.Arrived = time(NULL);
									Msg_WriteHeader(Number);
									Msg_UnLock();
								}
							}

						} while (Msg_Next(&Number) == TRUE);
					}
					Msg_Close();
					if (!do_quiet) {
						printf("      \b\b\b\b\b\b");
						fflush(stdout);
					}
				}
			}
		}
		fclose(pAreas);
	}

	sprintf(sAreas, "%s/etc/mareas.data", getenv("MBSE_ROOT"));
	if ((pAreas = fopen(sAreas, "r")) == NULL) {
		WriteError("Can't open %s", sAreas);
		return;
	}
	fread(&msgshdr, sizeof(msgshdr), 1, pAreas);

	while (fread(&msgs, msgshdr.recsize, 1, pAreas) == 1) {
		sysstart = ftell(pAreas);
		fseek(pAreas, msgshdr.syssize, SEEK_CUR);
		nextstart = ftell(pAreas);
		arearec++;

		if ((msgs.Active) && (msgs.Type == ECHOMAIL || msgs.Type == NETMAIL)) {
			if (!do_quiet) {
				colour(3, 0);
				printf("\r%5ld .. %-40s", arearec, msgs.Name);
				colour(13, 0);
				fflush(stdout);
			}

			if (Msg_Open(msgs.Base)) {
				if ((Total = Msg_Number()) != 0L) {
					Number = Msg_Lowest();

					do {
						if (CFG.slow_util && do_quiet)
							usleep(1);

						if (((Number % 10) == 0) && (!do_quiet)) {
							printf("%6lu\b\b\b\b\b\b", Number);
							fflush(stdout);
						}

						Msg_ReadHeader(Number);
						if (Msg.Local) {
							if (Msg_Lock(15L)) {
								Syslog('m', "Export %lu from area %ld", Number, arearec);

								/*
								 * Setup SEEN-BY lines
								 */
								if (msgs.Type == ECHOMAIL) {
									fill_list(&sbl, aka2str(msgs.Aka), NULL, FALSE);
									fseek(pAreas, sysstart, SEEK_SET);
									for (i = 0; i < (msgshdr.syssize / sizeof(sysconnect)); i++) {
										fread(&Link, sizeof(sysconnect), 1, pAreas);
										if ((Link.aka.zone) && (Link.sendto) && (!Link.pause)) {
											fill_list(&sbl, aka2str(Link.aka), NULL, FALSE);
										}
									}
									sort_list(&sbl);

									fseek(pAreas, sysstart, SEEK_SET);
									for (i = 0; i < (msgshdr.syssize / sizeof(sysconnect)); i++) {
										fread(&Link, sizeof(sysconnect), 1, pAreas);
										if (Link.aka.zone)
											ExportEcho(Link, Number, &sbl);
									}

									tidy_falist(&sbl);
								}
								if (msgs.Type == NETMAIL)
									ExportNet(Number, FALSE);

								Msg.Local = FALSE;
								Msg.Arrived = time(NULL);
								Msg_WriteHeader(Number);
								Msg_UnLock();
							}
						}

					} while (Msg_Next(&Number) == TRUE);
				}

				Msg_Close();

				if (!do_quiet) {
					printf("      \b\b\b\b\b\b");
					fflush(stdout);
				}
			}

			/*
			 * Make sure to start at the next area.
			 */
			fseek(pAreas, nextstart, SEEK_SET);
		}
	}

	fclose(pAreas);

	if (!do_quiet) {
		printf("\r                                                        \r");
		fflush(stdout);
	}
}



void ScanOne(char *path, unsigned long MsgNum)
{
	char		*p, *q, sAreas[81];
	FILE		*pAreas;
	long		sysstart;
	unsigned long	Total, Area = 0;
	int		i;
	sysconnect	Link;
	fa_list		*sbl = NULL;

	IsDoing("Scanning mail");

	if (!do_quiet) {
		colour(9, 0);
		printf("Scanning mail\n");
		colour(3, 0);
		fflush(stdout);
	}

#ifdef MAILDEBUG
	Syslog('m', "ScanOne(%s, %u)", path, MsgNum);
#endif

	if (strncmp(CFG.bbs_usersdir, path, strlen(CFG.bbs_usersdir)) == 0) {
#ifdef MAILDEBUG
		Syslog('m', "It's a users email area");
#endif
		if (Msg_Open(path)) {
			if (((Total = Msg_Number()) != 0L) && (Msg_ReadHeader(MsgNum)) && Msg.Local) {
				if (Msg_Lock(15L)) {
#ifdef MAILDEBUG
					Syslog('m', "ScanOne() starting export");
#endif
					scanned++;
					p = xstrcpy(path+strlen(CFG.bbs_usersdir)+1);
					q = strtok(p, "/");
#ifdef MAILDEBUG
					Syslog('m', "Extracted username \"%s\"", q);
#endif
					ExportEmail(MsgNum, q);
					free(p);
					Msg.Local = FALSE;
					Msg.Arrived = time(NULL);
					Msg_WriteHeader(MsgNum);
					Msg_UnLock();
				}

			}
			Msg_Close();
		} else {
			WriteError("Can't open %s", path);
		}
		return;
	}

	sprintf(sAreas, "%s/etc/mareas.data", getenv("MBSE_ROOT"));
	if ((pAreas = fopen(sAreas, "r")) == NULL) {
		WriteError("Can't open %s", sAreas);
		return;
	}
	fread(&msgshdr, sizeof(msgshdr), 1, pAreas);

	/*
	 * Seek the path we want
	 */
	while (TRUE) {
		if (fread(&msgs, msgshdr.recsize, 1, pAreas) != 1) {
			fclose(pAreas);
			Syslog('m', "ScanOne() reached end of areas");
			return;
		}
		Area++;
		sysstart = ftell(pAreas);
		fseek(pAreas, msgshdr.syssize, SEEK_CUR);
		if (strcmp(msgs.Base, path) == 0)
			break;
	}

#ifdef MAILDEBUG
	Syslog('m', "ScanOne() found area %d", Area);
#endif
	if ((msgs.Active) && (msgs.Type == ECHOMAIL || msgs.Type == NETMAIL)) {
#ifdef MAILDEBUG
		Syslog('m', "ScanOne() preparing for export");
#endif
		if (!do_quiet) {
			colour(3, 0);
			printf("\r%5ld .. %-40s", Area, msgs.Name);
			colour(13, 0);
			fflush(stdout);
		}

		if (Msg_Open(msgs.Base)) {
			if ((Total = Msg_Number()) != 0L) {
				if (Msg_ReadHeader(MsgNum)) {
					if (Msg.Local) {
						if (Msg_Lock(15L)) {
#ifdef MAILDEBUG
							Syslog('m', "ScanOne() starting export");
#endif
							scanned++;
							/*
							 * Setup SEEN-BY lines
							 */
							if (msgs.Type == ECHOMAIL) {
								fill_list(&sbl, aka2str(msgs.Aka), NULL, FALSE);
								fseek(pAreas, sysstart, SEEK_SET);
								for (i = 0; i < (msgshdr.syssize / sizeof(sysconnect)); i++) {
									fread(&Link, sizeof(sysconnect), 1, pAreas);
									if ((Link.aka.zone) && (Link.sendto) && (!Link.pause)) {
										fill_list(&sbl, aka2str(Link.aka), NULL, FALSE);
									}
								}
								sort_list(&sbl);

								fseek(pAreas, sysstart, SEEK_SET);
								for (i = 0; i < (msgshdr.syssize / sizeof(sysconnect)); i++) {
									fread(&Link, sizeof(sysconnect), 1, pAreas);
									if (Link.aka.zone) {
										ExportEcho(Link, MsgNum, &sbl);
									}
								}

								tidy_falist(&sbl);
							}
							if (msgs.Type == NETMAIL)
								ExportNet(MsgNum, FALSE);

							Msg.Local = FALSE;
							Msg.Arrived = time(NULL);
							Msg_WriteHeader(MsgNum);
							Msg_UnLock();
						}
					}

				}
			}

			Msg_Close();
		}
	} else {
		WriteError("Config error: area %d not active or not Echo/Netmail area", Area);
	}

	fclose(pAreas);

	if (!do_quiet) {
		printf("\r                                                        \r");
		fflush(stdout);
	}
}



/*
 *  Export message to downlink. The messagebase is locked.
 */
void ExportEcho(sysconnect L, unsigned long MsgNum, fa_list **sbl)
{
	char	*p;
	int	seenlen, oldnet, flags = 0;
	char	sbe[16];
	fa_list	*tmpl;
	FILE	*qp;
	faddr	*from, *dest;
	int	is_pid = FALSE;

	if ((!L.sendto) || L.pause || L.cutoff)
		return;

#ifdef MAILDEBUG
	Syslog('m', "Export to %s", aka2str(L.aka));
#endif

	if ((qp = OpenPkt(msgs.Aka, L.aka, (char *)"qqq")) == NULL)
		return;

	flags |= (Msg.Private)		? M_PVT : 0;
	from = fido2faddr(msgs.Aka);
	dest = fido2faddr(L.aka);
	AddMsgHdr(qp, from, dest, flags, 0, Msg.Written, Msg.To, Msg.From, Msg.Subject);
	tidy_faddr(from);
	tidy_faddr(dest);
	fprintf(qp, "AREA:%s\r", msgs.Tag);
#ifdef MAILDEBUG
	Syslog('m', "AREA:%s", msgs.Tag);
#endif

	if (Msg_Read(MsgNum, 78)) {
		if ((p = (char *)MsgText_First()) != NULL) {
			do {
				if ((strncmp(p, " * Origin:", 10) == 0) && !is_pid) {
					/*
					 * If there was no PID kludge, insert the TID
					 * kludge anyway.
					 */
					fprintf(qp, "\001TID: MBSE BBS %s\r", PIDVersion);
#ifdef MAILDEBUG
					Syslog('m', "\\001TID: MBSE BBS %s", PIDVersion);
#endif
				}
				fprintf(qp, "%s", p);
#ifdef MAILDEBUG
				Syslog('m', "%s", printable(p, 0));
#endif
				if (strncmp(p, " * Origin:", 10) == 0)
					break;

				/*
				 * Only append CR if not the last line
				 */
				fprintf(qp, "\r");

				/*
				 * Append ^aTID line
				 */
				if (strncmp(p, "\001PID", 4) == 0) {
					fprintf(qp, "\001TID: MBSE BBS %s\r", PIDVersion);
#ifdef MAILDEBUG
					Syslog('m', "\\001TID: MBSE BBS %s", PIDVersion);
#endif
					is_pid = TRUE;
				}

			} while ((p = (char *)MsgText_Next()) != NULL);
		}
	}

	seenlen = MAXSEEN + 1;
	/*
	 * Ensure that it will not match the first entry.
	 */
	oldnet = (*sbl)->addr->net - 1;
	for (tmpl = *sbl; tmpl; tmpl = tmpl->next) {
		if (tmpl->addr->net == oldnet)
			sprintf(sbe, " %u", tmpl->addr->node);
		else
			sprintf(sbe, " %u/%u", tmpl->addr->net, tmpl->addr->node);
		oldnet = tmpl->addr->net;
		seenlen += strlen(sbe);
		if (seenlen > MAXSEEN) {
			seenlen = 0;
			fprintf(qp, "\rSEEN-BY:");
			sprintf(sbe, " %u/%u", tmpl->addr->net, tmpl->addr->node);
			seenlen = strlen(sbe);
		}
		fprintf(qp, "%s", sbe);
	}
	fprintf(qp, "\r\001PATH: %u/%u\r", msgs.Aka.net, msgs.Aka.node);
#ifdef MAILDEBUG
	Syslog('m', "\\001PATH: %u/%u\r", msgs.Aka.net, msgs.Aka.node);
#endif
	putc(0, qp);
	fclose(qp);

	echo_out++;
}



/*
 *  Export Netmail message, the messagebase is locked.
 */
void ExportNet(unsigned long MsgNum, int UUCPgate)
{
	char		*p, ext[4];
	int		rc, flags = 0, first;
	FILE		*qp;
	fidoaddr	Dest, Route, *dest;
	time_t		now;
	struct tm	*tm;
	char		flavor;
	faddr		*from, *too, *ta;
	int		is_fmpt = FALSE, is_topt = FALSE, is_intl = FALSE;

	Syslog('m', "Export netmail to %s (%s) %s mode", Msg.ToAddress, 
		(Msg.Crash || Msg.Direct || Msg.FileAttach) ? "Direct" : "Routed", UUCPgate ? "UUCP" : "Netmail");

	if (UUCPgate) {
		memcpy(&Dest, &CFG.UUCPgate, sizeof(fidoaddr));
		memset(&msgs, 0, sizeof(msgs));
		memcpy(&msgs.Aka, &CFG.EmailFidoAka, sizeof(fidoaddr));
	} else {
		ta = parsefnode(Msg.ToAddress);
		dest = faddr2fido(ta);
		tidy_faddr(ta);
		memcpy(&Dest, dest, sizeof(fidoaddr));
		free(dest);
	}
	Dest.domain[0] = '\0';

	if (!(Msg.Crash || Msg.Immediate || Msg.Direct || Msg.FileAttach)) {
		if (!TrackMail(Dest, &Route)) {
			Syslog('!', "No route to %s, message orphaned", Msg.ToAddress);
			Msg.Orphan = TRUE;
			net_bad++;
			return;
		}
	}

	Msg.Sent = TRUE;
	if (Msg.KillSent)
		Msg.Deleted = TRUE;

	if (Msg.Crash || Msg.Direct || Msg.FileAttach || Msg.Immediate) {
		memset(&ext, 0, sizeof(ext));
		if (Msg.Immediate)
			sprintf(ext, (char *)"iii");
		else if (Msg.Crash)
			sprintf(ext, (char *)"ccc");
		else
			sprintf(ext, (char *)"nnn");
		if ((qp = OpenPkt(msgs.Aka, Dest, (char *)ext)) == NULL) {
			net_bad++;
			return;
		}

	} else {
		Syslog('m', "Route via %s", aka2str(Route));
		if (!SearchNode(Route, TRUE)) {
			WriteError("Routing node %s not in setup, aborting", aka2str(Route));
			return;
		}

		/*
		 *  Note that even if the exported netmail is not crash, that if
		 *  the routing node has crash status, this netmail will be send
		 *  crash.
		 */
		memset(&ext, 0, sizeof(ext));
		if (nodes.PackNetmail)
			sprintf(ext, (char *)"qqq");
		else if (nodes.Crash)
			sprintf(ext, (char *)"ccc");
		else if (nodes.Hold)
			sprintf(ext, (char *)"hhh");
		else
			sprintf(ext, (char *)"nnn");
		if ((qp = OpenPkt(msgs.Aka, Route, (char *)ext)) == NULL) {
			net_bad++;
			return;
		}
	}

	flags |= (Msg.Private)		? M_PVT   : 0;
	flags |= (Msg.Crash)		? M_CRASH : 0;
	flags |= (Msg.Hold)		? M_HOLD  : 0;
	flags |= (Msg.Immediate)	? M_CRASH : 0;
	flags |= (Msg.FileRequest)	? M_REQ   : 0;
	flags |= (Msg.FileAttach)	? M_FILE  : 0;
	flags |= (Msg.ReceiptRequest)	? M_RRQ   : 0;
	flags |= (Msg.ConfirmRequest)	? M_AUDIT : 0;

	too  = fido2faddr(Dest);
	from = fido2faddr(msgs.Aka);
	if (UUCPgate) {
		Syslog('m', "AddMsgHdr(%s, %s, %s)", (char *)"UUCP", Msg.From, Msg.Subject);
		rc = AddMsgHdr(qp, from, too, flags, 0, Msg.Written, (char *)"UUCP", Msg.From, Msg.Subject);
	} else {
		rc = AddMsgHdr(qp, from, too, flags, 0, Msg.Written, Msg.To, Msg.From, Msg.Subject);
	}
	tidy_faddr(from);
	tidy_faddr(too);

	if (rc) {
		WriteError("Create message failed");
		return;
	}

	/*
	 *  Analyze this message if it contains INTL, FMPT and TOPT kludges 
	 *  and check if we need them. If they are missing they are inserted.
	 *  GoldED doesn't insert them but MBSE does.
	 */
	if (Msg_Read(MsgNum, 78)) {
		if ((p = (char *)MsgText_First()) != NULL) {
			do {
				if (strncmp(p, "\001FMPT", 5) == 0)
					is_fmpt = TRUE;
				if (strncmp(p, "\001TOPT", 5) == 0)
					is_topt = TRUE;
				if (strncmp(p, "\001INTL", 5) == 0)
					is_intl = TRUE;
				if (strncmp(p, "--- ", 4) == 0)
					break;
			} while ((p = (char *)MsgText_Next()) != NULL);
		}
	}
	if (msgs.Aka.point && !is_fmpt)
		fprintf(qp, "\001FMPT %d\r", msgs.Aka.point);
	if (Dest.point && !is_topt)
		fprintf(qp, "\001TOPT %d\r", Dest.point);
	if (!is_intl)
		fprintf(qp, "\001INTL %d:%d/%d %d:%d/%d\r", Dest.zone, Dest.net, Dest.node, 
			msgs.Aka.zone, msgs.Aka.net, msgs.Aka.node);

	if (Msg_Read(MsgNum, 78)) {
		first = TRUE;
		if ((p = (char *)MsgText_First()) != NULL) {
			do {
				if (UUCPgate && first && (p[0] != '\001')) {
					/*
					 * Past the kludges at the message start.
					 * Add the To: name@dom.com and a blank line.
					 */
					fprintf(qp, "To: %s\r", Msg.To);
					fprintf(qp, "\r");
					first = FALSE;
				}
				fprintf(qp, "%s\r", p);
				if (strncmp(p, "--- ", 4) == 0)
					break;
			} while ((p = (char *)MsgText_Next()) != NULL);
		}
	}

	now = time(NULL);
	tm = gmtime(&now);
	fprintf(qp, "\001Via %s @%d%02d%02d.%02d%02d%02d.01.UTC MBSE BBS %s\r",
		aka2str(msgs.Aka), tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
		tm->tm_hour, tm->tm_min, tm->tm_sec, MBSEVersion);

	putc(0, qp);
	fclose(qp);

	if (Msg.FileAttach) {
		if (Msg.Crash)
			flavor = 'c';
		else
			flavor = 'f';

		ta = parsefnode(Msg.ToAddress);
		if (strlen(CFG.dospath)) {
			rc = attach(*ta, Dos2Unix(Msg.Subject), LEAVE, flavor);
			Syslog('+', "FileAttach %s %s", Dos2Unix(Msg.Subject), rc ? "Success":"Failed");
		} else {
			rc = attach(*ta, Msg.Subject, LEAVE, flavor);
			Syslog('+', "FileAttach %s %s", Msg.Subject, rc ? "Success":"Failed");
		}
		tidy_faddr(ta);
	}

	net_out++;
}



/*
 *  Export Email message, the messagebase is locked.
 */
void ExportEmail(unsigned long MsgNum, char *User)
{
        char	*p, *temp;
	int	header = TRUE;

	temp = calloc(2048, sizeof(char));

	Syslog('m', "Export email to %s", Msg.To);
	Msg.Sent = TRUE;
	if (Msg.KillSent)
		Msg.Deleted = TRUE;


	if (CFG.EmailMode == E_NOISP) {
		Syslog('m', "No ISP mode, sending to UUCP gate as netmail");
		ExportNet(MsgNum, TRUE);
		return;
	}

	/*
	 * Send to SMTP port 25 (Sendmail)
	 */
	if (smtp_connect() != -1) {
		sprintf(temp, "MAIL FROM: <%s@localhost>\r\n", User);
		if (smtp_cmd(temp, 250) == 0) {
			/*
			 * Setup default To: address, then try to find the headerline
			 * in the messagetext. If it's there we use that.
			 */
			sprintf(temp, "RCPT TO: <%s>\r\n", Msg.To);
			if (Msg_Read(MsgNum, 78)) {
				if ((p = (char *)MsgText_First()) != NULL) {
					do {
						if (strncmp(p, "\001To: ", 5) == 0) {
							sprintf(temp, "RCPT TO: <%s>\r\n", p+5);
							break;
						}
					} while ((p = (char *)MsgText_Next()) != NULL);
				}
			}
			if (smtp_cmd(temp, 250) == 0) {
				if (smtp_cmd((char *)"DATA\r\n", 354) == 0) {
					if (Msg_Read(MsgNum, 78)) {
						if ((p = (char *)MsgText_First()) != NULL) {
							do {
								if ((p[0] == '\001') && header) {
#ifdef MAILDEBUG
									Syslog('m', "%s", printable(p, 0));
#endif
									/*
									 * Depending on the local editor, most
									 * of these will never showup.
									 */
									if (strncmp(p, "\001MSGID:", 7) &&
									    strncmp(p, "\001PID:", 5) &&
									    strncmp(p, "\001INTL", 5) &&
									    strncmp(p, "\001FMPT", 5) &&
									    strncmp(p, "\001TOPT", 5) &&
									    strncmp(p, "\001FLAGS", 6) &&
									    strncmp(p, "\001CODEPAGE", 9) &&
									    strncmp(p, "\001ORIGCHRS", 9) &&
									    strncmp(p, "\001CHARSET", 8) &&
									    strncmp(p, "\001CHRS:", 6))
										sprintf(temp, "%s\r\n", p+1);
									else
										sprintf(temp, "X-FTN-%s\r\n", p+1);
									smtp_send(temp);
								} else {
									if (header) {
										/*
										 * End of headers, insert bank line
										 */
										header = FALSE;
										smtp_send((char *)"\r\n");
									}
									sprintf(temp, "%s\r\n", p);
									smtp_send(temp);
								}
							} while ((p = (char *)MsgText_Next()) != NULL);
						}
					}
					smtp_cmd((char *)".\r\n", 250);
				} else {
					WriteError("SMTP refused message DATA");
				}
			} else {
				WriteError("SMTP refused %s", temp);
			}
		} else {
			WriteError("SMTP refused destination");
		}
	} else {
		WriteError("SMTP connection refused");
	}

	free(temp);
	email_out++;
}

