Unit totLIST;
{$I Sys75.INC}

Interface

Uses DOS,
  totSYS, totLOOK, totFAST, totWIN, totINPUT, totLINK, totSTR, totIO1, totMISC;

Type
  tListAction = (Finish, Refresh, None);
  ListCharFunc = Function (Var K: Word; Var X, Y: Byte; HiPick: LongInt): tListAction;
  ListMsgFunc = Function (HiPick: LongInt): String;

pBrowseOBJ = ^BrowseOBJ;
BrowseOBJ = Object
              vWin: StretchWinPtr;
              vTopPick: LongInt;         {number of first pick in window}
              vTotPicks: LongInt;        {total number of picks}
              vListVisible: Boolean;     {is list on display}
              vListAssigned: Boolean;    {is data assigned to list}
              vActivePick: Integer;      {the offset of the active pick from the top}
              vRows: Integer;            {total number of visible rows}
              vStartCol : LongInt;       {string position of first character}
              vEndCol: LongInt;          {rightmost column for scrolling}
              vRealColWidth: Byte;       {max avail column width}
              vLastKey: Word;            {last key the user pressed}
              {methods ...}
              Constructor Init;
              Procedure   SetTopPick (TopPick: LongInt);
              Procedure   SetStartCol (Column: LongInt);
              Procedure   SetEndCol (Column: LongInt);
              Function    Win: StretchWinPtr;
              Procedure   DisplayPick (Pick: Integer);
              Procedure   DisplayAllPicks;
              Procedure   ScrollUp;
              Procedure   ScrollDown;
              Procedure   ScrollPgUp;
              Procedure   ScrollPgDn;
              Procedure   ScrollFirst;
              Procedure   ScrollLast;
              Procedure   SlideLeft;
              Procedure   SlideRight;
              Procedure   ScrollFarRight;
              Procedure   ScrollFarLeft;
              Procedure   ScrollJumpH (X, Y: Byte);
              Procedure   ScrollJumpV (X, Y: Byte);
              Function    LastKey: Word;
              Procedure   Remove;
              Procedure   Show;
              Procedure   ResetDimensions;
              Procedure   Go;
              Function    GetString (Pick, Start, Finish: LongInt): String;  Virtual;
              Destructor  Done;                                           Virtual;
            End; {BrowseOBJ}

pBrowseArrayOBJ = ^BrowseArrayOBJ;
BrowseArrayOBJ = Object (BrowseOBJ)
                   vArrayPtr: Pointer;
                   vStrLength: Byte;
                   {methods ...}
                   Constructor Init;
                   Procedure   AssignList (Var StrArray; Total: LongInt; StrLength: Byte);
                   Function    GetString (Pick, Start, Finish: LongInt): String;  Virtual;
                   Destructor  Done;                                           Virtual;
                 End; {BrowseArrayOBJ}

pBrowseLinkOBJ = ^BrowseLinkOBJ;
BrowseLinkOBJ = Object (BrowseOBJ)
                  vLinkList: ^DLLOBJ;
                  {methods ...}
                  Constructor Init;
                  Procedure   AssignList (Var LinkList: DLLOBJ);
                  Function    ListPtr: DLLPtr;
                  Function    GetString (Pick, Start, Finish: LongInt): String;  Virtual;
                  Destructor  Done;    Virtual;
                End; {BrowseLinkOBJ}

pBrowseFileOBJ = ^BrowseFileOBJ;
BrowseFileOBJ = Object (BrowseOBJ)
                  vStrList: ^StrDLLOBJ;
                  {methods ...}
                  Constructor Init;
                  Function    AssignFile (Filename: String): Integer;
                  Function    ListPtr: StrDLLPtr;
                  Function    GetString (Pick, Start, Finish: LongInt): String;  Virtual;
                  Destructor  Done;    Virtual;
                End; {BrowseFileOBJ}

pListOBJ = ^ListOBJ;
ListOBJ = Object
            vWin: WinPtr;              {pointer to a window}
            vMargin: tByteCoords;      {padding around window border}
            vZone: tByteCoords;        {outer window dimensions}
            vTopPick: LongInt;         {number of first pick in window}
            vTotPicks: LongInt;        {total number of picks}
            vAllowToggle: Boolean;     {can user select items in list}
            vListVisible: Boolean;     {is list on display}
            vListAssigned: Boolean;    {is data assigned to list}
            vLastChar: Word;           {last key user pressed}
            vColWidth: Byte;           {user set column width in list display: 0 = max}
            vNAttr: Byte;              {normal attribute/color}
            vSAttr: Byte;              {attribute for special items}
            vHAttr: Byte;              {highlighted topic attribute/color}
            vActivePick: Integer;      {the offset of the active pick from the top}
            vRows: Integer;            {total number of visible rows}
            vCols: Integer;            {Total number of visible columns}
            vRealColWidth: Byte;       {max avail column width}
            vLastColWidth: Byte;       {width of right most column}
            vUseLastCol: Boolean;      {use the last column for highlighting or too narrow}
            vLastKey: Word;            {last key the user pressed}
            vCharHook: ListCharFunc;   {character hook}
            vMsgHook: ListMsgFunc;     {message hook}
            vMsgActive: Boolean;       {is Msg hook enabled}
            vDualColors: Boolean;      {should list use SAttr and NAttr}
            {methods ...}
            Constructor Init;
            Procedure   SetTopPick (TopPick: LongInt);
            Procedure   SetActivePick (ThePick: LongInt);
            Procedure   SetTagging (On: Boolean);
            Procedure   SetColors (HAttr, NAttr, SAttr: Byte);
            Procedure   SetColWidth (Wid: Byte);
            Procedure   SetCharHook (Func: ListCharFunc);
            Procedure   SetMsgHook (Func: ListMsgFunc);
            Procedure   SetMsgState (On: Boolean);
            Procedure   SetDualColors (On: Boolean);
            Function    GetHiString: String;
            Function    GetSelectedPick: LongInt;
            Function    Win: WinPtr;
            Procedure   ResetDimensions;
            Procedure   DisplayPick (Pick: Integer; Hi: Boolean);
            Procedure   DisplayAllPicks;
            Procedure   RefreshList;
            Procedure   Remove;
            Procedure   ValidateActivePick;
            Procedure   ScrollUp;
            Procedure   ScrollDown;
            Procedure   JumpEngine (Tot, NewValue: LongInt);
            Procedure   ScrollJumpV (X, Y: Byte);
            Procedure   ScrollJumpH (X, Y: Byte);
            Procedure   ScrollLeft;
            Procedure   ScrollFarLeft;
            Procedure   ScrollRight;
            Procedure   ScrollFarRight;
            Procedure   ScrollPgDn;
            Procedure   ScrollPgUp;
            Procedure   ScrollFirst;
            Procedure   ScrollLast;
            Procedure   ToggleSelect;
            Function    TargetPick (X, Y: Byte): Integer;
            Procedure   MouseChoose (KeyX, KeyY: Byte);
            Function    LastKey: Word;
            Function    Go: Boolean;
            Procedure   Show;
            Function    CharTask (Var K: Word; Var X, Y: Byte;HiPick: LongInt): tListAction;          Virtual;
            Function    MessageTask (HiPick: LongInt): String;             Virtual;
            Function    GetString (Pick, Start, Finish: LongInt): String;  Virtual;
            Function    GetStatus (Pick: LongInt; BitPos: Byte): Boolean;   Virtual;
            Procedure   SetStatus (Pick: LongInt; BitPos: Byte; On: Boolean); Virtual;
            Procedure   TagAll (On: Boolean);                             Virtual;
            Destructor  Done;                                           Virtual;
          End; {ListOBJ}

pListArrayOBJ = ^ListArrayOBJ;
ListArrayOBJ = Object (ListOBJ)
                 vArrayPtr: Pointer;
                 vStrLength: Byte;
                 vLinkList: ^DLLOBJ;
                 {methods ...}
                 Constructor Init;
                 Procedure  AssignList (Var StrArray; Total: LongInt; StrLength: Byte; Selectable: Boolean);
                 Procedure  SetTagging (On: Boolean);
                 Function   GetString (Pick, Start, Finish: LongInt): String;  Virtual;
                 Function   GetStatus (Pick: LongInt; BitPos: Byte): Boolean;   Virtual;
                 Procedure  SetStatus (Pick: LongInt; BitPos: Byte; On: Boolean); Virtual;
                 Procedure  TagAll (On: Boolean);                             Virtual;
                 Destructor Done;                                           Virtual;
               End; {of object ListArrayOBJ}

pListLinkOBJ = ^ListLinkOBJ;
ListLinkOBJ = Object (ListOBJ)
                vLinkList: ^DLLOBJ;
                {methods ...}
                Constructor Init;
                Procedure   AssignList (Var LinkList: DLLOBJ);
                Function    ListPtr: DLLPtr;
                Function    GetString (Pick, Start, Finish: LongInt): String;  Virtual;
                Function    GetStatus (Pick: LongInt; BitPos: Byte): Boolean;   Virtual;
                Procedure   SetStatus (Pick: LongInt; BitPos: Byte; On: Boolean); Virtual;
                Procedure   TagAll (On: Boolean);                             Virtual;
                Destructor  Done;                                           Virtual;
              End; {ListLinkOBJ}

pListDirOBJ = ^ListDirOBJ;
ListDirOBJ = Object (ListOBJ)
               vFileList: ^FileDLLOBJ;
               vActiveDir: PathStr;
               vChangeDir: Boolean;
               {methods ...}
               Constructor Init;
               Procedure   SetChangeDir (On: Boolean);
               Procedure   ReadFiles (FileMasks: String; FileAttrib: Word);
               Function    GetHiString: String;
               Procedure   Go;
               Function    FileList: FileDLLPtr;
               Function    CharTask (Var K: Word; Var X, Y: Byte;
                             HiPick: LongInt): tListAction;          Virtual;
               Function    MessageTask (Hi: LongInt): String;                Virtual;
               Function    GetString (Pick, Start, Finish: LongInt): String;  Virtual;
               Function    GetStatus (Pick: LongInt; BitPos: Byte): Boolean;   Virtual;
               Procedure   SetStatus (Pick: LongInt; BitPos: Byte; On: Boolean); Virtual;
               Procedure   TagAll (On: Boolean);                             Virtual;
               Destructor  Done;                                           Virtual;
             End; {ListDirOBJ}

pListDirSortOBJ = ^ListDirSortOBJ;
ListDirSortOBJ = Object (ListDirOBJ)
                   Constructor Init;
                   Function    PromptAndSort: Boolean;
                   Function    CharTask (Var K: Word; Var X, Y: Byte;
                   HiPick: LongInt): tListAction;          Virtual;
                   Destructor  Done;                                           Virtual;
                 End; {ListDirSortOBJ}

Implementation
{|||||||||||||||||||||||||||||||||||||||||||||}
{                                             }
{     M i s c.  P r o c s   &   F u n c s     }
{                                             }
{|||||||||||||||||||||||||||||||||||||||||||||}
Function NoCharHook (Var K: Word; Var X, Y: Byte; HiPick: LongInt): tListAction; Far;
{}
Begin
  NoCharHook := None;
End; {NoCharHook}

Function NoMsgHook (HiPick: LongInt): String; Far;
{}
Begin
  NoMsgHook := '';
End; {NoEnterHook}

Procedure Error (Err: Byte);
Begin
  Halt (9);
End; {Error}
{||||||||||||||||||||||||||||||||||||||||||}
{                                          }
{    B r o w s e O B J   M E T H O D S     }
{                                          }
{||||||||||||||||||||||||||||||||||||||||||}
Constructor BrowseOBJ. Init;
{}
Begin
  New (vWin, Init);
  vWin^. SetScrollable (True, True);
  vTopPick := 1;
  vTotPicks := 1;
  vListAssigned := False;
  vListVisible := False;
  vStartCol := 1;
  vEndCol := 80;
  vActivePick := 1;
  vRows := 0;
End; {BrowseOBJ.Init}

Function BrowseOBJ. Win: StretchWinPtr;
{}
Begin
  Win := vWin;
End; {BrowseOBJ.Win}

Procedure BrowseOBJ. SetTopPick (TopPick: LongInt);
{}
Begin
  vTopPick := TopPick;
End; {BrowseOBJ.SetTopElement}

Procedure BrowseOBJ. SetStartCol (Column: LongInt);
{}
Begin
  vStartCol := Column;
End; {BrowseOBJ.SetStartCol}

Procedure BrowseOBJ. SetEndCol (Column: LongInt);
{}
Begin
  If (Column > vStartCol) Or (Column = 0) Then
    vEndCol := Column
  Else
    vEndCol := vStartCol;
End; {BrowseOBJ.SetEndCol}

Function BrowseOBJ. GetString (Pick, Start, Finish: LongInt): String;
{abstract}
Begin End;

Procedure BrowseOBJ. DisplayPick (Pick: Integer);
{}
Var
  PickStr: String;
Begin
  If Pred (vTopPick + Pick) <= vTotPicks Then
    PickStr := GetString (Pred (vTopPick + Pick), vStartCol, Pred (vStartCol) + vRealColWidth)
  Else
    PickStr := '';
  PickStr := padleft (PickStr, vRealColWidth, ' ');
  Screen^. WritePlain (1, Pick, PickStr);
End; {BrowseOBJ.DisplayPick}

Procedure BrowseOBJ. DisplayAllPicks;
{}
Var I : Integer;
Begin
  for I := 1 to vRows do
    DisplayPick (I);
End; {BrowseOBJ.DisplayAllPicks}

Procedure BrowseOBJ. ScrollUp;
{}
Begin
  If vTopPick > 1 Then
  Begin
    Dec (vTopPick);
    DisplayAllPicks;
  End;
End; {BrowseOBJ.ScrollUp}

Procedure BrowseOBJ. ScrollDown;
{}
Begin
  If vTopPick < vTotPicks Then
  Begin
    Inc (vTopPick);
    DisplayAllPicks;
  End;
End; {BrowseOBJ.ScrollDown}

Procedure BrowseOBJ. SlideLeft;
{}
Begin
  If vStartCol > 1 Then
  Begin
    Dec (vStartCol);
    DisplayAllPicks;
  End;
End; {BrowseOBJ.SlideLeft}

Procedure BrowseOBJ. SlideRight;
{}
Begin
  If (vEndCol = 0) Or (vStartCol < vEndCol) Then
  Begin
    Inc (vStartCol);
    DisplayAllPicks;
  End;
End; {BrowseOBJ.SlideRight}

Procedure BrowseOBJ. ScrollPgUp;
{}
Begin
  If vTopPick > 1 Then
  Begin
    Dec (vTopPick, vRows);
    If vTopPick < 1 Then
      vTopPick := 1;
    DisplayAllPicks;
  End;
End; {BrowseOBJ.ScrollPgUp}

Procedure BrowseOBJ. ScrollPgDn;
{}
Begin
  If Pred (vTopPick + vRows) < vTotPicks Then
  Begin
    Inc (vTopPick, vRows);
    DisplayAllPicks;
  End;
End; {BrowseOBJ.ScrollPgDn}

Procedure BrowseOBJ. ScrollFarRight;
{}
Var EndCol: LongInt;
Begin
  If (vEndCol = 0) Then
    EndCol := 255
  Else
    EndCol := vEndCol;
  If (vStartCol < EndCol - Pred (vRealColWidth) ) Then
  Begin
    vStartCol := EndCol - Pred (vRealColWidth);
    DisplayAllPicks;
  End;
End; {BrowseOBJ.ScrollFarRight}

Procedure BrowseOBJ. ScrollFarLeft;
{}
Begin
  If vStartCol > 1 Then
  Begin
    vStartCol := 1;
    DisplayAllPicks;
  End;
End; {BrowseOBJ.ScrollFarLeft}

Procedure BrowseOBJ. ScrollLast;
{}
Begin
  If Pred (vTopPick) + vRows <> vTotPicks Then
  Begin
    vTopPick := Succ (vTotPicks) - vRows;
    DisplayAllPicks;
  End;
End; {BrowseOBJ.ScrollLast}

Procedure BrowseOBJ. ScrollFirst;
{}
Begin
  If vTopPick <> 1 Then
  Begin
    vTopPick := 1;
    DisplayAllPicks;
  End;
End; {BrowseOBJ.ScrollFirst}

Procedure BrowseOBJ. ScrollJumpH (X, Y: Byte);
{}
Var NewStart: LongInt;
Begin
  If X = 1 Then
    NewStart := 1
  Else If X = Y Then
    NewStart := vEndCol
  Else
    NewStart := (X * vEndCol) Div Y;
  If NewStart < 1 Then                  {1.00j}
    NewStart := 1;
  If NewStart <> vStartCol Then
  Begin
    vStartCol := NewStart;
    DisplayAllPicks;
  End;
End; {BrowseOBJ.ScrollJumpH}

Procedure BrowseOBJ. ScrollJumpV (X, Y: Byte);
{}
Var NewTop: LongInt;
Begin
  If X = 1 Then
    NewTop := 1
  Else If X = Y Then
    NewTop := vTotPicks
  Else
    NewTop := (X * vTotPicks) Div Y;
  If NewTop < 1 Then              {1.00j}
    NewTop := 1;
  If NewTop <> vTopPick Then
  Begin
    vTopPick := NewTop;
    DisplayAllPicks;
  End;
End; {BrowseOBJ.ScrollJumpV}

Procedure BrowseOBJ. Go;
{}
Var
  Finished: Boolean;
  Mvisible: Boolean;
  K: Word;
  X, Y : Byte;
  CX, CY, CT, CB: Byte;
Begin
  If Monitor^. ColorOn Then
    With Screen^ do
    Begin
      CursSave;
      CX := WhereX;
      CY := WhereY;
      CT := CursTop;
      CB := CursBot;
      CursOff;
    End;
  Show;
  Finished := False;
  Repeat
    vWin^. DrawHorizBar (vStartCol, vEndCol);
    vWin^. DrawVertBar (vTopPick, vTotPicks);
    K := key^. GetKey;
    vWin^. Winkey (K, X, Y);
    If (K = LookTOT^. ListEndKey) Or (K = LookTOT^. ListEscKey) Then
      Finished := True
    Else
      Case K Of
        600: Finished := True; {window close}
        602: Begin
          ResetDimensions;
          DisplayAllPicks; {window stretched}
        End;
        610, 328, 584: ScrollUp; {1.00d}
        611, 336, 592: ScrollDown;
        612, 331, 589: SlideLeft;
        613, 333, 587: SlideRight;
        337: ScrollPgDn;
        329: ScrollPgUp;
        335: ScrollFarRight;
        327: ScrollFarLeft;
        388: ScrollFirst;
        374: ScrollLast;
        614: ScrollJumpV (X, Y);
        615: ScrollJumpH (X, Y);
      End; {case}
  Until Finished;
  vLastKey := K;
  If Monitor^. ColorOn Then
    With Screen^ do
    Begin
      GotoXY (CX, CY);
      CursSize (CT, CB);
    End;
End; {BrowseOBJ.Go}

Procedure BrowseOBJ. Remove;
{}
Begin
  vWin^. Remove;
End; {BrowseOBJ.Remove}

Function BrowseOBJ. LastKey: Word;
{}
Begin
  LastKey := vLastKey;
End; {BrowseOBJ.LastKey}

Procedure BrowseOBJ. ReSetDimensions;
{}
Var S: Byte;
Begin
  With vWin^ do
  Begin
    S := GetStyle;
    Case S Of
      0: vRows := Succ (vBorder. Y2 - vBorder. Y1);
      6: vRows := vBorder. Y2 - vBorder. Y1 - 3;
      Else vRows := Pred (vBorder. Y2 - vBorder. Y1)
    End; {case}
    If S in [0, 6] Then
      vRealColWidth := Succ (vBorder. X2 - vBorder. X1)
    Else
      vRealColWidth := Pred (vBorder. X2 - vBorder. X1);
  End; {with}
End; {Browse.ResetDimensions}

Procedure BrowseOBJ. Show;
{}
Begin
  If vListAssigned = False Then
    Error (1)
  Else
  Begin
    If Not vListVisible Then
    Begin
      vWin^. Draw;
      ResetDimensions;
      DisplayAllPicks;
      vListVisible := True
    End;
  End;
End; {BrowseOBJ.Show}

Destructor BrowseOBJ. Done;
{}
Begin
  Dispose (vWin, Done);
End; {BrowseOBJ.Done}
{|||||||||||||||||||||||||||||||||||||||||||||||||||||}
{                                                     }
{    B r o w s e A r r a y O B J    M E T H O D S     }
{                                                     }
{|||||||||||||||||||||||||||||||||||||||||||||||||||||}
Constructor BrowseArrayOBJ. Init;
{}
Begin
  BrowseObj. Init;
End; {BrowseArrayOBJ.Init}

Procedure BrowseArrayOBJ. AssignList (Var StrArray; Total: LongInt; StrLength: Byte);
{}
Var
  L : LongInt;
  Dummy: Byte;
  Result : Integer;
Begin
  vArrayPtr := @StrArray;
  vStrLength := StrLength;
  vTotPicks := Total;
  vListAssigned := True;
  vEndCol := StrLength;
End; {BrowseArrayOBJ.AssignList}

Function BrowseArrayOBJ. GetString (Pick, Start, Finish: LongInt): String;
{}
Var
  W : Word;
  TempStr : String;
  ArrayOffset: Word;
Begin
  {move array string to Temp}
  W := Pred (Pick) * Succ (vStrLength);
  ArrayOffset := Ofs (vArrayPtr^) + W;
  Move (Mem [Seg (vArrayPtr^): ArrayOffset], TempStr, 1);
  Move (Mem [Seg (vArrayPtr^): Succ (ArrayOffset) ], TempStr [1], Ord (TempStr [0] ) );
  If Start < 0 Then Start := 0;
  If Finish < 0 Then Finish := 0;
  {validate Start and Finish Parameters}
  If ( (Finish = 0) And (Start = 0) )
     Or (Start > Finish) 
  Then   {get full string}
  Begin
    Start := 1;
    Finish := 255;
  End
  Else If Finish - Start > 254 Then      {too long to fit in string}
    Finish := Start + 254;
  If Finish > vStrLength Then
    Finish := vStrLength;
  If (Start > vStrLength) Then
    GetString := ''
  Else
  Begin
    GetString := Copy (TempStr, Start, Succ (Finish - Start) );
  End;
End; {BrowseArrayOBJ.GetString}

Destructor BrowseArrayOBJ. Done;
{}
Begin
  BrowseObj. Done;
End; {BrowseArrayOBJ.Done}
{|||||||||||||||||||||||||||||||||||||||||||||||||||}
{                                                   }
{    B r o w s e L i n k O B J    M E T H O D S     }
{                                                   }
{|||||||||||||||||||||||||||||||||||||||||||||||||||}
Constructor BrowseLinkOBJ. Init;
{}
Begin
  BrowseObj. Init;
  vLinkList := Nil;
End; {BrowseLinkOBJ.Init}

Procedure BrowseLinkOBJ. AssignList (Var LinkList: DLLOBJ);
{}
Begin
  vLinkList := @LinkList;
  vTotPicks := LinkList. TotalNodes;
  vListAssigned := True;
  vEndCol := LinkList. GetMaxNodeSize;
End; {BrowseLinkOBJ.AssignList}

Function BrowseLinkOBJ. GetString (Pick, Start, Finish: LongInt): String;
{}
Var TempPtr : DLLNodePtr;
Begin
  TempPtr := vLinkList^. NodePtr (Pick);
  If TempPtr <> Nil Then
    vLinkList^. ShiftActiveNode (TempPtr, Pick);
  GetString := vLinkList^. GetStr (TempPtr, Start, Finish);
End; {BrowseLinkOBJ.GetString}

Function BrowseLinkOBJ. ListPtr: DLLPtr;
{}
Begin
  ListPtr := vLinkList;
End; {BrowseLinkOBJ.ListPtr}

Destructor BrowseLinkOBJ. Done; 
{}
Begin
  BrowseObj. Done;
End; {BrowseLinkOBJ.Done;}
{|||||||||||||||||||||||||||||||||||||||||||||||||||}
{                                                   }
{    B r o w s e F i l e O B J    M E T H O D S     }
{                                                   }
{|||||||||||||||||||||||||||||||||||||||||||||||||||}
Constructor BrowseFileOBJ. Init;
{}
Begin
  BrowseOBJ. Init;
End; {BrowseFileOBJ.Init}

Function BrowseFileOBJ. AssignFile (Filename: String): Integer;
{RetCodes:
         0   OK
         1   File not found
         2   Run out of memory
}
Var
  F : Text;
  Line : String;
  Result: Integer;
Begin
  Assign (F, Filename);
  {$I-}
  Reset (F);
  {$I+}
  If IOResult <> 0 Then
    AssignFile := 1
  Else
  Begin
    New (vStrList, Init);
    Result := 0;
    While (EoF (F) = False) And (Result = 0) do
    Begin
      ReadLn (F, Line);
      Result := vStrList^. Add (Line);
    End;
    {$I-}
    Close (F);         {1.00b}
    {$I+}
    If IOResult <> 0 Then
      Result := 1;
    vWin^. SetTitle (filename);
    vListAssigned := True;
    vTotPicks := vStrList^. TotalNodes;
    vEndCol := vStrList^. GetMaxNodeSize;
    If Result = 0 Then
      AssignFile := 0
    Else
      AssignFile := 1;
  End;
End; {BrowseFileOBJ.AssignFile}

Function BrowseFileOBJ. ListPtr: StrDLLPtr;
{}
Begin
  ListPtr := vStrList;
End; {BrowseFileOBJ.ListPtr}

Function BrowseFileOBJ. GetString (Pick, Start, Finish: LongInt): String;
{}
Var TempPtr : DLLNodePtr;
Begin
  TempPtr := vStrList^. NodePtr (Pick);
  If TempPtr <> Nil Then
    vStrList^. ShiftActiveNode (TempPtr, Pick);
  GetString := vStrList^. GetStr (TempPtr, Start, Finish);
End; {BrowseFileOBJ.GetString}

Destructor BrowseFileOBJ. Done;   
{}
Begin
  BrowseOBJ. Done;
  If vListAssigned Then {1.00a}
    Dispose (vStrList, Done);
End; {BrowseFileOBJ.Done}
{||||||||||||||||||||||||||||||||||||||}
{                                      }
{    L i s t O B J   M E T H O D S     }
{                                      }
{||||||||||||||||||||||||||||||||||||||}
Constructor ListOBJ. Init;
{}
Begin
  New (vWin, Init);
  vTopPick := 1;
  vTotPicks := 1;
  vActivePick := 1;
  vListVisible := False;
  vListAssigned := False;
  vMsgActive := False;
  vCharHook := NoCharHook;
  vMsgHook := NoMsgHook;
  vAllowToggle  := True;
  vColWidth := 0;
  vHAttr := LookTOT^. MenuHiNorm;
  vNAttr := LookTOT^. MenuLoNorm;
  vSAttr := LookTOT^. MenuOff;
  vWin^. SetColors (0, 0, vNattr, 0, 0, 0, 0);
  vDualColors := False;
End; {ListOBJ.Init}

Procedure ListOBJ. SetTopPick (TopPick: LongInt);
{}
Begin
  vTopPick := TopPick;
End; {ListOBJ.SetTopElement}

Procedure ListOBJ. SetActivePick (ThePick: LongInt);
{}
Begin
  vActivePick := ThePick;
End; {ListOBJ.SetTopElement}

Procedure ListOBJ. SetTagging (On: Boolean);
{}
Begin
  vAllowToggle := On;
End; {ListOBJ.SetTagging}

Procedure ListOBJ. SetDualColors (On: Boolean);
{}
Begin
  vDualColors := On;
End; {ListOBJ.SetDualColors}

Procedure ListOBJ. SetColors (HAttr, NAttr, SAttr: Byte);
{}
Begin
  vHAttr := HAttr;
  vNAttr := NAttr;
  vSAttr := SAttr;
  vWin^. SetColors (0, 0, vNattr, 0, 0, 0, 0);
End; {ListOBJ.SetColors}

Procedure ListOBJ. SetColWidth (Wid: Byte);
{}
Begin
  vColWidth := Wid;
End; {ListOBJ.SetColumnWidth}

Procedure ListOBJ. SetCharHook (Func: ListCharFunc);
{}
Begin
  vCharHook := Func;
End; {ListOBJ.SetCharHook}

Procedure ListOBJ. SetMsgHook (Func: ListMsgFunc);
{}
Begin
  vMsgHook := Func;
  vMsgActive := True;
End; {ListOBJ.SetMsgHook}

Procedure ListOBJ. SetMsgState (On: Boolean);
{}
Begin
  vMsgActive := On;
End; {ListOBJ.SetMsgState}

Function ListOBJ. GetHiString: String;
{}
Begin
  GetHiString := GetString (Pred (vTopPick + vActivePick), 0, 0);
End; {ListOBJ.GetHiString}

Function ListOBJ. GetSelectedPick: LongInt;       {1.00c}
{}
Begin
  GetSelectedPick := Pred (vTopPick + vActivePick);
End; {ListOBJ.GetSelectedPick}

Function ListOBJ. Win: WinPtr;
{}
Begin
  Win := vWin;
End; {ListOBJ.Win}

Procedure ListOBJ. ResetDimensions;
{adjusts the column and row settings based on the list window coords}
Var 
  ListWidth: Byte;
  Style: Byte;
Begin
  With vZone do
    vWin^. GetSize (X1, Y1, X2, Y2, Style);
  If Style = 0 Then
    FillChar (vMargin, SizeOf (vMargin), #0)
  Else
  Begin
    vMargin. X1 := 1;
    vMargin. X2 := 1;
    vMargin. Y2 := 1;
    If Style = 6 Then
      vMargin. Y1 := 3
    Else
      vMargin. Y1 := 1;
  End;
  If vColWidth < 5 Then
  Begin
    vRealColWidth := Succ (vZone. X2 - vZone. X1) - vMargin. X1 - vMargin. X2;
    vCols := 1;
    vLastColWidth := vRealColWidth;
  End
  Else
  Begin
    vRealColWidth := vColWidth;
    ListWidth := Succ (vZone. X2 - vZone. X1) - vMargin. X1 - vMargin. X2;
    If vRealColWidth > ListWidth Then
      vRealColWidth := ListWidth;
    vCols :=  ListWidth Div vRealColWidth;
    vLastColWidth := ListWidth - vCols * vRealColWidth;
    If vLastColWidth = 0 Then
      vLastColWidth := vRealColWidth
    Else
      Inc (vCols);
  End;
  vUseLastCol := (vCols = 1) Or (vLastColWidth = vRealColWidth);
  vRows := Succ (vZone. Y2 - vZone. Y1) - vMargin. Y1 - vMargin. Y2;
  If vMsgActive Then
    Dec (vRows, 2);  {make space for message}
End; {ListOBJ.ResetDimensions}

Procedure ListOBJ. DisplayPick (Pick: Integer; Hi: Boolean);
{}
Var
  X, Y, ATT, Pad, Max, L: Byte;
  W : LongInt;
  Partial,
  Selected: Boolean;
  PadLeft, PadRight: String [1];
  PickStr : String;
  LeftChar,
  RightChar,
  ToggleOnChar,
  ToggleOffChar : Char;
Begin
  If vTotPicks = 0 Then
    Exit;
  LeftChar := LookTOT^. ListLeftChar;
  RightChar := LookTOT^. ListRightChar;
  ToggleOnChar := LookTOT^. ListToggleOnChar;
  ToggleOffChar := LookTOT^. ListToggleOffChar;
  Partial := (vCols > 1) And (Pick > vRows * Pred (vCols) )
  And (vLastColWidth <> vRealColWidth);
  If Pred (vTopPick + Pick) > vTotPicks Then
  Begin
    ATT := vNAttr;
    If Not Partial Then
      PickStr := replicate (vRealColWidth, ' ')
    Else
      PickStr := replicate (vLastColWidth, ' ');
  End
  Else
  Begin
    Selected := False;
    Pad := Ord (LeftChar <> #0) + 2 * Ord (vAllowToggle);
    If Not Partial Then
      Pad := Pad + Ord (RightChar <> #0);
    If vAllowToggle Then
      Selected := GetStatus (Pred (vTopPick + Pick), 0);
    If Hi Then
      ATT := vHAttr
    Else
    Begin
      If vDualColors And GetStatus (Pred (vTopPick + Pick), 1) Then
        ATT := vSAttr
      Else
        ATT := vNAttr;
    End;
    If (vCols = 1) Or (Pick <= vRows * Pred (vCols) ) Then
    Begin
      Max := vRealColWidth;
      W := vRealColWidth - pad;
    End
    Else
    Begin
      Max := vLastColWidth;
      W := vLastColWidth - pad;
    End;
    If W < 0 Then
      PickStr := ''
    Else
    Begin
      PickStr := GetString (Pred (vTopPick + Pick), 1, W);
      L := Length (PickStr);
      If L < W Then {pad out the name}
        PickStr := PickStr + replicate (W - L, ' ');
    End;
    If vAllowToggle Then
    Begin
      If Selected Then
        PickStr :=  ToggleOnChar + PickStr
      Else
        PickStr :=  ToggleOffChar + PickStr;
    End;
    If Hi Then
    Begin
      If (LeftChar <> #0) Then
        PickStr := LeftChar + PickStr;
      If (RightChar <> #0) Then
        PickStr := PickStr + RightChar;
    End
    Else
    Begin
      If (LeftChar = #0) Then
        Padleft := ''
      Else
        PadLeft := ' ';
      If (RightChar = #0) Or Partial Then
        PadRight := ''
      Else
        PadRight := ' ';
      PickStr := PadLeft + PickStr + PadRight + ' ';
    End;
    If Length (PickStr) > Max Then
      PickStr := Copy (PickStr, 1, Max);
  End;
  If Pick <= vRows Then
    X := 1
  Else
    X := Succ (vRealColWidth * (Pred (Pick) Div vRows) );
  If Pick Mod vRows = 0 Then
    Y := vRows
  Else
    Y := (Pick Mod vRows);
  {now write the pick}
  Screen^. WriteAT (X, Y, ATT, PickStr);
  If Hi Then
  Begin
    Screen^. GotoXY (X, Y);
    If vMsgActive Then
    Begin
      PickStr := MessageTask (Pred (vTopPick + vActivePick) );
      Screen^. WriteAt (1, Succ (vZone. Y2 - vMargin. Y2 - vZone. Y1 - vMargin. Y1),
      vWin^. GetTitleAttr(True),
      PadCenter (PickStr, Succ (vZone. X2 - vZone. X1 - vMargin. X2 - vMargin. X1), ' ') );
    End;
  End;
End; {ListOBJ.DisplayPick}

Procedure ListOBJ. DisplayAllPicks;
{}
Var
  I, J: Integer;
Begin
  for I := 1 to vCols do
    for J := 1 to vRows do
      DisplayPick (Pred (I) * vRows + J, (Pred (I) * vRows + J) = vActivePick);
End; {ListOBJ.DisplayAllPicks}

Procedure ListOBJ. ValidateActivePick;
{}
Var I, J : Integer;
Begin
  If (vUseLastCol) Or (vCols = 1) Then
    I := vCols * vRows
  Else
    I := Pred (vCols) * vRows;
  If (vActivePick > I) Or (vActivePick < 1) Then
    vActivePick := 1;
End; {ListOBJ.ValidateActivePick}

Procedure ListOBJ. RefreshList;
{}
Begin
  ResetDimensions;
  ValidateActivePick;
  If vMsgActive Then
  Begin
    Screen^. HorizLine (1, Succ (vZone. X2 - vZone. X1 - vMargin. X2 - vMargin. X1),
    vZone. Y2 - vMargin. Y2 - vZone. Y1 - vMargin. Y1,
    Win^. GetBorderAttr(True),
    1);
  End;
  DisplayAllPicks;
End; {ListOBJ.RefreshList}

Procedure ListOBJ. ScrollDown;
{}
Var LastPick : Integer;
Begin
  If Pred (vTopPick + vActivePick) < vTotPicks Then {not end of list}
  Begin
    If (vUseLastCol) Or (vCols = 1) Then
      LastPick := vCols * vRows
    Else
      LastPick := Pred (vCols) * vRows;
    If vActivePick < LastPick Then
    Begin
      DisplayPick (vActivePick, False);
      Inc (vActivePick);
      DisplayPick (vActivePick, True);
    End
    Else
    Begin
      DisplayPick (vActivePick, False);
      Inc (vTopPick, vRows);
      Dec (vActivePick, Pred (vRows));
      DisplayAllPicks;
    End;
  End;
End; {ListOBJ.ScrollDown}

Procedure ListOBJ. ScrollUp;
{}
Begin
  If vActivePick = 1 Then
  Begin
    If vTopPick > 1 Then
    Begin
      DisplayPick (vActivePick, False);
      Dec (vTopPick, vRows);
      Inc (vActivePick, Pred (vRows));
      DisplayAllPicks;
    End;
  End
  Else
  Begin
    DisplayPick (vActivePick, False);
    Dec (vActivePick);
    DisplayPick (vActivePick, True);
  End;
End; {ListOBJ.ScrollUp}

Procedure ListObj. JumpEngine (Tot, NewValue: LongInt);
{}
Var I: Integer;
Begin
  If NewValue < 1 Then
    NewValue := 1;
  If (Tot < (vCols - Ord (Not vUseLastCol) ) * vRows)
     And (vTopPick <= NewValue) 
  Then {full list on display}
  Begin
    DisplayPick (vActivePick, False);
    vActivePick := NewValue - Pred (vTopPick);
    DisplayPick (vActivePick, True);
  End
  Else
  Begin
    vTopPick := NewValue;
    vActivePick := 1;
    DisplayAllPicks;
  End;
End; {JumpEngine}

Procedure ListOBJ. ScrollJumpV (X, Y: Byte);
{}
Var
  NewValue: LongInt;
Begin
  NewValue := (X * vTotPicks) Div Y;
  JumpEngine (vTotPicks, NewValue)
End; {ListOBJ.ScrollJumpV}

Procedure ListOBJ. ScrollJumpH (X, Y: Byte);
{}
Var
  NewValue: LongInt;
Begin
  NewValue := (X * vTotPicks) Div Y;
  JumpEngine (vTotPicks, NewValue)
End; {ListOBJ.ScrollJumpH}

Procedure ListOBJ. ScrollLeft;
{}
Begin
  If (vCols = 1) Or ( (vCols = 2) And Not vUselastCol) Then
    ScrollUp
  Else
    If vActivePick > vRows Then {not in first column}
    Begin
      DisplayPick (vActivePick, False);
      vActivePick := vActivePick - vRows;
      DisplayPick (vActivePick, True);
    End
  Else If vTopPick > vRows Then                      {leftmost column}
  Begin
    vTopPick := vTopPick - vRows;
    DisplayAllPicks;
  End
  Else
  Begin
    vTopPick := 1;
    vActivePick := 1;
    DisplayAllPicks;
  End;
End; {ListOBJ.ScrollLeft}

Procedure ListOBJ. ScrollRight;
{}
Begin
  If (vCols = 1) Or ( (vCols = 2) And Not vUselastCol) Then
    ScrollDown
  Else
    If (vActivePick < Pred (vCols - Ord (Not vUseLastCol) ) * vRows) {not in last column}
       Or (vTopPick + (vRows * (vCols - Ord (Not vUseLastCol) ) ) >= vTotPicks) 
    Then
    Begin
      DisplayPick (vActivePick, False);
      vActivePick := vActivePick + vRows;
      If vTopPick + Pred (vActivePick) > vTotPicks Then
        vActivePick := Succ (vTotPicks - vTopPick);
      DisplayPick (vActivePick, True);
    End
  Else 
  Begin
    vTopPick := vTopPick + vRows;
    If vTopPick + Pred (vActivePick) > vTotPicks Then
      vActivePick := Succ (vTotPicks - vTopPick);
    DisplayAllPicks;
  End;
End; {ListOBJ.ScrollRight}

Procedure ListOBJ. ScrollFarRight;
{}
Begin
  While (vActivePick < Pred (vCols - Ord (Not vUseLastCol) ) * vRows) do
    Inc (vActivePick, vRows);
  While (vTopPick + (vCols - Ord (Not vUseLastCol) ) * vRows < vTotPicks)
        And   (vTopPick + Pred (vActivePick) + vRows <= vTotPicks) 
  do
    Inc (vTopPick, vRows);
  DisplayAllPicks;
End; {ListOBJ.ScrollFarRight}

Procedure ListOBJ. ScrollFarLeft;
{}
Begin
  While vActivePick - vRows > 0 do
    Dec (vActivePick, vRows);
  vTopPick := 1;
  DisplayAllPicks;
End; {ListOBJ.ScrollFarLeft}

Procedure ListOBJ. ScrollPgDn;
{}
Begin
  If Pred (vTopPick + vRows) < vTotPicks Then
  Begin
    vTopPick := vTopPick + vRows;
    vActivePick := 1;
    DisplayAllPicks;
  End;
End; {ListOBJ.ScrollPgDn}

Procedure ListOBJ. ScrollPgUp;
{}
Begin
  If vTopPick > 1 Then
  Begin
    vTopPick := vTopPick - vRows;
    If vTopPick < 1 Then
      vTopPick := 1;
    DisplayAllPicks;
  End;
End; {ListOBJ.ScrollPgUp}

Procedure ListOBJ. ScrollLast;
{}
Begin
  If vTopPick + Pred ( (vCols - Ord (Not vUseLastCol) ) * vRows) >= vTotPicks Then {last node on display}
  Begin
    DisplayPick (vActivePick, False);
    vActivePick := Succ (vTotPicks - vTopPick);
    DisplayPick (vActivePick, True);
  End
  Else
  Begin
    vTopPick := vTotPicks;
    vActivePick := 1;
    DisplayAllPicks;
  End;
End; {ListOBJ.ScrollLast}

Procedure ListOBJ. ScrollFirst;
{}
Begin
  vTopPick := 1;
  vActivePick := 1;
  DisplayAllPicks;
End; {ListOBJ.ScrollFirst}

Procedure ListOBJ. ToggleSelect;
{}
Begin
  If GetStatus (Pred (vTopPick + vActivePick), 1) Then Exit;
  SetStatus (Pred (vTopPick + vActivePick), 0, Not GetStatus (Pred (vTopPick + vActivePick), 0) );
  If Pred (vTopPick + vActivePick) < vTotPicks Then
    ScrollDown
  Else
    DisplayPick (vActivePick, True);
End; {of ListOBJ.ToggleSelect}

Function ListOBJ. TargetPick (X, Y: Byte): Integer;
{return the pick number of the pick pointed to by
 the coordinates X,Y. If no pick is at those coordinates, a
 0 is returned}
Begin
  If  (X >= vZone. X1 + vMargin. X1)
      And (X <= vZone. X2 - vMargin. X2)
      And (Y >= vZone. Y1 + vMargin. Y1)
      And (Y <= vZone. Y1 + vMargin. Y1 + Pred (vRows) )
  Then
  Begin
    X := Succ (X - vZone. X1 - vMargin. X1);
    Y := Succ (Y - vZone. Y1 - vMargin. Y1);
    If X Mod vRealColWidth = 0 Then
      X := X Div vRealColWidth
    Else
      X := Succ (X Div vRealColWidth);
    If (X < vCols)
       Or ( (X = vCols) And vUseLastCol) 
    Then
    Begin
      If vTopPick + Pred (Pred (X) * vRows + Y) <= vTotPicks Then
      Begin
        TargetPick := Pred (X) * vRows + Y;
        Exit;
      End;
    End;
  End;
  TargetPick := 0;
End; {ListOBJ.TargetPick}

Procedure ListOBJ. MouseChoose (KeyX, KeyY: Byte);
{}
Var
  HitPick : Integer;
Begin
  HitPick := TargetPick (KeyX, KeyY);
  If HitPick <> 0 Then
  Begin
    DisplayPick (vActivePick, False);
    vActivePick := HitPick;
    SetStatus (Pred (vTopPick + vActivePick), 0, Not GetStatus (Pred (vTopPick + vActivePick), 0) );
    DisplayPick (vActivePick, True);
  End;
End; {ListOBJ.MouseChoose}

Procedure ListOBJ. Show;
{}
Begin
  If vListAssigned = False Then
    Error (1)
  Else
  Begin
    If Not vListVisible Then
    Begin
      vWin^. Draw;
      RefreshList;
      vListVisible := True
    End;
  End;
End; {ListOBJ.Show}

Function ListOBJ. Go: Boolean;
{}
Var
  Finished: Boolean;
  Mvisible: Boolean;
  Kdouble: Boolean;
  K: Word;
  X, Y : Byte;
  CursX, CursY: Byte;
  Msg : String;
  CX, CY, CT, CB: Byte;

       Procedure ProcessAction (Act: tListAction);
       {}
       Begin
         Case Act Of
           Finish:
                   Begin
                     K := 0;
                     Finished := True;
                   End;
           Refresh:
                    Begin
                      K := 0;
                      RefreshList;
                    End;
           None:; {nothing!}
         End; {case}
       End; {ProcessAction}

Var
  Bosdf: Boolean;
Begin
  Bosdf := True;
  If Monitor^. ColorOn Then
    With Screen^ do
    Begin
      CursSave;
      CX := WhereX;
      CY := WhereY;
      CT := CursTop;
      CB := CursBot;
      CursOff;
    End;
  Show;
  kDouble := key^. GetDouble;
  If Not kDouble Then
    key^. SetDouble (True);
  Finished := False;
  Repeat
    CursX := Screen^. WhereX;
    CursY := Screen^. WhereY;
    Screen^. GotoXY (CursX, CursY);
    K := key^. GetKey;
    vWin^. Winkey (K, X, Y);
    ProcessAction (CharTask (K, X, Y, Pred (vTopPick + vActivePick) ) );
    If K = 0 Then Continue;
    If (K = LookTOT^. ListEndKey) Or (K = LookTOT^. ListEscKey) Then
    Begin
      Finished := True;
      Bosdf := False;
    End
    Else If (K = LookTOT^. ListToggleKey) And vAllowToggle Then
      ToggleSelect
    Else If (K = LookTOT^. ListTagKey) And vAllowToggle Then
      TagAll (True)
    Else If (K = LookTOT^. ListUnTagKey) And vAllowToggle Then
      TagAll (False)
    Else
      Case K Of
        13: Begin
              Finished := True;
              Bosdf := True;
            End;
        600: Finished := True; {window close}
        601: ResetDimensions;
        602: RefreshList;
        610, 328, 584: ScrollUp; {1.00d}
        611, 336, 592: ScrollDown;
        612, 331, 589: ScrollLeft;
        613, 333, 587: ScrollRight;
        513: MouseChoose (X, Y);  {leftMouse}
        523: If TargetPick (X, Y) <> 0 Then
        Begin
          MouseChoose (X, Y);
          Finished := True;
        End;
        337: If (vCols = 1) Or ( (vCols = 2) And Not vUselastCol) Then {PgDn}
          ScrollPgDn
        Else
          ScrollRight;
        329: If (vCols = 1) Or ( (vCols = 2) And Not vUselastCol) Then {PgUp}
          ScrollPgUp
        Else
          ScrollLeft;
        335: ScrollFarRight;
        327: ScrollFarLeft;
        388: ScrollFirst;
        374: ScrollLast;
        614:
             Begin  {vertical scroll bar}
               If X = 1 Then
                 ScrollFirst
               Else If X = Y Then
                 ScrollLast
               Else
                 ScrollJumpV (X, Y); {vertical scroll bar}
             End;
        615:
             Begin {horizontal scroll bar}
               If X = 1 Then
                 ScrollFirst
               Else If X = Y Then
                 ScrollLast
               Else
                 ScrollJumpH (X, Y); {vertical scroll bar}
             End;
      End; {case}
  Until Finished;
  vLastKey := K;
  If Monitor^. ColorOn Then
    With Screen^ do
    Begin
      GotoXY (CX, CY);
      CursSize (CT, CB);
    End;
  key^. SetDouble (KDouble);
  go := bosdf;
End; {ListOBJ.Go}

Function ListOBJ. LastKey: Word;
{}
Begin
  LastKey := vLastKey;
End; {ListOBJ.LastKey}

Procedure ListOBJ. Remove;
{}
Begin
  vWin^. Remove;
End; {ListOBJ.Remove}

Function ListOBJ. CharTask (Var K: Word; Var X, Y: Byte; HiPick: LongInt): tListAction; 
{}
Begin
  CharTask := vCharHook (K, X, Y, HiPick);
End; {ListOBJ.CharTask}

Function ListOBJ. MessageTask (HiPick: LongInt): String; 
{}
Begin
  MessageTask := vMsgHook (HiPick);
End; {ListOBJ.MessageTask}

Function ListOBJ. GetString (Pick, Start, Finish: LongInt): String;
{abstract}
Begin End;

Function ListOBJ. GetStatus (Pick: LongInt; BitPos: Byte): Boolean;
{abstract}
Begin End;

Procedure ListObj. SetStatus (Pick: LongInt; BitPos: Byte; On: Boolean);
{abstract}
Begin End;

Procedure ListOBJ. TagAll (On: Boolean);
{}
Begin End;

Destructor ListOBJ. Done;
{}
Begin
  Dispose (vWin, Done);
End;  {ListOBJ.Done}
{|||||||||||||||||||||||||||||||||||||||||||||||||}
{                                                 }
{    L i s t A r r a y O B J    M E T H O D S     }
{                                                 }
{|||||||||||||||||||||||||||||||||||||||||||||||||}
Constructor ListArrayOBJ. Init;
{}
Begin
  ListObj. Init;
  vLinkList := Nil;
End; {ListArrayOBJ.Init}

Procedure ListArrayOBJ. AssignList (Var StrArray; Total: LongInt; StrLength: Byte; Selectable: Boolean);
{}
Var
  L : LongInt;
  Dummy: Byte;
  Result : Integer;
Begin
  vArrayPtr := @StrArray;
  vStrLength := StrLength;
  vTotPicks := Total;
  vListAssigned := True;
  vAllowToggle := Selectable;
  If vAllowToggle Then {assign a linked list to record selections}
  Begin
    New (vLinkList, Init);           {1.00i}
    With vLinkList^ do
    Begin
      Dummy := 0;
      for L := 1 to Total do
      Begin
        Result := Add (Dummy, 0);
        If Result <> 0 Then
        Begin
          Dispose (vLinkList, Done);
          vAllowToggle := False;
        End;
      End;
    End;
  End;
End; {ListArrayOBJ.AssignList}

Procedure ListArrayOBJ. SetTagging (On: Boolean);
{}
Begin
  If On And (vLinkList <> Nil) Then
    vAllowToggle := True
  Else
    vAllowToggle := False;
End; {ListOBJ.SetTagging}

Function ListArrayOBJ. GetString (Pick, Start, Finish: LongInt): String;
{}
Var
  W : LongInt;
  TempStr : String;
  ArrayOffset: Word;
Begin
  {move array string to Temp}
  W := Pred (Pick) * Succ (vStrLength);
  ArrayOffset := Ofs (vArrayPtr^) + W;
  Move (Mem [Seg (vArrayPtr^): ArrayOffset], TempStr, 1);
  Move (Mem [Seg (vArrayPtr^): Succ (ArrayOffset) ], TempStr [1], Ord (TempStr [0] ) );
  If Start < 0 Then Start := 0;
  If Finish < 0 Then Finish := 0;
  {validate Start and Finish Parameters}
  If ( (Finish = 0) And (Start = 0) )
     Or (Start > Finish) 
  Then   {get full string}
  Begin
    Start := 1;
    Finish := 255;
  End
  Else If Finish - Start > 254 Then      {too long to fit in string}
    Finish := Start + 254;
  If Finish > vStrLength Then
    Finish := vStrLength;
  If (Start > vStrLength) Then
    GetString := ''
  Else
  Begin
    GetString := Copy (TempStr, Start, Succ (Finish - Start) );
  End;
End; {ListArrayOBJ.GetString}

Function ListArrayOBJ. GetStatus (Pick: LongInt; BitPos: Byte): Boolean;
{}
Begin
  If vAllowToggle Then  {1.00f}
    GetStatus := vLinkList^. NodePtr (Pick)^. GetStatus (BitPos)
  Else
    getStatus := False;
End; {ListArrayOBJ.GetStatus}

Procedure ListArrayObj. SetStatus (Pick: LongInt; BitPos: Byte; On: Boolean);
{}
Begin
  If vAllowToggle Then  {1.00f}
    vLinkList^. NodePtr (Pick)^. SetStatus (BitPos, On);
End; {ListArrayObj.SetStatus}

Procedure ListArrayOBJ. TagAll (On: Boolean);
{}
Var NodeP : DLLNodePtr;
Begin
  NodeP := vLinkList^. StartNodePtr;
  While NodeP <> Nil do
  Begin
    NodeP^. SetStatus (0, On);
    NodeP := NodeP^. NextPtr;
  End;
  DisplayAllPicks;
End; {ListOBJ.TagAll}

Destructor ListArrayOBJ. Done;
{}
Begin
  If vLinkList <> Nil Then
    Dispose (vLinkList, Done);
  ListObj. Done;
End; {ListArrayOBJ.Done}
{|||||||||||||||||||||||||||||||||||||||||||||||}
{                                               }
{    L i s t L i n k O B J    M E T H O D S     }
{                                               }
{|||||||||||||||||||||||||||||||||||||||||||||||}
Constructor ListLinkOBJ. Init;
{}
Begin
  ListObj. Init;
  vLinkList := Nil;
End; {ListLinkOBJ.Init}

Procedure ListLinkOBJ. AssignList (Var LinkList: DLLOBJ);
{}
Begin
  vLinkList := @LinkList;
  vTotPicks := LinkList. TotalNodes;
  vListAssigned := True;
End; {ListLinkOBJ.AssignList}

Function ListLinkOBJ. ListPtr: DLLPtr;
{}
Begin
  ListPtr := vLinkList;
End; {ListLinkOBJ.ListPtr}

Function ListLinkOBJ. GetString (Pick, Start, Finish: LongInt): String;
{}
Var TempPtr : DLLNodePtr;
Begin
  TempPtr := vLinkList^. NodePtr (Pick);
  If TempPtr <> Nil Then
    vLinkList^. ShiftActiveNode (TempPtr, Pick);
  GetString := vLinkList^. GetStr (TempPtr, Start, Finish);
End; {ListLinkOBJ.GetString}

Function ListLinkOBJ. GetStatus (Pick: LongInt; BitPos: Byte): Boolean;
{}
Begin
  GetStatus := vLinkList^. NodePtr (Pick)^. GetStatus (BitPos);
End; {ListLinkOBJ.GetStatus}

Procedure ListLinkObj. SetStatus (Pick: LongInt; BitPos: Byte; On: Boolean);
{}
Begin
  vLinkList^. NodePtr (Pick)^. SetStatus (BitPos, On);
End;  {ListLinkObj.SetStatus}

Procedure ListLinkOBJ. TagAll (On: Boolean);
{}
Var NodeP : DLLNodePtr;
Begin
  NodeP := vLinkList^. StartNodePtr;
  While NodeP <> Nil do
  Begin
    NodeP^. SetStatus (0, On);
    NodeP := NodeP^. NextPtr;
  End;
  DisplayAllPicks;
End; {ListOBJ.TagAll}

Destructor ListLinkOBJ. Done;
{}
Begin
  ListObj. Done;
End; {ListLinkOBJ.Done}
{|||||||||||||||||||||||||||||||||||||||||||||}
{                                             }
{    L i s t D i r O B J    M E T H O D S     }
{                                             }
{|||||||||||||||||||||||||||||||||||||||||||||}
Constructor ListDirOBJ. Init;
{}
Begin
  ListObj. Init;
  New (vFileList, Init);
  vMsgActive := True;
  vDualColors := True;
  vColWidth := 14;
  vWin^. SetSize (10, 5, 71, 20, 1);
  vChangeDir := True; {1.00g}
End; {ListDirOBJ.Init}

Procedure ListDirOBJ. SetChangeDir (On: Boolean);   {1.00g}
{}
Begin
  vChangeDir := On;
End; {ListDirOBJ.SetChangeDir}

Procedure ListDirOBJ. ReadFiles (FileMasks: String; FileAttrib: Word);  {1.00k}
{}
Var B: Byte;
Begin
  vActiveDir := '';
  If FileMasks = '' Then
    FileMasks := '*.*'
  Else If (Pos (':', Filemasks) <> 0) Or (Pos ('\', Filemasks) <> 0) Then
  Begin
    B := Length (FileMasks);
    While Not (FileMasks [B] in [':', '\'] ) do
      Dec (B);
    vActiveDir := Copy (FileMasks, 1, B);
  End;
  vFileList^. SetFileDetails (Copy (FileMasks, Succ (B), 12), FileAttrib);
  If vActiveDir <> '' Then
  Begin
    {$I-}
    If vActiveDir [Length (vActiveDir)] <> '\' Then
      ChDir (vActiveDir)
    Else
      ChDir (Copy (vActiveDir, 1, Pred (Length (vActiveDir))));
    {$I-}
    If IOResult <> 0 Then
    Begin
      vActiveDir := '';
      Filemasks := Copy (FileMasks, Succ (B), 12);
    End;
  End;
  If vActiveDir = '' Then
  Begin
    GetDir (0, vActiveDir);
    If Not (vActiveDir [Length (vActiveDir) ] in [':', '\'] ) Then
      vActiveDir := vActiveDir + '\';
    Filemasks := vActiveDir + Filemasks;
  End;
  Win^. SetTitle (' '+ vActiveDir + ' ');
  Win^. SetClose (False);
  vFileList^. FillList;
  vTotPicks := vFileList^. TotalNodes;
  vListAssigned := True;
End; {ListDirOBJ.ReadFiles}

Function ListDirOBJ. GetString (Pick, Start, Finish: LongInt): String;
{}
Var TempPtr : DLLNodePtr;
Begin
  TempPtr := vFileList^. NodePtr (Pick);
  If TempPtr <> Nil Then
    vFileList^. ShiftActiveNode (TempPtr, Pick);
  GetString := vFileList^. GetStr (TempPtr, Start, Finish);
End; {ListDirOBJ.GetString}

Function ListDirOBJ. CharTask (Var K: Word; Var X, Y: Byte; HiPick: LongInt): tListAction;
{}
Var
  FileInfo: tFileInfo;
  HitPick : Integer;
Begin
  CharTask := vCharHook (K, X, Y, HiPick); {1.00h}
  If (K = 13) Or (K = 513) Then
  Begin
    If K = 513 Then
    Begin
      HitPick := TargetPick (X, Y);
      If HitPick <> 0 Then
        HiPick := Pred (vTopPick + HitPick)
      Else
        Exit;
    End;
    vFileList^. GetFileRecord (FileInfo, HiPick);
    If ((SubDirectory (FileInfo. Attr)) Or (Drive (FileInfo. Attr))) And vChangeDir Then {1.00g}
    Begin
      If Pos (':', FileInfo. Path) <> 0 Then Begin
        If Not DiskReady (FileInfo. Path [1]) Then Begin
          K := 0;
          Exit;
        End;
      End;
      {$I-}
      ChDir (FileInfo. Path);
      {$I+}
      If IOResult = 0 Then
      Begin
        vFileList^. FillList;
        vTotPicks := vFileList^. TotalNodes;
        vTopPick := 1;
        vActivePick := 1;
        CharTask := Refresh;
        GetDir (0, vActiveDir);
        If Not (vActiveDir [Length (vActiveDir) ] in [':', '\'] ) Then
          vActiveDir := vActiveDir + '\';
        Win^. UpdateTitle (' ' + vActiveDir + ' ');
      End;
    End
    Else If (K = 13) Or ( (K = 513) And (vAllowToggle = False) ) Then
      CharTask := Finish;
  End;
End; {ListDirOBJ.CharTask}

Function ListDirOBJ. GetHiString: String;
{}
Begin
  GetHiString := vActiveDir + GetString (Pred (vTopPick + vActivePick), 0, 0);
End; {ListDirOBJ.GetHiString}

Function ListDirOBJ. MessageTask (Hi: LongInt): String;
{}
Var TempPtr : DLLNodePtr;
Begin
  TempPtr := vFileList^. NodePtr (Hi);
  If TempPtr <> Nil Then
    vFileList^. ShiftActiveNode (TempPtr, Hi);
  MessageTask := vFileList^. GetLongStr (TempPtr);
End; {ListDirOBJ.MessageTask}

Function ListDirOBJ. GetStatus (Pick: LongInt; BitPos: Byte): Boolean;
{}
Begin
  GetStatus := vFileList^. NodePtr (Pick)^. GetStatus (BitPos);
End; {ListDirOBJ.GetStatus}

Procedure ListDirObj. SetStatus (Pick: LongInt; BitPos: Byte; On: Boolean);
{}
Begin
  vFileList^. NodePtr (Pick)^. SetStatus (BitPos, On);
End;  {ListDirObj.SetStatus}

Procedure ListDirOBJ. TagAll (On: Boolean);
{}
Var NodeP : DLLNodePtr;
Begin
  NodeP := vFileList^. StartNodePtr;
  While NodeP <> Nil do
  Begin
    NodeP^. SetStatus (0, On);
    NodeP := NodeP^. NextPtr;
  End;
  DisplayAllPicks;
End; {ListOBJ.TagAll}

Function ListDirOBJ. FileList: FileDLLPtr;
{}
Begin
  FileList := vFileList;
End; {ListDirOBJ.FileList}

Procedure ListDirOBJ. Go;
{}
Var
  D: String;
Begin
  GetDir (0, D);
  ListOBJ. Go;
  {$I-}
  ChDir (D);
  {$I+}
  If IOResult <> 0 Then
    {whogivesashit} ;
End; {ListDirOBJ.Go}

Destructor ListDirOBJ. Done;
{}
Begin
  ListObj. Done;
  Dispose (vFileList, Done);
End; {ListDirOBJ.Done}
{|||||||||||||||||||||||||||||||||||||||||||||||||||||}
{                                                     }
{    L i s t D i r S o r t O B J    M E T H O D S     }
{                                                     }
{|||||||||||||||||||||||||||||||||||||||||||||||||||||}
Constructor ListDirSortOBJ. Init;
{}
Begin
  ListDirObj. Init;
End; {ListDirSortOBJ.Init}

Function ListDirSortOBJ. PromptAndSort: Boolean;
{}
Var
  Manager: WinFormOBJ;
  Control:  ControlKeysIOOBJ;
  OK, Cancel: Strip3DIOOBJ;
  SField, SOrder: RadioIOOBJ;
  Result: tAction;
  SortField: Byte;
  SortOrder: Boolean;
Begin
  Control. Init; {Tab, STab, Enter, Esc}
  OK. Init (23, 5, '   ~O~K   ', Finished);
  OK. SetHotKey (79); {O}
  Cancel. Init (23, 8, ' ~C~ancel ', Escaped);
  Cancel. SetHotKey (67); {C}
  With SField do
  Begin
    Init (3, 2, 18, 6, 'Sort on:');
    AddItem ('Nat~u~ral DOS', Ord ('U'), vFileList^. vSortID = 0);
    AddItem ('~N~ame', Ord ('N'), vFileList^. vSortID = 1);
    AddItem ('~E~xt', Ord ('E'), vFileList^. vSortID = 2);
    AddItem ('~S~ize', Ord ('S'), vFileList^. vSortID = 3);
    AddItem ('~T~ime', Ord ('T'), vFileList^. vSortID = 4);
    SetID (1);
  End;
  With SOrder do
  Begin
    Init (3, 9, 18, 3, 'Sort Order:');
    AddItem ('~A~scending', Ord ('A'), vFileList^. vSortAscending);
    AddItem ('~D~escending', Ord ('D'), Not vFileList^. vSortAscending);
  End;
  With Manager do
  Begin
    Init;
    AddItem (Control);
    AddItem (SField);
    AddItem (SOrder);
    AddItem (OK);
    AddItem (Cancel);
    SetActiveItem (1);
    Win^. SetSize (25, 2, 58, 15, 1);
    Win^. SetTitle ('Directory Sort Options');
    Draw;
    Result := Go;
    SortField := Pred (Sfield. GetValue);
    SortOrder := (SOrder. GetValue = 1);
    Control. Done;
    OK. Done;
    Cancel. Done;
    SField. Done;
    SOrder. Done;
    Done;
  End;
  If Result = Finished Then
  Begin
    vFileList^. Sort (SortField, SortOrder);
    vTopPick := 1;
    vActivePick := 1;
    PromptAndSort := True;
  End
  Else
    PromptAndSort := False;
End; {ListDirSortOBJ.PromptAndSort}

Function ListDirSortOBJ. CharTask (Var K: Word; Var X, Y: Byte; HiPick: LongInt): tListAction;
{}
Var
  FileInfo: tFileInfo;
  D : String;
  MP: LongInt;
Begin
  CharTask := vCharHook (K, X, Y, HiPick); {1.00h}
  If (K = 83) Or (K = 115) Or (K = 514) Then {'S','s',rightbutton}
  Begin
    If PromptAndSort Then
      CharTask := Refresh
    Else
      CharTask := none;
  End
  Else
    CharTask := ListDirOBJ. CharTask (K, X, Y, HiPick);
End; {ListDirSortOBJ.CharTask}

Destructor ListDirSortOBJ. Done;
{}
Begin
  ListDirObj. Done;
End; {ListDirSortOBJ.Done}

End.



