program PROTOCOL;
{$S-,R-,V-,I-,B-,F-,A-}
{$M 16384, 0, 512000}

{Conditional defines that may affect this unit}
{$I APDEFINE.INC}

{Include OPro's define file if UseOPro is specified}
{$IFDEF UseOPro}
{$I OPDEFINE.INC}
{$ENDIF}

{$IFDEF UseStreams}
{.$DEFINE CreateStream}      {Define this to test protocol streams}
{$ENDIF}

uses
  Crt,
  Dos,
  FastW1,
  ApMisc,
  ApTimer,
  ApPort,
  ApUart,
  OoCom,
  OoAbsPcl,
  OoXModem,
  OoYModem,
  OoZModem,
  {OoKermit,}
  OoAscii,
  WinTTT5;

const
  {Number of entries to keep in the trace queue}
  TraceEntries = 10000;

  {Determines whether port is opened normal or with "keep"}
  BaudSpecified : Boolean = False;

  {Misc constants}
  WAttr : Byte = $1B;          {Window attribute}
  FAttr : Byte = $1E;          {Frame attribute}
  DAttr : Byte = $1F;          {Data attribute}
  StatusDelay = 2000;          {Delay 2 seconds for status messages}

type
  BufPtr = ^BufferArray;
  BufferArray = array[0..MaxInt] of Char;
  TransferModeType = (Transmit, Receive);

var
  SrcFile : File;
  SrcFilename : String;
  ComX : ComNameType;
  Baud : LongInt;
  ComPort : UartPortPtr;
  SkipPortDone : Boolean;
  Prot : Byte;
  TransferMode : TransferModeType;
  ResumeFile : Boolean;
  ClobberFile : Boolean;
  NewerLonger : Boolean;
  FLP : FileListPtr;
  SaveExit : Pointer;

  {$F+}
  procedure ProtocolExitProc;
  begin
    ExitProc := SaveExit;

    {$IFDEF Tracing}
    {Save the trace to PROTOCOL.TRC}
    DumpTrace('PROTOCOL.TRC');
    {$ENDIF}

    {$IFDEF EventLogging}
    {Save the trace to PROTOCOL.LOG}
    DumpEvents('PROTOCOL.LOG');
    {$ENDIF}
  end;
  {$F-}

  procedure Abort(Msg : String; Code : Word);
  begin
    if ComPort <> Nil then
      Dispose(ComPort, Done);
    Write(Msg);
    if Code <> 0 then
      WriteLn(Code)
    else
      WriteLn;
    oncursor;
    Halt(Code);
  end;

  function StUpcase(S : String) : String;
  var
    I : Byte;
  begin
    for I := 1 to Length(S) do
      S[I] := Upcase(S[I]);
    StUpcase := S;
  end;

  function CharStr(Ch : Char; Len : Byte) : string;
    {-Return a string of length len filled with ch}
  var
    S : string;
  begin
    if Len = 0 then
      CharStr[0] := #0
    else begin
      S[0] := Chr(Len);
      FillChar(S[1], Len, Ch);
      CharStr := S;
    end;
  end;

  function Long2Str(L : LongInt) : string;
    {-Convert a long/word/integer/byte/shortint to a string}
  var
    S : string;
  begin
    Str(L, S);
    Long2Str := S;
  end;

  function Long2StrBlank(L : LongInt) : string;
    {-Convert a long/word/integer/byte/shortint to a string}
  begin
    if L = 0 then
      Long2StrBlank := ''
    else
      Long2StrBlank := Long2Str(L);
  end;

  function Real2Str(R : Real; Width : Byte; Places : ShortInt) : string;
    {-Convert a real to a string}
  var
    S : string;
  begin
    Str(R:Width:Places, S);
    Real2Str := S;
  end;

  function Pad(S : string; Len : Byte) : string;
    {-Return a string right-padded to length len with ch}
  var
    o : string;
    SLen : Byte absolute S;
  begin
    if Length(S) >= Len then
      Pad := S
    else begin
      o[0] := Chr(Len);
      Move(S[1], o[1], SLen);
      if SLen < 255 then
        FillChar(o[Succ(SLen)], Len-SLen, ' ');
      Pad := o;
    end;
  end;

  function LeftPad(S : string; Len : Byte) : string;
    {-Return a string left-padded to length len}
  var
    o : string;
    SLen : Byte absolute S;
  begin
    if Length(S) >= Len then
      LeftPad := S
    else if SLen < 255 then begin
      o[0] := Chr(Len);
      Move(S[1], o[Succ(Word(Len))-SLen], SLen);
      FillChar(o[1], Len-SLen, ' ');
      LeftPad := o;
    end;
  end;

  procedure WriteHelp;
    {-Write help and halt}
  begin
    WriteLn('Usage: PROTOCOL [options] Filename'^M);
    WriteLn('  -B Baudrate  Baudrate ');
    WriteLn('  -C #         Comport name  [def = 1]');
    WriteLn('  -T           Transmit mode [def]');
    WriteLn('  -R           Receive mode');
    WriteLn('  -S           ASCII transfer');
    WriteLn('  -X           Xmodem/XmodemCRC [def]');
    WriteLn('  -K           Xmodem1K');
    WriteLn('  -L           Xmodem1KG');
    WriteLn('  -Y           Ymodem');
    WriteLn('  -G           YmodemG');
    WriteLn('  -Z           Zmodem');
    WriteLn('  -A           Zmodem option - resume interrupted transfer');
    WriteLn('  -N           Zmodem option - only overwrite if newer');
    WriteLn('  -O           Zmodem option - always overwrite files');
    {WriteLn('  -F           Kermit');}
    oncursor;
    Halt;
  end;

  procedure ParseCommandLine;
    {-Gets command line options and sets various parameters.}
  var
    Code : Word;
    Param : String;
    Cnt : Word;
    ComNum : Word;

  begin
    {Set defaults}
    ComX := Com1;
    Baud := 1200;
    TransferMode := Transmit;
    Prot := Xmodem;
    SrcFilename := '';
    ResumeFile := False;
    ClobberFile := False;
    NewerLonger := False;

    {Scan command line}
    if ParamCount = 0 then
      WriteHelp;
    Param := ParamStr(1);
    Cnt := 2;

    while True do begin
      case Param[1] of
        '/', '-' :
          if Length(Param) <> 2 then
            Abort('Invalid parameter: '+Param, 0)
          else
            case Upcase(Param[2]) of

              'B' : {Set baud rate}
                begin
                  BaudSpecified := True;
                  Param := ParamStr(Cnt);
                  Inc(Cnt);
                  Val(Param, Baud, Code);
                  if Code <> 0 then
                    Abort('Invalid baud rate: '+Param, 0);
                end;

              'C' : {Set Com port}
                begin
                  Param := ParamStr(Cnt);
                  Inc(Cnt);
                  Val(Param, ComNum, Code);
                  if Code <> 0 then
                    Abort('Invalid com port: '+Param, 0);
                  if (ComNum < 1) or (ComNum > 8) then
                    Abort('Com port number out of range: '+Param, 0);
                  ComX := ComNameType(ComNum-1);
                end;

              'T' : TransferMode := Transmit;
              'R' : TransferMode := Receive;

              'S' : Prot := Ascii;
              'X' : Prot := Xmodem;
              'K' : Prot := Xmodem1K;
              'L' : Prot := Xmodem1KG;
              'Y' : Prot := Ymodem;
              'G' : Prot := YmodemG;
              'Z' : Prot := Zmodem;
              {'F' : Prot := Kermit;}

              'A' : ResumeFile := True;
              'O' : ClobberFile := True;
              'N' : NewerLonger := True;

              '?' : {Request for help}
                WriteHelp;

            else
              Abort('Invalid parameter: '+Param, 0);
            end;
      else
        SrcFilename := Param;
      end;

      {Get next parameter}
      if Cnt > ParamCount then begin
        if (SrcFilename = '') and
           not ((TransferMode = Receive) and
                ((Prot = Ymodem) or
                (Prot = YmodemG) or
                {(Prot = Kermit) or}
                (Prot = Zmodem))) then
          WriteHelp;
        Exit;
      end;
      Param := ParamStr(Cnt);
      Inc(Cnt);
    end;
  end;

  function BuildWindow(XLow, YLow, XHigh, YHigh : Byte; Header : String) : Pointer;
    {-Saves the underlying screen, frames and clears a window}
  type
    FrameCharType = (ULeft, LLeft, URight, LRight, Horiz, Vert);
    FrameArray = array[FrameCharType] of Char;
  const
    FrameChars : FrameArray = 'Ըͳ';
  var
    CoversP : BufPtr;
    WordsPerRow : Word;
    BufBytes : Word;
    SrcPos : Word;
    DestPos : Word;
    Row : Word;
    HeaderLen : Byte absolute Header;
    Width, HeaderPos : Byte;
    Span : string[132];
    SpanLen : Byte absolute Span;

  begin
    BuildWindow := nil;

    {Compute number of words to move per row}
    WordsPerRow := Succ(XHigh-XLow);

    {Compute bytes needed for screen buffer}
    BufBytes := (WordsPerRow*Succ(YHigh-YLow)) shl 1;

    {Make sure enough memory is available}
    if not GetMemCheck(CoversP, BufBytes) then
      Exit;

    {Save current contents to the screen buffer}
    DestPos := 0;
    SrcPos := (Pred(YLow)*ScreenWidth+Pred(XLow)) shl 1;
    for Row := YLow to YHigh do begin
      MoveFromScreen(Mem[VideoSegment:SrcPos], CoversP^[DestPos], WordsPerRow);
      Inc(SrcPos, ScreenWidth shl 1);
      Inc(DestPos, WordsPerRow shl 1);
    end;

    {Calculate width of window and position of header}
    SpanLen := Succ(XHigh - XLow);
    Width := SpanLen-2;

    {construct the upper border and draw it}
    FillChar(Span[2], Width, FrameChars[Horiz]);
    Span[1] := FrameChars[ULeft];
    Span[SpanLen] := FrameChars[URight];
    FastWrite(Span, YLow, XLow, FAttr);

    {Draw the vertical bars}
    for Row := Succ(YLow) to Pred(YHigh) do begin
      FastWrite(FrameChars[Vert], Row, XLow, FAttr);
      FastWrite(FrameChars[Vert], Row, XHigh, FAttr);
    end;

    {Draw the bottom border}
    Span[1] := FrameChars[LLeft];
    Span[SpanLen] := FrameChars[LRight];
    FastWrite(Span, YHigh, XLow, FAttr);

    {Draw the header}
    if HeaderLen > 0 then begin
      if HeaderLen > Width then
        HeaderLen := Width;
      HeaderPos := (SpanLen-HeaderLen) shr 1;
      FastWrite(Header, YLow, XLow + HeaderPos, FAttr);
    end;

    {Fill in the window}
    for Row := Ylow+1 to YHigh-1 do
      FastWrite(CharStr(' ', Pred(XHigh-XLow)), Row, XLow+1, FAttr);

    BuildWindow := CoversP;
  end;

  procedure RemoveWindow(P : Pointer; XLow, YLow, XHigh, YHigh : Byte);
    {-Restore screen contents and deallocate buffer space if requested}
  var
    CoversP : BufPtr absolute P;
    WordsPerRow : Word;
    SrcPos : Word;
    DestPos : Word;
    Row : Word;
  begin
    {Compute number of words to move per row}
    WordsPerRow := Succ(XHigh-XLow);

    {Restore current contents to the screen buffer}
    DestPos := 0;
    SrcPos := (Pred(YLow)*ScreenWidth+Pred(XLow)) shl 1;
    for Row := YLow to YHigh do begin
      MoveToScreen(CoversP^[DestPos], Mem[VideoSegment:SrcPos], WordsPerRow);
      Inc(SrcPos, ScreenWidth shl 1);
      Inc(DestPos, WordsPerRow shl 1);
    end;

    {Deallocate buffer space}
    FreeMem(CoversP, (WordsPerRow*Succ(YHigh-YLow)) shl 1);
  end;

  function FormatMinSec(TotalSecs : LongInt) : String;
    {-Format TotalSecs as minutes:seconds}
  var
    Min, Sec : LongInt;
    S : String;
  begin
    Min := TotalSecs div 60;
    Sec := TotalSecs mod 60;
    Str(Sec:2, S);
    if S[1] = ' ' then
      S[1] := '0';
    FormatMinSec := Long2Str(Min) + ':' + S;
  end;

  procedure UpdateProgressBar(Row, Col, Len : Byte; Percent : Real);
    {-Fills in a progress bar with Percent complete}
  const
    CompleteChar = '';
  var
    CharPercent : Real;
    CharCount : Byte;
    BarStr : String;
  begin
    {Calculate "percent value" of each character space}
    CharPercent := 100.0 / Len;

    {Calculate how many chars we need to approach (but not exceed) Percent}
    CharCount := Trunc((Percent * 100) / CharPercent);

    {Make sure we don't go past Len}
    if CharCount > Len then
      CharCount := Len;

    {Write out the complete bar}
    FillChar(BarStr[1], CharCount, CompleteChar);
    BarStr[0] := Char(CharCount);
    if CharCount <> 0 then
      FastWrite(BarStr, Row, Col, DAttr);
  end;

  procedure UpdateStatusMsg(Row, Col, Len : Byte);
    {-Translate the current AsyncStatus into a status message}
  const
    LastStatus : Word = 65535;
    MaxMsgLen = 40;
  var
    Msg : String;
  begin
    if AsyncStatus <> LastStatus then begin
      FillChar(Msg[1], MaxMsgLen, ' ');
      Msg[0] := Char(MaxMsgLen);
      FastWrite(Msg, Row, Col, DAttr);
      Msg := StatusStr(AsyncStatus);
      FastWrite(Msg, Row, Col, DAttr);
    end;
  end;

  (*{$F+}
  procedure WindowStatus(AP : AbstractProtocolPtr;
                         Starting, Ending : Boolean);
    {-Windowed show status procedure}
  const
    XLow = 10;
    YLow = 4;
    XHigh = 69;
    YHigh = 22;
    P : Pointer = nil;
    DividerBar = 'Ĵ';
    NewProgBar = '';
    HeaderStr : array[TransferModeType] of String[19] =
      (' Protocol Upload ', ' Protocol Download ');
    ModeStr : array[TransferModeType] of String[9] =
      ('sent:', 'received:');
    OnOffStr : array[Boolean] of String[3] = ('Off', 'On ');
  var
    Blocks : Integer;
    Efficiency, MaxCPS, ActualCPS, R : Real;
    CurBlockSize : Word;
    CurFileSize : LongInt;
    CurBytesRemaining : LongInt;
    CurBytesTransferred : LongInt;
    CurProtocol : Byte;
    CurElapsedTics : LongInt;
    CurBlock : Word;
    S : String;
    I : Word;
    B : Boolean;
  begin
    if Starting then begin
      {Build and frame the window}
      P := BuildWindow(XLow, YLow, XHigh, YHigh, HeaderStr[TransferMode]);
      if P = nil then
        Abort('Insufficient memory ', 1);

      {Write out the fixed text strings}
      FastWrite('Protocol:', YLow+1, XLow+2, WAttr);
      FastWrite('Block check:', YLow+2, XLow+2, WAttr);
      FastWrite('File name:', YLow+3, XLow+2, WAttr);
      FastWrite('File size:', YLow+4, XLow+2, WAttr);
      FastWrite('Block size:', YLow+5, XLow+2, WAttr);
      FastWrite('Total blocks:', YLow+6, XLow+2, WAttr);

      FastWrite('Est. time:', YLow+8, XLow+2, WAttr);
      FastWrite('Elapsed time:', YLow+9, XLow+2, WAttr);
      FastWrite('Remaining time:', YLow+10, XLow+2, WAttr);

      FastWrite('Bytes '+ModeStr[TransferMode], YLow+1, XLow+33, WAttr);
      FastWrite('Bytes remaining:', YLow+2, XLow+33, WAttr);
      FastWrite('Blocks '+ModeStr[TransferMode], YLow+3, XLow+33, WAttr);
      FastWrite('Blocks remaining:', YLow+4, XLow+33, WAttr);
      FastWrite('Block errors:', YLow+5, XLow+33, WAttr);
      FastWrite('Total errors:', YLow+6, XLow+33, WAttr);

      FastWrite('Throughput:', YLow+8, XLow+33, WAttr);
      FastWrite('Efficiency:', YLow+9, XLow+33, WAttr);

      FastWrite('Progress:', YLow+12, XLow+2, WAttr);
      FastWrite('Status:', YLow+13, XLow+2, WAttr);

      FastWrite(DividerBar, YLow+14, XLow, FAttr);
      FastWrite('Baud:', YLow+15, XLow+2, WAttr);
      FastWrite('DataBits:', YLow+16, XLow+2, WAttr);
      FastWrite('Sfw Flow:', YLow+17, XLow+2, WAttr);

      FastWrite('StopBits:', YLow+15, XLow+33, WAttr);
      FastWrite('Parity:', YLow+16, XLow+33, WAttr);
      FastWrite('Hdw Flow:', YLow+17, XLow+33, WAttr);

      {Only update the port status on startup}
      with AP^.APort^.PR^ do begin
        FastWrite(LeftPad(Long2Str(CurBaud), 8), YLow+15, XLow+18, DAttr);
        FastWrite(LeftPad(Long2Str(CurDataBits), 8), YLow+16, XLow+18, DAttr);
        {$IFDEF UseSWFlow}
        B := AP^.APort^.SWFlowState <> fsOff;
        {$ELSE}
        B := False;
        {$ENDIF}
        FastWrite(OnOffStr[B], YLow+17, XLow+23, DAttr);
        FastWrite(LeftPad(Long2Str(CurStopBits), 8), YLow+15, XLow+50, DAttr);
        FastWrite(LeftPad(ParityString[CurParity], 8), YLow+16, XLow+50, DAttr);
        {$IFDEF UseHWFlow}
        B := AP^.APort^.HWFlowState <> fsOff;
        {$ELSE}
        B := False;
        {$ENDIF}
        FastWrite(OnOffStr[B], YLow+17, XLow+56, DAttr);
      end;
    end;

    {Update the data areas}
    with AP^ do begin
      {Store common status info in local variables}
      CurBlockSize := GetBlockSize;
      CurFileSize := GetFileSize;
      CurBytesRemaining := GetBytesRemaining;
      CurBytesTransferred := GetBytesTransferred;
      CurProtocol := GetProtocol;
      CurElapsedTics := GetElapsedTics;
      CurBlock := GetBlockNum;

      {Protocol and file name}
      FastWrite(ProtocolTypeString[CurProtocol], YLow+1, XLow+18, DAttr);
      case GetCheckType of
        bcNone      : S := bcsNone;
        bcChecksum1 : S := bcsChecksum1;
        bcChecksum2 : S := bcsChecksum2;
        bcCrc16     : S := bcsCrc16;
        bcCrc32     : S := bcsCrc32;
        bcCrcK      : S := bcsCrcK;
      end;
      FastWrite(S, YLow+2, XLow+18, DAttr);
      FastWrite(Pad(StUpcase(GetFileName), 12), YLow+3, XLow+18, DAttr);

      {File size, block size, block check and total blocks}
      FastWrite(LeftPad(Long2StrBlank(CurFileSize),8), YLow+4, XLow+18, DAttr);
      FastWrite(LeftPad(Long2Str(CurBlockSize),8), YLow+5, XLow+18, DAttr);
      if CurFileSize = 0 then
        I := 0
      else
        I := Succ(CurFileSize div CurBlockSize);
      FastWrite(LeftPad(Long2StrBlank(I),8), YLow+6, XLow+18, DAttr);

      {Estimated time, elapsed time and time remaining}
      FastWrite(FormatMinSec(EstimateTransferSecs(CurFileSize)),
                YLow+8, XLow+18, DAttr);
      FastWrite(FormatMinSec(Tics2Secs(CurElapsedTics)), YLow+9, XLow+18, DAttr);
      FastWrite(FormatMinSec(EstimateTransferSecs(CurBytesRemaining)),
                YLow+10, XLow+18, DAttr);

      {Bytes transferred and bytes remaining}
      FastWrite(LeftPad(Long2Str(CurBytesTransferred),8), YLow+1, XLow+50, DAttr);
      FastWrite(LeftPad(Long2StrBlank(CurBytesRemaining),8), YLow+2, XLow+50, DAttr);

      {Blocks transferred and blocks remaining}
      FastWrite(LeftPad(Long2Str(CurBlock),8), YLow+3, XLow+50, DAttr);
      Blocks := (CurBytesRemaining+Pred(CurBlockSize)) div CurBlockSize;
      FastWrite(LeftPad(Long2StrBlank(Blocks),8), YLow+4, XLow+50, DAttr);

      {Error counts}
      FastWrite(LeftPad(Long2Str(GetBlockErrors),8), YLow+5, XLow+50, DAttr);
      FastWrite(LeftPad(Long2Str(GetTotalErrors),8), YLow+6, XLow+50, DAttr);

      {Display an empty progress bar on startup}
      if CurBytesTransferred = 0 then
        FastWrite(NewProgBar, YLow+12, XLow+18, DAttr);

      {Update the progress bar (if the file size is known}
      if CurFileSize <> 0 then begin
        R := CurBytesRemaining;
        R := R / CurFileSize;
      end else
        R := 1;
      UpdateProgressBar(YLow+12, XLow+18, Length(NewProgBar), 1.0 - R);

      {Update status message}
      UpdateStatusMsg(YLow+13, XLow+18, 35);

      {Calculate and display throughput}
      if CurElapsedTics > 0 then begin
        R := CurBytesTransferred - GetInitialFilePos;
        ActualCPS := R / (CurElapsedTics / 18.2);
      end else
        ActualCPS := 0.0;
      FastWrite(LeftPad(Long2Str(Trunc(ActualCPS))+' CPS',9),
                YLow+8, XLow+49, DAttr);

      {Calculate and display efficiency}
      MaxCPS := APort^.PR^.CurBaud div 10;
      if MaxCPS > 0 then
        Efficiency := (ActualCPS / MaxCPS) * 100.0
      else
        Efficiency := 0.0;
      FastWrite(Real2Str(Efficiency, 7, 0)+'%', YLow+9, XLow+50, DAttr);
    end;

    {Remove the window on the last status call}
    if Ending then
      RemoveWindow(P, XLow, YLow, XHigh, YHigh);
  end;*)

{$F+}
procedure WindowStatus(AP : AbstractProtocolPtr; Starting, Ending : Boolean);
  var dir:dirstr;
      fil:namestr;
      ext:extstr;
begin
  if starting then begin
    FillScreen(1,1,80,24,white,blue,chr(176));
    mkwin (20,9,60,20,11,1,5);
    textcolor (11);
    textbackground (1);
    gotoxy (22,10);
    write ('Filename: '); {32,10}
    gotoxy (22,11);
    write ('Path: '); {28,11}
    gotoxy (22,12);
    write ('Total bytes: '); {35,12}
    gotoxy (22,13);
    write ('Bytes transferred: '); {41,13}
    gotoxy (22,14);
    write ('Bytes remaining: '); {39,14}
    gotoxy (22,15);
    write ('Block size: '); {34,15}
    gotoxy (22,16);
    write ('Time total: '); {34,16}
    gotoxy (22,17);
    write ('Time elapsed: '); {36,17}
    gotoxy (22,18);
    write ('Time remaining: '); {38,18}
    gotoxy (22,19);
    write ('Protocol: '); {32,19}
  end;
  OffCursor;
  textcolor (15);
  textbackground (1);
  gotoxy (32,10);
  write (ap^.getfilename);
  fsplit (ap^.getpathname,dir,fil,ext);
  gotoxy (28,11);
  write (dir);
  gotoxy (35,12);
  write (ap^.getfilesize);
  gotoxy (41,13);
  write (ap^.getbytestransferred);
  write ('           ');
  gotoxy (39,14);
  write (ap^.getbytesremaining);
  write ('           ');
  gotoxy (34,15);
  write (ap^.getblocksize);
  write ('           ');
  gotoxy (34,16);
  write (FormatMinSec(ap^.EstimateTransferSecs(ap^.getfilesize)));
  write ('           ');
  gotoxy (36,17);
  write (FormatMinSec(Tics2Secs(ap^.getelapsedtics)));
  write ('           ');
  gotoxy (38,18);
  write (FormatMinSec(ap^.EstimateTransferSecs(ap^.getbytesremaining)));
  write ('           ');
  gotoxy (32,19);
  write (protocoltypestring[ap^.getprotocol]);
  textbackground (0);
  if ending then begin
    rmwin;
    textbackground (0);
    clrscr;
    OnCursor;
  end;
end;

  function KbdAbort : Boolean;
    {-Default abort function}
  const
    Escape = #$1B;
  var
    Ch : Char;
  begin
    KbdAbort := False;
    if KeyPressed then begin
      Ch := ReadKey;
      if Ch = #0 then
        Ch := ReadKey;
      if Ch = Escape then begin
        KbdAbort := True;
        clrscr;
        Abort ('Aborted: ',1);
      end;
    end;
  end;

  procedure LogFileActivity(AP : AbstractProtocolPtr; LogFileStatus : LogFileType);
    {-Maintains a history of all file transmits and receives}
  var
    FLog : Text;
    F : File;
    FName : PathStr;
    Prot : Byte;
  begin
    with AP^ do begin
      Assign(FLog, 'PROTOCOL.HIS');
      Append(FLog);
      if IOResult = 2 then
        ReWrite(FLog);
      if IOResult <> 0 then
        Exit;
      FName := GetPathName;
      Prot := GetProtocol;
      case LogFileStatus of
        lfReceiveStart :
          {do nothing} ;
        lfReceiveOk :
          WriteLn(FLog, ProtocolTypeString[Prot], ' receive ', FName);
        lfReceiveFail :
          begin
            WriteLn(FLog, ProtocolTypeString[Prot], ' receive aborted ', FName);
            if GetProtocol <> Zmodem then begin
              Assign(F, FName);
              Erase(F);
              if IOResult <> 0 then ;
            end;
          end;
        lfReceiveSkip :
          WriteLn(FLog, ProtocolTypeString[Prot], ' receive skipped ', FName);
        lfTransmitStart :
          {do nothing} ;
        lfTransmitOk :
          WriteLn(FLog, ProtocolTypeString[Prot], ' transmit ', FName);
        lfTransmitFail :
          WriteLn(FLog, ProtocolTypeString[Prot], ' transmit aborted ', FName);
        lfTransmitSkip :
          WriteLn(FLog, ProtocolTypeString[Prot], ' transmit skipped ', FName);
      end;
      Close(FLog);
      if IOResult <> 0 then ;
    end;
  end;

  function ProtocolAcceptFile(AP : AbstractProtocolPtr) : Boolean;
    {-Test of file renaming with AcceptFile function}
  var
    FName : String[12];
  begin
    with AP^ do begin
      FName := GetFilename;
      FName[1] := '$';
      SetReceiveFileName(FName);
      ProtocolAcceptFile := True;
    end;
  end;

  procedure ProtocolErrorProc(P : Pointer; var StatusCode : Word);
  var
    AP : AbstractPortPtr absolute P;
    C : Char;
    S : String;
    W : Pointer;
  const
    BlankStr : String[78] =
  '                                                                              ';
  begin
    with AP^ do begin
      {Do nothing if a protocol is in progress}
      if ProtocolInProgress then
        Exit;
      if StatusCode mod 10000 <> 0 then begin
        {Build an error message}
        W := BuildWindow(1, 23, 80, 25, ' Press any key to continue ');
        if W = nil then begin
          {Not enough memory to show error -- just beep and exit}
          {Beep;}
          Exit;
        end;
        FastWrite(BlankStr, 24, 2, DAttr);
        Str(AsyncStatus, S);
        FastWrite(
          'Error during processing ('+S+'): '+StatusStr(AsyncStatus),
           24, 2, DAttr);
        C := ReadKey;
        FastWrite(BlankStr, 24, 2, DAttr);
        RemoveWindow(W, 1, 23, 80, 25);
      end;
    end;
  end;
  {$F-}

  {$IFDEF CreateStream}
  procedure CreateStream(PP : AbstractProtocolPtr);
    {-Store PP to a stream}
  var
    S : BufIdStream;
    Status : Word;
  begin
    {Create a new stream}
    if not S.Init('PROTOCOL.STM', SCreate, 1024) then
      Abort('Failed to make stream: ', InitStatus);

    {Register all protocol hierarchies}
    {S.RegisterHier(KermitProtocolStream);}
    S.RegisterHier(ZmodemProtocolStream);
    S.RegisterHier(YmodemProtocolStream);
    S.RegisterHier(XmodemProtocolStream);
    Status := S.GetStatus;
    if Status <> 0 then
      Abort('Error registering protocol object: ', Status);

    {Register our user procedures}
    S.RegisterPointer(1000, @WindowStatus);
    S.RegisterPointer(1001, @LogFileActivity);

    (* if this code is used the port object is stored *)
    {Register port hierarchy}
    S.RegisterHier(UartPortStream);
    S.RegisterPointer(ptErrorProc, @ProtocolErrorProc);
    S.RegisterPointer(ptAbortProc, @KbdAbort);
    Status := S.GetStatus;
    if Status <> 0 then
      Abort('Error registering protocol object: ', Status);

    (* if this code is used only a pointer code to the port object is stored
    {Register our port}
    S.RegisterPointer(ptPortPtr, ComPort);
    *)

    {Store the protocol}
    S.PutPtr(PP);
    Status := S.GetStatus;
    if Status <> 0 then
      Abort('Error storing protocol: ', Status);

    {Clean up}
    S.Done;
  end;

  procedure LoadStream(var PP : AbstractProtocolPtr);
    {-Load PP from a stream}
  var
    S : BufIdStream;
    Status : Word;
  begin
    {Re-open existing new stream}
    if not S.Init('PROTOCOL.STM', SOpen, 1024) then
      Abort('Failed to open stream: ', InitStatus);

    {Register all protocol hierarchies}
    {S.RegisterHier(KermitProtocolStream);}
    S.RegisterHier(ZmodemProtocolStream);
    S.RegisterHier(YmodemProtocolStream);
    S.RegisterHier(XmodemProtocolStream);
    Status := S.GetStatus;
    if Status <> 0 then
      Abort('Error registering protocol object: ', Status);

    {Register our user procedures}
    S.RegisterPointer(1000, @WindowStatus);
    S.RegisterPointer(1001, @LogFileActivity);

    (* if this is code is used the port object is stored *)
    {Register port hierarchy}
    S.RegisterHier(UartPortStream);
    S.RegisterPointer(ptErrorProc, @ProtocolErrorProc);
    S.RegisterPointer(ptAbortProc, @KbdAbort);
    Status := S.GetStatus;
    if Status <> 0 then
      Abort('Error registering protocol object: ', Status);

    (* if this code is used only a pointer code to the port is stored
    {Register our port}
    S.RegisterPointer(ptPortPtr, ComPort);
    *)

    {Load the protocol}
    RootPtr(PP) := S.GetPtr;
    Status := S.GetStatus;
    if Status <> 0 then
      Abort('Error loading protocol: ', Status);

    {Clean up}
    S.Done;
  end;
  {$ENDIF}

  procedure TransferFiles;
    {-Send or receive files}
  var
    PP : AbstractProtocolPtr;
  begin
    {Instantiate proper protocol type and add specific customizations}
    case Prot of
      Ascii :
        begin
          PP := New(AsciiProtocolPtr, Init(ComPort));
          if PP <> nil then begin
            if TransferMode = Transmit then begin
              AsciiProtocolPtr(PP)^.SetDelays(0, 100);
              PP^.SetFileMask(SrcFileName);
            end else
              PP^.SetReceiveFileName(SrcFileName);
          end;
        end;

      Xmodem, Xmodem1K, Xmodem1KG :
        begin
          PP := New(XmodemProtocolPtr, Init(ComPort,
                                            Prot = Xmodem1K,
                                            Prot = Xmodem1KG));
          if PP <> nil then
            with XmodemProtocolPtr(PP)^ do begin
              SetReceiveFileName(SrcFileName);
              SetBlockWait(RelaxedBlockWait);
            end;
        end;
      Ymodem, YmodemG :   {Ymodem derivatives}
        begin
          PP := New(YmodemProtocolPtr, Init(ComPort, True, Prot = YmodemG));
          YmodemProtocolPtr(PP)^.SetBlockWait(RelaxedBlockWait);
        end;
      ZModem :            {Zmodem protocol}
        begin
          PP := New(ZmodemProtocolPtr, Init(ComPort));
          if PP <> nil then
            with ZmodemProtocolPtr(PP)^ do begin
              if ClobberFile then
                SetFileMgmtOptions(True, False, WriteClobber);
              if NewerLonger then
                SetFileMgmtOptions(True, False, WriteNewerLonger);
              if ResumeFile then
                SetRecoverOption(True);
            end;
          end;
      (*Kermit :            {Kermit protocol}
        begin
          PP := New(KermitProtocolPtr, Init(ComPort));
        end;*)
    end;

    {Check results of initialization}
    if PP = nil then
      Abort('Failed to instantiate protocol ', AsyncStatus);

    with PP^ do begin
      {Add general customizations}
      SetShowStatusProc(WindowStatus);
      SetOverwriteOption(WriteRename);
      SetLogFileProc(LogFileActivity);
      SetFileMask(SrcFileName);

      (* Examples of AcceptFile procs
      SetAcceptFileFunc(ProtocolAcceptFile);
      SetAcceptFileFunc(AcceptOneFile);
      *)

      (* Examples of various options
      SetHandshakeWait(Secs2Tics(10), 5);
      apOptionsOn(apIncludeDirectory);
      *)

      (* Example of using file list
      SetNextFileFunc(NextFileList);
      MakeFileList(FLP, 500);
      AddFileToList(FLP, 'E:\ASYNC\APABSPCL.PAS');
      AddFileToList(FLP, 'E:\ASYNC\APCOM.PA1');
      AddFileToList(FLP, 'E:\ASYNC\APKERMIT.PAS');
      AddFileToList(FLP, 'E:\ASYNC\APLZH.PAS');
      AddFileToList(FLP, 'E:\ASYNC\APMISC.PAS');
      AddFileToList(FLP, 'E:\ASYNC\APUART.PAS');
      SetFileList(FLP);
      *)
    end;

    {$IFDEF CreateStream}
    CreateStream(PP);
    Dispose(PP, Done);
    LoadStream(PP);
    PP^.SetFileMask(SrcFileName);
    if PP^.DeallocPort then
      SkipPortDone := True;
    {$ENDIF}

    with PP^ do begin
      {Process the protocol request}
      case TransferMode of
        Transmit : ProtocolTransmit;
        Receive :  ProtocolReceive;
      end;

      {Cleanup}
      Dispose(PP, Done);
      if AsyncStatus = ecOk then
        Write(^M'Transfer complete'^M)
      else
        Write(^M'Transfer failed ', AsyncStatus,^M);
    end;
  end;

begin
  {Set ComPort to nil so abort can see if it needs to be disposed}
  ComPort := nil;

  {Get command line parameters}
  ParseCommandLine;

  {Make a port object}
  if BaudSpecified then New(ComPort, InitCustom(ComX, Baud, NoParity, 8, 1, 2048, 2048, DefPortOptions))
  else New(ComPort, InitKeep(ComX, 2048, 2048));

  ComPort^.ptOptionsOff({ptRestoreOnClose or }ptDropModemOnClose);
  ComPort^.SetDTR(True);
  ComPort^.SetRTS(True);

  if ComPort = nil then
    Abort('Failed to open port, Status = ', AsyncStatus);

  {$IFDEF Tracing}
  {Start a trace}
  InitTracing(TraceEntries);
  {$ENDIF}

  {$IFDEF EventLogging}
  {Start a log}
  InitEventLogging(5000);
  {$ENDIF}

  {$IFDEF UseHWFlow}
  {Turn on hardware flow control (CTS only)}
  ComPort^.HWFlowEnable(1800, 200, hfUseRTS or hfRequireCTS);
  {$ENDIF}

  {Set the port-level user abort function}
  ComPort^.SetAbortFunc(KbdAbort);

  {Set the port-level user error handler}
  ComPort^.SetErrorProc(ProtocolErrorProc);

  {Flag to avoid disposing ComPort twice when testing streams}
  SkipPortDone := False;

  {Set a user exit proc}
  SaveExit := ExitProc;
  ExitProc := @ProtocolExitProc;

  {Transfer file(s)}
  TransferFiles;

  {$IFDEF Tracing}
  {Save the trace to PROTOCOL.TRC}
  DumpTrace('PROTOCOL.TRC');
  {$ENDIF}

  {$IFDEF EventLogging}
  {Save the trace to PROTOCOL.LOG}
  DumpEvents('PROTOCOL.LOG');
  {$ENDIF}

  {Clean up}
  if not SkipPortDone then
    Dispose(ComPort, Done);
end.
