(* *************************************************************************
 *                                                                         *
 *       Program:  Fossil Routines                                         *
 *       Language: Borland Pascal V7.0                                     *
 *       Revision: V1.0 13/1/95                                            *
 *                                                                         *
 *       Copyright (C) Peter Davies 1995.  All Rights Reserved             *
 *                                                                         *
 *       Conditions of use                                                 *
 *          May be freely used and Modified.                               *
 *          Modified versions can not be distributed.                      *
 *          Any damage caused by this software, is at the users risk.      *
 *          The author takes no responsibility whatsover for the damage    *
 *          this software may, or may not cause                            *
 *                                                                         *
 ************************************************************************* *)


unit fossil;

interface

uses objects, Dos, Strings;

const
   ExtendedFossil : Boolean = False;
   CarrierValue   : Byte    = $80;
   CR                       = #$0D + #$0A;        (* Carriage Return *)
   BS                       = #$08 + #$20 + #$08; (* Backspace *)
   TimeOut        : Word    = 120;

type
   FossilInfoType = record
      InfoSize     : word;
      CurrFossil   : byte;            (* Fossil version number  *)
      CurrRev      : byte;            (*        revision number *)
      IDString     : PChar;           (* Fossils ID string      *)
      IBSize       : word;            (* Input buffer size      *)
      IBFree       : word;            (* free space in inbuf    *)
      OBSize       : word;            (* Output buffer size     *)
      OBFree       : word;            (* free space in outbuf   *)
      ScreenWidth  : byte;
      ScreenHeight : byte;
      Baud         : byte;            (* Baud mask, see Proc SetBaud *)
   end;

   PFossil = ^TFossil;
   TFossil = object(TObject)
    Private
      ComPort          : word;     (* 0 = Com1, 1 = Com2 *)
      FossilActive     : boolean;
      BaudRate         : longint;
      FossilInfo       : ^FossilInfoType;
      procedure        GetFossilInfo;

    Public
      constructor      Init(NewComPort : word);     (* 1 = Com1, 2 = Com2 *)
      destructor       Done; virtual;
      function         CarrierDetected   : boolean;
      function         CharAvailInOutputBuffer : boolean;
      procedure        ClearInputBuffer;
      procedure        ClearOutputBuffer;
      procedure        FlushOutputBuffer;
      function         FreeSpaceInInputBuffer  : word;
      function         FreeSpaceInOutputBuffer : word;
      function         GetBaudRate       : longint;
      procedure        GetBlock(var DataBlock;MaxBlockLen : word;var BlockLenRead : word);
      function         GetChar           : char;
      function         GetComPort        : word;
      function         GetFossilActive   : boolean;
      function         GetFossilIDString : string;
      function         GetFossilRevision : byte;
      function         GetFossilVersion  : byte;
      procedure        GiveTimeSlice;    virtual;
      procedure        HardwareFlowControl;
      function         InputBufferEmpty   : boolean;
      procedure        LowerDtr;
      function         OutputBufferEmpty  : boolean;
      procedure        PutBlock(var DataBlock;BlockLen : word;var BlockLenWritten : word);
      procedure        PutBlockCD(var DataBlock;BlockLen : word);
      procedure        PutChar(X : char);
      procedure        PutCharCD(X : char);
      procedure        PutStringCD(S : String);
      procedure        RaiseDtr;
      procedure        SetBaud(NewBaudRate : longint);
      procedure        SetTimerSecs(Secs : Longint);
      function         SizeOfInputBuffer  : word;
      function         SizeOfOutputBuffer : word;
      function         TimerExpired : boolean;
   end;

implementation

constructor TFossil.Init(NewComPort : Word);

var
   Regs : Registers;

begin
   ComPort := NewComPort - 1;
   Regs.ah := $04; (* Init Fossil *)
   Regs.dx := ComPort;
   Regs.bx := $00;
   Intr($14,Regs);
   FossilActive := (Regs.AX = $1954);
   if FossilActive and (not ((Regs.bh >= 5) and (Regs.bl >= $1b))) then begin
      FossilActive := False;
      Regs.dx := ComPort; (* DeInit Fossil *)
      Regs.ah := $05;
      intr($14,Regs);
   end;
   if FossilActive then begin
      new(FossilInfo);
      GetFossilInfo;
   end;
end;

destructor TFossil.Done;  (* kill fossil *)

var
   Regs : Registers;

begin
   if FossilActive then begin
      dispose(FossilInfo);
      Regs.AH := $05;
      Regs.DX := ComPort;
      Intr($14,Regs);
      FossilActive := False;
   end;
   inherited Done;
end;

function TFossil.CarrierDetected : boolean;

var
   Regs : Registers;

begin
   Regs.ah := $03;
   Regs.dx := ComPort;
   intr($14,Regs);
   CarrierDetected := ((Regs.al and CarrierValue) > 0)
end;

function TFossil.CharAvailInOutputBuffer : boolean;

var
   Regs : Registers;

begin
   Regs.ah := $03;
   Regs.dx := ComPort;
   intr($14,Regs);
   CharAvailInOutputBuffer := ((Regs.ah and $20) > 0); (* room in output buffer? *)
end;

procedure TFossil.ClearInputBuffer;

var
   Regs : Registers;

begin
   Regs.ah := $0A;
   Regs.dx := ComPort;
   intr($14,Regs);
end;

procedure TFossil.ClearOutputBuffer;

var
   Regs : Registers;

begin
   Regs.ah := $09;
   Regs.dx := ComPort;
   intr($14,Regs);
end;

procedure TFossil.FlushOutputBuffer;

begin
   SetTimerSecs(TimeOut);
   while (not OutputBufferEmpty) and CarrierDetected and (not TimerExpired) do
      GiveTimeSlice;
end;

function  TFossil.FreeSpaceInInputBuffer  : word;

begin
   GetFossilInfo;
   FreeSpaceInInputBuffer := FossilInfo^.IBFree;
end;

function  TFossil.FreeSpaceInOutputBuffer : word;

begin
   GetFossilInfo;
   FreeSpaceInOutputBuffer := FossilInfo^.OBFree;
end;

function TFossil.GetBaudRate : Longint;

begin
   GetBaudRate     := BaudRate;
end;

function TFossil.GetComPort : word;

begin
   GetComPort      := ComPort + 1;
end;

procedure TFossil.GetBlock(var DataBlock;MaxBlockLen : word;var BlockLenRead : word);

var
   Regs : Registers;

begin
   Regs.ah := $18;
   Regs.cx := MaxBlockLen;
   Regs.es := seg(DataBlock);
   Regs.di := ofs(DataBlock);
   Regs.dx := ComPort;
   intr($14,Regs);
   BlockLenRead := Regs.ax;
end;

function  TFossil.GetChar : char;

var
   Regs : Registers;

begin
   Regs.ah := $02;
   Regs.dx := ComPort;
   intr($14,Regs);
   GetChar := chr(Regs.al);
end;

function TFossil.GetFossilActive : boolean;

begin
   GetFossilActive := FossilActive;
end;

procedure TFossil.GetFossilInfo;

var
   Regs : Registers;

begin
   Regs.ah := $1b;
   Regs.cx := sizeof(FossilInfoType);
   Regs.dx := ComPort;
   Regs.es := seg(FossilInfo^);
   Regs.di := ofs(FossilInfo^);
   intr($14,Regs);
   (*

   if (FossilInfo^.InfoSize <> sizeof(FossilInfoType)) then begin
      ???? What to do ????
   end;

   *)
end;

function TFossil.GetFossilIDString : String;

begin
   GetFossilIDString := strpas(FossilInfo^.IDString);
end;

function TFossil.GetFossilRevision : byte;

begin
   GetFossilRevision := FossilInfo^.CurrRev;
end;

function TFossil.GetFossilVersion  : byte;

begin
   GetFossilVersion  := FossilInfo^.CurrFossil;
end;

procedure TFossil.GiveTimeSlice;

begin
   (* Override this function to give away time slices *)
end;

procedure TFossil.HardwareFlowControl;

var
   Regs : Registers;

begin
   Regs.ah := $0F;
   Regs.dx := ComPort;
   Regs.al := $02;
   intr($14,Regs);
end;

function TFossil.InputBufferEmpty : boolean;

var
   Regs : Registers;

begin
   Regs.ah := $03;
   Regs.dx := ComPort;
   intr($14,Regs);
   InputBufferEmpty := ((Regs.ah and $01) = 0);
end;

procedure TFossil.LowerDTR;

var
   Regs : Registers;

begin
   Regs.ah := $06;
   Regs.dx := ComPort;
   Regs.al := $00;
   intr($14,Regs);
end;

function TFossil.OutputBufferEmpty : boolean;

var
   Regs : Registers;

begin
   Regs.ah := $03;
   Regs.dx := ComPort;
   intr($14,Regs);
   OutputBufferEmpty := ((Regs.ah and $40) > 0);
end;

procedure TFossil.PutBlock(var DataBlock;BlockLen : word;var BlockLenWritten : word);

var
   Regs : Registers;

begin
   Regs.ah := $19;
   Regs.cx := BlockLen;
   Regs.es := seg(DataBlock);
   Regs.di := ofs(DataBlock);
   Regs.dx := ComPort;
   intr($14,Regs);
   BlockLenWritten := Regs.ax;
end;

procedure TFossil.PutBlockCD(var DataBlock;BlockLen : word);

type
   DataBlockType = array[0..65530] of char;

var
   BytesWritten      : Word;
   TotalBytesWritten : Word;
   DataBlockBytes    : DataBlockType absolute DataBlock;

begin
   SetTimerSecs(TimeOut);
   TotalBytesWritten := 0;
   while (TotalBytesWritten < BlockLen) and CarrierDetected and (not TimerExpired) do begin
      PutBlock(DataBlockBytes[TotalBytesWritten],BlockLen-TotalBytesWritten,BytesWritten);
      inc(TotalBytesWritten,BytesWritten);
      if (TotalBytesWritten < BlockLen) then
         GiveTimeSlice;
   end;
end;

procedure TFossil.PutChar(X : char);

var
   Regs : Registers;

begin
   Regs.ah := $01;
   Regs.dx := ComPort;
   Regs.al := ord(X);
   intr($14,Regs);
end;

procedure TFossil.PutCharCD(X : char);

label
   WaitChar;

var
   Regs : Registers;

begin
   SetTimerSecs(TimeOut);
   WaitChar :
      Regs.ah := $03;
      Regs.dx := ComPort;
      intr($14,Regs);
      if ((Regs.al and CarrierValue) > 0) then begin (* Carrier Detected *)
         if ((Regs.ah and $20) > 0) then begin (* Space in Output Buffer *)
            Regs.ah := $01;
            Regs.dx := ComPort;
            Regs.al := ord(X);
            intr($14,Regs);
         end else begin      (* No Space in Output Buffer *)
            if Not TimerExpired then begin
               GiveTimeSlice;
               goto WaitChar;
            end;
         end;
      end else
         ClearOutputBuffer; (* No Carrier *)
end;

procedure  TFossil.PutStringCD(S : String);

begin
   PutBlockCD(S[1],length(S));
end;

procedure  TFossil.RaiseDtr;

var
   Regs : Registers;

begin
   Regs.ah := $06;
   Regs.dx := ComPort;
   Regs.al := $01;
   intr($14,Regs);
end;

procedure  TFossil.SetBaud(NewBaudRate : longint);

var
   Regs : Registers;

begin
   if FossilActive then begin
      BaudRate := NewBaudRate;
      if ExtendedFossil then begin
         Regs.AH := $1E;
         Regs.AL := $00;
         Regs.BH := $00;
         Regs.BL := $00;
         Regs.CH := $00;
         if (BaudRate = 300) then
            Regs.CL := $03
         else if (BaudRate = 1200) then
            Regs.CL := $04
         else if (BaudRate = 2400) then
            Regs.CL := $05
         else if (BaudRate = 4800) then
            Regs.CL := $06
         else if (BaudRate = 9600) then
            Regs.CL := $07
         else if (BaudRate = 19200) then
            Regs.CL := $08
         else if (BaudRate = 28800) then
            Regs.CL := $80
         else if (BaudRate = 38400) then
            Regs.CL := $81
         else if (BaudRate = 57600) then
            Regs.CL := $82
         else if (BaudRate = 76800) then
            Regs.CL := $83
         else if (BaudRate = 115200) then
            Regs.CL := $84;
         Regs.DX := ComPort;
         Intr($14,Regs);
      end else begin
         Regs.AH := $00;
         Regs.DX := ComPort;
         regs.AL := $00;
         if (BaudRate = 9600) then
            Regs.Al := Regs.Al or $E0
         else if (BaudRate = 300) then
            Regs.Al := Regs.Al or $40
         else if (BaudRate = 1200) then
            Regs.Al := Regs.Al or $80
         else if (BaudRate = 4800) then
            Regs.Al := Regs.Al or $C0
         else if (BaudRate = 38400) then
            Regs.Al := Regs.Al or $20
         else if (BaudRate = 2400) then
            Regs.Al := Regs.Al or $A0;
         Regs.Al := Regs.Al or $3;        (* Set N,8,1 *)
         Intr($14,Regs);
      end;
   end;
end;

procedure TFossil.SetTimerSecs(Secs : Longint);

begin
   (* Override this procedure with your own timer functions *)
end;

function  TFossil.SizeOfInputBuffer : word;

begin
   SizeOfInputBuffer := FossilInfo^.IBSize;
end;

function  TFossil.SizeOfOutputBuffer : word;

begin
   SizeOfOutputBuffer := FossilInfo^.OBSize;
end;

function  TFossil.TimerExpired : boolean;

begin
   (* Override this function with your own timer functions *)
   TimerExpired := False;
end;

end.
