/* This is a program made for BBS sysops, which will read a door
 * application's node configuration file and copy it to configuration
 * files for other nodes, changing the path to the drop file in the
 * configuration files for each node.
 *
 * I wrote this application because many BBS doors require a
 * separate configuration file for each node, and often, the
 * only thing that changes in the configuration file is the
 * path to the BBS node/drop file.
 *
 * Program usage:
 *  NodeConfigCopier <configuration filename> <node path template> <total # of nodes>
 *  The first parameter specifies the configuration file to copy.  The node path
 *  template specifies the part of the node path that is the same for all nodes
 *  (i.e., D:\SBBS\NODE).  The third parameter specifies the total number of nodes
 *  that the BBS has.
 *
 * Date       Author            Description
 * 2010-01-31 Eric Oulashin     Started
 * 2011-02-01 Eric Oulashin     Added try/catch blocks in certain places
 * 2011-04-25 Eric Oulashin     Updated to output an error if the node path template
 *                              was not found in the source configuration file.
 */

#include <iostream>
#include <string>
#include <sstream>
#include <fstream>
#include <list>
#include <cctype> // For isdigit()
using std::cout;
using std::cerr;
using std::endl;
using std::string;
using std::istringstream;
using std::ostringstream;
using std::ifstream;
using std::ofstream;
using std::list;

//////////////////////////////////////////////////////////////////////////////////////
// Constants/defines/etc.

#define VERSION "1.01"
#define VERSION_DATE "2011-04-25"

//////////////////////////////////////////////////////////////////////////////////////
// Function declarations

// Converts a C-style string to an integer.
//
// Parameters:
//  pStr: The string to convert
//
// Return value: The string converted to an integer
int strToInt(const char* pStr);

// Replaces the node number in a drop file path.
//
// Parameters:
//  pPath: The path in which to replace the node number
//  pPathTemplate: The path template string (i.e., "D:\BBS\NODE")
//  pNodeNumStr: The new node number, as a string
//
// Return value: The path string with the node number replaced
string replaceNodeNumInPath(const string& pPath, const string& pPathTemplate, const string& pNodeNumStr);

// Replaces a number in a string with a new number.  It is assumed that
// the string will only have one instance of a number in it.
//
// Parameters:
//  pStr: The string in which to replace the number
//  pNewNum: The new number to put in its place
//
// Return value: The string with the number replaced
string replaceNumInStr(const string& pStr, int pNewNum);

// Returns the index of the first digit in a string, or string::npos if
// none is found.
//
// Parameters:
//  pStr: The string to search
//
// Return value: The index of the first digit in the string, or
//               string::npos if none is found.
size_t findFirstDigitIndex(const string& pStr);

// Returns the index of the last digit in a string, or string::npos if
// none is found.
//
// Parameters:
//  pStr: The string to search
//
// Return value: The index of the last digit in the string, or
//               string::npos if none is found.
size_t findLastDigitIndex(const string& pStr);

// Splits a configuration filename into its parts.
//
// Parameters:
//  pConfigFilename: The name of a configuration file (i.e., "NODE1.CFG")
//  pPart1: Will hold the first part of the filename.
//  pNodeNum: Will hold the node number from the filename.
//  pPart3: Will hold the 3rd part of the filename
void splitConfigFilename(const string& pConfigFilename, string& pPart1, int& pNodeNum, string& pPart3);

//////////////////////////////////////////////////////////////////////////////////////
// Main function definition
int main(int argc, char* argv[])
{
	cout << "Eric Oulashin's BBS door config copier version " << VERSION
		 << " (" << VERSION_DATE << ")" << endl
		 << endl;

	// Command line parameters:
	// 1: Name of the config file to copy
	// 2: Path template (i.e., "D:\BBS\NODE")
	// 3: Number of nodes
	if (argc != 4)
	{
		cout << "Usage:" << endl
			 << "dncc <config filename> <node path template> <total number of nodes>" << endl
			 << endl
			 << "The config filename is assumed to have a number in it (i.e.," << endl
			 << "NODE1.CFG), and the node path template is assumed to have" << endl
			 << "the node number at the end (i.e., \"D:\\BBS\\NODE1\")." << endl;
		return 1;
	}

	// Convert the # of nodes parameter into an integer
	const int numNodes = strToInt(argv[3]);
	// If the # of nodes is below 2, then output an error and exit.
	if (numNodes < 2)
	{
		cerr << "ERROR: The total number of nodes must be 2 or more." << endl
			 << "Specified: " << numNodes << endl;
		return 2;
	}

	cout << "Configuration file to read: " << argv[1] << endl
		 << "        Node path template: " << argv[2] << endl
		 << "     Total number of nodes: " << numNodes << endl
		 << endl;

	// Open the config file and read its contents into a collection of strings.
	list<string> cfgFileLines;     // To store the contents of the config file
	int cfgNodePathLineIndex = -1; // The config file line # (0-based) that has the node # path
	try
	{
		ifstream srcCfgFile;
		srcCfgFile.open(argv[1], std::ios::in);
		if (srcCfgFile.is_open())
		{
			string fileLine; // A line read from the file
			int lineNum = 0; // File line number
			while (!srcCfgFile.eof() && (srcCfgFile.peek() != EOF))
			{
				std::getline(srcCfgFile, fileLine);
				cfgFileLines.push_back(fileLine);
				// Check to see if this line contains the node path
				if (fileLine.find(argv[2]) != string::npos)
					cfgNodePathLineIndex = lineNum;
				++lineNum;
			}
			srcCfgFile.close();
		}
		// If unable to read the source configuration file, then output an
		// error and exit.
		else
		{
			cerr << "Unable to open " << argv[1] << " for reading - Cannot continue." << endl;
			return 3;
		}
	}
	catch (const std::exception& pExc)
	{
		cerr << "Error reading " << argv[1] << ": " << pExc.what() << endl;
		return 4;
	}
	cout << " Read " << argv[1] << endl << "-----" << endl;

	// If the node path template was not found in the configuration file, then
	// output an error and exit.
	if (cfgNodePathLineIndex == -1)
	{
		cerr << "Error: The node path template was not found in the file!" << endl;
		return 5;
	}

	int returnValue = 0; // Value to return when the program exits

	// Split the specified configuration filename up into its 3 parts
	string cfgFilenamePart1, cfgFilenamePart3;
	int specifiedConfigFileNode = 0;
	splitConfigFilename(argv[1], cfgFilenamePart1, specifiedConfigFileNode, cfgFilenamePart3);

	// Write the other config files.
	string configFileName;
	ostringstream oss; // For converting a number to a string
	ofstream outFile;
	list<string>::iterator iter; // For iterating through cfgFileLines
	int lineNum = 0;             // File line number
	for (int nodeNum = 1; nodeNum <= numNodes; ++nodeNum)
	{
		// Skip the node # of the specified filename
		if (nodeNum == specifiedConfigFileNode)
			continue;

		try
		{
			// Construct the configuration file name for this node
			oss.str("");
			oss << nodeNum;
			configFileName = cfgFilenamePart1 + oss.str() + cfgFilenamePart3;
			// Open & write to the output configuration file
			outFile.open(configFileName.c_str(), std::ios::out);
			if (outFile.is_open())
			{
				lineNum = 0;
				for (iter = cfgFileLines.begin(); iter != cfgFileLines.end(); ++iter)
				{
					if (lineNum == cfgNodePathLineIndex)
						outFile << replaceNodeNumInPath(*iter, argv[2], oss.str());
					else
						outFile << *iter;
					// If this is not the last line, then also output a newline to
					// the config file.
					if (lineNum < (int)cfgFileLines.size()-1)
						outFile << endl;

					++lineNum;
				}
				outFile.close();
				cout << "Wrote " << configFileName << endl;
			}
			else
			{
				cerr << "Warning: Unable to open " << configFileName << " for writing!" << endl;
				returnValue = 6;
			}
		}
		catch (const std::exception& pExc)
		{
			cerr << "Error when writing node " << nodeNum << " configuration file: " << pExc.what() << endl;
		}
		// In case an exception was thrown before the file was closed:
		if (outFile.is_open())
			outFile.close();
	}

	cout << endl << "Done." << endl;

	return returnValue;
}

//////////////////////////////////////////////////////////////////////////////////////
// Helper function definitions

int strToInt(const char* pStr)
{
	istringstream iss(pStr);
	int strInt = 0;
	iss >> strInt;
	return strInt;
}

string replaceNodeNumInPath(const string& pPath, const string& pPathTemplate, const string& pNodeNumStr)
{
	// If pPath is blank, then just return it.
	if (pPath == "")
		return pPath;

	// Search for the last digit in pPath
	size_t lastDigitIndex = findLastDigitIndex(pPath);

	// If lastDigitIndex is npos, then just return pPath.
	if (lastDigitIndex == string::npos)
		return pPath;

	// Start constructing the new path
	string newPathStr = pPathTemplate + pNodeNumStr;
	// If there is anything after the last digit in pPath, then
	// copy that part to the new path string.
	if (lastDigitIndex < pPath.length()-1)
		newPathStr += pPath.substr(lastDigitIndex+1);

	return newPathStr;
}

string replaceNumInStr(const string& pStr, int pNewNum)
{
	// Find the indexes of the first & last digits in pStr
	size_t firstDigitIndex = findFirstDigitIndex(pStr);
	size_t lastDigitIndex = string::npos;
	if (firstDigitIndex != string::npos)
		lastDigitIndex = findLastDigitIndex(pStr);

	// If both index variables are string::npos, then just return pStr.
	if ((firstDigitIndex == string::npos) && (lastDigitIndex == string::npos))
		return pStr;

	// Convert pNewNum to a string
	ostringstream oss;
	oss << pNewNum;

	// Return the new string
	return pStr.substr(0, firstDigitIndex) + oss.str() + pStr.substr(lastDigitIndex+1);
}

size_t findFirstDigitIndex(const string& pStr)
{
	size_t digitIndex = string::npos;
	size_t strLen = pStr.length();
	for (size_t i = 0; i < strLen; ++i)
	{
		if (isdigit(pStr[i]))
		{
			digitIndex = i;
			break;
		}
	}
	return digitIndex;
}

size_t findLastDigitIndex(const string& pStr)
{
	size_t digitIndex = string::npos;
	for (int i = (int)pStr.length()-1; i >= 0; --i)
	{
		if (isdigit(pStr[i]))
		{
			digitIndex = i;
			break;
		}
	}
	return digitIndex;
}

void splitConfigFilename(const string& pConfigFilename, string& pPart1, int& pNodeNum, string& pPart3)
{
	// Default the OUT parameters.
	pPart1 = "";
	pNodeNum = 0;
	pPart3 = "";


	// If pConfigFilename is blank, then just return.
	if (pConfigFilename == "")
		return;

	// Find the indexes of the first & last digits in pConfigFilename
	size_t firstDigitIndex = findFirstDigitIndex(pConfigFilename);
	size_t lastDigitIndex = string::npos;
	if (firstDigitIndex != string::npos)
		lastDigitIndex = findLastDigitIndex(pConfigFilename);

	// If both index variables are string::npos, then return now.
	if ((firstDigitIndex == string::npos) && (lastDigitIndex == string::npos))
		return;

	// Convert the number found into an integer and set pNodeNum.
	pNodeNum = strToInt(pConfigFilename.substr(firstDigitIndex, lastDigitIndex-firstDigitIndex+1).c_str());

	// Set pPart1 and pPart3
	pPart1 = pConfigFilename.substr(0, firstDigitIndex);
	pPart3 = pConfigFilename.substr(lastDigitIndex+1);
}