/* String Editor 
 * For Synchronet BBS 3.15a or higher.
 * 
 * Version: 0.9b
 * Created: 2008-05-26
 * Author: Mark Zolton (blakdeth@masqueradebbs.com)
 *
 * Copyright 2008 Mark C. Zolton
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/* Includes */
load("sbbsdefs.js");

/* Constants */
var VERSION = "0.9b";
var IN_FILE = "..\\ctrl\\text.dat";
var OUT_FILE = IN_FILE;
var MAX_STRINGS = 5;
var MAX_INPUT = 10;

/* Globals */
var stringIndex = 0;
var stringData = new Array();
var clipboard = new Array();
var prevSearch = null;
var quit = false;

/** Text String Object */
function TextString(id, name, string, more) 
{
   this.id = id;
   this.name = name;
   this.string = new Array();
   this.old = new Array();
   this.more = more;
}

function filterString(string)
{
    // Clear screen.
    var clearScreen = /\001L/gi;   
    string = string.replace(clearScreen, "\\1L");
    
    return string;
}

function parseString(str)
{
    // Get the components of a text string.
	// var reg = new RegExp(/"(.*)"(\\*).*([0-9]{3}) (\w+)/g);
	var reg = new RegExp(/"(.*)"(\\*)(.*([0-9]{3})\s(.*))?/g);
    var found = reg.exec(str);
	
	var textString = new TextString();
	
	if(found != null)
	{
	    // The ID and Name may not appear.
	    if(found[4] != undefined) { textString.id = found[4]; }
	    if(found[5] != undefined) { textString.name = found[5]; }
	    
	    // String should always be present.
	    textString.string.push(filterString(found[1]));
	    
	    // Does the string continue to the next line?
	    if(found[2] == "\\")
	    {
	        textString.more = true;
	    }
	    else
	    {
	        textString.more = false;
	    }
	}
	
	return textString;
}

function loadStrings(filename)
{
    var append = false;
    var strings = new Array();
    var f = new File(filename);
    
	if(!f.open("r")) 
	{ 
	    alert("Error opening " + filename); 
	    return;
	}

    var text = f.readAll();
     
	for (var i = 0; i < text.length; i++) 
	{
	    var string = parseString(text[i]);
	
		if(string.string[0] != undefined)
		{	
    		if(append)
    		{
    		    var index = strings.length - 1;
    		
    		    // Append this string to the last.  
    		    // strings[index].string += "###" + string.string;
    		    strings[index].string.push(string.string[0])
    		}
    		else
    		{
    		    // Add a new string.
    		    strings.push(string);
    		}
    		
    		// Should we append the next string?
    		append = string.more;
    	}
    	else
    	{
    	    // Reset the append bit.
    	    append = false;
    	}
	}
	
    f.close();
    return strings;
}

function saveStrings(filename)
{
    var f = new File(filename);
    
	if(!f.open("w")) 
	{ 
	    alert("Error opening " + filename); 
	    return;
	} 
	
	for(var i = 0; i < stringData.length; i++)
	{
	    var current = stringData[i];

	    var line = "\"" + current.string[0] + "\"";
	    if(current.string.length > 1) { line += "\\"; }
	    // line += "[20C" + current.id + " " + current.name + "\n";
	    line += " " + current.id + " " + current.name + "\n";
	    
	    // Write additional lines.
	    if(current.string.length > 1)
	    {
	        for(var j = 1; j < current.string.length; j++)
	        {
	            line += "\"" + current.string[j] + "\"";
	            
	            // Is there another line after this one?
	            if(j != current.string.length - 1) { line += "\\" }
	            
	            line += "\n";
	        }
	    }
	   
	    f.write(line);
	}
	
	f.close();
}

function displayMenu()
{
    console.print("\r\n\1n\1h\1k::: [\1w[\1k]\1nPrevious \1h\1k[\1w]\1k]\1nNext \1h\1k[\1w1-5\1k]\1nEdit \1h\1k[\1w=\1k]\1nJump \1h\1k[\1w/\1k]\1nSearch");
    console.print("\r\n\1n\1h\1k::: [\1wR\1k]\1nReset \1h\1k[\1wC\1k]\1nCopy \1h\1k[\1wP\1k]\1nPaste \1h\1k[\1wQ\1k]\1nQuit \1h\1k% \1nCommand\1h\1w: ");
}

function displayStrings()
{
    // Header
    // console.print("\1n\1h\1k::: \1wSynchronet String Editor v" + VERSION + " \1nby \1hBlakdeth\1n");
    // console.crlf();

    for(var i = 0; i < MAX_STRINGS; i++)
    {
        var string = stringData[stringIndex + i];
        var num = i + 1;
    
        console.print("\r\n\1n\1h" + num + "\1h\1k: \1n" + string.id + "\1h\1k, \1w" + string.name);
        // console.print("\r\n\1n" + string.string);
        
        for(var j = 0; j < string.string.length; j++)
        {
            console.print("\r\n\1n" + string.string[j]);
        }
        
        console.crlf();
    }
}

function setIndex(value)
{
    // Make sure the value is some kind of number.
    if(!isNaN(value))
    {
        stringIndex = value;

        if(stringIndex < 0) 
        { 
            stringIndex = 0; 
        }   
        else if(stringIndex + MAX_STRINGS > stringData.length) 
        { 
            stringIndex = stringData.length - MAX_STRINGS;
        }
    }
}

function searchFor(searchString)
{
    var index = -1;

    if((searchString == null) && (prevSearch == null))
    {
        return index;
    }
    
    if(searchString == null && prevSearch != null)
    {
        searchString = prevSearch;
    }
      
    if(searchString != null) 
    {
         // Store the query for later use.
        prevSearch = searchString;
    
        var reg = new RegExp(searchString, "i");
    
        // Start searching with the next string.
        search:
        for(var i = stringIndex + 1; i < stringData.length; i++)
        {
            var current = stringData[i];
         
            for(var j = 0; j < current.string.length; j++)
            {
                if(reg.test(current.string[j]))
                {   
                    index = i;
                    break search;
                }
            
            }
        }
    }
    
    return index;
}

function editString(index)
{
    index = stringIndex + (index - 1);
    
    console.clear();
    console.print("\r\n\1n\1h" + stringData[index].id + "\1k, \1w" + stringData[index].name);
    console.print("\1n\1h\1k (\1n" + MAX_INPUT + " lines max\1h\1k). \1nBlank line saves.");
    console.print("\r\n\1n\1h\1k------------------------------------------------------------------------------- "); 
    
    // Backup the old string.
    stringData[index].old = stringData[index].string;
    
    // Destroy the existing string.
    stringData[index].string = new Array();
    
    for(var i = 0; i < MAX_INPUT; i++)
    {
        console.print("\1n\1h");
        var newString = console.getstr(stringData[index].old[i], 79, K_EDIT|K_MSG);
       
        // TODO: Add ability to capture commands like save, abort, and go back a line.
       
        // If the user entered a blank line, just quit, otherwise, save the line.
        if(newString == "") { break; }
        else { stringData[index].string[i] = filterString(newString); }
    }
}

function resetString(index)
{
    index = stringIndex + (index - 1);
    
    // Make sure there is old data to reset. If not, do nothing.
    if(stringData[index].old.length > 0)
    {
        // Copy the old data over the current data.
        stringData[index].string = stringData[index].old;
    }
}

function copyString(index)
{
    index = stringIndex + (index - 1);
    
    // Copy specified string data to the clipboard.
    clipboard = stringData[index].string;
}

function pasteString(index)
{
    index = stringIndex + (index - 1);
    
    // Backup existing string data.
    stringData[index].old = stringData[index].string;
    
    // Paste from clipboard only if there is data present.
    if(clipboard.length > 0)
    {
        stringData[index].string = clipboard;
    }
}

function getInput()
{
    // Get input.
    var input = console.getkeys("[]=/Q12345RCP");
        
    switch (input) 
    {
        // Previous
        case "[":
            setIndex(stringIndex - MAX_STRINGS);
            break;
        
        // Next
        case "]":
            setIndex(stringIndex + MAX_STRINGS);
            break;
            
        // Jump
        case "=":
            var input = prompt("\r\n\1n\1h\1k--- \1nEnter the string number to jump to \1h\1k[\1w1-" + stringData.length + "\1k]\1w");
            setIndex(parseInt(input) - 1);
            break;
            
        // Search 
        case "/":
            var input = prompt("\r\n\1n\1h\1k--- \1nEnter text to search for \1h\1k[\1nenter\1h\1k/\1wnext\1k]\1w");
            var index = searchFor(input);
            if(index != -1) { setIndex(index); }
            break;
            
        // Edit
        case "1":
        case "2":
        case "3":
        case "4":
        case "5":
            editString(parseInt(input));
            break;
            
        // Reset
        case "R":
            var input = prompt("\r\n\1n\1h\1k--- \1nEnter the string number to reset \1h\1k[\1w1-5\1k]\1w");
            resetString(parseInt(input));
            break;

        // Copy
        case "C":
            var input = prompt("\r\n\1n\1h\1k--- \1nEnter the string number to copy \1h\1k[\1w1-5\1k]\1w");
            copyString(parseInt(input));
            break;
  
        // Paste
        case "P":
            var input = prompt("\r\n\1n\1h\1k--- \1nEnter the string number to paste to \1h\1k[\1w1-5\1k]\1w");
            pasteString(parseInt(input));
            break;
  
        // Quit
        case "Q":
            quit = true;
            break;
            
        default:
            // Do nothing.
    }
}

function displayRefresh()
{
    console.clear();
    displayStrings();
    displayMenu();
}

/* Main */
function main()
{
    // Load strings from the data file.
    stringData = loadStrings(IN_FILE);
    
    while((bbs.online) && !quit)
    {
        displayRefresh();
        getInput();
    }
    
    console.crlf();
    
    //  Should we save everything?    
    if(console.yesno("Save changes"))
    {
       saveStrings(OUT_FILE);
    }
}

main();