use std::ffi::CString;
use std::fs;
use std::path::{Path, PathBuf};

fn fnmatch_simple(pattern: &str, string: &str) -> bool {
    let pattern = CString::new(pattern).unwrap();
    let string = CString::new(string).unwrap();

    matches!(
        unsafe { libc::fnmatch(pattern.as_ptr(), string.as_ptr(), 0) },
        0
    )
}

fn get_ignores(path: &Path) -> Vec<String> {
    let content = fs::read_to_string(path.join(".FILES_IGNORED"));
    if content.is_err() {
        return Vec::new();
    }

    content
        .unwrap()
        .split(':')
        .map(|i| i.trim().to_owned())
        .collect()
}

fn is_ignored(path: &Path, ignores: &[String]) -> bool {
    let filename = path.file_name().unwrap().to_string_lossy().into_owned();

    filename.starts_with('.')
        || ignores
            .iter()
            .any(|pattern| fnmatch_simple(pattern, &filename))
}

pub fn read_dir(path: &Path) -> Vec<PathBuf> {
    let entries_iter = fs::read_dir(path).expect("Could not read_dir");
    let mut paths: Vec<_> = entries_iter.filter_map(|r| r.ok()).collect();
    paths.sort_by_key(|entry| entry.path());

    if path.join(".DIR_REVERSED").is_file() {
        paths.reverse();
    }

    let ignores = get_ignores(path);

    let mut result = Vec::new();

    for p in paths {
        let p = p.path();
        if !p.is_dir() && !p.is_file() {
            continue;
        }

        if is_ignored(&p, &ignores) {
            continue;
        }

        result.push(p);
    }

    result
}
