use std::collections::VecDeque;
use std::fs;
use std::path::Path;

use crate::index_parser::parse_index;
use crate::menu_types::determine_menu_type;
use crate::path::normalize_if_allowed;
use crate::read_dir::read_dir;
use crate::settings::Settings;

fn sanitize(s: &str) -> String {
    s.trim().replace("\t", "    ")
}

fn search_through_file(path: &Path, base_path: &Path, search: &str, output: &mut String) -> bool {
    const NUM_CONTEXT: usize = 2;

    let search_lower = search.to_lowercase();

    let content = fs::read_to_string(path).unwrap_or(String::from(""));
    let mut had_match = false;

    let mut previous_lines: VecDeque<String> = VecDeque::new();
    let mut remaining: usize = 0;

    for line in content.lines() {
        let line_lower = line.to_lowercase();

        if line_lower.contains(&search_lower) {
            if !had_match {
                output.push_str(&format!(
                    "[{}|{}|{}|server|port]\n",
                    determine_menu_type(path),
                    path.file_name().unwrap().to_string_lossy(),
                    path.strip_prefix(base_path).unwrap().to_string_lossy(),
                ));
            }

            // Previous context lines.
            if remaining == 0 {
                for prev in &previous_lines {
                    output.push_str(&format!(" :  {}\n", sanitize(prev)));
                }

                remaining = NUM_CONTEXT;
            }

            // Location of match.
            output.push_str(&format!("->  {}\n", sanitize(line)));

            had_match = true;
        } else if remaining > 0 {
            // Trailing context lines.
            output.push_str(&format!(" :  {}\n", sanitize(line)));
            remaining -= 1;

            if remaining == 0 {
                output.push_str("----\n");
            }
        }

        previous_lines.push_back(line.to_owned());
        if previous_lines.len() > NUM_CONTEXT {
            previous_lines.pop_front();
        }
    }

    had_match
}

fn print_search_recursive(
    settings: &Settings,
    path: &Path,
    base_path: &Path,
    search: &str,
    output: &mut String,
) -> bool {
    #[cfg(debug_assertions)]
    eprintln!("search looking at path [{}]", path.to_string_lossy());

    let mut had_matches = false;

    for entry in read_dir(path) {
        let entry = path.join(entry);

        if let Ok(norm_path) = normalize_if_allowed(settings, &entry) {
            if norm_path.is_file() {
                had_matches |= search_through_file(&norm_path, base_path, search, output);
            } else if norm_path.is_dir() {
                had_matches |=
                    print_search_recursive(settings, &norm_path, base_path, search, output);
            }
        }
    }

    had_matches
}

fn print_search_make_index(settings: &Settings, path: &Path, search: &str) -> String {
    let mut output = String::new();

    if search.is_empty() {
        output.push_str("Please provide a search term.\n");
    } else {
        output.push_str(&format!("Search results for \"{search}\":\n"));
        output.push('\n');

        if !print_search_recursive(settings, path, path, search, &mut output) {
            output.push_str("No hits.\n");
        }
    }

    #[cfg(debug_assertions)]
    eprintln!("Final generated index:\n\n{output}");

    output
}

pub fn print_search(settings: &Settings, path: &Path, search: &str) {
    let relative_to = path.strip_prefix(&settings.docroot).unwrap();
    print!(
        "{}",
        parse_index(
            settings,
            &print_search_make_index(settings, path, search),
            relative_to
        )
    );
}
