use std::path::Path;
use std::sync::LazyLock;

struct TypeMap<'a> {
    extensions: Vec<&'a str>,
    menu_type: &'a str,
}

// This is not a "lock", it is a thread-safe (hence "locked") lazily initialized cell.
static MAP_EXTENSION_TO_MENU_TYPE: LazyLock<Vec<TypeMap>> = LazyLock::new(|| {
    vec![
        TypeMap {
            extensions: vec![
                ".asc", ".atom", ".c", ".groff", ".h", ".md", ".roff", ".sh", ".sig", ".txt",
            ],
            menu_type: "0",
        },
        TypeMap {
            extensions: vec![".gif"],
            menu_type: "g",
        },
        TypeMap {
            extensions: vec![".htm", ".html"],
            menu_type: "h",
        },
        TypeMap {
            extensions: vec![".bmp", ".ff", ".jpeg", ".jpg", ".png", ".tiff"],
            menu_type: "I",
        },
        TypeMap {
            extensions: vec![".flac", ".mp3", ".ogg", ".wav"],
            menu_type: "s",
        },
    ]
});

pub fn determine_menu_type(path: &Path) -> String {
    if path.is_dir() {
        return String::from("1");
    }

    let lossy_name = path.file_name().unwrap().to_string_lossy();

    for map in &*MAP_EXTENSION_TO_MENU_TYPE {
        for ext in &map.extensions {
            if lossy_name.ends_with(ext) {
                return map.menu_type.to_string();
            }
        }
    }

    String::from("9")
}
