use std::ffi::OsString;
use std::fs;
use std::path::Path;

use crate::index_parser::parse_index;
use crate::menu_types::determine_menu_type;
use crate::read_dir::read_dir;
use crate::settings::Settings;

fn maybe_add_annotation(index: &mut String, path: &Path) {
    let mut filename_ann = OsString::from(".");
    filename_ann.push(path.file_name().unwrap());
    filename_ann.push(OsString::from(".ANN"));

    let mut path_ann = path.to_owned();
    path_ann.set_file_name(filename_ann);

    maybe_add_file_content(index, &path_ann, false);
}

fn maybe_add_file_content(index: &mut String, filename: &Path, extra_empty_line: bool) {
    if let Ok(content) = fs::read_to_string(filename) {
        index.push_str(&content);
    }
    if extra_empty_line {
        index.push('\n');
    }
}

pub fn print_autoindex(settings: &Settings, path: &Path) {
    let mut index = String::new();

    maybe_add_file_content(&mut index, &path.join(".HEADER"), true);

    if path.join(".SEARCH").is_file() {
        // parse_index() will make this relative to the current directory.
        index.push_str("[7|Search these files|.SEARCH|server|port]\n");
        index.push('\n');
    }

    for entry in read_dir(path) {
        maybe_add_annotation(&mut index, &entry);

        let filename = entry.file_name().unwrap().to_string_lossy();
        index.push_str(&format!(
            "[{}|{}|{}|server|port]\n",
            determine_menu_type(&entry),
            filename,
            filename,
        ));
    }

    let relative_to = path.strip_prefix(&settings.docroot).unwrap();
    print!("{}", parse_index(settings, &index, relative_to));
}
