use std::env::{args, set_current_dir};
use std::path::PathBuf;
use std::process::ExitCode;

mod index_parser;
mod input_reader;
mod logging;
mod menu_types;
mod path;
mod print_autoindex;
mod print_file;
mod print_index_file;
mod print_search;
mod read_dir;
mod settings;

#[cfg(target_os = "openbsd")]
mod openbsd;

use crate::input_reader::read_input;
use crate::logging::{find_remotehost, log_request};
use crate::path::normalize_if_allowed;
use crate::print_autoindex::print_autoindex;
use crate::print_file::print_file;
use crate::print_index_file::print_index_file;
use crate::print_search::print_search;
use crate::settings::Settings;

#[cfg(target_os = "openbsd")]
use crate::openbsd::{pledge_promises, unveil, unveil_final};

fn process_request(settings: &Settings) {
    let rtype;

    let (selector, search) = match read_input() {
        Some((in_selector, in_search)) => (in_selector, in_search),
        None => {
            rtype = "TIMEOUT";
            log_request(settings, rtype, "", "");
            return;
        }
    };

    let mut requested_path = String::new();
    requested_path.push_str(&settings.docroot);
    requested_path.push('/');
    requested_path.push_str(&selector);
    let requested_path = PathBuf::from(&requested_path);

    let mut success = true;

    if let Some(url) = selector.strip_prefix("URL:") {
        print!(
            "<!DOCTYPE html>
            <html lang=\"en\"><head>
                <title>Goodbye</title>
                <meta http-equiv=\"refresh\" content=\"2; url={url}\"/>
            </head><body>
            <p>You are now leaving Gopher space: <a href=\"{url}\">{url}</a></p>
            </body></html>",
        );
        rtype = "HTTPREDIRECT";
    } else if let Ok(norm_path) = normalize_if_allowed(settings, &requested_path) {
        if norm_path.is_file() {
            if norm_path.file_name().unwrap() == ".SEARCH" {
                print_search(settings, norm_path.parent().unwrap(), &search);
                rtype = "SEARCH";
            } else {
                print_file(&norm_path);
                rtype = "FILE";
            }
        } else if norm_path.is_dir() {
            let index_path = norm_path.as_path().join(".INDEX");
            if index_path.is_file() {
                print_index_file(settings, &index_path);
                rtype = "INDEX";
            } else {
                print_autoindex(settings, &norm_path);
                rtype = "AUTOINDEX";
            }
        } else {
            rtype = "ERROR_FILE_NOT_FOUND";
            success = false;
        }
    } else {
        rtype = "ERROR_PATH_NORM";
        success = false;
    }

    if !success {
        print!("3Invalid request.\tError\tError\t0\r\n");
    }

    log_request(settings, rtype, &selector, &search);
}

fn main() -> ExitCode {
    #[cfg(target_os = "openbsd")]
    if !pledge_promises("stdio rpath unveil inet") {
        panic!("pledge() failed");
    }

    let remotehost = find_remotehost();

    #[cfg(target_os = "openbsd")]
    if !pledge_promises("stdio rpath unveil") {
        panic!("pledge() failed");
    }

    let mut settings = Settings {
        servername: String::from("localhost"),
        serverport: 70,
        docroot: String::from("/srv/gopher"),
        logremoteip: false,
        remotehost,
    };

    let args: Vec<String> = args().collect();

    let mut i = 1;
    while i < args.len() {
        if args[i] == "-l" {
            settings.logremoteip = true;
        } else if args[i] == "-n" && i < args.len() - 1 {
            i += 1;
            settings.servername = String::from(&args[i]);
        } else if args[i] == "-p" && i < args.len() - 1 {
            i += 1;
            settings.serverport = args[i].parse::<i32>().unwrap();
        } else if args[i] == "-r" && i < args.len() - 1 {
            i += 1;
            settings.docroot = String::from(&args[i]);
        } else {
            panic!("Invalid command line argument: {}", args[i]);
        }

        i += 1;
    }

    // Only after processing the command line arguments, we can lock down the process even more
    // with unveil(), because we didn't know the docroot before. (Both unveil() and pledge()
    // *could* be done in one step at this point right here, because we still haven't processed any
    // client data. Still, let's do pledge() as early as possible.)
    #[cfg(target_os = "openbsd")]
    {
        if !unveil(&settings.docroot, "r") {
            panic!("unveil() failed");
        }
        if !unveil_final() {
            panic!("unveil() failed");
        }
    }

    #[cfg(debug_assertions)]
    eprintln!("{settings:?}");

    if set_current_dir(&settings.docroot).is_err() {
        eprintln!("Cannot chdir to docroot");
        return ExitCode::from(1);
    }

    process_request(&settings);

    ExitCode::from(0)
}
