use std::ffi::CString;
use std::mem::{MaybeUninit, size_of};
use std::ptr::null_mut;

use crate::settings::Settings;

#[cfg(target_os = "openbsd")]
use crate::openbsd::sendsyslog_simple;

pub fn find_remotehost() -> String {
    // inetd does not tell us anything about the remote host via env vars, like other super servers
    // do. Instead, we run getpeername() and getnameinfo() on file descriptor 0, which gives us all
    // the information that we need.

    let unknown = String::from("<UNKNOWN>");

    let mut ss = MaybeUninit::<libc::sockaddr_storage>::uninit();
    let ss_usize: usize = size_of::<libc::sockaddr_storage>();
    let mut ss_len: libc::socklen_t = ss_usize.try_into().unwrap();

    let bufsize = 128;
    let host = CString::new(" ".repeat(bufsize)).unwrap();

    let r;
    unsafe { r = libc::getpeername(0, ss.as_mut_ptr() as *mut libc::sockaddr, &mut ss_len) };
    if r != 0 {
        return unknown;
    }

    let host_ptr = host.into_raw();
    let r;
    unsafe {
        r = libc::getnameinfo(
            ss.as_mut_ptr() as *mut libc::sockaddr,
            ss_len,
            host_ptr,
            bufsize.try_into().unwrap(),
            null_mut(),
            0,
            libc::NI_NUMERICHOST | libc::NI_NUMERICSERV,
        )
    };
    if r != 0 {
        return unknown;
    }

    let host_back = unsafe { CString::from_raw(host_ptr) };

    host_back.to_string_lossy().into_owned()
}

pub fn log_request(settings: &Settings, rtype: &str, selector: &str, search: &str) {
    let remote_host = match settings.logremoteip {
        true => &settings.remotehost,
        false => "<ANON>",
    };

    let msg = format!("[rophy] '{remote_host}' '{rtype}' '{selector}' '{search}'");

    // On OpenBSD, we talk to syslog.
    #[cfg(target_os = "openbsd")]
    sendsyslog_simple(&msg);

    // On other systems, we expect stderr to be connected to some logging daemon.
    #[cfg(not(target_os = "openbsd"))]
    eprintln!("{msg}");
}
